// @abstract(Streams for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(January 12, 2006)
// @lastmod(January 12, 2006)
//
// Implements GZ compressed input and output streams.
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>

{$INCLUDE valkyrie.inc}
unit vgzstream;
interface
uses vstream, vutil, gzio;

// GZFile input stream class. Loades the whole file to memory.
type TGZFileInputStream = class(TInputInterfaceStream)
       // Opens stream for reading.
       constructor Open(const fname : string); reintroduce;
       // Read a string. The first byte is assumed to have the string lenght.
       function getName : string; override;
       // Closes stream.
       destructor Close;
       private
       Filename : AnsiString;
       Size     : LongInt;
     end;

// GZFile output stream class.
type TGZFileOutputStream = class(TOutputInterfaceStream)
       // Create a file named fname for writing.
       constructor Create(const fname : string);
       // Write one byte.
       procedure WriteByte(const b : byte); override;
       // Write one word.
       procedure WriteWord(const w : word); override;
       // Write one longint.
       procedure WriteLong(const l : LongInt); override;
       // Write a flagset (set of byte).
       procedure WriteFlags(const f : TFlags); override;
       // Write one char.
       procedure WriteChar(const c : char); override;
       // Write a string. The first byte will be the string lenght.
       procedure WriteString(const s : string); override;
       // Write a string. The first word will be the string lenght.
       procedure WriteAnsi(const s : Ansistring); override;
       // Write amount bytes to file from source.
       procedure Write(const amount : Longint; const source : Pointer); override;
       // Returns the name of the stream. Ex: filename.
       function getName : string; override;
       // Closes stream.
       destructor Close;
       private
       Filename : AnsiString;
       OutFile  : gzFile;
     end;

implementation

uses sysutils;

// Opens stream for reading.
constructor TGZFileInputStream.Open(const fname : string);
type TByteBlock = array[1..2048] of Byte;
var GZ     : gzFile;
    block  : TByteBlock;
    nRead  : LongInt;
    memPtr : PByte;
   function gzsize(gzf : gzFile) : LongInt;
   var bl      : TByteBlock;
       sum,red : LongInt;
   begin
     sum := 0;
     repeat
       red := gzread(gzf,@bl,2048);
       if red = -1 then CritError('Error while checking the size of '+fname+'.');
       sum := sum + red;
     until red = 0;
     gzrewind(gzf);
     gzsize := sum;
   end;

var
  OldFileMode: Byte;
begin
  Filename := fname;
  
  { Workaround for gzio (in paszlib) bug: when file is opened in 'r' mode,
    actually it may be opened in read-write if FileMode doesn't indicate
    read-only. Reason: gzOpen just calls Reset. This creates unnecessary
    problems if user doesn't have write access to the file. }
  OldFileMode := FileMode;
  FileMode := fmOpenRead;
  try
    GZ := gzio.gzOpen(fname,'r');
  finally
    FileMode := OldFileMode;
  end;

  if GZ = nil then CritError('Can''t open compressed file '+fname+'!');
  Size     := gzSize(GZ);
  GetMem(Source,Size);
  if Source = nil then CritError('Not enough memory to load '+fname+' (Needed : '+IntToStr(Size)+'b)');
  memPtr := Source;
  repeat
    nRead := gzRead(GZ,@block,Sizeof(block));
    if nRead <= 0  then Break;
    Move(block,memptr^,nRead);
    Inc(memptr,nRead);
  until (nRead <> Sizeof(block));
  if nRead = -1           then CritError('Error while reading '+fname+'.');
  if gzio.gzClose(GZ) > 0 then CritError('Can''t close (?) compressed file '+fname+'.');
  inherited Open(Source);
end;

// Returns the name of the stream. Ex: filename.
function TGZFileInputStream.GetName : string;
begin
  Exit(FileName);
end;

// Closes stream.
destructor TGZFileInputStream.Close;
begin
  FreeMem(Source,Size);
end;

constructor TGZFileOutputStream.Create(const fname : string);
begin
  Filename := fname;
  if FileExists(fname) then DeleteFile(fname);
  OutFile := gzOpen(fname,'wf-');
  if OutFile = nil then CritError('Can''t create compressed file '+fname);
end;

// Write one byte.
procedure TGZFileOutputStream.WriteByte(const b : byte);
begin
  gzWrite(OutFile,@b,1);
end;

// Write one word.
procedure TGZFileOutputStream.WriteWord(const w : word);
begin
  gzWrite(OutFile,@w,2);
end;

// Write one longint.
procedure TGZFileOutputStream.WriteLong(const l : LongInt);
begin
  gzWrite(OutFile,@l,4);
end;

// Write a flagset (set of byte).
procedure TGZFileOutputStream.WriteFlags(const f : TFlags);
begin
  Write(sizeof(f),@f);
end;

// Write one char.
procedure TGZFileOutputStream.WriteChar(const c : char);
begin
  gzWrite(OutFile,@c,1);
end;

// Write a string. The first byte will be the string lenght.
procedure TGZFileOutputStream.WriteString(const s : string);
begin
  WriteByte(length(s));
  Write(length(s),@s[1]);
end;

// Write a ansistring. The first byte will be the string lenght.
procedure TGZFileOutputStream.WriteAnsi(const s : Ansistring);
begin
  WriteWord(length(s));
  Write(length(s),@s[1]);
end;

// Write amount bytes to file from source.
procedure TGZFileOutputStream.Write(const amount : Longint; const source : Pointer);
var written : Longint;
begin
  written := gzWrite(OutFile,source,amount);
//  if written <> amount then CritError('Can''t write to compressed file '+Filename+'... disk full?');
end;

// Returns the name of the stream. Ex: filename.
function TGZFileOutputStream.GetName : string;
begin
  Exit(FileName);
end;

// Closes stream.
destructor TGZFileOutputStream.Close;
begin
  if gzClose(OutFile) > 0 then CritError('Can''t close (?) CMP File. ');
end;


end.

