//  @abstract(FreePascal Valkyrie - Valkyrie GL Font)
//  @author  (Kornel Kisielewicz)
//  
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
{$INCLUDE valkyrie.inc}
unit vglfont;
interface
uses vds, vrltools, vnode, vsdl, vsdlsurface, vsystem, SDL_image, GL, GLu;

// Abstract base class for all GL Font classes
// All new font systems should inherit this class. All methods are abstract and
// should be overloaded.
type

{ TFont }

TGLFont = class(TVObject)
       // Creates a surface with the given text - warning, uses immidate mode
       procedure RenderText(const text : ansistring; pos : TCoord2D; fg : TColor); virtual; abstract;
       // Returns the estimated text width
       function GetWidth(const text : ansistring) : Integer; virtual; abstract;
       // Returns the estimated text height
       function GetHeight(const text : ansistring) : Integer; virtual; abstract;
     end;

// Class for GL bitmap fonts
type

{ TGLBitmapFont }

TGLBitmapFont = class(TGLFont)
       // Loads the given TTF font
       constructor Create(const filename : ansistring; const metrics : AnsiString);
       // Creates a surface with the given text
       procedure RenderText(const text : ansistring; pos : TCoord2D; fg : TColor); override;
       // Returns the estimated text width
       function GetWidth(const text : ansistring) : Integer; override;
       // Returns the estimated text height
       function GetHeight(const text : ansistring) : Integer; override;
       // Frees memory
       destructor Destroy; override;
       private
       Texture : TSurface;
       Widths  : array[0..255] of Byte;
     end;

{// Class for GL bitmap fonts
type TTTFFont = class(TGLFont)
       // Loads the given TTF font
       constructor Create(const filename : ansistring);
       // Creates a surface with the given text
       procedure RenderText(const text : ansistring; fg : TColor); override;
       // Returns the estimated text width
       function GetWidth(const text : ansistring) : Integer; override;
       // Returns the estimated text height
       function GetHeight(const text : ansistring) : Integer; override;
       // Frees memory
       destructor Destroy; override;
       private
     end;
}

type TFontArray = specialize TManagedArray<TGLFont>;

// The System class for the Font Engine
// The TGLFonts class must be initaited before any font classes are used.
// Currently except taking care of initialization and finalization this system
// has no other uses.
type TGLFonts = class(TSystem)
       private
       function getFont(index : DWord) : TGLFont;
       procedure setFont(index : DWord; const nFont : TGLFont);
       public
       // Registers the font system.
       constructor Create; override;
       // Closes the font system.
       destructor Destroy; override;
       // Access property for handling fonts.
       property Font[index : DWord] : TGLFont read getFont write setFont; default;
       private
       FontArray : TFontArray;
     end;

const GLFonts : TGLFonts = nil;

implementation
uses SysUtils{, sdl_ttf};

type A4GLfloat = array[0..3] of GLfloat;

function TGLFonts.GetFont(index : DWord) : TGLFont;
begin
  if FontArray[index] = nil then CritError('Font ID#'+IntToStr(index)+' not found!');
  Exit(FontArray[index])
end;

procedure TGLFonts.SetFont(index : DWord; const nFont : TGLFont);
begin
  if FontArray[index] <> nil       then CritError('Trying to reassign font ID#'+IntToStr(index)+'!');
  if not nFont.InheritsFrom(TGLFont) then CritError('Non font class passed to TFonts!');
  FontArray[index] := nFont;
end;

constructor TGLFonts.Create;
begin
  inherited Create;
  //TTF_Init;
  FontArray := TFontArray.Create;
end;

destructor TGLFonts.Destroy;
begin
  FreeAndNil(FontArray);
  //TTF_Quit;
  inherited Destroy;
end;


{constructor TTTFFont.Create(const filename : ansistring; ptsize : integer; quality : TFontQuality );
begin
  FontData    := TTF_OpenFont(PChar(filename), ptsize);
  FontQuality := Quality;
end;

function TTTFFont.TypeText(const text : ansistring; fg : TColor) : PSDL_Surface;
const BLACK : TSDL_Color = (r:0;g:0;b:0;);
begin
  case FontQuality of
    FQSolid   : Exit(TTF_RenderText_Solid( FontData , PChar(text) , fg.SDLColor ));
    FQShaded  : Exit(TTF_RenderText_Shaded( FontData , PChar(text) , fg.SDLColor , BLACK));
    FQBlended : Exit(TTF_RenderText_Blended( FontData , PChar(text) , fg.SDLColor ));
  end;
end;

// Returns the estimated text width
function TTTFFont.GetWidth(const text : ansistring) : Integer;
var temp : Integer;
begin
  TTF_SizeText( FontData, PChar(text), GetWidth, temp );
end;

// Returns the estimated text height
function TTTFFont.GetHeight(const text : ansistring) : Integer;
var temp : Integer;
begin
  TTF_SizeText( FontData, PChar(text), temp, GetHeight);
end;


destructor TTTFFont.Destroy;
begin
  TTF_CloseFont(FontData);
end;}

{ TGLBitmapFont }

constructor TGLBitmapFont.Create(const filename: ansistring; const Metrics : AnsiString);
var T : Text;
    i : Byte;
    s : String;
begin
  Texture := TSurface.Create( filename, true );
  Texture.RenderGL();
  Assign(T, Metrics);
  Reset(T);
  Readln(T);
  for i := 0 to 255 do
  begin
    Readln(T,s);
    Delete(S,1,Pos('=',s));
    Widths[i] := StrToInt(s);
  end;
  Close(T);
end;

procedure TGLBitmapFont.RenderText(const text: ansistring; pos : TCoord2D; fg: TColor);
var c     : Word;
    idx   : Byte;
    px,py : Single;
    funit : Single;
    iunit : Byte;
begin
  glBindTexture( GL_TEXTURE_2D, Texture.GLTexture );
  glColor3ub(fg.SDLColor.r,fg.SDLColor.g,fg.SDLColor.b);
  glBegin( GL_QUADS );
  funit := Texture.GLWidth / 16;
  iunit := Texture.SDLSurface^.w div 16;
  for c := 1 to Length(text) do
  begin
    idx := Ord(text[c]) - 32;
    px := (idx mod 16) * funit;
    py := (idx div 16) * funit;

    glTexCoord2f(px,py);
    glVertex2i(pos.x, pos.y);

    glTexCoord2f(px,py + funit);
    glVertex2i(pos.x, pos.y + iunit);

    glTexCoord2f(px + funit,py + funit);
    glVertex2i(pos.x + iunit, pos.y + iunit);

    glTexCoord2f(px + funit,py);
    glVertex2i(pos.x + iunit, pos.y);

    pos.x := pos.x + Widths[ idx ];
  end;
  glEnd();
end;

function TGLBitmapFont.GetWidth(const text: ansistring): Integer;
var c     : Word;
begin
  GetWidth := 0;
  for c := 1 to Length(text) do
    GetWidth += Widths[ Ord(text[c]) - 32 ];
end;

function TGLBitmapFont.GetHeight(const text: ansistring): Integer;
begin
  Exit( Texture.SDLSurface^.h div 16 );
end;

destructor TGLBitmapFont.Destroy;
begin
  FreeAndNil( Texture );
  inherited Destroy;
end;

end.

