{$INCLUDE valkyrie.inc}
// @abstract(Online Console classes for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @cvs($Author: chaos-dev $)
// @cvs($Date: 2008-01-14 22:16:41 +0100 (Mon, 14 Jan 2008) $)
//
// Introduces a console window, and console system.
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>

unit vconsole;

interface

uses SysUtils, vutil, vnode, vsystem;
  
type TConsoleWindow = class(TVObject)
  // Initializes the system.
  constructor Create(nBufferSize : Word = 1000; const nOptions : TFlags32 = []); virtual;
  // Adds a new message line to the system.
  procedure Add(MessageText : Ansistring);
  // Clears the buffer.
  procedure ClearAll;
  // Draws the messages to screen.
  procedure Draw; virtual; abstract;
  // Get's a added message. The messages are indexed from the last added to
  // the last one in the buffer. That is -- Get(1) returns the last added
  // message, Get(2) returns the previous and so on...
  function Get(Index : Word) : Ansistring;
  // Reads the input
  function ReadInput : string; virtual; abstract;
  // Saves the screen
  procedure SaveScreen; virtual;
  // Restores the screen
  procedure RestoreScreen; virtual;
  protected
  function GetCommand(Index : Word) : Ansistring;
  protected
  // Holds the messages
  Data : array of AnsiString;
  // Holds recent commands
  Commands : array of AnsiString;
  // Determinates the current position of adding in the Data message buffer.
  Position    : Word;
  // Holds the size of the Data array
  BufferSize  : Word;
  // Determinates the current position of adding in the Command message buffer.
  CPosition   : Word;
  // Holds the size of the Command array
  CBufferSize : Word;
  // Last
  // Width of the console in characters
  Width       : Word;
  // Options for the message window
  Options     : TFlags32;
end;

type

{ TConsole }

TConsole = class(TSystem)
  Window  : TConsoleWindow;
  Running : Boolean;
  constructor Create(ConsoleWindow : TConsoleWindow); reintroduce;
  procedure Call; override;
  procedure Print(text : AnsiString);
  procedure Command(cmd : Ansistring); virtual;
  procedure DumpHistory( const FileName : AnsiString );
  procedure LoadHistory( const FileName : AnsiString );
  destructor Destroy; override;
end;

var Console : TConsole = nil;

implementation

{ TConsoleWindow }

constructor TConsoleWindow.Create(nBufferSize : Word = 1000; const nOptions : TFlags32 = []);
var Count : Word;
begin
  BufferSize := nBufferSize;
  CBufferSize:= 20;
  Position   := 0;
  CPosition  := 0;
  Options    := nOptions;
  SetLength(Data,BufferSize);
  SetLength(Commands,CBufferSize);
  for Count := 0 to BufferSize - 1 do Data[Count] := '';
  for Count := 0 to CBufferSize - 1 do Commands[Count] := '';
  Width      := 76;
end;

procedure TConsoleWindow.Add(MessageText : Ansistring);
var Temp,Rest,Line : Ansistring;
begin
  Rest := MessageText;

  if Length(Rest) >= Width then
  while Rest <> '' do
  begin
    Temp := Rest;
    Split(Temp,Line,Rest,' ',Width-1);
    Add(Line);
  end
  else
  begin
    Data[Position] := Rest;
    Inc(Position);
    if Position = BufferSize then Position := 0;
  end;
end;

procedure TConsoleWindow.ClearAll;
var Count : Word;
begin
  for Count := 0 to BufferSize - 1 do Data[Count] := '';
  Position := 0;
end;

function TConsoleWindow.Get(Index : Word) : Ansistring;
var RelPos : LongInt;
begin
  RelPos := Position - Index;
  while RelPos < 0 do RelPos := RelPos + BufferSize;
  Exit(Data[RelPos]);
end;

function TConsoleWindow.GetCommand(Index : Word) : Ansistring;
var RelPos : LongInt;
begin
  RelPos := CPosition - Index;
  while RelPos < 0 do RelPos := RelPos + CBufferSize;
  Exit(Commands[RelPos]);
end;

procedure TConsoleWindow.SaveScreen;
begin
end;

procedure TConsoleWindow.RestoreScreen;
begin
end;

constructor TConsole.Create(ConsoleWindow : TConsoleWindow);
begin
  inherited Create;
  Window := ConsoleWindow;
  Running := False;
end;


procedure TConsole.Call;
var CommandStr : String;
begin
  Running := not Running;
  if not Running then Exit;

  Window.SaveScreen;
  while Running do
  begin
    Window.Draw;
    CommandStr := Window.ReadInput;
    Command(CommandStr);
  end;
  Window.RestoreScreen;
end;

procedure TConsole.Print(text : AnsiString);
begin
  Window.Add(text);
end;

procedure TConsole.Command(cmd : Ansistring);
begin
  cmd := Trim(cmd);
  Print(cmd);
end;

procedure TConsole.DumpHistory(const FileName: AnsiString);
var T : Text;
    C : DWord;
    S : AnsiString;
begin
  Assign( T, FileName );
  Rewrite(T);
  for C := Window.CBufferSize downto 1 do
  begin
    S := Window.GetCommand(C);
    if S <> '' then
    begin
      Writeln( T, S );
    end;
  end;
  Close( T );
end;

procedure TConsole.LoadHistory(const FileName: AnsiString);
var T : Text;
    C : DWord;
    S : AnsiString;
begin
  if not FileExists( FileName ) then Exit;
  Assign( T, FileName );
  Reset(T);
  Window.CPosition := 0;
  for C := 1 to Window.CBufferSize do
  begin
    Readln(T,S);
    if (S = '') or EoF(T) then Break;
    Window.Commands[Window.CPosition] := S;
    Inc(Window.CPosition);
  end;
  Close(T);
end;

destructor TConsole.Destroy;
begin
  Console := nil;
  FreeAndNil(Window);
  inherited Destroy;
end;


end.

