// @abstract(BerserkRL -- User Interface class)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Jan 9, 2007)
// @lastmod(Jan 9, 2007)
//
// This unit holds the User Interface class of Berserk!. Every input/output
// related task should be done via this class. This unit opens a possibility
// of a different UI for Berserk!, for example with tiles.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

unit brui;
interface

uses vsystem, voutput, vrltools, vtextut,
     bruichar, bruitext, vui;

const
    // Option that makes the name always "random"
    Option_AlwaysRandomName : Boolean = False;
    // Option that turns on message clearing
    Option_ClearMessages    : Boolean = False;
    // Option that turns on message coloring
    Option_MessageColoring  : Boolean = False;
    // Option for the kill count showing
    Option_KillCount        : Boolean = False;
    // Option of the message buffer size
    Option_MessageBuffer    : DWord = 100;
    // Option of the number of last messages in mortem.
    Option_MortemMessages   : DWord = 10;

    // Input command value -- Quit command
    COMMAND_QUIT       = 1;
    // Input command value -- walk up command
    COMMAND_WALKNORTH  = 2;
    // Input command value -- walk down command
    COMMAND_WALKSOUTH  = 3;
    // Input command value -- walk right command
    COMMAND_WALKEAST   = 4;
    // Input command value -- walk left command
    COMMAND_WALKWEST   = 5;
    // Input command value -- walk up right command
    COMMAND_WALKNE     = 6;
    // Input command value -- walk down right command
    COMMAND_WALKSE     = 7;
    // Input command value -- walk down left command
    COMMAND_WALKNW     = 8;
    // Input command value -- walk up left command
    COMMAND_WALKSW     = 9;
    // Input command value -- walk in place
    COMMAND_WAIT       = 10;
    // Input command value -- dialog escape
    COMMAND_ESCAPE     = 11;
    // Input command value -- dialog ok
    COMMAND_OK         = 12;
    // Input command value -- enter feature?
    COMMAND_ENTER      = 13;
    // Input command value -- look around command
    COMMAND_LOOK       = 21;
    // Input command value -- help system invocation
    COMMAND_HELP       = 25;
    // Input command value -- player info command
    COMMAND_PLAYERINFO = 26;
    // Input command value -- crossbow firing command
    COMMAND_FIRE       = 27;
    // Input command value -- knife throwing command
    COMMAND_KNIFE      = 28;
    // Input command value -- crossbow reload command
    COMMAND_CRRELOAD   = 29;
    // Input command value -- use fairy dust
    COMMAND_FAIRYDUST  = 30;
    // Input command value -- throw bomb
    COMMAND_BOMB       = 31;
    // Input command value -- throw bomb
    COMMAND_RUNNING    = 32;
    // Input command value -- use sweep attack
    COMMAND_SKILLSWEEP = 33;
    // Input command value -- fire cannon
    COMMAND_CANNON     = 34;
    // Input command value -- cannon reload
    COMMAND_CARELOAD   = 35;
    // Input command value -- Past messages viewer.
    COMMAND_MESSAGES   = 36;
    // Input command value -- Whirlwind attack skill.
    COMMAND_SKILLWHIRL = 47;
    // Input command value -- Whirlwind attack skill.
    COMMAND_SKILLIMPALE= 48;
    // Input command value -- Jump attack skill.
    COMMAND_SKILLJUMP  = 49;


    // Set of move commands
    COMMANDS_MOVE        = [COMMAND_WALKNORTH,COMMAND_WALKSOUTH,
                            COMMAND_WALKEAST, COMMAND_WALKWEST,
                            COMMAND_WALKNE,   COMMAND_WALKSE,
                            COMMAND_WALKNW,   COMMAND_WALKSW];
                            
// Color constants
const Black        = 0;    DarkGray     = 8;
      Blue         = 1;    LightBlue    = 9;
      Green        = 2;    LightGreen   = 10;
      Cyan         = 3;    LightCyan    = 11;
      Red          = 4;    LightRed     = 12;
      Magenta      = 5;    LightMagenta = 13;
      Brown        = 6;    Yellow       = 14;
      LightGray    = 7;    White        = 15;

// Picture
type TPicture = Word;

type TCmdSet = Set of Byte;

type TBerserkCharGenClass = class of TBerserkCharGen;
     TBerserkTextClass    = class of TBerserkText;

type
  // Berserk User interface class. Responsible for keeping all the UI in one
  // place. After some twidling, this could allow for a graphics version of
  // Berserk!. All display/input commands should go here.
  TBerserkUI = class(TUIElement)
    // The CharGen subclass, static.
    CharGen : TBerserkCharGenClass;
    // The CharGen subclass, static.
    Text    : TBerserkTextClass;
    // Playback mode
    Playback  : Boolean;
    // Holds initial Random Seed
    ThisSeed  : Cardinal;
    // Holds the current screen info
    Screen  : (Game,Menu);
    // Initialization of all data.
    constructor Create; reintroduce;
    // Adds a message for the message buffer
    procedure Msg(MsgText : Ansistring);
    // Show past messages.
    procedure MsgPast;
    // Kills last message from the message buffer.
    procedure MsgKill;
    // Dump last messages to file.
    procedure MsgDump(var TextFile : Text);
    // Writes a tile description in the msg area.
    procedure MsgCoord( Coord : TCoord2D ); virtual; abstract;
    // Reads a key from the keyboard and returns it's Command value.
    function GetCommand(Valid : TCmdSet = []) : byte;
    // Returns the key for the given command.
    function GetKeybinding(Command : Byte) : string;
    // Interface function for choosing direction. Returns the direction, or 0 if escaped.
    function ChooseDirection : TDirection;
    // Waits for enter key.
    procedure PressEnter;
    // Draws the level, player status, messages, and updates the screen.
    procedure Draw; override;
    // Graphical effect of a screen flash, of the given color, and Duration in
    // miliseconds.
    procedure Blink(Color : Byte; Duration : Word = 100); virtual; abstract;
    // Delays for effects.
    procedure Delay(Time : Word); virtual; abstract;
    // Cleaning up everything that was initialized in TBerserkUI.Create.
    destructor Destroy; override;
    // Converts Commands to Direction values
    function CommandDirection(Command : byte) : TDirection;
    // Draws the whole level to the screen. You need to Update the screen
    // afterwards to see it.
    procedure DrawLevel; virtual; abstract;
    // Moves the cursor to the position x,y in terms of Map coords.
    procedure Focus( Where : TCoord2D );
    // Renders an explosion on the screen.
    procedure Explosion( Where : TCoord2D; Color : byte; Range : byte;  Step : byte;  DrawDelay : Word = 50); virtual; abstract;
    // Draws a firey background
    procedure DrawFire; virtual; abstract;
    // Starts recording keystrokes
    procedure StartRecord;
    // Saves keystroke record
    procedure SaveRecord;
    // Loads keystroke record
    procedure LoadRecord;
    // Checks wether replay exists.
    function ReplayExists : boolean;
    // Sends missile
    procedure SendMissile( Source, Target : TCoord2D; mtype : Byte); virtual; abstract;
    // Draws target X
    procedure Target( Where : TCoord2D; color : Byte); virtual; abstract;
    // Renders a breath weapon attack
    procedure Breath( Where : TCoord2D; Direction : TDirection; Color : byte; Range : byte; Step : byte;
  DrawDelay : Word = 50);  virtual; abstract;
    // Window drawing procedure, call Clear afterwards
    procedure Window(x1,y1,x2,y2 : Word); virtual; abstract;
    // Clears all screen and windows
    procedure Clear; virtual; abstract;
    private
    // A singleton valiable for the vrltools.TMessages class.
    Messages : TTextModeMessages;
    // Draws the player status information.
    procedure DrawPlayerStatus;
  end;


// Singleton for the TBerserkUI class
const UI : TBerserkUI = nil;

implementation
uses SysUtils, math,
     vsystems, vinput, vini, zstream,
     brlevel, brplayer, brdata, brmain;

function InputCommandParser(const Token : shortstring) : byte; forward;

{ TBerserkUI }

constructor TBerserkUI.Create;
var INIFile : TINI;
begin
//  inherited Create;
  Output.SetTitle('Berserk!','Berserk!');
  Output.UI := Self;
  Input.ResetRecorder;
  INIFile := TINI.Load('berserk.ini');
  INIFile.SetSection('KeyBindings');
  Input.Load(INIFile,@InputCommandParser);
  INIFile.SetSection('General');
  Option_AlwaysRandomName := INIFile.GetBoolean('ALWAYSRANDOMNAME');
  Option_ClearMessages    := INIFile.GetBoolean('CLEARMESSAGES');
  Option_MessageColoring  := INIFile.GetBoolean('MESSAGECOLORING');
  Option_MessageBuffer    := INIFile.GetNumber ('MESSAGEBUFFER');
  Option_KillCount        := INIFile.GetBoolean('KILLCOUNT');
  Option_MortemMessages   := INIFile.GetNumber ('MORTEMMESSAGES');

  if Option_ClearMessages
    then Messages := TTextModeMessages.Create(52,17,28,8,Option_MessageBuffer,[MSG_MORE,MSG_UPDATECLEAR])
    else Messages := TTextModeMessages.Create(52,17,28,8,Option_MessageBuffer,[MSG_MORE]);

  if Option_MessageColoring then
  begin
    INIFile.SetSection('Messages');
    Messages.LoadColors(INIFile);
  end;
  FreeAndNil(INIFile);
  
  Playback := False;
  
  Screen := Menu;

  Msg('Berserk!');
  Msg('Press @<'+Input.CommandToString(COMMAND_HELP)+'@> for help.');
end;

procedure TBerserkUI.Msg(MsgText : Ansistring);
begin
  Messages.Add(MsgText);
end;

procedure TBerserkUI.MsgKill;
begin
  Messages.KillLast;
end;

procedure TBerserkUI.MsgPast;
begin
  Messages.ShowRecent;
  Draw;
end;

procedure TBerserkUI.MsgDump(var TextFile : Text);
var Count : Word;
begin
  for Count := Option_MortemMessages downto 1 do
    if Messages.Get(Count) <> '' then
      Writeln(TextFile,' '+Output.Strip(Messages.Get(Count)));
end;


function TBerserkUI.GetCommand(Valid : TCmdSet) : byte;
begin
  if Valid = [] then Exit(Input.GetCommand);
  Exit(Input.GetCommand(Valid));
end;

function TBerserkUI.GetKeybinding(Command : Byte) : string;
begin
  Exit(Input.CommandToString(Command));
end;


function TBerserkUI.ChooseDirection : TDirection;
begin
  Messages.Draw;
  Exit(CommandDirection(Input.GetCommand(COMMANDS_MOVE+[COMMAND_ESCAPE])));
end;

procedure TBerserkUI.Draw;
begin
  Output.Clear;
  Focus(Player.Position);
  Level.Vision.Run(Player.Position,Player.LightRadius);
  DrawLevel;
  DrawPlayerStatus;
  Messages.Draw;
  Messages.Update;
end;


destructor TBerserkUI.Destroy;
begin
  FreeAndNil(Messages);
  inherited Destroy;
end;

function TBerserkUI.CommandDirection(Command : byte) : TDirection;
begin
  case Command of
    COMMAND_WALKWEST  : CommandDirection.Create(4);
    COMMAND_WALKEAST  : CommandDirection.Create(6);
    COMMAND_WALKNORTH : CommandDirection.Create(8);
    COMMAND_WALKSOUTH : CommandDirection.Create(2);
    COMMAND_WALKNW    : CommandDirection.Create(7);
    COMMAND_WALKNE    : CommandDirection.Create(9);
    COMMAND_WALKSW    : CommandDirection.Create(1);
    COMMAND_WALKSE    : CommandDirection.Create(3);
    COMMAND_WAIT      : CommandDirection.Create(5);
    else CommandDirection.Create(0);
  end;
end;

procedure TBerserkUI.DrawPlayerStatus;
var Full,Full2,Count : Byte;
function BColor(Cnt,Full : Word) : Byte;
begin
  if Cnt <= Full then Exit(Red) else Exit(LightRed);
end;
begin
with Player do
begin
  Output.DrawString(52,1,Yellow,Name+', the Berserker');
  Output.DrawString(52,2,DarkGray,'Str:@<@1@> Dex:@<@2@> End:@<@3@> Wil:@<@4@>',[Strength,Dexterity,Endurance,Willpower]);
  Output.DrawString(52,4,DarkGray,'Health');
  Output.DrawString(58,4,DarkGray,' : @R@1@d/@R@2',[HP,HPMax]);

  Full  := math.Max(Round(HP*28/HPMax),0);
  Full2 := math.Max(Round(BerserkPT*28/HPMax),0);
  if Full > 0  then
    for Count := 1 to Full do
      Output.DrawChar(51+Count, 5,BColor(Count,Full2),'#');
  if Full < 28 then
    for Count := Full+1 to 28 do
      Output.DrawChar(51+Count, 5,BColor(Count,Full2),'-');

  Output.DrawString(52, 6,DarkGray,'Energy');
  Output.DrawString(58, 6,DarkGray,' : @y@1@d/@y@2',[EN,ENMax]);

  Full := math.Max(Round(EN*28/ENMax),0);
  if Full > 0  then
    for Count := 1 to Full do
      Output.DrawChar(51+Count, 7,Yellow,'#');
  if Full < 28 then
    for Count := Full+1 to 28 do
      Output.DrawChar(51+Count, 7,Yellow,'-');

  Output.DrawString(52,9 ,DarkGray,'Crossbow (@<@1@>/@<@2@>)',[Crossbow,CrossMis]);
  Output.DrawString(68,9 ,DarkGray,'Cannon (@<@1@>/@<@2@>)',[Cannon,CannonChg]);
  Output.DrawString(52,10,DarkGray,'Knives (@<@1@>)',[Knives]);
  Output.DrawString(68,10,DarkGray,'Bombs (@<@1@>)',[Bombs]);
  Output.DrawString(52,11,DarkGray,'Fairydust (@<@1@>)',[FairyDust]);

  if Pain > 0    then Output.DrawString(52,13,Red,'Pain (@<-@1@>)',[Pain]);
  if Freeze > 0  then Output.DrawString(64,13,LightBlue,'Freeze (@<-@1@>)',[Freeze]);
  if isRunning   then Output.DrawString(52,14,Yellow,'RUNNING');
  if isBerserk   then Output.DrawString(62,14,Red,'BERSERK');
  if isBerserk   then
    if EnemiesAround div 2 > 0 then
      Output.DrawString(62,13,Red,'Berserker (@<+@1@>)',[EnemiesAround div 2]);

  Output.DrawString(52,16,DarkGray,'---------------------------');
  if Mode <> modeMassacre then
  begin
    Full := Max(Min(Round((Level.TickCount/NIGHTDURATION)*28),28),1);
    Output.DrawString(52+Full-1,16,LightGray,'=');
    if Full < 14 then Output.LeftDrawString(77,16,DarkGray,' Night @1 ',[Player.Night])
                 else Output.DrawString(53,16,DarkGray,' Night @1 ',[Player.Night]);
  end;
  

  if Option_KillCount then
    Output.LeftDrawString(80,25,DarkGray,'[@r@1@>]',[Kills.All]);
end;
end;

procedure TBerserkUI.Focus( Where : TCoord2D );
begin
  Output.MoveCursor(Where.x+MAP_POSX-1,Where.y+MAP_POSX-1);
end;

procedure TBerserkUI.PressEnter;
begin
  Input.GetKey([VKEY_ENTER]);
end;

function InputCommandParser(const Token : shortstring) : byte;
begin
  if Token = 'QUIT'        then Exit(COMMAND_QUIT);
  if Token = 'WALKNORTH'   then Exit(COMMAND_WALKNORTH);
  if Token = 'WALKSOUTH'   then Exit(COMMAND_WALKSOUTH);
  if Token = 'WALKEAST'    then Exit(COMMAND_WALKEAST);
  if Token = 'WALKWEST'    then Exit(COMMAND_WALKWEST);
  if Token = 'WALKNE'      then Exit(COMMAND_WALKNE);
  if Token = 'WALKSE'      then Exit(COMMAND_WALKSE);
  if Token = 'WALKNW'      then Exit(COMMAND_WALKNW);
  if Token = 'WALKSW'      then Exit(COMMAND_WALKSW);
  if Token = 'ESCAPE'      then Exit(COMMAND_ESCAPE);
  if Token = 'OK'          then Exit(COMMAND_OK);
  if Token = 'ENTER'       then Exit(COMMAND_ENTER);
  if Token = 'WAIT'        then Exit(COMMAND_WAIT);
  if Token = 'LOOK'        then Exit(COMMAND_LOOK);
  if Token = 'HELP'        then Exit(COMMAND_HELP);
  if Token = 'FIRE'        then Exit(COMMAND_FIRE);
  if Token = 'PLAYERINFO'  then Exit(COMMAND_PLAYERINFO);
  if Token = 'KNIFE'       then Exit(COMMAND_KNIFE);
  if Token = 'CRRELOAD'    then Exit(COMMAND_CRRELOAD);
  if Token = 'FAIRYDUST'   then Exit(COMMAND_FAIRYDUST);
  if Token = 'BOMB'        then Exit(COMMAND_BOMB);
  if Token = 'RUNNING'     then Exit(COMMAND_RUNNING);
  if Token = 'SKILLSWEEP'  then Exit(COMMAND_SKILLSWEEP);
  if Token = 'SKILLWHIRL'  then Exit(COMMAND_SKILLWHIRL);
  if Token = 'SKILLIMPALE' then Exit(COMMAND_SKILLIMPALE);
  if Token = 'SKILLJUMP'   then Exit(COMMAND_SKILLJUMP);
  if Token = 'CANNON'      then Exit(COMMAND_CANNON);
  if Token = 'CARELOAD'    then Exit(COMMAND_CARELOAD);
  if Token = 'MESSAGES'    then Exit(COMMAND_MESSAGES);
  Exit(0);
end;

procedure TBerserkUI.StartRecord;
begin
  if Playback then Exit;
  Randomize;
  ThisSeed := RandSeed;
  Input.RecorderStart;
end;

procedure TBerserkUI.SaveRecord;
var RecFile  : TGZFileStream;
begin
  if Playback then Exit;
  RecFile := TGZFileStream.Create('berserk.rec',gzOpenWrite);
  RecFile.Write(ThisSeed,SizeOf(ThisSeed));
  Input.RecorderDump(RecFile);
  FreeAndNil(RecFile);
end;

procedure TBerserkUI.LoadRecord;
var PlayFile : TGZFileStream;
begin
  PlayFile := TGZFileStream.Create('berserk.rec',gzOpenRead);
  PlayFile.Read(ThisSeed,SizeOf(ThisSeed));
  RandSeed := ThisSeed;
  Input.PlaybackLoad(PlayFile);
  Input.PlaybackStart;
  FreeAndNil(PlayFile);
  PLAYBACK := True;
end;

function TBerserkUI.ReplayExists : boolean;
begin
  Exit(FileExists('berserk.rec'));
end;



end.

