// @abstract(BerserkRL -- Textmode User Interface class)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Apr 11, 2007)
// @lastmod(Apr 11, 2007)
//
// This unit holds the texmode User Interface class of Berserk!.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

unit brtextui;
interface
uses SysUtils, vmath, vrltools, brui;

type

{ TBerserkTextUI }

TBerserkTextUI = class(TBerserkUI)
    // Initialization of all data.
    constructor Create; reintroduce;
    // Sends missile
    procedure SendMissile( Source, TargetPos : TCoord2D; mtype : Byte); override;
    // Draws target X
    procedure Target( Where : TCoord2D; color : Byte ); override;
    // Renders an explosion on the screen.
    procedure Explosion( Where : TCoord2D; Color : byte; Range : byte; Step : byte;  DrawDelay : Word = 50); override;
    // Renders a breath weapon attack
    procedure Breath( Where : TCoord2D; Direction : TDirection; Color : byte; Range : byte;  Step : byte;
  DrawDelay : Word = 50); override;
    // Graphical effect of a screen flash, of the given color, and Duration in
    // miliseconds.
    procedure Blink(Color : Byte; Duration : Word = 100); override;
    // Delays for effects.
    procedure Delay(Time : Word); override;
    // Writes a tile description in the msg area.
    procedure MsgCoord( Coord : TCoord2D ); override;
    // Draws a firey background
    procedure DrawFire; override;
    // Draws the whole level to the screen. You need to Update the screen
    // afterwards to see it.
    procedure DrawLevel; override;
    // Window drawing procedure, call Clear afterwards
    procedure Window(x1,y1,x2,y2 : Word); override;
    // Clears all screen and windows
    procedure Clear; override;
    private
    // Marks the given tile with the specified gylph. Use MarkDisplay afterwards.
    procedure MarkTile(x,y : byte; atr : byte; chr : char);
  end;


implementation

uses video, vsystems, voutput, vinput, vtoutput, vtinput, brlevel, brdata, brbeing, brplayer, vtextut, vvision;

{ TBerserkTextUI }

constructor TBerserkTextUI.Create;
begin
  Systems.Add(Output,TTextModeOutput.Create(80,25,[VTMO_FORCECLEAR]));
  Systems.Add(Input,TTextModeInput.Create);
  inherited Create;
  Output.UI := nil;
end;

procedure TBerserkTextUI.SendMissile( Source, TargetPos : TCoord2D; mtype : Byte);
const HITDELAY  = 50;
var Dist : byte;
    Ray  : TVisionRay;
    Old  : TCoord2D;
    Scan      : Word;
    DrawDelay : Word;
begin
  DrawDelay := 10;
  case mtype of
    MTBOLT  : DrawDelay := 10;
    MTKNIFE : DrawDelay := 20;
    MTBOMB  : DrawDelay := 50;
    MTENERGY: DrawDelay := 20;
    MTICE   : DrawDelay := 30;
    MTSPORE : DrawDelay := 80;
  end;

  Ray.Init(Level,Source,TargetPos);
  repeat
    Old := Ray.GetC;
    Ray.Next;
    if not Level.properCoord( Ray.GetC ) then Exit;

    if Level.Vision.isVisible( Ray.GetC ) then // Draw when visible
    begin
      case MTYPE of
        MTBOLT  : MarkTile(Ray.GetX,Ray.GetY,LightGray,DirPicture(CreateDir(Old.X,Old.Y,Ray.GetX,Ray.GetY)));
        MTKNIFE : MarkTile(Ray.GetX,Ray.GetY,White,DirPicture(CreateDir(Old.X,Old.Y,Ray.GetX,Ray.GetY)));
        MTBOMB  : MarkTile(Ray.GetX,Ray.GetY,Brown,'*');
        MTENERGY: case Ray.cnt mod 3 of
                    0 : MarkTile(Ray.GetX,Ray.GetY,White,'*');
                    1 : MarkTile(Ray.GetX,Ray.GetY,LightBlue,'*');
                    2 : MarkTile(Ray.GetX,Ray.GetY,Blue,'*');
                  end;
        MTICE   : case Ray.cnt mod 2 of
                    0 : MarkTile(Ray.GetX,Ray.GetY,White,'*');
                    1 : MarkTile(Ray.GetX,Ray.GetY,LightBlue,'*');
                  end;
        MTSPORE : case Ray.cnt mod 2 of
                    0 : MarkTile(Ray.GetX,Ray.GetY,Green,'*');
                    1 : MarkTile(Ray.GetX,Ray.GetY,LightGreen,'*');
                  end;
      end;
      Output.Update;
      Delay(DRAWDELAY);
      DrawLevel;
    end;
  until Ray.Done;
  
  Scan := Player.TryMove( Ray.GetC );
  if Scan > 500 then // Missile hits non-passable feature
    if Level.Vision.isVisible( Ray.GetC ) then
      begin
        MarkTile(Ray.GetX, Ray.GetY, LightGray, '*');
        Output.Update;
        Delay(HITDELAY);
        DrawLevel;
      end;

  if Scan > 0 then // Missile hits Being
  begin
    MarkTile(Ray.GetX, Ray.GetY, Red, '*');
    Output.Update;
    Delay(HITDELAY);
    DrawLevel;
  end;

end;

procedure TBerserkTextUI.Explosion( Where : TCoord2D; Color : byte; Range : byte;Step : Byte;
  DrawDelay : Word = 50);
var ax,ay    : Byte;
    Coord    : TCoord2D;
    d        : Byte;
    c1,c2,c3 : Byte;
    Vis      : boolean;
begin
  Vis := False;
  case Color of
    Blue    : begin c1 := Blue;    c2 := LightBlue;  c3 := White; end;
    Magenta : begin c1 := Magenta; c2 := Red;        c3 := Blue; end;
    Green   : begin c1 := Green;   c2 := LightGreen; c3 := White; end;
    LightRed: begin c1 := LightRed; c2 := Yellow;   c3 := White; end;
     else     begin c1 := Red; c2 := LightRed; c3 := Yellow; end;
  end;

  DrawLevel;
  for ax := Max(1,Where.x-Range) to Min(MAP_MAXX,Where.x+Range) do
    for ay := Max(1,Where.y-Range) to Min(MAP_MAXY,Where.y+Range) do
    begin
      Coord.Create( ax, ay );
      if Level.Vision.isVisible( Coord ) then Vis := True else Continue;
      if not Level.isEyeContact( Coord, Where ) then Continue;
      d := Distance( Coord, Where );
      if d > Range then Continue;
      if d = Step   then MarkTile(ax,ay,c1,'*');
      if d = Step-1 then MarkTile(ax,ay,c2,'*');
      if d = Step-2 then MarkTile(ax,ay,c3,'*');
      if d = Step-3 then MarkTile(ax,ay,c2,'*');
    end;
  if Vis then
  begin
    Output.Update;
    Delay(DrawDelay);
    DrawLevel;
  end;
  Output.Update;
end;


procedure TBerserkTextUI.Breath( Where : TCoord2D; Direction : TDirection; Color : byte;Range : byte; Step : byte;
  DrawDelay : Word = 50);
var ax,ay : Byte;
    dx,dy    : ShortInt;
    d        : Byte;
    c1,c2,c3 : Byte;
    Vis      : boolean;
    Angle    : Real;
    Coord    : TCoord2D;
begin
  Vis := False;
  case Color of
    Blue    : begin c1 := Blue;    c2 := LightBlue;  c3 := White; end;
    Magenta : begin c1 := Magenta; c2 := Red;        c3 := Blue; end;
    Green   : begin c1 := Green;   c2 := LightGreen; c3 := White; end;
    LightRed: begin c1 := LightRed; c2 := Yellow;   c3 := White; end;
     else     begin c1 := Red; c2 := LightRed; c3 := Yellow; end;
  end;

  UI.DrawLevel;
  for ax := Max(1,Where.x-Range) to Min(MAP_MAXX,Where.x+Range) do
    for ay := Max(1,Where.y-Range) to Min(MAP_MAXY,Where.y+Range) do
    begin
      Coord.Create(ax,ay);
      d := Distance( Coord, Where );
      if d = 0 then Continue;

      if Direction.x <> 0 then if Sgn(ax-Where.x) = -Direction.x then Continue;
      if Direction.y <> 0 then if Sgn(ay-Where.y) = -Direction.y then Continue;
      if Direction.x = 0 then begin if Abs(ay-Where.y) < Abs(ax-Where.x) then Continue; end;
      if Direction.y = 0 then begin if Abs(ax-Where.x) < Abs(ay-Where.y) then Continue; end;

      angle := ((ax-Where.x)*Direction.x + (ay-Where.y)*Direction.y)/(vmath.RealDistance(Where.x,Where.y,ax,ay)*vmath.RealDistance(Where.x,Where.y,Where.x+Direction.x,Where.y+Direction.y));
      if angle < 0.76+(d*0.02) then Continue;

      if Level.Vision.isVisible( Coord ) then Vis := True else Continue;
      if not Level.isEyeContact( Coord, Where ) then Continue;
      if d > Range then Continue;
      
      if Random(2) = 0 then d := d + Random(2);
      if d = Step   then MarkTile(ax,ay,c1,'*');
      if d = Step-1 then MarkTile(ax,ay,c2,'*');
      if d = Step-2 then MarkTile(ax,ay,c3,'*');
      if d = Step-3 then MarkTile(ax,ay,c2,'*');
      if d = Step-4 then MarkTile(ax,ay,c1,'*');
    end;
  if Vis then
  begin
    Output.Update;
    Delay(DrawDelay);
    DrawLevel;
  end;
  Output.Update;
end;


procedure TBerserkTextUI.Target( Where : TCoord2D; color : Byte);
begin
  MarkTile(where.x,where.y,color,'X');
  Output.Update;
end;

procedure TBerserkTextUI.MsgCoord(Coord : TCoord2D);
begin
  Output.DrawString(52,24,LightGray,'                           ');
  if Level.Vision.isVisible( Coord ) then
    if Level.Being[ Coord ] <> nil then
      Output.DrawString(52,24,White,Level.Being[ Coord ].LookDescribe)
    else
      Output.DrawString(52,24,White,Level.Terrain[ Coord ].Name)
  else
    Output.DrawString(52,24,White,'nothing');
  Output.Update;
end;


procedure TBerserkTextUI.MarkTile(x, y : byte; atr : byte; chr : char);
begin
  Output.DrawChar(x+MAP_POSX-1,y+MAP_POSY-1,atr,chr);
end;


procedure TBerserkTextUI.Blink(Color : Byte; Duration : Word);
var vx,vy : byte;
    pchr   : Char;
    pcol   : Byte;
begin
  pchr := '';
  pcol := color;
  for vx := 1 to 80 do for vy := 1 to 25 do Output.DrawPicture(vx,vy,pcol*256 + Ord(pchr));
  Output.Update;
  Delay(Duration);
  Draw;
  Output.Update;
end;

type TGFXScreen = array[1..25,1..80] of Word;

procedure TBerserkTextUI.DrawFire;
var Temp  : TGFXScreen;
    x,y,Count,limit : byte;
    shift : shortint;
const RedFire    = Ord('#') + 256*Red;
      LRedFire   = Ord('#') + 256*LightRed;
      YellowFire = Ord('#') + 256*Yellow;
  procedure Draw(xx,vy,lim : byte);
  var vx : byte;
  begin
    Shift := 0;
    for vy := 25 downto lim do
    begin
      case Shift of
        -1: if Random(4) = 0 then Shift := 0;
        0 : if Random(5) = 0 then Shift := Random(2)*2-1;
        1 : if Random(4) = 0 then Shift := 0;
      end;
      vx := Min(Max(4,xx + Shift),77);
      case vy - lim of
        0..4  : Temp [vy,vx] := RedFire;
        5..7  : begin
                  Temp [vy,vx-1] := RedFire;
                  Temp [vy,vx]   := RedFire;
                end;
        8..11 : begin
                  Temp [vy,vx-1] := RedFire;
                  Temp [vy,vx]   := RedFire;
                  Temp [vy,vx+1] := RedFire;
                end;
        12..14 : begin
                  Temp [vy,vx-1] := RedFire;
                  Temp [vy,vx]   := LRedFire;
                  Temp [vy,vx+1] := RedFire;
                end;
        15..16 : begin
                  Temp [vy,vx-2] := RedFire;
                  Temp [vy,vx-1] := LRedFire;
                  Temp [vy,vx]   := YellowFire;
                  Temp [vy,vx+1] := RedFire;
                end;
        17..18  : begin
                  Temp [vy,vx-2] := RedFire;
                  Temp [vy,vx-1] := LRedFire;
                  Temp [vy,vx]   := YellowFire;
                  Temp [vy,vx+1] := LRedFire;
                  Temp [vy,vx+2] := RedFire;
                end;
        19..20 : begin
                  Temp [vy,vx-3] := RedFire;
                  Temp [vy,vx-2] := LRedFire;
                  Temp [vy,vx-1] := YellowFire;
                  Temp [vy,vx]   := YellowFire;
                  Temp [vy,vx+1] := LRedFire;
                  Temp [vy,vx+2] := RedFire;
                end;
        21..25 : begin
                  Temp [vy,vx-3] := RedFire;
                  Temp [vy,vx-2] := LRedFire;
                  Temp [vy,vx-1] := YellowFire;
                  Temp [vy,vx]   := YellowFire;
                  Temp [vy,vx+1] := YellowFire;
                  Temp [vy,vx+2] := LRedFire;
                  Temp [vy,vx+3] := RedFire;
                end;
       end;
     end;
  end;
  function Cycle(b : shortint) : byte;
  begin
    if b < 1  then b := 1;
    if b > 80 then b := 80;
    Exit(b);
  end;

begin
  Output.Clear;
  for x := 1 to 80 do for y := 1 to 25 do Temp [y,x] := Ord(' ')+LightGray;

  for Count := 1 to 20 do
  begin
    x := Max(Min(76,Random(4)-2+Count*4),4);
    limit := Random(10)+5;
    Draw(x,y,limit);
  end;
  for Count := 1 to 20 do
  begin
    x := Max(Min(76,Random(8)-4+Count*4),4);
    limit := Random(15)+1;
    Draw(x,y,limit);
  end;

  for y := 1 to 25 do
    for x := 1 to 80 do Output.DrawPicture(x,y,Temp[y,x]);
  Output.Update;
end;

procedure TBerserkTextUI.Delay(Time : Word);
begin
  Sleep(Time);
end;

procedure TBerserkTextUI.DrawLevel;
var X, Y, CMod : Word;
    iCoord     : TCoord2D;
  function Color(Pic : Word; Light : Boolean = False) : Word;
  begin
    if CMod <> 0 then Exit(Pic+CMod);
    if not Player.isBerserk then Exit(Pic)
    else if Light then Exit( Pic mod 256 + 256*LightRed )
                  else Exit( Pic mod 256 + 256*Red );
  end;
begin
  if Player.HP < 1 then CMod := 256*16*Red else CMod := 0;
  for X := 1 to MAP_MAXX do
    for Y := 1 to MAP_MAXY do
    begin
      iCoord.Create( X, Y );
      with Level.Terrain[ iCoord ] do
        if not Level.Vision.isVisible( iCoord )
          then Output.DrawPicture(X+MAP_POSX-1,Y+MAP_POSY-1,Color(DarkPic))
          else if Level.Being[ iCoord ] <> nil then Output.DrawPicture(X+MAP_POSX-1,Y+MAP_POSY-1,Color(Level.Being[ iCoord ].Picture, True))
                                               else Output.DrawPicture(X+MAP_POSX-1,Y+MAP_POSY-1,Color(Picture,TF_HIGHLIGHT in Level.getFlags( iCoord )));
    end;
end;

procedure TBerserkTextUI.Window(x1, y1, x2, y2: Word);
begin
  Output.ClearRect(x1,y1,x2,y2);
end;

procedure TBerserkTextUI.Clear;
begin
  Output.Clear;
end;



end.

