// @abstract(BerserkRL -- Main Application class)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Oct 16, 2006)
// @lastmod(Oct 22, 2006)
//
// This unit just holds the TBerserk class -- the application framework
// for BerserkRL.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

{$include brinclude.inc}

unit brmain;
{$mode objfpc}
interface

uses vsystem, brlua, brlevel, brplayer,brdata,zstream,
{$IFDEF GRAPHICS}
     brgui;
{$ELSE}
     brtextui;
{$ENDIF}


type
// Main application class.
// TBerserk is responsible initialization, running, and cleaning up afterwards
TBerserk = class(TSystem)
       // If set to true, means that the player requested a quit action.
       Escape    : Boolean;
       // Holds id number of the arena on which the player plays -- temporary
       Arena     : Byte;
       // Holds the berserk lua state
       Lua       : TBerserkLua;
       // Initialization of all data.
       constructor Create; override;
       // Saves the data after night is over
       procedure Save;
       // Loads the data
       procedure Load;
       // Checks wether a save file exists
       function SaveExists : Boolean;
       // Running the program -- main application loop is here.
       procedure   Run;
       // Cleaning up everything that was initialized in TBerserk.Create.
       destructor  Destroy; override;
     end;

var Berserk : TBerserk = nil;

implementation

uses SysUtils,
     vutil, vsystems, vparams, vrltools,
     brui, brhof;

{ TBerserk }

constructor TBerserk.Create;
var CP : TParams;
begin
  inherited Create;
  Berserk := Self;
  Version := ReadVersion('version.txt');
  CP := TParams.Create;
  {$IFDEF GRAPHICS}
  UI := TBerserkGUI.Create(CP.isSet('fullscreen'));
  {$ELSE}
  UI := TBerserkTextUI.Create;
  {$ENDIF}
  Systems.Add(HOF,THOF.Create);

  Lua := TBerserkLua.Create;
  Lua.Load();

  UI.Screen := Menu;

  if CP.isSet('god')  then GodMode := True;
  if CP.isSet('play') then
    if UI.ReplayExists then UI.LoadRecord
    else
    begin
      Writeln('No berserk.rec exists! Play a game, and try again!');
      Writeln('Press <Enter>...');
      Readln;
      Halt(0);
    end;
  if CP.isSet('quick') then QuickStart := True;
  FreeAndNil(CP);

  UI.StartRecord;
  Level  := TLevel.Create;
  Escape := False;
  Player := TPlayer.Create( NewCoord2D( 1,1 ) );
  Arena     := 1;
  Player.Night     := 0;

  Level.Beings[1] := Player;
  Lua.TableExecute('core','initialize');
  Level.Beings[1] := nil;
end;

procedure TBerserk.Save;
var SaveFile  : TGZFileStream;
begin
  if UI.Playback then Exit;
  Dec(Player.Night);
  SaveFile := TGZFileStream.Create('berserk.sav',gzOpenWrite);
  SaveFile.Write(Pointer(Player)^,Player.InstanceSize);
  SaveFile.Destroy;
end;

procedure TBerserk.Load;
var SaveFile : TGZFileStream;
begin
  if UI.Playback then Exit;
  SaveFile := TGZFileStream.Create('berserk.sav',gzOpenRead);
  SaveFile.Read(Pointer(Player)^,Player.InstanceSize);
  SaveFile.Destroy;
  DeleteFile('berserk.sav');
end;

function TBerserk.SaveExists: Boolean;
begin
  Exit(FileExists('berserk.sav'));
end;

procedure TBerserk.Run;
begin
  UI.Text.Intro;
  
  if SaveExists then Load
                else Player.CreateCharacter;
  
  repeat
    Inc(Player.Night);
    Level.Clear;
    case Player.Mode of
      modeMassacre : Level.Generate(Arena,Player.Mode,1);
      modeEndless  : begin
          if Player.Night > 1 then
          begin
            UI.Text.Night;
            if SaveExists then Break;
            Player.Advance;
          end;
          Level.Generate(1,Player.Mode,Player.Night);
        end;
    end;
    UI.Screen := Game;
    repeat
      Level.Tick;
    until Escape or Level.Cleared;
    UI.Screen := Menu;
  until Escape;
  if not SaveExists then UI.Text.HOF;
  UI.Text.Outro;
  UI.SaveRecord;
end;

destructor TBerserk.Destroy;
begin
  FreeAndNil(Lua);
  FreeAndNil(Level);
  FreeAndNil(Player);
  inherited Destroy;
end;

finalization

  if (ExitCode <> 0) then
  begin
    Writeln('Abnormal program termination! Please write down the above');
    Writeln('to help get rid Berserk! of all those bugs! You only need');
    Writeln('to write down the filenames and linenumbers.');
    if (Player <> nil) and (Player.Mode <> ModeMassacre) then Berserk.Save;
    Readln;
  end;
end.

