// @abstract(BerserkRL -- Main Application class)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Oct 16, 2006)
// @lastmod(Oct 22, 2006)
//
// This unit just holds the TBerserk class -- the application framework
// for BerserkRL.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>
{$include brinclude.inc}
unit brlua;
interface
uses Classes, SysUtils, vlua;

type TBerserkLua = class(TLua)
  constructor Create;
  procedure Load;
  procedure LoadCells;
  destructor Destroy; override;
  procedure LuaError;
end;

implementation
uses lua, strutils,
     vdebug, vinput, vrltools,
     brmain, brui, brdata, brbeing, brlevel;

procedure CheckObjectFunc( L: Plua_State; FuncName : AnsiString; Args : Byte );
begin
  if lua_gettop(L) < Args    then Berserk.Lua.Error(FuncName+' has wrong amount of parameters, at least '+IntToStr( Args )+' expected!');
  if not lua_isuserdata(L,1) then Berserk.Lua.Error(FuncName+' udata pointer expected!');
end;

function lua_berserk_require( L: Plua_State ) : Integer; cdecl;
var Arg      : AnsiString;
    Module   : AnsiString;
    Path     : AnsiString;
begin
  if lua_gettop(L) <> 1 then Berserk.Lua.Error('Require has wrong amount of parameters!');
  Arg := lua_tostring( L, 1 );
  if Berserk.Lua.ModuleNames.Exists(Arg) then Exit(0);

  Path := Arg;
  Module := ExtractDelimited( 1, Path, [':'] );
  Delete( Path, 1, Length( Module ) + 1 );

  Path := Path+'.lua';
  if not FileExists(Path) then
  begin
    Path := 'lua'+PathDelim+Path;
    if not FileExists(Path) then
      raise ELuaException.Create('require : File "'+Path+'" not found!');
  end;
  Berserk.Lua.LoadFile(Path);
  Berserk.Lua.ModuleNames[ Arg ] := True;
  Exit( 0 );
end;

function lua_core_log(L: Plua_State): Integer; cdecl;
begin
  if lua_gettop(L) = 0 then Exit(0);
  Log('Lua : ' + lua_tostring(L, 1));
  Result := 0;
end;

function lua_ui_msg(L: Plua_State): Integer; cdecl;
begin
  if lua_gettop(L) = 0 then Exit(0);
  UI.Msg(lua_tostring(L, 1));
  UI.Draw;
  Result := 0;
end;

function lua_ui_delay(L: Plua_State): Integer; cdecl;
begin
  Sleep(Round(lua_tonumber(L,1)));
  Result := 0;
end;


function lua_being_property_set(L: Plua_State): Integer; cdecl;
var Being : TBeing;
    id    : Word;
    arg   : Variant;
begin
  CheckObjectFunc(L,'being_property_set',3);
  Being := TObject(lua_touserdata(L,1)) as TBeing;
  id  := Round(lua_tonumber(L, 2));
  arg := lua_tovariant(L, 3);
  Being.setProperty(id, arg);
  Result := 0;
end;

function lua_being_property_get(L: Plua_State): Integer; cdecl;
var Being : TBeing;
    id    : Word;
begin
  CheckObjectFunc(L,'being_property_get',2);
  Being := TObject(lua_touserdata(L,1)) as TBeing;
  id  := Round(lua_tonumber(L, 2));
  lua_pushvariant( L, Being.getProperty(id) );
  Result := 1;
end;

function lua_being_flags_set(L: Plua_State): Integer; cdecl;
var Being : TBeing;
    id    : Byte;
begin
  CheckObjectFunc(L,'being_flags_set',3);
  Being := TObject(lua_touserdata(L,1)) as TBeing;
  id  := Round(lua_tonumber(L, 2));
  if lua_toboolean(L, 3)
    then Include( Being.Flags, id )
    else Exclude( Being.Flags, id );
  Result := 0;
end;

function lua_being_flags_get(L: Plua_State): Integer; cdecl;
var Being : TBeing;
    id    : Byte;
begin
  CheckObjectFunc(L,'being_flags_get',2);
  Being := TObject(lua_touserdata(L,1)) as TBeing;
  id  := Round(lua_tonumber(L, 2));
  lua_pushboolean( L, id in Being.Flags );
  Result := 1;
end;

function lua_being_die(L: Plua_State): Integer; cdecl;
var Being : TBeing;
begin
  CheckObjectFunc(L,'being_die',1);
  Being := TObject(lua_touserdata(L,1)) as TBeing;
  Being.Die;
  Result := 0;
end;

function lua_being_move(L: Plua_State): Integer; cdecl;
var Being : TBeing;
begin
  CheckObjectFunc(L,'being_move',3);
  Being := TObject(lua_touserdata(L,1)) as TBeing;
  Being.Move(NewCoord2D(Round(lua_tonumber(L, 2)), Round(lua_tonumber(L, 3))));
  Result := 0;
end;

function lua_being_send_missile(L: Plua_State): Integer; cdecl;
var Being : TBeing;
begin
  CheckObjectFunc(L,'being_send_missile',4);
  Being := TObject(lua_touserdata(L,1)) as TBeing;
  Being.SendMissile(NewCoord2D(Round(lua_tonumber(L, 2)), Round(lua_tonumber(L, 3))),Round(lua_tonumber(L, 4)));
  Result := 0;
end;


function lua_being_apply_damage(L: Plua_State): Integer; cdecl;
var Being : TBeing;
begin
  CheckObjectFunc(L,'being_apply_damage',3);
  Being := TObject(lua_touserdata(L,1)) as TBeing;
  Being.ApplyDamage(Round(lua_tonumber(L,2)),Round(lua_tonumber(L,3)));
  Result := 0;
end;


function lua_level_get_being(L: Plua_State): Integer; cdecl;
var Args  : Byte;
    Being : TBeing;
begin
  Args := lua_gettop(L);
  if Args = 0 then Exit(0);
  if (Args = 1) or lua_isnil(L,2) then
    Being := Level.Beings[ Round(lua_tonumber(L, 1)) ]
  else
    Being := Level.Being[ NewCoord2D(Round(lua_tonumber(L, 1)), Round(lua_tonumber(L, 2))) ];
  if Being = nil
    then lua_pushnil( L )
    else lua_pushlightuserdata(L, Being);
  Result := 1;
end;

function lua_level_summon(L: Plua_State): Integer; cdecl;
var Args  : Byte;
    Being : TBeing;
    dlvl  : Word;
begin
  Args := lua_gettop(L);
  if Args < 3 then Exit(0);
  dlvl := 0;
  if (args > 3) and (not lua_isnil(L,4)) then dlvl := Round(lua_tonumber(L, 4));

  Being := Level.Summon(lua_tostring(L,1),NewCoord2D(Round(lua_tonumber(L, 2)), Round(lua_tonumber(L, 3))), dlvl);
  if Being = nil
    then lua_pushnil( L )
    else lua_pushlightuserdata(L, Being);
  Result := 1;
end;

function lua_level_explosion(L: Plua_State): Integer; cdecl;
var Args  : Byte;
    Being : TBeing;
    dlvl  : Word;
begin
  Args := lua_gettop(L);
  if Args < 3 then Exit(0);
  Level.Explosion(
    NewCoord2D(Round(lua_tonumber(L, 1)), Round(lua_tonumber(L, 2))),
    Round(lua_tonumber(L, 3)), // color
    Round(lua_tonumber(L, 4)), // range
    Round(lua_tonumber(L, 5)), // strength
    Round(lua_tonumber(L, 6)), // draw delay
    Round(lua_tonumber(L, 7)) // damage type
  );
  Result := 0;
end;


function lua_level_set_cell (L: Plua_State): Integer; cdecl;
var CellID : Word;
    Coord  : TCoord2D;
begin
  //cellid, x, y,
  if lua_gettop(L) < 3 then Exit(0);
  CellID  := Round(lua_tonumber(L, 1));
  Coord.x := Round(lua_tonumber(L, 2));
  Coord.y := Round(lua_tonumber(L, 3));
  Level.Cell[Coord] := CellID;
  Result := 0;
end;

function lua_level_get_cell (L: Plua_State): Integer; cdecl;
var Coord  : TCoord2D;
begin
  //x, y
  if lua_gettop(L) < 2 then Exit(0);
  Coord.x := Round(lua_tonumber(L, 1));
  Coord.y := Round(lua_tonumber(L, 2));
  lua_pushnumber(L, Level.Cell[Coord] );
  Result := 1;
end;


constructor TBerserkLua.Create;
begin
  inherited Create;
end;

procedure TBerserkLua.Load;
var Amount, Count : DWord;
begin
  Register('core_log',        @lua_core_log);
  Register('ui_msg',          @lua_ui_msg);

  Register('being_property_get',@lua_being_property_get);
  Register('being_property_set',@lua_being_property_set);
  Register('being_flags_get',@lua_being_flags_get);
  Register('being_flags_set',@lua_being_flags_set);

  Register('being_die',@lua_being_die);
  Register('being_apply_damage', @lua_being_apply_damage);
  Register('being_move',@lua_being_move);
  Register('being_send_missile',@lua_being_send_missile);

  Register('level_get_being',@lua_level_get_being);
  Register('level_summon',   @lua_level_summon);
  Register('level_explosion',@lua_level_explosion);
  Register('level_get_cell', @lua_level_get_cell);
  Register('level_set_cell', @lua_level_set_cell);

  try
    Register('require',         @lua_berserk_require);
    LoadFile('lua/main.lua');
  except
    on e : ELuaException do
      CritError( e.Message );
  end;



  LoadCells;
end;

procedure TBerserkLua.LoadCells;
var Amount, Count : DWord;
  function Resolve( const CellID : AnsiString ) : Word;
  begin
    if CellID = '' then Exit(0);
    Exit( IndexedTable['cells',CellID,'nid'] );
  end;
begin
  Amount := Table['cells','__counter'];
  SetLength( TerraData, Amount+1 );
  for Count := 1 to Amount do
  with TLuaTable.Create(Self,'cells',Count) do
  try
    with TerraData[ Count ] do
    begin
      Name      := GetString('name');
      Picture   := Ord(GetString('picture')[1]) + 256 * GetNumber('color');
      DarkPic   := Ord(GetString('dpicture')[1]) + 256 * GetNumber('dcolor');
      DR        := GetNumber('dr');
      BloodID   := Resolve( GetString('blood_id') );
      DestroyID := Resolve( GetString('destroy_id') );
      ActID     := Resolve( GetString('act_id') );
      SpriteB   := GetNumber('spriteb');
      Sprite    := GetNumber('sprite');
      MoveCost  := GetNumber('move_cost');
      Flags     := GetFlags('flags');

      TerraID[ GetString('id') ] := Count;
    end;
  finally
    Free;
  end;

end;

destructor TBerserkLua.Destroy;
begin
  inherited Destroy;
end;

procedure TBerserkLua.LuaError;
begin
  UI.Msg('LuaError: '+ErrorStr);
  Log('LuaError: '+ErrorStr);
end;

end.

