// @abstract(BerserkRL -- TBeing class unit)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Oct 16, 2006)
// @lastmod(Oct 22, 2006)
//
// This unit just holds the TBeing class -- a class representing the creatures
// you fight in Berserk. It is also a base class for the TPlayer class, that
// can be found in brplayer.pas.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

{$include brinclude.inc}

unit brbeing;
{$mode objfpc}
interface
uses SysUtils, vutil, vnode, vmath, vrltools, brdata, brui;


type TBeingHook  = ( Being_OnCreate, Being_OnAction, Being_OnDie );
     TBeingHooks = set of TBeingHook;
const BeingHooks : array[ TBeingHook ] of string = ( 'OnCreate', 'OnAction', 'OnDie' );


type

// The TBeing class represents all beings in the Berserk game. By itself it is
// a class representing monsters, but it also is the Parent class for TPlayer,
// the Player class.

{ TBeing }

TBeing = class(TVObject)
  // Being name
  Name       : string[20];
  // Identifier of the monster race. This number is an index for the BeingData
  // array found in brdata.pas.
  id         : string[20];
  // Target being nid
  lid        : Word;
  // The being array identifier. This is an index for the TLevel.Beings array.
  // It is important that it's value always matches the value in Beings -- that
  // is Level.Beings[x].nid = x.
  nid        : Word;
  // This is the coordinate of the Being. It must be always matched by a proper
  // entry in the TLevel.Map table, that is
  // Level.Beings[Level.Map[Pos.x,Pos.y].Being].Pos = Pos
  Position   : TCoord2D;
  // SpeedCount is a part of Berserks speed system. Each Tick, the SpeedCount is
  // increased by the value of Speed. If it reaches SPEEDLIMIT then the being
  // is allowed to take an action.
  SpeedCount : Word;
  // Speed is the amount that SpeedCount is increased each Tick.
  Speed      : Word;
  
  // AI Type of the being
  AI : Byte;
  
  // Target coord - used by targeting code
  TargetCoord : TCoord2D;
  // Target being nid - used by targeting code
  tid : Word;

  // Strength represents the Beings physical strenght. Mainly it affects damamge.
  Strength   : Word;
  // Endurance represents the Beings strudiness. If affects how the being
  // recovers from damage.
  Endurance  : Word;
  // Dexterity affects the beings chances to hit in combat, and to defend.
  Dexterity  : Word;
  // Willpower affects mainly the player -- it affects the chances of berserking,
  // and the psychical defence.
  Willpower  : Word;
  
  // Pain represents the way that damage dealt to the being affects it chances
  // to do actions. It is deducted from most "skill" rolls like attack or defense.
  Pain       : Word;
  
  // Ice based attacks cause freezing. Freezing reduces overall speed.
  Freeze     : Word;

  // HP is Hit Points. It's the amount of damage a being can take before it dies.
  HP         : Integer;
  // HPMax represents the initial and maximum value of HP.
  HPMax      : Word;
  // EN is Energy -- it represents how much stanima you have for heavy tasks like
  // special attacks.
  EN         : Integer;
  // ENMax represents the initial and maximum value of EN. The being regenerates
  // EN up to this value.
  ENMax      : Word;

  // WillCount determinates the speed of Willpower based regenerations (Pain
  // and Energy). Each turn a amount based on Willpower is added, and when
  // the count reaches WILLLIMIT then regeneration takes place.
  WillCount  : Word;

  
  // Used to calculate Knockback values.
  Weight     : Byte;
  
  // Reduces the damage by given amount.
  Armor      : Byte;
  
  // Picture is a value that holds both the ASCII gylph of the being, and it's
  // Color.
  Picture    : TPicture;
  
  // Graphical mode only : The sprite ID for the being.
  Sprite     : Word;
  
  // Graphical mode only : Values of 1.0,1.0,1.0 is the natural color
  ColorOverlay : TColorOverlay;
  
  // Used by explosions and the like to see if they have been already affected.
  Affected   : Boolean;
  
  // Counts the number of Attacks against a being.
  AttackCount: Byte;
  
  // Special abilities go into here. See constants starting with BF_
  Flags      : TFlags;
  
  // Facing for sprite rendering - used only in graphics mode
  FaceLeft   : Boolean;

  // Creates a new being. aid is the nid value of the being. It MUST be equal to
  // the TLevel.Beings index where the being was initialized. mid is the id value
  // of the creature, and index to the BeingData array. posX and posY are the
  // coordinates of the being on the map. This coordinate MUST be free of beings.
  constructor Create( aid : Word; const mid : AnsiString; pos : TCoord2D; lvl : Byte = 0 );
  
  // Chooses a suitable target based on the given AI set. Target is stored in tid,tx,ty.
  procedure ClosestTarget(AISet : TAITypeSet; Radius : Byte);
  
  // Sets the beings stats based on the id. In a bigger game this should be read
  // from a file, but for the scope of Berserk! it's just fine.
  procedure ApplyTemplate( const bid : AnsiString; lvl : byte );
  
  // The death method. Handles being death accordingly then runs Free/Destroy,
  // and destroys the being.
  procedure Die; virtual;
  
  // Action method -- called by Tick if enough SpeedCount is accumulated.
  procedure Action; virtual;
  
  // Tick method -- called by TLevel.Tick. Processes SpeedCount accumulation and
  // regeneration rates is applicable.
  procedure Tick;
  
  // Melee Attack against the Target being
  procedure Attack(Target : TBeing; AFlags : TFlags = []);
  
  // Attacks given square. Flags as in TBeing.Attack
  procedure Attack(Coord : TCoord2D; AFlags : TFlags = []);
  
  // Applies damage to the being. DR and other resistances are applied here.
  procedure ApplyDamage(Amount : Word; DType : Byte); virtual;
  
  // Scans the mx,my coordinates for a potential move. Returns the ScanCode (see
  // brdata.pas for values). Also returns 0 if move possible, or being.nid if
  // a being is on the target coord.
  function tryMove( NewCoord : TCoord2D ) : Word;
  
  // Moves the Being to the given coords. No error checking, so run TryMove
  // first.
  procedure Move( NewCoord : TCoord2D );
  
  // Returns true if the being is the player, false otherwise.
  function isPlayer : Boolean; virtual;
  
  // Returns wether he being is visible, based on the last TLevel.DrawLOS
  // calculation
  function isVisible : Boolean;
  
  // Returns wether lx,ly is in LOS. For the player it uses visibility map, for
  // the monsters it runs isEyeContact.
  function isInLOS( Coord : TCoord2D ) : Boolean;
  
  // Returns Dodge value
  function getDodge : Byte;
  
  // Returns beings name
  function getName : string;
  
  // Returns Parry value
  function getParry : Byte;
  
  // Returns damage dice based on Strength
  function getDamageDice : Byte; virtual;
  
  // Returns damage mod based on Strength
  function getDamageMod : ShortInt;
  
  // Knockes back the being in the given direction with the given strength.
  // Checks wether possible, so safe to call.
  procedure Knockback( Direction : TDirection;  Damage : Integer );
  
  // Sends a missile of type mtype towards the target.
  procedure SendMissile( Target : TCoord2D; mtype : Byte );
  
  // Returns the string shown in look mode and targeting.
  function LookDescribe : string;
  
  // Frees all allocated data. Also removes the being refrence from TLevel.Map
  // and TLevel.Beings structires.
  destructor Destroy; override;

  // Returns a property value
  function getProperty( PropertyID : Byte ) : Variant; virtual;

  // Sets a property value
  procedure setProperty( PropertyID : Byte; Value : Variant ); virtual;

  // Runs a hook with thing set to self
  function CallHook( Hook : TBeingHook; const Args : array of Variant ) : Variant;

  protected
  procedure UpdateFacing(NewX : Word);

  public
  property Properties[ PropertyID : Byte ] : Variant read getProperty write setProperty; default;

  private
  Hooks       : TBeingHooks;
end;

implementation
uses variants, vvision, vlua, lua, brlevel,brmain, brplayer;


{ TBeing }

constructor TBeing.Create(aid : Word;  const mid : AnsiString; pos : TCoord2D; lvl : Byte = 0 );
begin
  Affected := False;
  SpeedCount := 4000 + Random(500);
  WillCount  := 0;
  AttackCount := 0;
  nid := aid;
  Position := Pos;

  tid := 0;
  TargetCoord := ZeroCoord2D;

  FaceLeft := True;
  
  Pain := 0;
  Freeze := 0;

  Name := '';
  Hooks := [];

  ApplyTemplate(mid,lvl);
  Move( Position );
end;

procedure TBeing.ApplyTemplate( const bid : AnsiString; lvl : byte );
var Hook : TBeingHook;
begin
  with TLuaTable.Create( Berserk.Lua, 'beings', bid ) do
  try
    Picture    := Ord(GetString('picture')[1]) + 256 * GetNumber('color');
    Name       := GetString('name');
    Sprite     := GetNumber('sprite');
    Strength   := GetNumber('st');
    Dexterity  := GetNumber('dx');
    Willpower  := GetNumber('wp');
    Endurance  := GetNumber('en');

    AI         := GetNumber('ai');

    HPMax      := GetNumber('hp');
    HP         := HPMax;
    ENMax      := GetNumber('energy');
    EN         := ENMax;
    Speed      := GetNumber('speed');
    Armor      := GetNumber('armor');
    Weight     := GetNumber('weight');

    Flags      := GetFlags('flags');
    id         := GetString('id');
    lid        := GetNumber('nid');


    Hooks := [];
    for Hook := Low(TBeingHooks) to High(TBeingHooks) do
      if isFunction(BeingHooks[Hook]) then
        Include(Hooks,Hook);
  finally
    Free
  end;
  ColorOverlay := NOCOLOROVERLAY;
  CallHook( Being_OnCreate, [lvl] );
end;

procedure TBeing.ClosestTarget(AISet: TAITypeSet; Radius : Byte);
var scx, scy, t,ran : Word;
    Target : TCoord2D;
begin
  ran := 1000;
  for scx := Max(1,Position.x-Radius-1) to Min(MAP_MAXX,Position.x+Radius+1) do
    for scy := Max(1,Position.y-Radius-1) to Min(MAP_MAXY,Position.y+Radius+1) do
      begin
        Target := NewCoord2D( scx,scy );
        if Level.Being[ Target ] <> nil then
          if Level.Being[ Target ].AI in AISet then
            if isInLOS( Target ) then
            begin
              t := Distance( Target, Position );
              if t <= ran then
              begin
                TargetCoord := Target;
                ran  := t;
                tid := Level.Being[ Target ].nid;
              end;
            end;
      end;
  if ran = 1000 then
  begin
    TargetCoord := Position;
    tid := 0;
  end;
end;


procedure TBeing.Die;
begin
  if Player.tid = nid then Player.tid := 0;
  
  if HP <= 0 then
  begin
    Player.Kills.Add(lid);
    if isVisible then UI.Msg('The '+getName+' dies!');
  end else HP := 0;

  CallHook( Being_OnDie, [] );
  
  if (BF_DEFILED in Flags) and (EN = ENMAX) then
    Level.Explosion(Position,Green,2,3,50,DAMAGE_SPORE);

  if (not (TF_NOCORPSE in Level.getFlags( Position ))) and (not (BF_NOCORPSE in Flags)) then
  begin
    Level.Bleed( Position, 2 );
    Level.Cell[ Position ] := TerraID['bloody_corpse'];
  end;
  Free;
end;

procedure TBeing.Action;
var mx, my     : Integer;
    MoveResult : Word;
    TargetDist : Word;
    TargetVis  : Boolean;
    Count      : Word;
    MoveCoord  : TCoord2D;
    WCoord     : TCoord2D;
    Direction  : TDirection;
    Being      : TBeing;
begin
  AttackCount := 0;

  case AI of
    AI_CIVILIAN :
      begin
        ClosestTarget( [AI_MONSTER], 4 );
        if tid = 0 then
          TargetCoord := Level.Area.Center
        else
          TargetCoord := Position + NewDirection( TargetCoord, Position );
      end;
    AI_ALLY :
      begin
        ClosestTarget([AI_MONSTER],6);
        if tid = 0 then
        begin
          TargetCoord := Player.Position;
          if Distance( TargetCoord, Position ) < 3 then TargetCoord := Position;
        end;
      end;
    AI_MONSTER :
      begin
        ClosestTarget([AI_Civilian,AI_Ally,AI_Player],2);
        if tid = 0 then
        begin
          tid := 1;
          TargetCoord := Player.Position;
        end;
      end;
  end;

  TargetDist  := Distance( Position, TargetCoord );
  TargetVis   := Level.isEyeContact( Position, TargetCoord );

  if Being_OnAction in Hooks then
    if not CallHook( Being_OnAction, [TargetVis,TargetDist] ) then Exit;


  if BF_DEFILED in Flags then
  begin
    case EN of
      1..4  : Picture := Picture mod 256 + 256*Green;
      5..10 : Picture := Picture mod 256 + 256*LightGreen;
    end;
    {$IFDEF GRAPHICS}
      ColorOverlay[1] := 1.0 * (12-EN) / 12;
      ColorOverlay[3] := 1.0 * (12-EN) / 12;
    {$ENDIF}
  end;

   // Impale attack
  if (BF_IMPALE in Flags) and (EN > 20) and (TargetDist = 2) then
    if ((TargetCoord.x-Position.x) mod 2 = 0) and ((TargetCoord.y-Position.y) mod 2 = 0) then
    begin
      MoveCoord := Position + NewDirection( Position, TargetCoord );
      if tryMove( MoveCoord ) = 0 then
      begin
        EN -= 20;
        if isVisible then UI.Msg('The '+getName+' charges!');
        Move( MoveCoord );
        Attack(Level.Beings[tid],[AF_IMPALE]);
        SpeedCount -= 1500;
        Exit;
      end;
    end;

  // BF_REGEN regeneration.
  if (BF_REGEN in Flags) and (HP < HPMax) and (EN > 2) then
  begin
    EN -= 2;
    Inc(HP);
  end;
  
  
  mx := Sgn(TargetCoord.x-Position.x);
  my := Sgn(TargetCoord.y-Position.y);
  MoveCoord.Create( Position.X + mx, Position.Y + my );
  WCoord := ZeroCoord2D;
  MoveResult := TryMove( MoveCoord );
  
  if MoveResult = SCAN_BLOCK then WCoord := MoveCoord;
  if (MoveResult <> 0) and (MoveResult <> tid) then
  begin
    if mx = 0 then
    begin
      MoveCoord.x := Position.x+(Random(2)*2-1);
      MoveCoord.y := Position.y+my;
      MoveResult := TryMove( MoveCoord );
    end
    else
    begin
      MoveCoord.x := Position.x+mx;
      MoveCoord.y := Position.y;
      MoveResult := TryMove( MoveCoord );
    end;
  end;
  if (MoveResult = SCAN_BLOCK) and (WCoord.X = 0) then WCoord := MoveCoord;
  if (MoveResult <> 0) and (MoveResult <> tid) then
  begin
    if my = 0 then
    begin
      MoveCoord.x := Position.x+mx;
      MoveCoord.y := Position.y+(Random(2)*2-1);
      MoveResult := TryMove( MoveCoord );
    end
    else
    begin
      MoveCoord.x := Position.x;
      MoveCoord.y := Position.y+my;
      MoveResult := TryMove( MoveCoord );
    end;
  end;
  if (MoveResult = SCAN_BLOCK) and (WCoord.X = 0) then WCoord := MoveCoord;
  if MoveResult = 0 then Move( MoveCoord ) else
    if AI <> AI_Civilian then
    begin
      if MoveResult = tid then Attack(Level.Beings[tid])
      else if (WCoord.X <> 0) and (AI = AI_MONSTER) then Attack(WCoord);
    end;
  SpeedCount -= 1000; //TEMPORARY
end;

procedure TBeing.Tick;
begin
  if Freeze > Speed-10 then Freeze := Speed - 10;
  SpeedCount += Speed - Freeze;
  while SpeedCount > SPEEDLIMIT do Action;
  if (Pain > 0) or (Freeze > 0) or (EN < ENMAX) then
  begin
    WillCount += Willpower*Willpower;
    while WillCount > WILLLIMIT do
    begin
      Dec(WillCount,WILLLIMIT);
      if Pain > 0   then Dec(Pain);
      if Freeze > 0 then Dec(Freeze);
      if EN < ENMax then Inc(EN);
    end;
  end
  else WillCount := 0;
end;

procedure TBeing.Attack(Target : TBeing; AFlags : TFlags = []);
var ToHit   : Integer;
    ToWound : Integer;
    Defense : Integer;
    Damage  : Integer;
begin
  if Target = nil then Exit;
  UpdateFacing(Target.Position.X);
  if Target.isPlayer then Player.LastAttack := lid;
  ToHit := RollDice;

  if (TF_WATER in Level.getFlags(Position)) and (not (BF_WATER in Flags)) then Inc(ToHit,3);
  
  if BF_BERSERK in Flags then Dec(ToHit,3);

  if AF_SWEEP   in AFlags then Inc(ToHit,3);
  if AF_IMPALE  in AFlags then Dec(ToHit,1);

  if ToHit > Dexterity+3-Pain then
  begin
    if IsPlayer then UI.Msg('You miss the '+Target.getName+'.')
                else if Target.isPlayer then UI.Msg('The '+getName+' misses you.');
    Exit;
  end;
  
  Defense := 0;
  Defense := Target.getParry;

  if Defense = 0 then Defense := Target.getDodge;
  
  if Target.isPlayer then Defense += Player.DefBonus;

  ToWound := RollDice;

  if ToWound <= Defense-(Target.Pain div 2 + Target.Freeze div 5)-Target.AttackCount then
  begin
    if IsPlayer then UI.Msg('The '+Target.getName+' evades your blow.')
                else if Target.isPlayer then UI.Msg('You evade.');
    Exit;
  end;
  
//  msg.add('Dice('+IntToStr(getDamageDice)+',6)'+'+'+IntToStr(getDamageMod+2));

  Damage := Dice(getDamageDice,6)+getDamageMod+2; // The +1 represents the weapon
  if isPlayer then Damage += 3;                   // The +3 represents the Dragonslayer
  
  if AF_SWEEP  in AFlags then Damage -= 2;
  if AF_IMPALE in AFlags then Damage := Round(Damage * 1.5);

  if ToHit < 0 then Damage *= 2; // Critical Hit

  if ToHit < 0 then
    if Target.IsPlayer then UI.Msg('The '+getName+' critically hits you! '+GodStr('D:'+IntToStr(Damage)))
                       else if isPlayer then UI.Msg('You critically hit the '+Target.getName+'! '+GodStr('D:'+IntToStr(Damage)))
                                        else UI.Msg(getName+' critically hits the '+Target.getName+'. '+GodStr('D:'+IntToStr(Damage)))
  else
  if Target.IsPlayer then UI.Msg('The '+getName+' hits you! '+GodStr('D:'+IntToStr(Damage)))
                     else if isPlayer then UI.Msg('You hit the '+Target.getName+'! '+GodStr('D:'+IntToStr(Damage)))
                                      else if isVisible then UI.Msg('The '+getName+' hits the '+Target.getName+'. '+GodStr('D:'+IntToStr(Damage)));

  if not (AF_NOKNOCKBACK in Flags) then
    if AF_DKNOCKBACK in Flags
      then Target.Knockback( NewDirection( Position, Target.Position ),2*Damage)
      else Target.Knockback( NewDirection( Position, Target.Position ),Damage);
  Target.ApplyDamage(Damage,DAMAGE_SLASH);
end;

procedure TBeing.Attack( Coord : TCoord2D; AFlags : TFlags = [] );
begin
  UpdateFacing( Coord.x );
  if not Level.ProperCoord( Coord ) then Exit;
  Level.DamageTile( Coord, Dice(getDamageDice,6)+getDamageMod);
  if Level.Being[ Coord ] = nil then Exit;
  Attack(Level.Being[ Coord ], AFlags );
end;

procedure TBeing.Knockback( Direction : TDirection; Damage : Integer);
var Str   : Word;
    Coord : TCoord2D;
begin
  if Damage < 3 then Exit;
  Str := Max(Round(Int((Damage / KNOCKBACKVALUE) / (Weight / 10))),0);
  
  Coord := Position;

  while Str > 0 do
  begin
    if TryMove( Coord + Direction ) <> 0 then Break;
    Coord += Direction;
    Dec(Str);
  end;
  Move( Coord );
end;

// NOTE: Stray hits (hits that hit not the intended target) happen on a flat
// rate of 40% if the missile flies through that square.
//
// Hitting the target from range 4 always succeeds.
procedure TBeing.SendMissile( Target : TCoord2D; MType : Byte);
const HITDELAY  = 50;
var Dist : byte;
    Roll : integer;
    Ray  : TVisionRay;
    Scan      : Word;
    DrawDelay : Word;
    Tgt  : TBeing;
begin
  UpdateFacing(TargetCoord.X);
  Player.LastAttack := lid;
  Dist := Distance(Position,Target);
  DrawDelay := 10;
  case mtype of
    MTBOLT  : DrawDelay := 10;
    MTKNIFE : DrawDelay := 20;
    MTBOMB  : DrawDelay := 50;
    MTENERGY: DrawDelay := 20;
    MTICE   : DrawDelay := 30;
    MTSPORE : DrawDelay := 80;
  end;


  if Dist > 4 then
  begin
    Roll := RollDice;
    if Roll > Dexterity-((Dist-5) div 3) then
      if Dist < 9 then
        Target.RandomShift(1)
      else
        Target.RandomShift(2);
  end;

  Ray.Init(Level,Position,Target);
  repeat
    Ray.Next;
    if not Level.properCoord(Ray.GetC) then Exit;

    Scan := TryMove(Ray.GetC);
    if Scan > 500 then // Missile hits non-passable feature
    begin
      UI.SendMissile(Position,Ray.GetC,mtype);
      Break;
    end;
    if Scan > 0 then // Missile hits Being
    begin
      Tgt := Level.Beings[Scan];
      if (not (BF_RUNNING in Tgt.Flags)) or (Random(10) < 5) then
      if (Ray.GetC = TargetCoord) or (Dice(1,10) > 6) then
      begin
        UI.SendMissile(Position,Ray.GetC,mtype);
        if isPlayer then begin if isVisible then UI.Msg('You hit the '+Tgt.getName+'!') end
                    else if (Ray.GetC = Player.Position) then UI.Msg('You''re hit!');
        case mtype of
          MTBOLT   : Tgt.ApplyDamage(Dice(2,4),DAMAGE_PIERCE);
          MTKNIFE  : Tgt.ApplyDamage(Dice(getDamageDice,6)+getDamageMod+1,DAMAGE_PIERCE);
          MTENERGY : Tgt.ApplyDamage(Dice(1,6),DAMAGE_ENERGY);
          MTICE    : Tgt.ApplyDamage(Dice(2,5)+2,DAMAGE_FREEZE);
          MTSPORE  : if Tgt.isPlayer then
                       Tgt.ApplyDamage(Dice(3,6),DAMAGE_SPORE)
                     else
                       if not (BF_SPORERES in Tgt.Flags) then
                       begin
                         Tgt.ENMAX := 5;
                         Tgt.EN    := 0;
                         Include(Tgt.Flags,BF_DEFILED);
                       end;
        end;
        Break;
      end;
    end;

    if (mtype in [MTBOMB,MTSPORE]) then
      if Ray.Done then
      begin
        UI.SendMissile(Position,Ray.GetC,mtype);
        Break;
      end;
  until False;
  if mtype = MTBOMB then Level.Explosion(Ray.GetC,Red,4,6);
  if (mtype = MTSPORE) and (Level.Being[Ray.GetC] = nil) then
    Level.Summon('spore',Ray.GetC);
end;


function TBeing.LookDescribe : string;
var Wounds : string;
begin
  case Round(HP*100/HPMax) of
    -20..10  : Wounds := '@ralmost dead';
    11..30   : Wounds := '@rmortally wounded';
    31..50   : Wounds := '@Rseverely wounded';
    51..70   : Wounds := '@ywounded';
    71..90   : Wounds := '@ybruised';
    91..99   : Wounds := '@lscratched';
    100      : Wounds := '@Lunhurt';
    101..1000: Wounds := '@Bboosted';
  end;
  Exit(getName + ' (' +Wounds+'@>)');
end;

procedure TBeing.ApplyDamage(Amount : Word; DType : Byte);
begin
  if HP <= 0 then Exit;
  Amount := Max(1,Amount-Armor);
  if (BF_FLAMABLE in Flags) and (DType = DAMAGE_FIRE) then Amount *= 2;
  if BF_SKELETAL in Flags then
  begin
    if DType = DAMAGE_PIERCE then Amount := Amount div 3;
    if DType = DAMAGE_SPORE  then Exit;
  end;
  if DType = DAMAGE_FREEZE then
  begin
    Freeze += Amount;
    Amount := Amount div 2;
    {$IFDEF GRAPHICS}
    if Freeze = 0 then ColorOverlay := NOCOLOROVERLAY else
    begin
      ColorOverlay[1] := 0.01*(100-Freeze);
      ColorOverlay[2] := 0.01*(100-Freeze);
    end;
    {$ENDIF}
  end;
  HP -= Amount;
  if DType = DAMAGE_FIRE   then Freeze := Max(0,Freeze-2*Amount);
  Pain += Amount div 2;
  if not (BF_SKELETAL in Flags) then
    Level.Bleed(Position);
  if HP <= 0 then Die;
end;

function TBeing.TryMove( NewCoord : TCoord2D ) : Word;
begin
  TryMove := SCAN_OK;
  if not Level.properCoord( NewCoord ) then Exit( SCAN_OUT );
  if Level.Map[ NewCoord.x, NewCoord.y ].Being <> 0 then Exit( Level.Map[NewCoord.x,NewCoord.y].Being );
  if not Level.Area.Shrinked.Contains( NewCoord ) then Exit( SCAN_OUT );
  if TF_NOMOVE in Level.getFlags( NewCoord ) then TryMove += SCAN_BLOCK;
end;

procedure TBeing.Move( NewCoord : TCoord2D );
begin
  if Level.Being[ NewCoord ] <> nil then Exit;
  UpdateFacing( NewCoord.X );
  Level.Map[ Position.X, Position.Y ].Being := 0;
  Position := NewCoord;
  Level.Map[ Position.X, Position.Y ].Being := nid;
end;

function TBeing.isPlayer : Boolean;
begin
  Exit(False);
end;

function TBeing.isVisible : Boolean;
begin
  Exit(Level.Vision.isVisible( Position ))
end;

function TBeing.isInLOS( Coord : TCoord2D ): Boolean;
begin
  if isPlayer then Exit(Level.Vision.isVisible(Coord)) else Exit(Level.isEyeContact(Position,Coord));
end;

function TBeing.getDodge : Byte;
begin
  if isPlayer then Exit((Dexterity+Endurance) div 4+4)
              else Exit((Dexterity+Endurance) div 4+2);
end;

function TBeing.getName: string;
begin
  Exit(Name);
end;

function TBeing.getParry : Byte;
begin
  if isPlayer then Exit(((Dexterity+3) div 2)+4)
              else Exit(0);
end;

function TBeing.getDamageDice : Byte;
begin
  case Strength of
     0..12 : Exit(1);
    13..16 : Exit(2);
  else Exit(((Strength-13) div 4) + 2);
  end;
end;

function TBeing.getDamageMod : ShortInt;
begin
  case Strength of
    0..8 : Exit(-2);
    9 : Exit(-1);
    10: Exit(0);
    11: Exit(1);
    else Exit(((Strength-9) mod 4) - 1);
  end;
end;

destructor TBeing.Destroy;
begin
  if Level <> nil then
  begin
    Level.Map[Position.X,Position.Y].Being := 0;
    Level.Beings[nid]    := nil;
  end;
  inherited Destroy;
end;

function TBeing.getProperty(PropertyID: Byte): Variant;
begin
  case PropertyID of
    PROP_ID          : Exit( ID );
    PROP_NAME        : Exit( Name );
    PROP_POS_X       : Exit( Position.X );
    PROP_POS_Y       : Exit( Position.Y );
    PROP_TARGET_X    : Exit( TargetCoord.X );
    PROP_TARGET_Y    : Exit( TargetCoord.Y );
    PROP_SPEED_COUNT : Exit( SpeedCount );
    PROP_SPEED       : Exit( Speed );
    PROP_AI          : Exit( AI );
    PROP_ST          : Exit( Strength );
    PROP_EN          : Exit( Endurance );
    PROP_DX          : Exit( Dexterity );
    PROP_WP          : Exit( Willpower );
    PROP_PAIN        : Exit( Pain );
    PROP_FREEZE      : Exit( Freeze );
    PROP_HP          : Exit( HP );
    PROP_HP_MAX      : Exit( HPMax );
    PROP_ENERGY      : Exit( En );
    PROP_ENERGY_MAX  : Exit( EnMax );
    PROP_WEIGHT      : Exit( Weight );
    PROP_ARMOR       : Exit( Armor );
    PROP_PICTURE     : Exit( Chr(Picture mod 256) );
    PROP_COLOR       : Exit( Picture div 256 );
    PROP_SPRITE      : Exit( Sprite );
    PROP_OVERLAY_R   : Exit( ColorOverlay[1] );
    PROP_OVERLAY_G   : Exit( ColorOverlay[2] );
    PROP_OVERLAY_B   : Exit( ColorOverlay[3] );
    PROP_VISIBLE     : Exit( isVisible );

    else
      CritError( 'Unknown property #@1 requested!', [PropertyID] );
  end;
  Exit( Null );
end;

procedure TBeing.setProperty(PropertyID: Byte; Value: Variant);
begin
  case PropertyID of
    PROP_NAME        : Name := Value;
    PROP_SPEED_COUNT : SpeedCount := Value;
    PROP_SPEED       : Speed := Value;
    PROP_TARGET_X    : TargetCoord.X := Value;
    PROP_TARGET_Y    : TargetCoord.Y := Value;
    PROP_AI          : AI := Value;
    PROP_ST          : Strength := Value;
    PROP_EN          : Endurance := Value;
    PROP_DX          : Dexterity := Value;
    PROP_WP          : Willpower := Value;
    PROP_PAIN        : Pain := Value;
    PROP_FREEZE      : Freeze := Value;
    PROP_HP          : HP := Value;
    PROP_HP_MAX      : HPMax := Value;
    PROP_ENERGY      : En := Value;
    PROP_ENERGY_MAX  : EnMax := Value;
    PROP_WEIGHT      : Weight := Value;
    PROP_ARMOR       : Armor := Value;
    PROP_PICTURE     : Picture := (Picture div 256)*256 + Ord(VarToStr(Value)[1]);
    PROP_COLOR       : Picture := Picture mod 256 + Value * 256;
    PROP_SPRITE      : Sprite  := Value;
    PROP_OVERLAY_R   : ColorOverlay[1] := Value;
    PROP_OVERLAY_G   : ColorOverlay[2] := Value;
    PROP_OVERLAY_B   : ColorOverlay[3] := Value;
  else
    CritError( 'Unknown property #@1 requested!', [PropertyID] );
  end;
end;

function TBeing.CallHook( Hook : TBeingHook; const Args: array of Variant): Variant;
var NArgs,i : Integer;
    L       : PLua_State;
    msg     : AnsiString;
begin
  if not (Hook in Hooks) then Exit(VarNull);
  L := Berserk.Lua.LuaState;
  with TLuaTable.Create( Berserk.Lua, 'Being' ) do
  try
    lua_pushstring(L, 'call_hook');
    lua_rawget(L, -2);
    if not lua_isfunction( L, -1) then Exception.Create(BeingHooks[Hook]+' not found!');
    lua_pushlightuserdata(L, Self);
    lua_pushstring(L, ID);
    lua_pushstring(L, BeingHooks[Hook]);
    NArgs := High(Args);
    if NArgs >= 0 then
    for i:=0 to NArgs do
      lua_pushvariant(l, args[i]);
    if lua_pcall(l, NArgs+4, 1, 0) <> 0 then
    begin
      msg := lua_tostring(l, -1);
      lua_pop(l, 1);
      raise Exception.Create(msg);
    end;
    CallHook := lua_tovariant(L, -1);
    lua_pop(l, 1);
  finally
    Free;
  end
end;

procedure TBeing.UpdateFacing(NewX: Word);
begin
  if NewX > Position.X then FaceLeft := False;
  if NewX < Position.X then FaceLeft := True;
end;

end.

