#include <gtk/gtk.h>
#include <hildon/hildon.h>

#include <hal/libhal.h>
#include <dbus/dbus.h>

#include <stdio.h>
 
#include <libhildondesktop/libhildondesktop.h>

#include "battery.h"

#include "config.h"

HD_DEFINE_PLUGIN_MODULE(BatteryPlugin, battery_plugin, HD_TYPE_HOME_PLUGIN_ITEM)

static gboolean battery_plugin_on_timeout(gpointer data)
{
	//g_warning("Updating...");
	
	GtkWidget *widget = GTK_WIDGET(data);
	BatteryPlugin* battery = BATTERY_HOME_PLUGIN(widget);

	DBusConnection *connection;
	DBusError error;
	LibHalContext *ctx;
	dbus_error_init(&error);
	connection = dbus_bus_get(DBUS_BUS_SYSTEM,&error);
	
	if ( dbus_error_is_set(&error) )
	{
		printf("Unable to connect to DBus: %s\n",error.message);
		dbus_error_free(&error);
		return 1;
	}
	
	ctx = libhal_ctx_new();
	
	if ( !libhal_ctx_set_dbus_connection(ctx, connection) )
	{
		printf("Error %s\n",error.message);
		dbus_error_free(&error);
		return 1;
	}
	
	if ( !libhal_ctx_init(ctx, &error) )
	{
		if (battery->percent < 100)
			battery->percent += 1;
		else
			battery->percent = 0;
	
		//printf("Hal context initializing failure %s\n",error.message);
		//return 1;
	}
	else
	{
		battery->percent = libhal_device_get_property_int(ctx, "/org/freedesktop/Hal/devices/bme", "battery.charge_level.percentage", NULL);
	}
	
	//GtkWidget *widget = GTK_WIDGET(data);

	if (!widget->window)
		return TRUE;
	
	//BatteryPlugin* battery = BATTERY_HOME_PLUGIN(widget);

	GdkRegion *region = gdk_drawable_get_clip_region(widget->window);
	if (!region)
		return TRUE;
	    
	gdk_window_invalidate_region(widget->window, region, TRUE);
	gdk_window_process_updates(widget->window, TRUE);
	gdk_region_destroy(region);

	return TRUE;
}

/*static void battery_plugin_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
	requisition->width = 170;
	requisition->height = 102;
}

static void battery_plugin_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
	widget->allocation = *allocation;

	if (GTK_WIDGET_REALIZED(widget))
	{
		gdk_window_move_resize(
			widget->window,
			allocation->x, allocation->y,
			allocation->width, allocation->height
		);
	}
}*/

/* gint size: 1 = large, 2 = medium, 3 = small */
static void battery_change_size(BatteryPlugin* battery, gdouble size)
{
	/* calculate stuff */
	battery->bat_size = size;
	
	gint w = 170 * size;
	gint h = 102 * size;
	
	gtk_widget_set_size_request(GTK_WIDGET(battery), w, h);
	gtk_window_resize(GTK_WIDGET(battery), w, h);
}

static void battery_plugin_show_settings(GtkWidget *widget, BatteryPlugin* battery)
{
	//g_warning("Show settings\n");
	
	GtkWidget* dialog;
	dialog = gtk_dialog_new_with_buttons("Settings", NULL, 0, GTK_STOCK_SAVE, GTK_RESPONSE_OK, NULL);

	gtk_window_set_title(GTK_WINDOW(dialog), "Battery settings");
	
	GtkWidget* vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), vbox);
	
	GtkWidget* scale_size = gtk_hscale_new_with_range(0.6, 1.0, 0.1);
	gtk_scale_set_value_pos(GTK_SCALE( scale_size), GTK_POS_RIGHT);
	gtk_range_set_value(GTK_RANGE(scale_size), battery->bat_size);
	
	gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new("Size of the widget"), FALSE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), scale_size, FALSE, TRUE, 0);
	
	gtk_widget_show_all(GTK_WIDGET(dialog));
	
	//gtk_widget_set_size_request (GTK_WIDGET (battery), 103, 40);
	//gtk_window_resize (GTK_WINDOW (battery), 103, 40);
	
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
	{
		battery_change_size(battery, gtk_range_get_value(GTK_RANGE(scale_size)));
			
		/* Write settings to a file */
		GKeyFile* settings = NULL;
		settings = g_key_file_new();
		
		gchar* file = g_strdup_printf("%s/.battery-widget", g_get_home_dir());
		g_key_file_load_from_file(settings, file, G_KEY_FILE_KEEP_COMMENTS, NULL);
		
		g_key_file_set_double(settings, "general", "size", gtk_range_get_value(GTK_RANGE(scale_size)));
		
		gchar* data = g_key_file_to_data(settings, NULL, NULL);
		g_file_set_contents(file, data, -1, NULL);
		
		g_free(data);
		g_free(file);
		g_key_file_free(settings);
	}
	
	gtk_widget_destroy(dialog);
}

static void battery_plugin_current_desktop(BatteryPlugin* battery, gpointer data)
{
	if (battery->active)
	{
		//g_warning("Showing widget...");
		battery->timeout_handler = g_timeout_add(60000, battery_plugin_on_timeout, battery);
		battery_plugin_on_timeout(battery);
		battery->active = FALSE;
	}
	else
	{
		//g_warning("Hiding widget...");
		g_source_remove(battery->timeout_handler);
		battery->timeout_handler = 0;
		battery->active = TRUE;
	}
}

static void battery_plugin_init(BatteryPlugin* battery)
{
	battery->image_base   = cairo_image_surface_create_from_png("/opt/battery/batterybase.png");
	battery->image_green  = cairo_image_surface_create_from_png("/opt/battery/batteryhigh.png");
	battery->image_yellow = cairo_image_surface_create_from_png("/opt/battery/batterymed.png");
	battery->image_red    = cairo_image_surface_create_from_png("/opt/battery/batterylow.png");
	
	/* Load settings */
	GKeyFile* settings = NULL;
	settings = g_key_file_new();
	
	gchar* file = g_strdup_printf("%s/.battery-widget", g_get_home_dir());
	gboolean exists = g_key_file_load_from_file(settings, file, G_KEY_FILE_KEEP_COMMENTS, NULL);
	
	gdouble size;
	if (exists)
		size = g_key_file_get_double(settings, "general", "size", NULL);
	else
		size = 1.0;

	g_free(file);
	g_key_file_free(settings);
		
	battery_change_size(battery, size);
	
	hd_home_plugin_item_set_settings(battery, TRUE);
	g_signal_connect(G_OBJECT(battery), "show-settings", G_CALLBACK(battery_plugin_show_settings), (gpointer)battery);
	
	g_signal_connect(G_OBJECT(battery), "notify::is-on-current-desktop", G_CALLBACK(battery_plugin_current_desktop), battery);
	
	battery->active = TRUE;
}

static void battery_plugin_realize(GtkWidget *widget)
{
	GdkScreen* screen = gtk_widget_get_screen(widget);
	gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
	gtk_widget_set_app_paintable(widget, TRUE);

	GTK_WIDGET_CLASS(battery_plugin_parent_class)->realize(widget);
}

static gboolean battery_plugin_expose_event(GtkWidget *widget, GdkEventExpose *event)
{
	BatteryPlugin* battery = BATTERY_HOME_PLUGIN(widget);

	cairo_t *cr;

	cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));
	gdk_cairo_region(cr, event->region);
	cairo_clip(cr);

	cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
	cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 0.0);
	cairo_paint(cr);
	
	cairo_scale(cr, battery->bat_size, battery->bat_size);
	
	cairo_set_source_surface(cr, battery->image_base, 0, 0);
	cairo_paint(cr);

	float width = (cairo_image_surface_get_width(battery->image_green) / 100.0f) * battery->percent;
	float height = cairo_image_surface_get_height(battery->image_green);
	
	if (battery->percent > 66)
		cairo_set_source_surface(cr, battery->image_green, 12, 12);
	else if (battery->percent < 33)
		cairo_set_source_surface(cr, battery->image_red, 12, 12);
	else
		cairo_set_source_surface(cr, battery->image_yellow, 12, 12);
	
	cairo_rectangle(cr, 12, 12, width, height);
	cairo_clip(cr);
	cairo_paint(cr);

	/* Draw percent */
	cairo_reset_clip(cr);
	
	const char* percent_str = g_strdup_printf("%d %%", battery->percent);
	cairo_text_extents_t te;
	cairo_set_source_rgb (cr, 1.0, 1.0, 1.0);
	cairo_select_font_face (cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
	cairo_set_font_size (cr, 32.0);
	cairo_text_extents (cr, percent_str, &te);
	cairo_move_to (cr, 170 / 2 - te.width / 2 - te.x_bearing, 102 / 2 - te.height / 2 - te.y_bearing);
	cairo_show_text (cr, percent_str);
	g_free(percent_str);

	cairo_destroy(cr);

	return GTK_WIDGET_CLASS(battery_plugin_parent_class)->expose_event(widget, event);
}

static void battery_plugin_dispose (GObject *object)
{
	G_OBJECT_CLASS(battery_plugin_parent_class)->dispose(object);
}

static void battery_plugin_finalize(GObject *object)
{
	//g_print("removed... destroying images...");
	
	BatteryPlugin* battery = BATTERY_HOME_PLUGIN(object);

	cairo_surface_destroy(battery->image_base);
	cairo_surface_destroy(battery->image_green);
	cairo_surface_destroy(battery->image_yellow);
	cairo_surface_destroy(battery->image_red);
	
	if (battery->timeout_handler)
	{
		g_source_remove(battery->timeout_handler);
	}

	G_OBJECT_CLASS(battery_plugin_parent_class)->finalize(object);
}

static void battery_plugin_class_init(BatteryPluginClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	object_class->dispose = battery_plugin_dispose;
	object_class->finalize = battery_plugin_finalize;

	widget_class->realize = battery_plugin_realize;
	//widget_class->size_request = battery_plugin_size_request;
	//widget_class->size_allocate = battery_plugin_size_allocate;
	widget_class->expose_event = battery_plugin_expose_event;
}

static void battery_plugin_class_finalize(BatteryPluginClass* klass)
{

}
