#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm

try:
    import sqlite3
except ImportError, e:
    import sys
    sys.stderr.write(str(e) + "\n")
    sys.stderr.write("In ScratchBox, run with 'run-standalone.sh python2.5 (scriptname.py)'" "\n")
    sys.exit(1)

# profiling
import time
import datetime

class DataStorage(object):
    def __init__(self, path, dataSources):
        self.path = path
        self.con = sqlite3.connect(self.path)
        self.createTables()
        self.dataSources = dataSources
        self.storedTypes = {}
        self.debug = False
        for ds in self.dataSources:
            for k,v in ds.getProvidedValues().items():
                self.storedTypes[k] = v
        self.checkAndAddTypes(self.storedTypes)

    def checkAndAddTypes(self, types):
        cursor = self.con.cursor()
        for type, unit in types.items():
            try:
                cursor.execute('insert into observation_type (name, unit) values (?, ?)',
                               (type, unit))
            except sqlite3.IntegrityError:
                # It already exists
                pass
        self.con.commit()

    def addObservation(self, type, value, unixtime, commit=True):
        cursor = self.con.cursor()
        cursor.execute('insert into observation (type, value, timestamp) values ((select id from observation_type where name = ?), ?, ?)',
                       (type, value, int(unixtime)))
        if commit:
            self.con.commit()

    def __getObservations(self, cursor, type, start, end):
        started = time.time()
        if isinstance(type, (tuple, list)):
            cursor.execute('''select timestamp, value, observation_type.name
                              from observation
                              left join observation_type on (observation_type.id = observation.type)
                              where type in
                                    (select id from observation_type where name in (%s))
                              and timestamp between ? and ?
                              order by timestamp asc''' % ','.join(['?']*len(type)),
                           tuple(type) + (start, end))
        else:
            cursor.execute('''select timestamp, value, observation_type.name
                              from observation
                              left join observation_type on (observation_type.id = observation.type)
                              where type =
                                    (select id from observation_type where name = ?)
                              and timestamp between ? and ?
                              order by timestamp asc''',
                              (type, start, end))
        ret = cursor.fetchall()
        self.took('Query', started)
        return ret

    def getObservations(self, type, start, end):
        started = time.time()
        obs = self.__getObservations(self.con.cursor(), type, start, end)
        def intOrNone(input):
            try:
                return int(input)
            except TypeError:
                return None
        ret = map(lambda row: (row[0], intOrNone(row[1]), row[2]), obs)
        self.took('getObservations', started)
        return ret

    def getObservationsWithBreaks(self, type, start, end):
        started = time.time()
        cursor = self.con.cursor()
        ret = []
        breaks = self.__getObservations(cursor, 'internal.graph.break', start, end)
        breaks.append((end+1, None, 'internal.graph.break'))
        intervalStart = start
        for b in breaks:
            obs = self.__getObservations(cursor, type, intervalStart+1, b[0]-1)
            ret.append(map(lambda row: (row[0], int(row[1]), row[2]), obs))
            intervalStart = b[0]
        self.took('getObservationsWithBreaks', started)
        return ret

    def getPreviousObservationTime(self, timestamp):
        cursor = self.con.cursor()
        cursor.execute('select max(timestamp) from observation where timestamp < ?', (timestamp, ))
        try:
            return cursor.fetchone()[0]
        except:
            return None

    def getMaxCharge(self):
        cursor = self.con.cursor()
        cursor.execute('''select max(cast(value as integer))
                          from observation
                          where type = (select id from observation_type where name = ?)''',
                       ('hal.battery.reporting.design', ))
        try:
            return cursor.fetchone()[0]
        except:
            return None

    def getOldestObservationTime(self):
        cursor = self.con.cursor()
        cursor.execute('''select min(timestamp) from observation''')
        try:
            return cursor.fetchone()[0]
        except:
            return 0

    def __getObservationInInterval(self, cursor, interval, type, order):
        cursor.execute('''select timestamp, cast(value as integer)
                          from observation
                          where timestamp between ? and ?
                          and type = (select id from observation_type where name = ?)
                          order by timestamp %s
                          limit 1''' % order, (interval[0], interval[1], type))
        try:
            return cursor.fetchone()
        except:
            return None

    def getFirstObservationForType(self, interval, type):
        cursor = self.con.cursor()
        return self.__getObservationInInterval(cursor, interval, type, 'asc')

    def getLastObservationForType(self, interval, type):
        cursor = self.con.cursor()
        return self.__getObservationInInterval(cursor, interval, type, 'desc')

    def countObservations(self):
        cursor = self.con.cursor()
        cursor.execute('select count(id) from observation')
        return cursor.fetchone()[0]

    def deleteObservations(self, olderThan):
        cursor = self.con.cursor()
        cursor.execute('delete from observation where timestamp < ?',
                       (olderThan, ))
        self.con.commit()
        cursor.execute('vacuum')

    def setConfig(self, key, value):
        cursor = self.con.cursor()
        cursor.execute('insert or replace into config (name, value) values (?, ?)',
                       (key, value))
        self.con.commit()
        
    def getConfig(self, key, default = None):
        cursor = self.con.cursor()
        cursor.execute('select value from config where name = ?',
                       (key, ))
        try:
            return cursor.fetchone()[0]
        except:
            return default

    def createTables(self):
        tables = ['''
            config (
                id integer primary key autoincrement,
                name text not null,
                value text
            )''','''
            observation (
                id integer primary key autoincrement,
                type integer,
                value text,
                timestamp integer
            )''','''
            observation_type (
                id integer primary key autoincrement,
                name text,
                unit text,

                unique (name)
            )''']
        indexes = ['timestamp_index on observation (timestamp)',
                   'type_timestamp_index on observation (type, timestamp)']
        uniques = ['config_unique_index on config (name)']
        cursor = self.con.cursor()
        for table in tables:
            cursor.execute('create table if not exists %s' % table)
        for index in indexes:
            cursor.execute('create index if not exists %s' % index)
        for unique in uniques:
            cursor.execute('create unique index if not exists %s' % unique)

    def debugPrint(self, msg):
        if self.debug:
            print "%s %s" % (str(datetime.datetime.now()), msg)

    def took(self, name, started):
        if self.debug:
            self.debugPrint("%s: %dms" % (name, int(round((time.time() - started) * 1000))))

