#!/usr/bin/env python

#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm

import os

import gtk
import hildon

import pango

import beye

import time
import datetime

class MainWindow(hildon.StackableWindow):
    def __init__(self):
        hildon.StackableWindow.__init__(self)
        self.program = hildon.Program.get_instance()
        self.program.set_can_hibernate(True)

        self.connect("delete_event", lambda a, b: self.quit())
        self.program.add_window(self)

        self.layout = gtk.Fixed()
        self.add(self.layout)

        self.graph = Graph()
        self.layout.put(self.graph, 0, 0)

        gtk.set_application_name("battery-eye")
        self.setupMenu()

    def setupMenu(self):
        self.menu = hildon.AppMenu()
        self.filters = []
        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, None))
        self.filters[0].set_label('%')
        self.filters[0].set_mode(False)
        self.filters[0].connect("clicked",
                                lambda button: self.graph.changeDataset('hal.battery.charge_level.percentage'))

        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, self.filters[0]))
        self.filters[1].set_label('mAh')
        self.filters[1].set_mode(False)
        self.filters[1].connect("clicked", lambda button: self.graph.changeDataset('hal.battery.reporting.current'))

        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, self.filters[1]))
        self.filters[2].set_label('mV')
        self.filters[2].set_mode(False)
        self.filters[2].connect("clicked", lambda button: self.graph.changeDataset('hal.battery.voltage.current'))
        
        for f in self.filters:
            self.menu.add_filter(f)

        self.zoomButton = hildon.PickerButton(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.zoomButton.set_title('Zoom')
        self.zoomSelector = hildon.TouchSelector(text = True)
        self.zoomSelector.append_text('1 day')
        self.zoomSelector.append_text('2 days')
        self.zoomSelector.append_text('3 days')
        self.zoomSelector.append_text('5 days')
        self.zoomSelector.append_text('7 days')
        self.zoomSelector.set_active(0, 0)
        self.zoomSelector.connect("changed", self.changeZoom)
        self.zoomButton.set_selector(self.zoomSelector)

        self.menu.append(self.zoomButton)

        self.deleteButton = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.deleteButton.set_label('Delete data...')
        self.deleteButton.connect("clicked", self.deleteData)

        self.menu.append(self.deleteButton)

        self.aboutButton = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.aboutButton.set_label('About')
        self.aboutButton.connect("clicked", lambda x: AboutDialog(self))

        self.menu.append(self.aboutButton)

        self.menu.show_all()
        self.set_app_menu(self.menu)

    def deleteData(self, widget):
        dlg = gtk.Dialog()
        dlg.set_title("Delete all graph data older than")

        selector = hildon.TouchSelector(text = True)
        selector.append_text('3 days')
        selector.append_text('7 days')
        selector.append_text('14 days')
        selector.append_text('28 days')
        selector.append_text('Delete all')
        selector.connect("changed", self.doDelete, dlg)
        dlg.vbox.pack_start(selector, True, True)
        dlg.vbox.set_size_request(300, 300)
        dlg.set_transient_for(self)

        dlg.show_all()
        dlg.run()
        dlg.destroy()

    def doDelete(self, selector, user_data, parent):
        if selector.get_current_text() == 'Delete all':
            msg = 'All graph data will be deleted. Continue?'
        else:
            msg = 'Graph data older than %s will be deleted. Continue?' % selector.get_current_text()
        note = hildon.hildon_note_new_confirmation(parent, msg)
        response = note.run()
        if response == gtk.RESPONSE_OK:
            # Delete
            parent.response(gtk.RESPONSE_OK)
            now = int(time.time())
            values = {'3 days': now - 3 * 86400,
                      '7 days': now - 7 * 86400,
                      '14 days': now - 14 * 86400,
                      '28 days': now - 28 * 86400,
                      'Delete all': now + 1}
            dataStorage.deleteObservations(values[selector.get_current_text()])
            self.graph.updateData()
        note.destroy()
       
    def changeZoom(self, selector, user_data):
        values = {'1 day': 25, '2 days': 25*2, '3 days': 25*3, '5 days': 25*5, '7 days': 25*7}
        self.graph.changeVisibleHours(values[selector.get_current_text()])

    def run(self):
        self.show_all()
        self.graph.jump_to(self.graph.drawAreaInfo.canvasSize[0]-1, 0)
        try:
            gtk.main()
        except KeyboardInterrupt:
            print ""
            print "Ctrl-C"

    def quit(self):
        # set_app_menu(None) gets rid of glib assertion errors on exit
        self.set_app_menu(None)
        gtk.main_quit()

class DrawAreaInfo(object):
    def __init__(self, canvasSize, drawAreaSize, offset, windowSize):
        self.canvasSize = canvasSize
        self.drawAreaSize = drawAreaSize
        self.offset = offset
        self.windowSize = windowSize

class EraseableBuffer(object):
    def __init__(self, pixmap):
        self.pixmap = pixmap
        self.backbuffer = gtk.gdk.Pixmap(None, pixmap.get_size()[0], pixmap.get_size()[1], pixmap.get_depth())
        self.lastDrawParams = None # (x, y, w, h)

    def draw(self, gc, dst, x, y, w, h):
        if w == -1:
            w = self.pixmap.get_size()[0]
        if h == -1:
            h = self.pixmap.get_size()[1]
        self.backbuffer.draw_drawable(gc, dst,
                                      x, y,
                                      0, 0,
                                      w, h)
        dst.draw_drawable(gc, self.pixmap,
                          0, 0,
                          x, y,
                          w, h)
        self.lastDrawParams = (x, y, w, h)
        return gtk.gdk.Rectangle(x, y, w, h)

    def undraw(self, gc, dst):
        if self.lastDrawParams == None:
            return
        x, y, w, h = self.lastDrawParams
        dst.draw_drawable(gc, self.backbuffer,
                          0, 0,
                          x, y,
                          w, h)
        self.lastDrawParams = None
        return gtk.gdk.Rectangle(x, y, w, h)

    def get_size(self):
        return self.pixmap.get_size()

class Graph(hildon.PannableArea):

    def __init__(self):
        hildon.PannableArea.__init__(self)
        self.graphWidth = 800
        self.graphHeight = 424
        self.set_property('mov-mode', hildon.MOVEMENT_MODE_HORIZ)
        self.set_property('vscrollbar-policy', gtk.POLICY_NEVER)
        self.set_size_request(self.graphWidth, self.graphHeight)
        self.area = gtk.DrawingArea()
        self.add_with_viewport(self.area)

        self.area.connect("expose-event", self.redrawGraph)

        self.get_hadjustment().connect("value-changed", self.moveLabels)

        self.bgColor = self.area.get_colormap().alloc_color(0, 0, 0)
        self.dataColor = self.area.get_colormap().alloc_color(0, 255*256, 0)
        self.gridColor = self.area.get_colormap().alloc_color(40*256, 40*256, 40*256)
        self.labelsColor = self.area.get_colormap().alloc_color(120*256, 120*256, 120*256)

        self.chargingColor = self.area.get_colormap().alloc_color(25*256, 35*256, 0)
        self.fullColor = self.area.get_colormap().alloc_color(10*256, 40*256, 0)

        self.dataset = 'hal.battery.charge_level.percentage'
        self.visibleHours = 25

        self.requestJump = None

        self.updateData()

    def changeDataset(self, dataset):
        self.dataset = dataset
        self.updateData()

    def changeVisibleHours(self, visibleHours):
        self.visibleHours = visibleHours
        self.updateData()
        self.requestJump = self.drawAreaInfo.canvasSize[0]-1

    def updateData(self):
        if self.dataset == 'hal.battery.charge_level.percentage':
            yMin = 0
            yMax = 100
            yLabelsInterval = 20
            yLabels = ['0%', '20%', '40%', '60%', '80%', '100%']
        elif self.dataset == 'hal.battery.reporting.current':
            yMin = 0
            yMax = 1320
            yLabelsInterval = 200
            yLabels = ['0mAh', '200mAh', '400mAh', '600mAh', '800mAh', '1000mAh', '1200mAh', '1320mAh']
        elif self.dataset == 'hal.battery.voltage.current':
            yMin = 3600
            yMax = 4200
            yLabelsInterval = 100
            yLabels = ['3600mV', '3700mV', '3800mV', '3900mV', '4000mV', '4100mV', '4200mV']

        now = int(time.time())

        rawData = dataStorage.getObservationsWithBreaks(self.dataset, now - 30*86400, now)

        self.xMax = now + 2*3600

        firstObservationTime = None
        for segment in rawData:
            if len(segment) > 0:
                firstObservationTime = segment[0][0]
                break

        if firstObservationTime == None:
            self.xMin = self.xMax - self.visibleHours*3600
        else:
            self.xMin = min(firstObservationTime, self.xMax - self.visibleHours*3600)

        if self.visibleHours < 48:
            self.xGridInterval = 1
            self.xLabelInterval = 2
        elif self.visibleHours < 100:
            self.xGridInterval = 2
            self.xLabelInterval = 6
        else:
            self.xGridInterval = 6
            self.xLabelInterval = 24
           
        areaWidth = max((self.xMax - self.xMin) / 3600 * (self.graphWidth/self.visibleHours), self.graphWidth - 10)

        self.drawAreaInfo = DrawAreaInfo((areaWidth, self.graphHeight),
                                         (areaWidth, self.graphHeight-40),
                                         (0, 14),
                                         (self.graphWidth, self.graphHeight-10))

        self.pixmap = gtk.gdk.Pixmap(None, areaWidth, self.graphHeight, 16)
        self.area.set_size_request(areaWidth, self.graphHeight)

        self.datas = []
        for segment in rawData:
            self.datas.append(DataRendererValue(segment, self.drawAreaInfo,
                                                (self.xMin, self.xMax),
                                                (yMin, yMax)))
        self.chargeInfo = []
        for segment in self.getChargeInfo(now):
            self.chargeInfo.append(DataRendererStatus(segment, self.drawAreaInfo, (self.xMin, self.xMax)))
            
        self.yGrid = self.yGridSegments(self.drawAreaInfo, yMin, yMax, yLabelsInterval)
        self.xGrid = self.xGridSegments(self.drawAreaInfo, self.xMin, self.xMax, self.xGridInterval)
        self.yLabels = self.createYLabels(yLabels)
        self.xLabels = self.createXLabels(self.xMin, self.xMax, self.xLabelInterval)

        gc = self.pixmap.new_gc()

        gc.foreground = self.bgColor
        self.pixmap.draw_rectangle(gc, True,
                                   0, 0,
                                   self.drawAreaInfo.canvasSize[0],
                                   self.drawAreaInfo.canvasSize[1])

        for d in self.chargeInfo:
            d.render(self.pixmap, {(True, True): self.fullColor, (True, False): self.chargingColor})
        
        gc.foreground = self.gridColor
        self.pixmap.draw_segments(gc, self.yGrid)
        self.pixmap.draw_segments(gc, self.xGrid)

        for d in self.datas:
            d.render(self.pixmap, self.dataColor)

        gc.foreground = self.labelsColor
        self.drawYLabels(gc, self.yLabels, self.yGrid)
        self.drawXLabels(gc, self.xLabels, self.xLabelInterval)

        self.area.queue_draw()

    def redrawGraph(self, widget, event):
        self.gc = widget.style.fg_gc[gtk.STATE_NORMAL]

        area = event.area

#        print "Redraw (%d, %d)->(%d, %d)" % (area.x, area.y, area.x + area.width, area.y + area.height)

        widget.window.draw_drawable(self.gc, self.pixmap,
                                    area.x, area.y,
                                    area.x, area.y,
                                    area.width, area.height)
        if (self.requestJump != None):
            self.jump_to(self.requestJump, 0)
            self.requestJump = None

    def moveLabels(self, adjustment):
        area = self.drawYLabels(self.pixmap.new_gc(), self.yLabels, self.yGrid)
        self.area.queue_draw_area(area.x, area.y, area.width, area.height)

    def createYLabels(self, labels):
        pContext = self.area.create_pango_context()
        pContext.set_font_description(pango.FontDescription('sans normal 10'))
        labelPixmaps = []
        for label in labels:
            layout = pango.Layout(pContext)
            layout.set_text(label)
            pixmap = gtk.gdk.Pixmap(None, layout.get_pixel_size()[0],
                                          layout.get_pixel_size()[1], 16)
            gc = pixmap.new_gc()
            pixmap.draw_rectangle(gc, True, 0, 0, pixmap.get_size()[0], pixmap.get_size()[1])
            gc.foreground = self.labelsColor
            pixmap.draw_layout(gc, 0, 0, layout)
            labelPixmaps.append(EraseableBuffer(pixmap))
        return labelPixmaps

    def unionRects(self, rects):
        r = None
        for rect in rects:
            if rect != None:
                if r == None:
                    r = rect
                else:
                    r = r.union(rect)
        return r

    def clearYLabels(self, gc, pixmaps):
        invalidated = []
        for pixmap in pixmaps:
            invalidated.append(pixmap.undraw(gc, self.pixmap))
        return self.unionRects(invalidated)

    # Returns a gtk.gdk.Rectangle describing the area that is invalided
    # by draws.
    def drawYLabels(self, gc, pixmaps, yGrid):
        wPos = int(self.get_hadjustment().get_value())
        invalidated = []
        for pixmap, position in zip(pixmaps, yGrid):
            invalidated.append(pixmap.undraw(gc, self.pixmap))
            invalidated.append(pixmap.draw(gc, self.pixmap,
                               self.drawAreaInfo.windowSize[0] - pixmap.get_size()[0] + wPos, position[1]-15,
                               -1, 15))
        return self.unionRects(invalidated)
            
    def yGridSegments(self, areaInfo, yMin, yMax, yInterval):
        y = []
        j = yMin
        while j < yMax:
            y.append(j)
            j += yInterval
        y.append(yMax)
        points = self.rescaleData(zip(range(len(y)), y), areaInfo, 0, len(y), yMin, yMax)
        ret = []
        for trash,y in points:
            ret.append((areaInfo.offset[0], y, areaInfo.drawAreaSize[0]-1, y))
        return ret

    def getChargeInfo(self, now):
        rawData = dataStorage.getObservations(('hal.battery.rechargeable.is_charging',
                                               'hal.battery.rechargeable.is_discharging',
                                               'internal.graph.break'),
                                               now - 30*86400, now)
        if len(rawData) == 0:
            return []
        segment = []
        segments = [segment]
        for point in rawData:
            segment.append(point)
            if point[2] == 'internal.graph.break':
                segment = []
                segments.append(segment)

        # Some serious list processing voodoo follows.

        statuses = [] #[((charging, discharging), start, end), (...), ...]
        for segment in [x for x in segments if len(x) > 0]:
            intervalStart = segment[0][0]
            if segment[-1][2] == 'internal.graph.break':
                intervalEnd = dataStorage.getPreviousObservationTime(segment[-1][0])
            else:
                intervalEnd = dataStorage.getPreviousObservationTime(now)
            if intervalEnd == None or intervalStart >= intervalEnd:
                continue

            statusList = []
            lastStatus = ((False, True), intervalStart, intervalStart)
            for point in segment:
                if point[2] == 'hal.battery.rechargeable.is_charging':
                    if point[1] != lastStatus[0][0]:
                        if point[0] > lastStatus[2]:
                            statusList.append((lastStatus[0], lastStatus[1], point[0]))
                        lastStatus = ((bool(point[1]), lastStatus[0][1]), point[0], point[0])
                    else:
                        lastStatus = (lastStatus[0], lastStatus[1], point[0])
                elif point[2] == 'hal.battery.rechargeable.is_discharging':
                    if point[1] != lastStatus[0][1]:
                        if point[0] > lastStatus[2]:
                            statusList.append((lastStatus[0], lastStatus[1], point[0]))
                        lastStatus = ((lastStatus[0][0], bool(point[1])), point[0], point[0])
                    else:
                        lastStatus = (lastStatus[0], lastStatus[1], point[0])
            statusList.append((lastStatus[0], lastStatus[1], intervalEnd))
            statuses.append([x for x in statusList if x[1] < x[2]])
        return [x for x in statuses if len(x) > 0]

#        for segment in statuses:
#            print ""
#            for status in segment:
#                if status[0] == (True, False):
#                    descr = "Charging"
#                elif status[0] == (True, True):
#                    descr = "Full"
#                elif status[0] == (False, True):
#                    descr = "Discharging"
#                else:
#                    descr = "W00t? No battery?"
#                print "%s (%d minutes) %s - %s" % (descr, (status[2] - status[1]) / 60,
#                                                   datetime.datetime.fromtimestamp(status[1], beye.utc).astimezone(beye.local),
#                                                   datetime.datetime.fromtimestamp(status[2], beye.utc).astimezone(beye.local))
       
    def createXLabels(self, xMin, xMax, modulo):
        hours = self.getHoursInInterval(xMin, xMax, 1)
        pContext = self.area.create_pango_context()
        pContext.set_font_description(pango.FontDescription('sans normal 10'))
        labelPixmaps = []
        for hour in hours:
            if hour.hour % modulo == 0:
                layout = pango.Layout(pContext)
                if hour.hour == 0:
                    layout.set_text(hour.strftime('%d. %b'))
                else:
                    layout.set_text(hour.strftime('%H:%M'))
                pixmap = gtk.gdk.Pixmap(None, layout.get_pixel_size()[0],
                                              layout.get_pixel_size()[1], 16)
                gc = pixmap.new_gc()
                pixmap.draw_rectangle(gc, True, 0, 0, pixmap.get_size()[0], pixmap.get_size()[1])
                gc.foreground = self.labelsColor
                pixmap.draw_layout(gc, 0, 0, layout)
                labelPixmaps.append(pixmap)
        return labelPixmaps

    def drawXLabels(self, gc, pixmaps, modulo):
        hours = self.getHoursInInterval(self.xMin, self.xMax, 1)
        labeledHours = []
        for hour in hours:
            if hour.hour % modulo == 0:
                labeledHours.append(hour)
        y = self.drawAreaInfo.offset[1] + self.drawAreaInfo.drawAreaSize[1] + 1
        for pixmap, hour in zip(pixmaps, labeledHours):
            x = self.rescaleData([(time.mktime(hour.timetuple()), 1)], self.drawAreaInfo, self.xMin, self.xMax, 0, 1)[0][0]
            self.pixmap.draw_drawable(gc, pixmap,
                                      0, 0,
                                      x - int(pixmap.get_size()[0]/2),
                                      y,
                                      -1, -1)

    def xGridSegments(self, areaInfo, xMin, xMax, modulo = 1):
        hours = self.getHoursInInterval(xMin, xMax, modulo)
        points = []
        for hour in hours:
            points.append(time.mktime(hour.timetuple()))
        points = self.rescaleData(zip(points, range(len(points))), areaInfo, xMin, xMax, 0, 1)
        ret = []
        for x,thrash in points:
            ret.append((x, areaInfo.offset[1] + 1, x, areaInfo.offset[1] + areaInfo.drawAreaSize[1]-1))
        return ret

    def rescaleData(self, points, areaInfo, xMin, xMax, yMin, yMax):
        if len(points) == 0:
            return []
        dxOffset = xMin
        dxMax = xMax
        dxDistance = dxMax - dxOffset
        yDistance = yMax - yMin
        if dxDistance <= 0 or yDistance <= 0:
            # Let's not divide by zero
            return []

        def rescalePoint(point):
            x = (areaInfo.drawAreaSize[0]-1) * float(point[0] - dxOffset) / dxDistance
            y = (areaInfo.drawAreaSize[1]-1) * float(point[1] - yMin) / yDistance
            return (int(x) + areaInfo.offset[0], areaInfo.drawAreaSize[1] - int(y) + areaInfo.offset[1])

        return map(rescalePoint, points)
  
    def getHoursInInterval(self, start, end, modulo):
        hour = datetime.timedelta(0, 0, 0, 0, 0, 1)
        
        startDt = self.nextHour(datetime.datetime.fromtimestamp(start, beye.utc).astimezone(beye.local))
        endDt = self.nextHour(datetime.datetime.fromtimestamp(end, beye.utc).astimezone(beye.local))
        
        current = startDt
        ret = []
        while current < endDt:
            if current.hour % modulo == 0:
                ret.append(current)
            current += hour
        return ret

    def nextHour(self, dt):
        deltaBack = datetime.timedelta(0, dt.second, dt.microsecond, 0, dt.minute)
        deltaForward = datetime.timedelta(0, 0, 0, 0, 0, 1)
        return dt + deltaForward - deltaBack

class AboutDialog(gtk.AboutDialog):
    def __init__(self, parent):
        gtk.Dialog.__init__(self)
        self.set_name("battery-eye")
        self.set_logo_icon_name("battery-eye")

        self.set_copyright(beye.__copyright__)
        self.set_version(beye.__version__)
        self.set_website(beye.__url__)

        stat = os.stat(beye.dbFilePath)
        self.set_comments('Database size: %dkB, %d records' % (stat[6] / 1024, dataStorage.countObservations()))

        #gtk.about_dialog_set_url_hook(lambda dialog, link, user_data: webbrowser.open(link), None)
        self.set_license('''The battery-eye software is distributed under GPL version 3. For more information, see
<http://www.gnu.org/licenses/>

The battery-eye icon is a derivative based on original work by
rutty <http://www.flickr.com/photos/rutty/438995413>, and is distributed under a Creative Commons license:
<http://creativecommons.org/licenses/by-nc-sa/2.0/>

battery-eye is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
''')
        self.set_wrap_license(True)
        
        self.show_all()
        self.run()
        self.destroy()

class DataRendererValue(object):
    def __init__(self, rawData, areaInfo, timeBounds, valueBounds):
        self.rawData = rawData
        self.areaInfo = areaInfo
        self.timeBounds = timeBounds
        self.valueBounds = valueBounds
        self.scaledData = self.rescaleData(self.rawData, self.areaInfo, self.timeBounds, self.valueBounds)

    def rescaleData(self, points, areaInfo, xBounds, yBounds):
        if len(points) == 0:
            return []
        xDistance = xBounds[1] - xBounds[0]
        yDistance = yBounds[1] - yBounds[0]
        if xDistance <= 0 or yDistance <= 0:
            # Let's not divide by zero
            return []

        scaleX = float(areaInfo.drawAreaSize[0]-1) / xDistance
        scaleY = float(areaInfo.drawAreaSize[1]-1) / yDistance

        offX = self.areaInfo.offset[0]
        offY = self.areaInfo.offset[1]

        def rescalePoint(point):
            x = scaleX * float(point[0] - xBounds[0])
            y = areaInfo.drawAreaSize[1] - scaleY * float(point[1] - yBounds[0])
            return (int(x) + offX, int(y) + offY)

        return map(rescalePoint, points)

    def render(self, drawable, color):
        gc = drawable.new_gc(foreground=color, line_width=1, join_style=gtk.gdk.JOIN_ROUND)
        color2 = drawable.get_colormap().alloc_color(int(color.red*0.5),
                                                     int(color.green*0.5),
                                                     int(color.blue*0.5))
        gc2 = drawable.new_gc(foreground=color2, line_width=2, join_style=gtk.gdk.JOIN_ROUND)

        if len(self.scaledData) == 1:
            drawable.draw_points(gc2, self.scaledData)
            drawable.draw_points(gc, self.scaledData)
        elif len(self.scaledData) > 1:
            drawable.draw_lines(gc2, self.scaledData)
            drawable.draw_lines(gc, self.scaledData)

class DataRendererStatus(object):
    def __init__(self, rawData, areaInfo, timeBounds):
        self.rawData = rawData
        self.areaInfo = areaInfo
        self.timeBounds = timeBounds
        self.scaledData = self.rescaleData(self.rawData, self.areaInfo, self.timeBounds)

    def rescaleData(self, statuses, areaInfo, xBounds):
        if len(statuses) == 0:
            return []
        xDistance = xBounds[1] - xBounds[0]
        if xDistance <= 0:
            # Let's not divide by zero
            return []

        scaleX = float(areaInfo.drawAreaSize[0]-1) / xDistance

        offX = self.areaInfo.offset[0]

        def rescaleStatus(status):
            x1 = scaleX * float(status[1] - xBounds[0])
            x2 = scaleX * float(status[2] - xBounds[0])
            return (status[0], int(x1), int(x2))

        return map(rescaleStatus, statuses)

    def render(self, drawable, colorMap):
        for status in self.scaledData:
            if (not colorMap.has_key(status[0])) or colorMap[status[0]] == None:
                continue
            gc = drawable.new_gc(foreground=colorMap[status[0]])
            drawable.draw_rectangle(gc, True,
                                    status[1], self.areaInfo.offset[1],
                                    status[2] - status[1], self.areaInfo.offset[1] + self.areaInfo.drawAreaSize[1] - 14)

if __name__ == "__main__":
    dataStorage = beye.DataStorage(beye.setupAndGetDbPath(), [beye.DataSourceHal()])
    mainwin = MainWindow()
    mainwin.run()
