#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm


import sqlite3

class DataStorage(object):
    def __init__(self, path, dataSources):
        self.path = path
        self.con = sqlite3.connect(self.path)
        self.createTables()
        self.dataSources = dataSources
        self.storedTypes = {}
        for ds in self.dataSources:
            for k,v in ds.getProvidedValues().items():
                self.storedTypes[k] = v
        self.checkAndAddTypes(self.storedTypes)

    def checkAndAddTypes(self, types):
        cursor = self.con.cursor()
        for type, unit in types.items():
            try:
                cursor.execute('insert into observation_type (name, unit) values (?, ?)',
                               (type, unit))
            except sqlite3.IntegrityError:
                # It already exists
                pass
        self.con.commit()

    def addObservation(self, type, value, unixtime):
        cursor = self.con.cursor()
        cursor.execute('insert into observation (type, value, timestamp) values ((select id from observation_type where name = ?), ?, ?)',
                       (type, value, int(unixtime)))
        self.con.commit()

    def getObservations(self, type):
        cursor = self.con.cursor()
        if isinstance(type, (tuple, list)):
            cursor.execute('''select timestamp, value, observation_type.name
                              from observation
                              left join observation_type on (observation_type.id = observation.type)
                              where type in
                                    (select id from observation_type where name in (%s))
                              order by timestamp asc''' % ','.join(['?']*len(type)),
                           type)
        else:
            cursor.execute('''select timestamp, value, observation_type.name
                              from observation
                              left join observation_type on (observation_type.id = observation.type)
                              where type =
                                    (select id from observation_type where name = ?)
                              order by timestamp asc''',
                              (type, ))
        ret = []
        for row in cursor.fetchall():
            ret.append((int(row[0]), int(row[1]), row[2]))
        return ret
        
    def createTables(self):
        tables = ['''
            config (
                id integer primary key autoincrement,
                name text not null,
                value text
            )''','''
            observation (
                id integer primary key autoincrement,
                type integer,
                value text,
                timestamp integer
            )''','''
            observation_type (
                id integer primary key autoincrement,
                name text,
                unit text,

                unique (name)
            )''']
        cursor = self.con.cursor()
        for table in tables:
            cursor.execute('create table if not exists %s' % table)


