/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2010  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QTimer>
#include <QDebug>
#include <QMessageBox>


MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    // load settings
    itsSettings = new Settings();

    // setup audio monitor
    itsAudioMonitor = new AudioMonitor(itsSettings, this);
    connect(itsAudioMonitor, SIGNAL(update(int, int)), this, SLOT(refreshAudioData(int, int)));
    itsMonitorActive = false;
    itsNotificationPending = false;

    // setup phone call handler
    itsUserNotifier = new UserNotifier(itsSettings, this);
    connect(itsUserNotifier, SIGNAL(notifyFinished()), this, SLOT(notifyFinished()));

    // setup GUI
    ui->setupUi(this);

    // connect UI dialogs
    connect(ui->actionSettings, SIGNAL(triggered()), itsSettings, SLOT(ShowDialog()));

    // setup default UI values
    ui->lineEdit_phone->setText(itsSettings->itsPhonenumber);

    // setup graphs
    itsAudioLevelGraphics = new AudioLevelGraphicsScene(ui->graphicsViewLevel, itsSettings, this);
    itsAudioCounterGraphics = new AudioLevelGraphicsScene(ui->graphicsViewCounter, itsSettings, this);

    // start audio capturing
    startAudio();
}


MainWindow::~MainWindow()
{
    // get graphics dimensions
    qDebug() << ui->graphicsViewLevel->geometry().width() << ui->graphicsViewLevel->geometry().height();

    // save settings
    itsSettings->Save();

    delete ui;
}


void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}


void MainWindow::on_actionAbout_triggered()
{
    QMessageBox msgBox;
    msgBox.setWindowTitle("Babyphone");
    msgBox.setText(tr("The babyphone is a baby monitoring application that measures the audio noise level. If the volume exceeds a configurable threshold the phone initiates a call to a predefined phone number (the parents).\n\n(c) 2010, Roman Morawek"));
    msgBox.exec();
}


void MainWindow::on_lineEdit_phone_textChanged()
{
    // update settings
    itsSettings->itsPhonenumber = ui->lineEdit_phone->text();
}


void MainWindow::on_pushButton_clicked()
{
    if (itsMonitorActive) {
        itsMonitorActive = false;

        // update GUI
        ui->pushButton->setText(tr("Inactive - Click to start"));
        ui->menuSettings->setEnabled(true);
        ui->lineEdit_phone->setEnabled(true);
    }
    else {
        // check whether we have a valid phone number
        if (itsSettings->itsPhonenumber.size() > 0) {
            itsMonitorActive = true;

            // update GUI
            ui->pushButton->setText("Active - Click to stop");
            ui->menuSettings->setEnabled(false);
            ui->lineEdit_phone->setEnabled(false);
        }
        else {
            // incomplete settings
            // open dialog to warn user
            QMessageBox::warning(this, tr("Babyphone"),
                  tr("No valid parent's phone number set. Please adjust the application settings."));

            // reset the button color again
            ui->pushButton->setChecked(false);
        }
    }
}


void MainWindow::notifyFinished()
{
    // notifcation finished
    itsNotificationPending = false;

    // update GUI
    ui->pushButton->setEnabled(true);

    // reset audio monitor warning
    itsAudioMonitor->itsCounter = 0;

    // start monitor after timeout
    QTimer::singleShot(itsSettings->itsRecallTimer*1000, this, SLOT(startAudio()));
}


void MainWindow::startAudio()
{
    itsAudioMonitor->start();

    // setup safety timer
    QTimer::singleShot(itsSettings->AUDIO_ON_TIMER, this, SLOT(stopAudio()));
}


void MainWindow::stopAudio()
{
    // suspend audio monitoring
    itsAudioMonitor->stop();

    // in case of notification, we get an event on its end
    // otherwise we must restart automatically via a timer
    if (!itsNotificationPending)
        QTimer::singleShot(itsSettings->AUDIO_OFF_TIMER, this, SLOT(startAudio()));
}


void MainWindow::refreshAudioData(int counter, int value)
{
    // update GUI
    itsAudioLevelGraphics->AddValue(value);
    itsAudioCounterGraphics->AddValue(counter);

    // check for noise
    if (itsMonitorActive && (counter > itsSettings->THRESHOLD_VALUE)) {
        qDebug() << tr("Audio threshold reached. Notifying user.");

        // store event
        itsNotificationPending = true;

        // update GUI
        ui->pushButton->setEnabled(false);

        // stop monitoring
        stopAudio();

        // notify user
        if (itsUserNotifier->Notify() == false)
        {
            // the notify command yielded an error
            QMessageBox *msgBox = new QMessageBox(QMessageBox::Critical, tr("Babyphone"),
                  tr("Could not establish phone call. Please check the phone number settings."));
            // the message box needs a timeout to keep the application active for retry
            QTimer *msgBoxCloseTimer = new QTimer(msgBox);
            msgBoxCloseTimer->setInterval(itsSettings->MSG_BOX_TIMEOUT);
            msgBoxCloseTimer->setSingleShot(true);
            connect(msgBoxCloseTimer, SIGNAL(timeout()), msgBox, SLOT(accept())); // or reject
            msgBoxCloseTimer->start();
            msgBox->exec();

            // restart audio capturing
            notifyFinished();
        }
    }
}
