#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

import gtk
import gobject
#import hildon
import pango
import logging
import urllib

import dbus

# for tests
import time
import sys

from atabake.lib.errors import *
from atabake.lib.media_engine  import MediaEngine
from atabake.lib.player_session import PlayerSession

logging.basicConfig(level=logging.DEBUG,
                format=("### %(asctime)s %(name)-18s \t%(levelname)-8s "
                        "\t%(message)s"),datefmt="%Y-%m-%d %H:%M:%S")

log = logging.getLogger("atabake.player_ui")

# FIXME: replicating
(ATABAKE_STATE_NONE,
 ATABAKE_STATE_PLAYING,
 ATABAKE_STATE_PAUSED,
 ATABAKE_STATE_ERROR,
 ATABAKE_STATE_HOLDED) = range(5)

#class PlayerUI(hildon.Window):
class PlayerUI(gtk.Window):

    def __init__(self, app=None):
        #hildon.Window.__init__(self)
        gtk.Window.__init__(self)
        self.resize(600, 400)

        # FIXME: fullscreen test
        self.idle_call = 0

        self.app = app
        self.url = None
        #app.add_window(self)
        self.actions = {}
        self.is_fullscreen = False
        self.last_played = None

        self.accel_group = gtk.AccelGroup()
        self.add_accel_group(self.accel_group)
        self.action_group = gtk.ActionGroup("PlayerUI")
        self._setup_actions()

        self.media_length = None
        self.media_title = None

        self._setup_toolbar()
        self._setup_ui()
        self._setup_connections()

        # setup connection to media-engine
        self._setup_media_engine()
        self._setup_media_player()

    def _setup_media_engine(self):
        self.bus = dbus.SessionBus()
        media_engine_obj = self.bus.get_object(MediaEngine.DBUS_SERVICE_NAME,
                                               MediaEngine.DBUS_OBJ_PATH,
                                               introspect=False)

        self.media_engine_iface = dbus.Interface(media_engine_obj,
                                                 MediaEngine.DBUS_IFACE)

        # creating None player
        self.player_iface = None

    def _setup_media_player(self):
        self.player_obj_path = self.media_engine_iface.create_session("")
        log.info("Player dbus obj path: %s" % self.player_obj_path)

        # instanciate player

        # resolving unique bus name for player session
        bus_object = self.bus.get_object('org.freedesktop.DBus',
                                         '/org/freedesktop/DBus',
                                         introspect=False)
        self.named_service = bus_object.GetNameOwner(
                               PlayerSession.DBUS_SERVICE_NAME,
                               dbus_interface='org.freedesktop.DBus')

        player_obj = self.bus.get_object(self.named_service,
                                         self.player_obj_path,
                                         introspect=False)

        self.player_iface = dbus.Interface(player_obj, PlayerSession.DBUS_IFACE)

        try:
            self.player_iface.set_video_window(self.video_container.window.xid)
            self.player_iface.set_player(self.selected_player)

        except PlayerError, e:
            log.error(e)

        else:
            self.player_state = ATABAKE_STATE_NONE
            self.player_iface.connect_to_signal("state_signal",
                                                self._player_state_changed)
            self.player_iface.connect_to_signal("eos_signal",
                                                self._player_eos)
            self.player_iface.connect_to_signal("error_signal",
                                                self._player_error)
            self.player_iface.connect_to_signal("buffering_signal",
                                                self._buffering)



    def _add_stock_action(self, name, label=None, tooltip=None, stock=None):
        act = gtk.Action(name, label, tooltip, stock)

        self.action_group.add_action_with_accel(act, None)
        act.set_accel_group(self.accel_group)
        act.connect_accelerator()

        self.actions[name] = act

    def _setup_actions(self):
        self._add_stock_action("Quit", stock=gtk.STOCK_QUIT)
        self._add_stock_action("Preferences", stock=gtk.STOCK_PREFERENCES)
        self._add_stock_action("Open", stock=gtk.STOCK_OPEN)
        self._add_stock_action("Play", stock=gtk.STOCK_MEDIA_PLAY)
        self._add_stock_action("Stop", stock=gtk.STOCK_MEDIA_STOP)
        self._add_stock_action("Hold", stock=gtk.STOCK_MEDIA_PAUSE)


        act = gtk.ToggleAction("Pause", None, None, gtk.STOCK_MEDIA_PAUSE)
        self.action_group.add_action_with_accel(act, None)
        act.set_accel_group(self.accel_group)
        act.connect_accelerator()
        self.actions["Pause"] = act

    def _setup_toolbar(self):
        tb = gtk.Toolbar()
        tb.set_orientation(gtk.ORIENTATION_HORIZONTAL)
        tb.set_style(gtk.TOOLBAR_BOTH_HORIZ)

        self.toolbar = tb
        self.show_all()

        model = gtk.ListStore(str, str)

        # TODO: get available player from media engine
        model.append(("MPlayer", "mplayer"))
        model.append(("Osso Media Engine", "oms"))
        model.append(("GStreamer backend", "gst"))

        ti = gtk.ToolItem()
        self.source_selector = gtk.ComboBox(model)
        self.source_selector.set_active(0)
        self.selected_player = "mplayer"

        self.source_selector.connect("changed", self._player_selector_changed)
        r = gtk.CellRendererText()
        r.set_property("scale", pango.SCALE_SMALL)
        r.set_property("scale-set", True)
        self.source_selector.pack_start(r, True)
        self.source_selector.add_attribute(r, "text", 0)
        ti.add(self.source_selector)
        ti.show_all()
        tb.insert(ti, -1)

        ti = self.actions["Open"].create_tool_item()
        ti.show()
        tb.insert(ti, -1)

        self.infotext = gtk.Label()
        self.infotext.set_use_markup(True)
        self.infotext.set_justify(gtk.JUSTIFY_LEFT)
        self.infotext.set_alignment(xalign=0.0, yalign=0.5)
        self.infotext.set_markup("<small>Idle.</small>")

        ti = gtk.ToolItem()
        ti.add(self.infotext)
        ti.set_expand(True)
        tb.insert(ti, -1)
        ti.show_all()

        ti = gtk.ToolItem()

        # Seek bar
        self.seek_adjust = gtk.Adjustment(0,0,100,0,1,1)
        self.seek_timeout_id = None
        self.seek_scale = gtk.HScale(self.seek_adjust)
        self.seek_scale.set_draw_value(False)
        self.seek_scale.set_sensitive(False)
        self.seek_scale.show()

        ti.add(self.seek_scale)
        ti.set_expand(True)
        tb.insert(ti, -1)
        ti.show()

        self.cur_time = gtk.Label("00:00");
        self.separator= gtk.Label(" / ");
        self.total_time = gtk.Label("00:00");

        hbox = gtk.HBox(homogeneous=False, spacing=1)
        hbox.pack_start(self.cur_time, fill=True, expand=False)
        hbox.pack_start(self.separator, fill=True, expand=False)
        hbox.pack_start(self.total_time, fill=True, expand=False)

        ti = gtk.ToolItem()
        ti.add(hbox)
        ti.set_expand(False)
        tb.insert(ti, -1)
        ti.show_all()

    def _setup_ui(self):
        self.top_layout = gtk.VBox(homogeneous=False, spacing=0)
        self.add(self.top_layout)

        self._setup_ui_player()
        self.top_layout.show_all()

    def _setup_ui_player(self):
        self.play_layout = gtk.HBox(homogeneous=False, spacing=2)
        self.top_layout.pack_start(self.play_layout, fill=True, expand=True)

        self.top_layout.pack_start(self.toolbar, fill=False, expand=False)

        vbox = gtk.VBox(homogeneous=True, spacing=5)
        vbox.set_size_request(80, -1)
        for name, type in (("Play", gtk.Button),
                           ("Pause", gtk.ToggleButton),
                           ("Stop", gtk.Button),
                           ("Hold", gtk.Button)):
            act = self.actions[name]
            btn = type()
            btn.set_image(act.create_icon(gtk.ICON_SIZE_DIALOG))
            vbox.pack_start(btn, fill=False, expand=False)
            act.connect_proxy(btn)

        # Volume bar
        self.volume_adjust = gtk.Adjustment(0,0,100,0,1,1)
        self.volume_scale = gtk.VScale(self.volume_adjust)
        self.volume_scale.set_digits(0)
        self.volume_scale.set_value(50)
        self.volume_scale.show()

        self.volume_scale.connect("value_changed", self._volume_changed)

        vbox.pack_start(self.volume_scale, fill=True, expand=False)

        vbox.show_all()
        self.play_layout.pack_start(vbox, fill=True, expand=False)
        self.controls = vbox

        self.video_container = gtk.EventBox()
        self.video_container.set_property("can-focus", True)
        self.video_container.set_events(gtk.gdk.ALL_EVENTS_MASK)

        self.video_container.modify_bg(gtk.STATE_NORMAL,
                                       gtk.gdk.color_parse("#000000"))
        self.play_layout.pack_start(self.video_container, expand=True,
                                    fill=True)

        self.video_container.show()

    def _setup_connections(self):
        self.connect("destroy", self.quit)
        self.connect("window-state-event", self._window_state_changed)
        self.connect("key-press-event", self._key_pressed)

        def action_connect(name, callback):
            self.actions[name].connect("activate", callback)

        action_connect("Quit", self.quit)
        action_connect("Preferences", self._do_preferences)
        action_connect("Open", self._do_open)
        action_connect("Play", self._do_play)
        action_connect("Pause", self._do_pause)
        action_connect("Stop", self._do_stop)
        action_connect("Hold", self._do_hold)


        self.video_container.connect_after("realize",
                                           self._video_container_realize)

        self.seek_scale.connect("button-press-event", self.on_timebar_press)
        self.seek_scale.connect("button-release-event", self.on_timebar_release)

    # Callbacks

    def on_timebar_press(self, widget, event):
        print 'bar press'
        if self.seek_timeout_id:
            gobject.source_remove(self.seek_timeout_id)
            self.seek_timeout_id = None

    def on_timebar_release(self, widget, event):
        try:
            position = self.player_iface.get_position()
            if position:
                log.info("position = %d" % position)
            else:
                log.info("player returned NO position")

            duration = self.player_iface.get_duration()
            if duration:
                print 'duration = %d' % duration
            else:
                print 'player returned NO duration'

            pos = long(self.seek_scale.get_value() * duration / 100)
            log.info("pos to seek = %s" % pos)
            if (self._do_seek(pos) and
                self.player_state == ATABAKE_STATE_PLAYING):
                self.seek_timeout_id = gobject.timeout_add(500, self.update_gui)

        except PlayerError, e:
            log.error(e)

    def update_gui(self, pos=None, duration=None):
        def format_time(time):
            time_seconds = time / 1000.0
            time_f = '%02d:%02d' % (time_seconds / 60, time_seconds % 60)

            return time_f

        result = True

        try:
            if not pos:
                pos = self.player_iface.get_position()
            if not duration:
                duration = self.player_iface.get_duration()

        except PlayerError, e:
            log.error(e)

        else:
            if pos >= 0 and duration > 0 and pos <= duration:
                self.seek_adjust.set_value(pos * 100.0 / duration)
                self.cur_time.set_text(format_time(pos))
                self.total_time.set_text(format_time(duration))

            else:
                log.debug("Did not receive (pos, dur), aborting seek_timeout")
                return False

            return True

    def _do_seek(self, pos):
        return self.player_iface.seek(pos)

    def _volume_changed(self, volumebar):
        value = int(volumebar.get_value())
        log.info("setting volume to %d" % value)

        try:
            self.player_iface.set_volume(value)

        except PlayerError, e:
            log.error(e)

    def _window_state_changed(self, window, event):
        state = event.new_window_state & gtk.gdk.WINDOW_STATE_FULLSCREEN
        self.is_fullscreen = bool(state)

        if self.is_fullscreen:
            self.controls.hide()
            self.toolbar.hide()
        else:
            self.controls.show()
            self.toolbar.show()

        return True

    def _player_selector_changed(self, combo):
        index = combo.get_active()
        if index < 0:
            return

        self.selected_player = combo.get_model()[index][1]

        log.info("player selector changed, active = %s" \
              % (self.selected_player))

        try:
            self.player_iface.set_player(self.selected_player)

        except PlayerError, e:
            log.error(e)

    def _key_pressed(self, window, event):
        log.debug("Key pressed %s, event=%s, keyval=%s" % \
                  (window, event, event.keyval))

        if event.keyval == gtk.keysyms.F6 and not self.idle_call:
            try:
                if self.player_iface.get_state() == ATABAKE_STATE_PLAYING:
                    self.player_iface.pause()
                    self.idle_call = gobject.idle_add(self._idle_resume_play)

                self.player_iface.set_fullscreen(not self.is_fullscreen)

            except PlayerError, e:
                log.error(e)

            else:
                if self.is_fullscreen:
                    self.unfullscreen()
                else:
                    self.fullscreen()

                self.is_fullscreen = not self.is_fullscreen

        elif event.keyval == gtk.keysyms.F8:
            level = self.volume_scale.get_value()
            level = max(0, level - 10)
            self.volume_scale.set_value(level)

        elif event.keyval == gtk.keysyms.F7:
            level = self.volume_scale.get_value()
            level = min(100, level + 10)
            self.volume_scale.set_value(level)

        return True

    def _idle_resume_play(self):
        gtk.gdk.threads_enter()
        self._do_pause(1)
        self.idle_call = 0
        gtk.gdk.threads_leave()

    def _video_container_realize(self, wid):
        if not wid.window:
            return

        log.info("in video realize")
        if self.player_iface:
            log.info("setting xid to 0x%x " % wid.window.xid)
            self.player_iface.set_video_window(wid.window.xid)

    def _do_open(self, wid):
        self.choose_file()

    def _do_play(self, wid):
        if not self.url:
            self.choose_file()

        # create player
        log.info("Requesting playback of url: %s to %s backend" % \
                 (self.url, self.selected_player))

        try:
            self.player_iface.play()
        except PlayerError, e:
            log.error(e)

    def _buffering(self, value):
        if int(value) == 100:
            self.seek_scale.set_sensitive(True)
            self.seek_timeout_id = gobject.timeout_add(500, self.update_gui)

    def _do_pause(self, wid):
        log.debug("player state in do_pause = %d" % self.player_state)

        try:
            self.player_iface.pause()

        except PlayerError, e:
            log.error(e)
        else:
            if (self.player_state == ATABAKE_STATE_PLAYING and \
                    self.seek_timeout_id):
                gobject.source_remove(self.seek_timeout_id)
                self.seek_timeout_id = None

            elif (self.player_state == ATABAKE_STATE_PAUSED and \
                      not self.seek_timeout_id):
                self.seek_timeout_id = gobject.timeout_add(500, self.update_gui)

    def _do_stop(self, wid=None):
        try:
            self.player_iface.stop()
        except PlayerError, e:
            log.error(e)
        else:
            if self.seek_timeout_id:
                gobject.source_remove(self.seek_timeout_id)
                self.seek_timeout_id = None

            self._reset_pos()
            self.media_length = None

    def _do_hold(self, wid=None):
        try:
            self.player_iface.hold()
        except PlayerError, e:
            log.error(e)
        else:
            if self.seek_timeout_id:
                gobject.source_remove(self.seek_timeout_id)
                self.seek_timeout_id = None

    def quit(self, *ignored):
        self._do_stop()
        gtk.main_quit()

    def _player_eos(self):
        log.info("received EOS signal from player")
        if self.seek_timeout_id:
            gobject.source_remove(self.seek_timeout_id)
            self.seek_timeout_id = None

        self._reset_pos()
        self.media_length = None

    def _player_error(self, error_code):
        log.info("received error signal from player")

        # TODO: switch error code
        if error_code != AtabakeException.ERROR_STOPPING and \
                error_code != AtabakeException.ERROR_PLAYER_UNAVAILABLE:
            self._do_stop()

        if self.seek_timeout_id:
            gobject.source_remove(self.seek_timeout_id)
            self.seek_timeout_id = None

        self._reset_pos()
        self.media_length = None

        self.infotext.set_label("<small>Stopped due player error.</small>")

    @staticmethod
    def _time_to_str(time):
        time = int(time)
        seconds = time % 60
        t = time / 60
        minutes = t % 60
        hours = t / 60
        return "%02d:%02d:%02d" % (hours, minutes, seconds)

    def _player_pos(self, player, pos):
        elapsed = self._time_to_str(pos)
        self.timepos.set_text(elapsed)
        if self.media_length:
            self.timepos.set_fraction(pos / self.media_length)

    def _player_state_changed(self, state):

        log.info("In player state changed to %d" % state)

        self.player_state = state

        if state == ATABAKE_STATE_NONE:
            msg = "<small>Idle.</small>"
            self.media_length = None
            self._reset_pos()
        elif state == ATABAKE_STATE_PLAYING:
            msg = "<small>Playing.</small>"
        elif state == ATABAKE_STATE_PAUSED:
            msg = "<small>Paused.</small>"
        elif state == ATABAKE_STATE_HOLDED:
            msg = "<small>Holded.</small>"

        self.infotext.set_markup(msg)

    def _player_media_info(self, player, d):
        #self.media_length = d.get("length", None)
        #self.media_title = d.get("media_title", None)
        pass

    def get_url_from_chooser(self):
        # TODO: hildonize
        d = gtk.FileChooserDialog("Choose media file", self,
                                  gtk.FILE_CHOOSER_ACTION_OPEN,
                                  (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                   gtk.STOCK_OPEN, gtk.RESPONSE_OK))

        self.gui_url_chooser = d
        self.gui_url_chooser.show()
        r = self.gui_url_chooser.run()
        self.gui_url_chooser.hide()
        if r == gtk.RESPONSE_OK:
            escaped_url = self.gui_url_chooser.get_uri()
            self.url = urllib.url2pathname(escaped_url)
            log.info("got url = %s" % self.url)
            return self.url
        else:
            log.info("got NO url")
            return ""

    def choose_file(self):
        url = self.get_url_from_chooser()
        if self.player_iface:
            self.player_iface.set_uri(url, False)

    def _reset_pos(self):
        self.cur_time.set_text("00:00");
        self.total_time.set_text("00:00");
        self.seek_adjust.set_value(0.0)
        if self.player_iface:
            self.player_iface.reset_session()

    def _do_preferences(self, wid):
        self.show_preferences()

    def show_preferences(self, focus_widget=None):
        if self.pref_dialog.run(focus_widget):
            self._save_settings()


if __name__ == "__main__":
    ui = PlayerUI()
    ui.show()
    gtk.main()
