#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

import os
import sys
import gobject
import logging
import errno

from atabake.lib.errors import *

log = logging.getLogger("atabake.daemon")

class Daemon(object):
    """Class to daemonize the media engine

    This class uses the double-fork technique in order to daemonize
    our media engine. It also has basic functions to be executed by
    the class that wants to be daemonized.

    """
    def __init__(self, pidfile):
        self.pidfile = pidfile

    def _daemonize(self):
        """Daemonize the subclass using 'double-fork' trick"""
        # do first fork
        try:
            pid = os.fork()

            if pid > 0:
                # must exit first parent
                os._exit(0)
        except OSError, e:
            log.critical("Fork #1 failed %d (%s)", e.errno, e.strerror)
            os._exit(1)

        os.setsid() # run a program in a new session
        os.umask(0)

        # do second fork
        try:
            pid = os.fork()

            if pid > 0:
                # must exit first parent
                os._exit(0)
        except OSError, e:
            log.critical("Fork #2 failed %d (%s)", e.errno, e.strerror)
            os._exit(1)

        # write pid to file
        pid = str(os.getpid())
        file(self.pidfile,'w+').write("%s\n" % pid)

    def _getpid(self):
        """Get the pid inside pidfile"""
        try:
            pidfile = open(self.pidfile, "r")
            pid = int(pidfile.readline())
            pidfile.close()
        except IOError:
            pid = None

        if pid and not self._pid_exists(pid):
            self._delpid()
            pid = None

        return pid

    def _pid_exists(self, pid):
        """Check whether certain pid is valid (has a process with it)"""
        try:
            os.kill(pid, 0)
            return True
        except OSError, err:
            return err.errno == errno.EPERM

    def _delpid(self):
        """Delete the pidfile"""
        os.remove(self.pidfile)
        return True

    def start(self):
        """Start the daemon"""
        pid = self._getpid()

        if pid:
            # the daemon is running or we have a problem
            raise DaemonStartError(pid)

        # Start the daemon
        self._daemonize()
        self.run()
        return True

    def stop(self):
        """Stop the daemon, killing the process"""
        # Try to get pid from file
        pid = self._getpid()

        if pid:

            if pid == os.getpid():
                gobject.idle_add(self.mainloop.quit)
                self._delpid()
                log.info("Process %s@%s stopped by dbus", pid, self.pidfile)
                return True
            else:
                log.warning("You should not try to stop MediaEngine"
                            " through another process")

        else:
            # When we are restarting the daemon this is not an error
            log.info("Pidfile %s does not exist. Are you sure the "
                     "daemon is running ?", self.pidfile)
            return True

    def run(self):
        """This will be called when the daemon starts, so you should override
        this when subclassing Daemon. If you want to make the daemon run
        continuously, you'll need a mainloop also.
        """
        raise NotImplementedError("You must implement run() "
                                  "to daemonize something")

    def __str__(self):
        return "%s(pidfile=%s)" % (self.__class__.__name__, self.pidfile)
