#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

import os
import sys
import gobject
import logging

import dbus
import dbus.service
# setting glib as default mainloop in
# for backwards-compatible code (maemo)
import dbus.glib

import atabake.players
from atabake.lib.errors import *
from atabake.lib.daemon import Daemon
from atabake.lib.player import Player
from atabake.lib.player_set import PlayerSet
from atabake.lib.player_session import PlayerSession

# timeout of 5 minutes (300 ms) to die
timeout = 300 * 1000
log = logging.getLogger("atabake.engine")
home_dir = os.path.expanduser(os.path.join("~", ".atabake"))

class MediaEngine(Daemon, dbus.service.Object):
    DBUS_SERVICE_NAME = "br.org.indt.atabake.MediaEngine"
    DBUS_OBJ_PATH     = "/br/org/indt/atabake/MediaEngine"
    DBUS_IFACE        = "br.org.indt.atabake.MediaEngine"

    def __init__(self, pidfile):
        Daemon.__init__(self, pidfile)
        self.session_bus = dbus.SessionBus()
        self.bn = dbus.service.BusName(self.DBUS_SERVICE_NAME,
                                       bus=self.session_bus)

        dbus.service.Object.__init__(self, self.bn, self.DBUS_OBJ_PATH)

        self.players = PlayerSet(Player)
        self.holded = None
        self.sessions = {}

    def load_players(self, directory):
        """Load the players (players) under that directory"""
        self.players.load_from_directory(directory)

    def _add_session(self, sender_id, session):
        """Adds a session to the current engine"""
        self.sessions.setdefault(sender_id, []).append(session)

    def _delete_session(self, sender_id=None, obj_path=None):
        """Delete just one session for a given application"""
        if not sender_id or not obj_path:
            raise ValueError("Can't delete a session without "
                             "a sender_id or the object's path")
        try:
            sessions = self.sessions[sender_id]
            for session in sessions:
                if session.session_path == obj_path:
                    if not session.is_idle():
                        session.stop()
                    sessions.remove(session)
                    session.delete()
                    del session
                    break
            log.debug("Remaining sessions for %s: %s", sender_id, self.sessions)

        except KeyError, e:
            raise InvalidSessionError(sender_id)

    @dbus.service.method(DBUS_IFACE)
    def delete_sessions(self, sender_id=None):
        """Delete all sessions of a given application"""
        try:
            sessions = self.sessions.pop(sender_id)
            for session in sessions:
                if not session.is_idle():
                    session.stop()
            log.debug("Remaining sessions: %s", self.sessions)

        except KeyError, e:
            raise InvalidSessionError(sender_id)

    @dbus.service.method(DBUS_IFACE, sender_keyword='sender')
    def create_session(self, uri=None, sender=None):
        """Create a session on the bus queue and return it's id"""
        try:
            session = PlayerSession(self.session_bus, sender,
                                    uri, self.players)

        except PlayerSessionError, e:
            # Can be PlayerSessionInitError
            # or PlayerSessionCreatePlayerError
            log.error("Error while creating session: %s", e)
            raise

        session.dispose_cb = self._delete_session
        self._add_session(sender, session)
        log.info("Created session:%s %s", sender, session)
        return session.session_path

    @dbus.service.method(DBUS_IFACE)
    def hold_player(self):
        """Stop the active player if there is one. It's
        important to note that there should not be more
        than one active player as our architecture does
        not allow that due the lack of resources."""
        log.debug("Sessions: %s", self.sessions)
        try:
            for sessions in self.sessions.itervalues():
                for session in sessions:
                    if session.is_playing() or session.is_paused():
                        self.holded = session
                        self.holded.hold()

        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

    @dbus.service.method(DBUS_IFACE)
    def resume_player(self):
        """Resume the last player"""
        try:
            if self.holded:
                self.holded.resume()
                self.holded = None
            else:
                log.info("No player to resume")

        except ResumePlayerError, e:
            log.error("It's not possible to resume "
                      "a player that was not holded: %s", e, exc_info=True)
            raise

        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

    @dbus.service.method(DBUS_IFACE)
    def start(self):
        """Start our Media Engine daemon"""
        Daemon.start(self)

    @dbus.service.method(DBUS_IFACE)
    def stop(self):
        """Stop our Media Engine daemon and it's players"""
        try:
            if os.path.exists(self.pidfile):
                for sessions in self.sessions.itervalues():
                    for session in sessions:
                        if session.is_playing() or session.is_paused():
                            session.stop()

        except PlayerError, e:
            log.error(e, exc_info=True)
            raise

        log.info("Stopped MediaEngine")
        Daemon.stop(self)

    def check_sessions(self):
        """Check for Atabake's clients inactivity"""
        if len(self.sessions) == 0:
            log.debug("Atabake quit due to inactivity")
            self.stop()
        return False

    def name_owner_changed(self, obj, old, new):
        """Function to handle when a client disconnect from the engine"""
        obj = str(obj)
        old = str(old)
        new = str(new)

        try:
            if obj in self.sessions and old == obj and not new:
                sessions = self.sessions[obj]
                self.delete_sessions(obj)
                log.info("Removed session(s) %s %s", obj, sessions)
                if len(self.sessions) == 0:
                    gobject.timeout_add(timeout, self.check_sessions)
        except InvalidSessionError, e:
            log.warning(e, exc_info=True)
            raise

    def run(self):
        """Implementation of method run for class Daemon"""
        # dbus stuff
        try:
            self.session_bus.add_signal_receiver(self.name_owner_changed,
                                                 "NameOwnerChanged")
            log.info("Starting MediaEngine")

            # load our players (players)
            try:
                players_directory = atabake.players.__path__[0]
                self.load_players(players_directory)
                self.load_players(os.path.join(home_dir, "plugins"))
            except OSError, e:
                log.warning("Could not load some plugins: %s", e)

            self.mainloop = gobject.MainLoop()
            self.mainloop.run()
        except Exception, e:
            log.error(e, exc_info=True)
            raise
