#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"
__version__ = "0.1.6"

import os
import logging
import ConfigParser

log = logging.getLogger("atabake.cofig_file")

class AtabakeConfigFile(ConfigParser.RawConfigParser):
    def __init__(self):
        ConfigParser.RawConfigParser.__init__(self)
        self._user_path = os.path.expanduser(os.path.join("~", ".atabake",
                                                          "config"))
        self._system_path = os.path.join("/", "etc", "atabake", "config")
        self.read([self._system_path, self._user_path])
        self._check_config_file()

    def _check_config_file(self):
        check_list = [self._check_other_section,
                      self._check_fallback_section,
                      self._check_media_section,
                      self._check_mplayer_section,
                      self._check_version]

        result = [True for x in check_list if x()]
        if True in result:
            self.save()

    def _check_fallback_section(self):
        if not self.has_section("Fallback"):
            self.add_section("Fallback")
            self.set("Fallback", "audio", "oms")
            self.set("Fallback", "video", "mplayer")
            self.set("Fallback", "general", "mplayer")
            log.info("Created fallback section")
            return True
        return False

    def _set_medias(self):
        medias = [("mp4", "oms"),
                  ("m4a", "oms"),
                  ("3gp", "oms"),
                  ("rv", "oms"),
                  ]
        for media,player in medias:
            self.set("Media", media, player)

    def _check_media_section(self):
        if not self.has_section("Media"):
            self.add_section("Media")
            # hack regarding media types on maemo
            self._set_medias()
            log.info("Created media section")
            return True
        elif self.get("Other", "version") < __version__:
            self._set_medias()
            log.info("Updated media section")
            return True
        return False

    def _check_other_section(self):
        if not self.has_section("Other"):
            self.add_section("Other")
            # version == 0.0 to mark this as "dirty"
            # so we update everything that this version needs
            self.set("Other", "version", "0.0")
            return True
        return False

    def _check_mplayer_section(self):
        if not self.has_section("Mplayer"):
            self.add_section("Mplayer")
            self.set("Mplayer", "vo", "xv")
            self.set("Mplayer", "#opts", "-dr -nomenu")
            self.set("Mplayer", "#vo", "xv:ck-method=auto")
            return True
        return False

    def _check_version(self):
        if self.get("Other", "version") < __version__:
            self.set("Other", "version", __version__)
            return True
        return False

    def save(self):
        dir = os.path.dirname(self._user_path)
        if not os.path.exists(dir):
            os.makedirs(dir, 0700)
        file = open(self._user_path, "w+")
        self.write(file)
        file.close()
        log.info("Saved config file for Atabake")
