#!/usr/bin/env python

""" Copyright 2009 Edwin Marsahll 

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
from sys import argv
from os import makedirs, environ
from os.path import isdir, abspath
from subprocess import call, PIPE
from optparse import OptionParser, Option
from ConfigParser import RawConfigParser

#TODO: Replace all instances of cal/PIPE with getoutput
from commands import getoutput

# get information about the currently logged in user
user = environ["HOME"].split("/")[-1]
uid = getoutput("id -u %s" % user)
gid = getoutput("id -g %s" % user)

# create config directory if it doesn't already exist
config_dir = environ["HOME"] + "/.config/aptly/"

try:
    makedirs(config_dir)
    getoutput("chown  %s:%s %s" %(uid, gid, config_dir))
except OSError:
    pass

config = RawConfigParser()
settings_file = config_dir + "settings"
available_settings = "cache", "force-clean"
try: 
    open(settings_file)
    config.read(settings_file)
except IOError:
    # create a default settings file if one can't be found
    config.add_section("General")
    config.write(open(settings_file, "wb"))
    getoutput("chown %s:%s %s" %(uid, gid, settings_file))

def install_callback(option, opt_str, value, parser):
    """ Handles installing of remote and local packages using apt-get and
        dpkg(-deb) respectively.

        If installing from a local deb file, this function calculates the
        dependencies of that deb file and calls apt-get accordingly before
        calling dpkg on it. In general, packages are installed 
        simultaneously. If a deb file is among the list of apps to install
        it is installed separately (though it's dependencies are installed
        simultaneously. If you would rather install all apps seperately,
        you may invoke multiple -i/--install commands.
     """
        
    assert value is None
    local_apps = []
    remote_apps = []

    # install from a catalog
    def install_remote(apps): 
       command = ["apt-get", "install"] + apps
       if parser.values.cache is not None:
           command = command[:1] + parser.values.cache + command[1:]

       if parser.values.install is not True:
           retcode = call(" ".join(command), shell=True)
       else:
           retcode = 1

       if retcode:
           parser.values.install = False
       else:
           parser.values.install = True

    # install from a local deb package
    def install_local(app):
        p = call(["dpkg-deb", "-I", app], stdout=PIPE)
        deps = ""

        for line in p.stdout.readlines():
            if "Depends" in line:
                deps = line.split("Depends: ")[1]
                deps = deps.split(", ")
                for i, dep in enumerate(deps):
                    deps[i] = dep.split()[0]

        install_remote(deps)
        call(["dpkg", "-i", app])

    # detect variable args 
    for arg in parser.rargs:
        if len(arg) == 2 or "--" in arg:
            break
        else:
            if arg[-4:] == ".deb":
                local_apps.append(arg)
            else:
                remote_apps.append(arg)

    unique_local_apps = dict.fromkeys(local_apps).keys() 
    for app in unique_local_apps:
        install_local(app)

    unique_remote_apps = dict.fromkeys(remote_apps).keys() 
    install_remote(unique_remote_apps)


def cache_callback(option, opt_str, value, parser):
    """ Handles downloading packages to an different cache directory using
        apt-get.

        This function will download the deb packages for the apps associated with
        the -i/--install option to the directory specified immediately after
        it. If no directory is specified, ~/.config/aptly/cache is used
        instead.  If a directory was specified, its location is then saved to
        the settings file located at ~/.config/aptly/settings so that should
        the -c/--clean option be specified, aplty will know where to look for
        the deb packages.
    """

    assert value is None
    value = []

    for arg in parser.rargs:
        if len(arg) == 2 or "--" in arg:
            break
        else:
            value.append(arg)

    if not value:
        prompt = "No cache directory was specified. Please enter " \
                 "a directory path or press enter to use the " \
                 "default.\n" \
                 "==> "
        dir = raw_input(prompt)

        if isdir(dir):
            value.append(dir)
        
    if value and isdir(value[0]):
        custom_cache_dir = abspath(value[0])
        if custom_cache_dir[-1] != "/":
            custom_cache_dir += "/"

        config.set("General", "cache", custom_cache_dir)
        config.write(open(settings_file, "wb"))
        getoutput("chown %s:%s %s" %(uid, gid, settings_file))

        partial_dir = custom_cache_dir + "partial"
        try:
            makedirs(partial_dir)
        except OSError:
            pass

        cache_option = "dir::cache::archives='%s'" % custom_cache_dir
        parser.values.cache = ["-o", cache_option]

    else:
        if "cache" not in config.options("General"):
            cache_dir = config_dir + "cache/"
            try:
                makedirs(cache_dir)
            except OSError:
                pass

            partial_dir = cache_dir + "partial/"
            try:
                makedirs(partial_dir)
            except OSError:
                pass

            custom_cache = "dir::cache::archives='%s'" % cache_dir
        else:
            custom_cache_dir = config.get("General", "cache")
            custom_cache = "dir::cache::archives='%s'" % custom_cache_dir

        parser.values.cache = ["-o", custom_cache]

def remove_callback(option, opt_str, value, parser):
    """ Handles uninstalling of local packages using apt-get. """

    assert value is None
    apps = []

    # detect variable args 
    for arg in parser.rargs:
        if len(arg) == 2 or "--" in arg:
            break
        else:
            apps.append(arg)

    unique_apps = dict.fromkeys(apps).keys() 
    command = ["apt-get", "remove"] + unique_apps
    call(command)

def search_callback(option, opt_str, value, parser):
    """ Handles searching for packages using apt-cache. """

    assert value is None
    apps = []

    
    # detect variable args 
    for arg in parser.rargs:
        if len(arg) == 2 or "--" in arg:
            break
        else:
            apps.append(arg)

    unique_apps = dict.fromkeys(apps).keys() 
    command = ["apt-cache", "search"] + unique_apps
    call(command)

def about_callback(option, opt_str, value, parser):
    """ Handles searching for packages details using apt-cache.

        Unlike apt-cache, aptly is capable of returning information on
        multiple packages. """

    assert value is None
    apps = []
    
    # detect variable args 
    for arg in parser.rargs:
        if len(arg) == 2 or "--" in arg:
            break
        else:
            apps.append(arg)

    unique_apps = dict.fromkeys(apps).keys() 
    for app in unique_apps:
        call(["apt-cache", "showpkg", app])

def update_callback(option, opt_str, value, parser):
    """ Handles updating the list of available packages using apt-get. """

    assert value is None

    retcode = call(["apt-get", "update"])
    if retcode:
        parser.values.update = False
    else:
        parser.values.update = True

def upgrade_callback(option, opt_str, value, parser):
    """ Handles upgrading installed packages using apt-get. """

    assert value is None
    
    if parser.values.update is None:
        prompt = "Would you like to update the list of available packages " \
                 "before upgrading? (y/N) "
        update = raw_input(prompt)
        if update.lower() in "yes":
            update_callback("-U/--update", "-U", None, parser)

    elif parser.values.update is False:
        print "\nUpdate was unsucessful, aborting upgrade..."
        parser.values.upgrade = False
        return

    retcode = call(["apt-get", "upgrade"])

    if retcode:
        parser.values.upgrade = False
    else:
        parser.values.upgrade = True

def clean_callback(option, opt_str, value, parser):
    """ Handles cleaning the cache directory using apt-get.

        If a custom cache directory was used (and can be found in the settings
        file, package files will be removed from there as well.
    """

    assert value is None

    print "Attempting to remove packages from cache directory..."
    call(["apt-get", "clean"])

    if "cache" in config.options("General"):
        cache_dir = config.get("General", "cache")
    else:
        cache_dir = config_dir + "cache"
    
    call("rm %s/*.deb" % cache_dir, shell=True, stderr=PIPE)
    call("rm -r %s/partial" % cache_dir, shell=True, stderr=PIPE)
    call("rm %s/lock" % cache_dir, shell=True, stderr=PIPE)

def autoclean_callback(option, opt_str, value, parser):
    """ Handles autocleaning the cache directory using apt-get. """

    assert value is None

    call(["apt-get", "autoclean"])

def option_callback(option, opt_str, value, parser):
    """ Writes various options to the settings file.

        This unintuitively named command line option allows you to write
        options to the settings file without having to open it up and edit it
        manually.  While some settings are automatically saved, this option
        gives you the ability to set them explicitly, which is helpful if, for
        example, the option that was automatically set is wrong.
    """
    
    assert value is None
    value = []
    
    # detect variable args 
    for arg in parser.rargs:
        if len(arg) == 2 or "--" in arg or "=" not in arg:
            break
        else:
           value.append(arg)

    for option in value:
        setting, new_value = option.split("=")
        if setting in available_settings:
            config.set("General", setting, new_value)
        else:
            print "\"%s\" not a valid option, ignoring..." % setting

    config.write(open(settings_file, "wb"))

def main():
    """ Aptly is an apt-get, apt-cache, and dpkg wrapper. 

        In addition to wrapping the most frequently used functions
        of the mentioned programs into a single tool, aptly enhances
        some of them. For example, installing from a deb package now
        automatically resolves dependencies, and it is also possible to
        both update and upgrade in one command without the use of "&&".
    """

    #usage = "usage: %prog [options] pkg1 [pkg2 ...]"
    usage = "usage: %prog [options]\n" \
            "       %prog --install pkg1 [pkg2 ...]\n" \
            "       %prog --remove pkg1 [pkg2 ...]\n" \
            "       %prog --about pkg1 [pkg2 ...]\n\n" \
            "aptly is a tool that combines the most commonly used " \
            "apt-get apt-cache, and dpkg."

    version = "%prog 2009-12-28.1"
    parser = OptionParser(usage=usage, version=version)

    parser.add_option("-i", "--install", 
                      action="callback",
                      callback=install_callback,
                      dest="install", 
                      metavar="LIST",
                      help="Install LIST to your system.")

    parser.add_option("-r", "--remoev", 
                      action="callback",
                      callback=remove_callback,
                      dest="remove", 
                      metavar="LIST",
                      help="remove LIST from your system")

    parser.add_option("-s", "--search", 
                      action="callback",
                      callback=search_callback,
                      dest="search", 
                      metavar="PATTERN",
                      help="search for a package whose name matches PATTERN")
    
    parser.add_option("-A", "--about",
                      action="callback",
                      callback=about_callback,
                      dest="about",
                      metavar="LIST",
                      help="show detailed information about LIST")

    parser.add_option("-U", "--update", 
                      action="callback", 
                      callback=update_callback,
                      dest="update", 
                      help="update each catalog's available app list")

    parser.add_option("-u", "--upgrade",
                      action="callback",
                      callback=upgrade_callback,
                      dest="upgrade",
                      help="upgrade all installed apps")
    parser.add_option("-c", "--clean",
                      action="callback",
                      callback=clean_callback,
                      dest="clean",
                      help="clean cache directory")
    parser.add_option("-a", "--autoclean",
                      action="callback",
                      callback=autoclean_callback,
                      dest="autoclean",
                      help="autoclean cache directory")
    parser.add_option("-C", "--cache",
                      action="callback",
                      callback=cache_callback,
                      dest="cache",
                      help="download packages to an alternate cache direcotry")
    parser.add_option("-o", "--option",
                      action="callback",
                      callback=option_callback,
                      dest="optioin",
                      help="Write an option to the settings file.")


    if argv[1:] == []:
        parser.parse_args(["-h"])
    else:
        if "force-clean" in config.options("General") \
        and config.get("General", "force-clean"):
           argv.append("-c") 
        (options, args) = parser.parse_args()
    
if __name__ == "__main__":
    main()
