#include "repostat.h"
#include "fremantlerepochooser.h"
#include "rangepicker.h"
#include <QDate>
#include <QFile>
#include <QFileInfo>
#include <QTextStream>
#include <QDebug>
#include <QNetworkAccessManager>
#include <QNetworkDiskCache>

RepoStat::RepoStat(QWidget* parent): QMainWindow(parent), settings("AppWatch", "RepoStat")
{
    ui.setupUi(this);
//    m_fullscreen = new QAction(QIcon::fromTheme("general_fullsize"), QString(), this);
/*    m_fullscreen = new QAction(QIcon(), QString(), this);
    m_fullscreen->setCheckable(true);
    connect(m_fullscreen, SIGNAL(toggled(bool)), this, SLOT(toggleFullScreen(bool)));
    QToolBar *tb = new QToolBar(this);
    tb->addAction(m_fullscreen);
    addToolBar(tb);
*/
    connect(ui.action_Settings, SIGNAL(triggered()), this, SLOT(showRepoChooser()));
    connect(ui.action_Date_range, SIGNAL(triggered()), this, SLOT(showRangeChooser()));

    manager = new QNetworkAccessManager(this);
    qDebug() << settings.fileName();
    QNetworkDiskCache *diskCache = new QNetworkDiskCache(this);
    diskCache->setCacheDirectory(QFileInfo(settings.fileName()).absolutePath() + "/cacheDir");
    manager->setCache(diskCache);

    getData();

    connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(loadStat(QNetworkReply*)));

    ////
    ////  Setup view & scene
    ////

    ui.graphicsView->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    ui.graphicsView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    ui.graphicsView->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    ui.graphicsView->setRenderHint(QPainter::Antialiasing);

}

void RepoStat::getData()
{
    QString interval = settings.value("interval", QVariant("m")).toString();

    // do a request preferred from cache
    QNetworkRequest request(QUrl(QString("http://app.cache.rs/repostat/line?scale=%1&fmt=z").arg(interval)));
//    request.setAttribute(QNetworkRequest::CacheLoadControlAttribute, QNetworkRequest::AlwaysCache);
    request.setAttribute(QNetworkRequest::CacheLoadControlAttribute, QNetworkRequest::PreferCache);
    manager->get(request);
}

void RepoStat::showRangeChooser()
{
    RangePicker rp(settings.value("interval", QVariant("m")).toString());
    if (rp.exec() > 0)
        settings.setValue("interval", rp.range());
        getData();
}

void RepoStat::showRepoChooser()
{
    FremantleRepoChooser rc("RepoStat");
    if (rc.exec() > 0)
        getData();
}

void RepoStat::loadStat(QNetworkReply* reply)
{

    QVariant fromCache = reply->attribute(QNetworkRequest::SourceIsFromCacheAttribute);
    qDebug() << "stat page from cache?" << fromCache.toBool();

    QGraphicsScene* scene = ui.graphicsView->scene();
    if (scene)
        delete scene;
    scene = new QGraphicsScene();
    ui.graphicsView->setScene(scene);

    ////
    ////  Load graph data
    ////

    if (settings.allKeys().length() == 0) { // none selected, cludgy way to get defaults
        FremantleRepoChooser* rc = new FremantleRepoChooser("RepoStat");
        rc->saveChoice();
        delete rc;
    }

    ////
    ////  Load graph data
    ////
    if (!reply->open(QIODevice::ReadOnly | QIODevice::Text)){
       return;
    }
    QTextStream in(reply);
    QString lastline = "";
    bool skipds = true; // set this to true to avoid referencing a datasourec before the first one
    int elemcount = 0;
    int prevx, prevy = 0;
    QPen pen;
    pen.setWidth(2);
    pen.setCosmetic(true);
    while (!in.atEnd()) {
       QString line = in.readLine();
       QStringList list = line.split(",");
       if (list.length() < 1 || line.isEmpty()) {
           lastline = line;
           continue;
       }
       if (lastline.isEmpty()) { // header line
           skipds = false;
           if (settings.value(list[0], QVariant(0)).toInt() == 0){
               skipds = true;
           }
           pen.setColor(QColor::fromRgb(QRgb(list[1].toInt(NULL, 16))));
           elemcount = 0;
       } else if (!skipds) {
           if (elemcount > 0) {
               scene->addLine(prevx, -prevy, list[0].toInt(), -list[1].toInt(), pen);
           }
           prevx = list[0].toInt();
           prevy = list[1].toInt();
           elemcount++;
       }
       lastline = line;
    }

    reply->deleteLater();
    this->show();
    ui.graphicsView->fitInView(scene->sceneRect(), Qt::IgnoreAspectRatio);
}

void RepoStat::resizeEvent(QResizeEvent* event)
{
    ui.graphicsView->fitInView(ui.graphicsView->sceneRect(), Qt::IgnoreAspectRatio);
}

void RepoStat::toggleFullScreen(bool b)
{
    setWindowState(b ? windowState() | Qt::WindowFullScreen
                     : windowState() & ~Qt::WindowFullScreen);
}

RepoStat::~RepoStat()
{
    delete manager;

}
