using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using System.Xml;
using System.Globalization;

namespace AGS.Types
{
    public class SerializeUtils
    {
        /// <summary>
        /// Wrapper function for SelectSingleNode that throws an exception
        /// mentioning the node name if it is not found. Returns the node text
        /// if successful.
        /// </summary>
        public static string GetElementString(XmlNode node, string elementName)
        {
            XmlNode foundNode = node.SelectSingleNode(elementName);
            if (foundNode == null) 
            {
                throw new InvalidDataException("Missing XML element: " + elementName);
            }
            return foundNode.InnerText;
        }

        /// <summary>
        /// Wrapper function for SelectSingleNode that returns the node text
        /// or some default text if it does not exist.
        /// </summary>
        public static string GetElementStringOrDefault(XmlNode node, string elementName, string defaultValue)
        {
            XmlNode foundNode = node.SelectSingleNode(elementName);
            if (foundNode == null)
            {
                return defaultValue;
            }
            return foundNode.InnerText;
        }

        /// <summary>
        /// Wrapper function for SelectSingleNode that throws an exception
        /// mentioning the node name if it is not found. Returns the node's
        /// children if successful.
        /// </summary>
        public static XmlNodeList GetChildNodes(XmlNode parent, string elementName)
        {
            XmlNode foundNode = parent.SelectSingleNode(elementName);
            if (foundNode == null)
            {
                throw new InvalidDataException("Missing XML element: " + elementName);
            }
            return foundNode.ChildNodes;
        }

        public static void SerializeToXML(object obj, XmlTextWriter writer)
        {
            SerializeToXML(obj, writer, true);
        }

        public static void SerializeToXML(object obj, XmlTextWriter writer, bool writeEndElement)
        {
            writer.WriteStartElement(obj.GetType().Name);
            PropertyInfo[] properties = obj.GetType().GetProperties();
            foreach (PropertyInfo prop in properties)
            {
                if (prop.GetCustomAttributes(typeof(AGSNoSerializeAttribute), true).Length > 0)
                {
                    continue;
                }
                if ((prop.CanRead) && (prop.CanWrite))
                {
                    bool canSerializeClass = false;
                    if (prop.GetCustomAttributes(typeof(AGSSerializeClassAttribute), true).Length > 0)
                    {
                        canSerializeClass = true;
                    }

                    if (prop.GetValue(obj, null) == null)
                    {
                        writer.WriteStartElement(prop.Name);
                        writer.WriteAttributeString("IsNull", "True");
                        writer.WriteEndElement();
                    }
                    else if ((prop.PropertyType.IsClass) && (canSerializeClass))
                    {
                        object theClass = prop.GetValue(obj, null);
                        theClass.GetType().GetMethod("ToXml").Invoke(theClass, new object[] { writer });
                    }
                    else
                    {
                        if (prop.PropertyType == typeof(string))
                        {
                            writer.WriteStartElement(prop.Name);
							string propValue = prop.GetValue(obj, null).ToString();
                            if (propValue.IndexOf(' ') >= 0)
                            {
                                writer.WriteAttributeString("xml", "space", null, "preserve");
                            }
                            writer.WriteString(propValue);
                            writer.WriteEndElement();
                        }
						else if (prop.PropertyType == typeof(DateTime))
						{
							writer.WriteElementString(prop.Name, ((DateTime)prop.GetValue(obj, null)).ToString("yyyy-MM-dd"));
						}
						else
						{
							writer.WriteElementString(prop.Name, prop.GetValue(obj, null).ToString());
						}
                    }
                }
            }
            if (writeEndElement)
            {
                writer.WriteEndElement();
            }
        }

        public static void DeserializeFromXML(object obj, XmlNode node)
        {
            XmlNode mainNode = node;

            if (node.Name != obj.GetType().Name)
            {
                mainNode = node.SelectSingleNode(obj.GetType().Name);
                if (mainNode == null)
                {
                    throw new InvalidDataException("Node not found in XML: " + obj.GetType().Name);
                }
            }

            foreach (XmlNode child in mainNode.ChildNodes)
            {
                string elementValue = child.InnerText;
                PropertyInfo prop = obj.GetType().GetProperty(child.Name);
                if (prop == null)
                {
                    throw new InvalidDataException("The property '" + child.Name + "' could not be read. This game may require a newer version of AGS.");
                }
                else if (!prop.CanWrite)
                {
                    // do nothing, read-only
                }
                else if (child.Attributes.GetNamedItem("IsNull") != null)
                {
                    prop.SetValue(obj, null, null);
                }
                else if (prop.PropertyType == typeof(Boolean))
                {
                    prop.SetValue(obj, Convert.ToBoolean(elementValue), null);
                }
                else if (prop.PropertyType == typeof(int))
                {
                    prop.SetValue(obj, Convert.ToInt32(elementValue), null);
                }
                else if (prop.PropertyType == typeof(short))
                {
                    prop.SetValue(obj, Convert.ToInt16(elementValue), null);
                }
                else if (prop.PropertyType == typeof(string))
                {
                    prop.SetValue(obj, elementValue, null);
                }
				else if (prop.PropertyType == typeof(DateTime))
				{
                    // Must use CultureInfo.InvariantCulture otherwise DateTime.Parse
                    // crashes if the system regional settings short date format has
                    // spaces in it (.NET bug)
					prop.SetValue(obj, DateTime.Parse(elementValue, CultureInfo.InvariantCulture), null);
				}
                else if (prop.PropertyType.IsEnum)
                {
                    prop.SetValue(obj, Enum.Parse(prop.PropertyType, elementValue), null);
                }
                else if (prop.PropertyType.IsClass)
                {
                    ConstructorInfo constructor = prop.PropertyType.GetConstructor(new Type[] { typeof(XmlNode) });
                    prop.SetValue(obj, constructor.Invoke(new object[] { child }), null);
                }
                else
                {
                    throw new InvalidDataException("Unknown data type: " + prop.PropertyType.Name);
                }
            }
        }

        public static string GetAttributeString(XmlNode node, string attrName)
        {
            if (node.Attributes[attrName] == null)
            {
                throw new InvalidDataException("Missing attribute: " + attrName);
            }
            return node.Attributes[attrName].InnerText;
        }

        public static int GetAttributeInt(XmlNode node, string attrName)
        {
            return Convert.ToInt32(GetAttributeString(node, attrName));
        }
    }
}
