/* -*- mode:c; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of maemo-security-certman
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
 \file appid.cpp
 \ingroup libcertman
 \brief A command-line placeholder utility to acquire application id
        by process id

 This program is used temporarily to acquire the application id
 for binaries. It is to be replaced by proper functionality from
 the Aegis security framework.
 
*/

#include <stdio.h>
#include <string>
using namespace std;

#ifdef AEGIS_DEBUG_ENABLED
#undef AEGIS_DEBUG_ENABLED
#endif
#define AEGIS_SHOW_ERRORS
#include "aegis_common.h"
#include "aegis_crypto.h"

enum cmd_type {
	cmd_pid,
	cmd_pids,
	cmd_bin,
	cmd_bins,
	cmd_usage
} cmd = cmd_usage;



static void
usage(void)
{
	printf(
		"Usage:\n"
		"appid [-rPB] | [-p pid] | [-b pathname]\n"
		" -p pid to print the application id of the given pid\n"
		" -b pid to print the application id of the given binary\n"
		" -P to read pids from stdin and print appids to stdout\n"
		" -B to read binary names stdin and print appids to stdout\n"
		);
}

extern int resolve_symlinks;

int main(int argc, char* argv[])
{
	int a;
	char* binary = NULL;
	pid_t pid = (pid_t)-1;
	char* appid_str = NULL;
	string tbname = "";

    while (1) {
		a = getopt(argc, argv, "p:b:PBl");
		if (a < 0) {
			break;
		}
		switch(a) 
		{
		case 'p':
			pid = (pid_t)atol(optarg);
			cmd = cmd_pid;
			break;
		case 'b':
			binary = optarg;
			cmd = cmd_bin;
			break;
		case 'P':
			cmd = cmd_pids;
			break;
		case 'B':
			cmd = cmd_bins;
			break;
		case 'l':
			resolve_symlinks = 0;
            AEGIS_DEBUG(1, "Do not resolve symlinks");
			break;
		default:
			usage();
			return(-1);
		}
	}

	if (0 == aegis_crypto_init()) {
		printf("Crypto init failed, exit\n%s\n", aegis_crypto_last_error_str());
		return -1;
	}

	switch (cmd) 
	{
	case cmd_pid:
		aegis_application_id(pid, &appid_str);
		printf("%s\n", appid_str);
		aegis_crypto_free(appid_str);
		break;
	case cmd_bin:
		aegis_application_id_of_bin(binary, &appid_str);
		printf("%s\n", appid_str);
		aegis_crypto_free(appid_str);
		break;
	case cmd_pids:
		/*
		 * Run in a loop reading pids from stdin and 
		 * writing appids to stdout. Flush after output
		 * to make sure that this also works with pipes.
		 */
		while (true) {
			if (0 < scanf("%d", &pid)) {
				aegis_application_id(pid, &appid_str);
				printf("%s\n", appid_str);
				aegis_crypto_free(appid_str);
				fflush(stdout);
			}
		}
		break;
	case cmd_bins:
	    {
			char filename[512];
			while (NULL != fgets(filename, sizeof(filename), stdin)) {
				if (strlen(filename) && '\n' == filename[strlen(filename) - 1])
					filename[strlen(filename) - 1] = '\0';
				aegis_application_id_of_bin(filename, &appid_str);
				printf("%s\n", appid_str);
				aegis_crypto_free(appid_str);
				fflush(stdout);
			}
		}
		break;
	case cmd_usage:
		usage();
		break;
	default:
		;
	}

	aegis_crypto_finish();
	return 0;
}
