/* -*- mode:c; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of maemo-security-certman
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
 \file aecli.c
 \ingroup libcertman
 \brief A command-line utility for BB5-assisted cryptography

 This command-line utility can be used to encrypt, decrypt, sign 
 and verify data by using the primitives provided by BB5.
 
*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <pwd.h>
#include <termios.h>
#include <ctype.h>
#include <fcntl.h>
#include <signal.h>
#include <wait.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <grp.h>

#define LBB5_MAC_WITH_PROGRAMID
#ifdef USE_BB5
#define HEADER_SHA_H
#include <libbb5.h>
#define SHA_DIGEST_LENGTH 20
#endif
#ifdef USE_CREDS
#include <sys/creds.h>
#endif
#ifdef USE_BB5
#include <libbb5.h>
#endif

#include <openssl/pem.h>
#include <openssl/pkcs12.h>
#include <openssl/evp.h>
#include <openssl/aes.h>
#define EVPOK 1
#include <openssl/err.h>
#include <openssl/ssl.h>

#ifndef sk_STORE_OBJECT_num
#define sk_STORE_OBJECT_num(st) SKM_sk_num(STORE_OBJECT, (st))
#endif

#define ENABLE_MAEMO_DEBUG_OUTPUT

#define AEGIS_SHOW_ERRORS
#include <aegis_common.h>
#include <aegis_crypto.h>
#include <aegis_storage.h>

#if 0
#undef AEGIS_ERROR
#define AEGIS_ERROR(format,args...) \
	do {\
        fprintf(stderr, "%s(%d)[%s]: ERROR: " format "\n", \
                bare_file_name(__FILE__), __LINE__, __func__ ,##args);  \
	} while (0)
#endif

typedef enum {
    fmt_raw,
    fmt_hex,
    fmt_base64
} signature_format;

/* For debugging: sign directly data, not hash over it.
 */
static int direct = 0;

/* Output only parsable data
 */
static int quiet = 0;

/* Run crypto performance tests
 */
#define MIN_BUF 16
#define MAX_BUF 65536
static struct cresult {
    int8_t md[SHA_DIGEST_LENGTH];
} results[16];
#define min(x,y) (x<y?x:y)

static unsigned char aes_sink[MAX_BUF];
#ifdef USE_BB5
static bb5_aes_ctx *gctx1 = NULL;
#endif
static AES_KEY gctx2;
static unsigned char symkey[32];
static unsigned char ivec[AES_BLOCK_SIZE];
static int inc_factor = 0;

#define NBROF_TESTS 8
#define NBROF_LENS 13
static double tput[NBROF_TESTS * NBROF_LENS];

const char *test_names [] = {
	"HMAC by FPROT",
	"SHA1 by OMAP3",
	"SHA1 by OpenSSL",
	"AES-CBC by FPROT",
	"AES-CBC by OMAP3",
	"AES-CBC by OpenSSL",
	"AES-CBC by OMAP3 (bare)",
	"AES-CBC by OpenSSL (bare)",
};

#define HMAC_BY_FPROT 0
#define SHA1_BY_OMAP3 1
#define SHA1_BY_OPENSSL 2
#define AES_CBC_BY_FPROT 3
#define AES_CBC_BY_OMAP3 4
#define AES_CBC_BY_OPENSSL 5
#define AES_CBC_BY_OMAP3_BARE 6
#define AES_CBC_BY_OPENSSL_BARE 7

typedef int crypto_test_cb_t(int8_t *data, size_t len, int8_t *result);

#define CRYPTO_TEST(o,m,c) do {											\
        int i = 0;														\
        if (!quiet) printf("%s\n", test_names[o]);						\
        for (len = min_buf; len <= max_buf; ) {							\
            if (!quiet) printf("    %6u bytes ", len);                  \
            if (0 != run_one_crypto_test(o, i, m, rounds, data,			\
										 len, results[i].md, c))		\
                break;													\
            i++;                                                        \
			if (!inc_factor) len <<= 1 ; else len += inc_factor;		\
        }                                                               \
    } while (0)

static int64_t unow()
{
    struct timeval now;
    if (0 > gettimeofday(&now, NULL)) {
        AEGIS_ERROR("%s: getttimeofday (%s)", __func__, strerror(errno));
        return 0;
    }
    return (int64_t)now.tv_sec * 1000000 + (int64_t)now.tv_usec;
}

static int
get_random_data(int8_t *buf, size_t len)
{
	int fd;
	ssize_t res = 0;
	fd = open("/dev/urandom", O_RDONLY);
	if (fd != -1) {
		res = read(fd, buf, len);
		if (0 < res) {
			len -= res;
		} else
			AEGIS_ERROR("cannot read /dev/urandom");
		close(fd);
	}
	while (len--) {
		*buf++ = rand() % 256;
		res++;
	}
	return(0);
}


#ifdef USE_BB5
static int
sha1_by_omap3(int8_t *data, size_t len, int8_t *result)
{
    bb5_sha_ctx *shactx;
    if (NULL != (shactx = bb5_sha1_init())) {
        bb5_sha1_update(shactx, (unsigned char*)data, len);
        bb5_sha1_final(shactx, (unsigned char*)result);
        return 0;
    }
    AEGIS_ERROR("%s: failed", __func__);
    return 1;
}


static int
aes_by_omap3(int8_t *data, size_t len, int8_t *result) 
{
    int rc;
    bb5_aes_ctx *ctx = bb5_aes_cbc_init(symkey, sizeof(symkey));
    if (NULL == ctx) {
        AEGIS_ERROR("%s: bb5_aes_cbc_init failed", __func__);
        return 1;
    }
    if (0 != (rc = bb5_aes_encrypt(ctx, (const unsigned char*)data, aes_sink, len, ivec))) {
        AEGIS_ERROR("%s: bb5_aes_encrypt failed (%d)", __func__, rc);
        return 1;
    }
    if (0 != (rc = bb5_aes_cleanup(ctx))) {
        AEGIS_ERROR("%s: bb5_aes_cleanup failed (%d)", __func__, rc);
        return 1;
    }
    memcpy(result, &aes_sink[len - AES_BLOCK_SIZE], AES_BLOCK_SIZE);
    return 0;
}

static int
aes_by_omap3_lesskeyset(int8_t *data, size_t len, int8_t *result) 
{
    int rc;
    if (0 != (rc = bb5_aes_encrypt(gctx1, (const unsigned char*)data, aes_sink, len, ivec))) {
        AEGIS_ERROR("%s: bb5_aes_encrypt failed (%d)", __func__, rc);
        return 1;
    }
    memcpy(result, &aes_sink[len - AES_BLOCK_SIZE], AES_BLOCK_SIZE);
    return 0;
}
#endif

static int
hmac_by_fprot(int8_t *data, size_t len, int8_t *result)
{
	struct aegis_signature_t signature;
	aegis_crypto_result rc = aegis_crypto_sign(data, len, NULL, &signature);
	if (aegis_crypto_ok != rc) {
        AEGIS_ERROR("%s: aegis_crypto_sign failed (%s)", __func__, 
					aegis_crypto_last_error_str());
        return 1;
	}
    memcpy(result, &signature, AES_BLOCK_SIZE);
	return 0;
}


static int
aes_by_fprot(int8_t *data, size_t len, int8_t *result)
{
	RAWDATA_PTR ciphertext = NULL;
	size_t ciplen = 0;
	aegis_crypto_result rc = aegis_crypto_encrypt(data, len, NULL, 
												  &ciphertext, &ciplen);
	if (NULL != ciphertext) {
		memcpy(result, ciphertext, AES_BLOCK_SIZE);
		aegis_crypto_free(ciphertext);
	}
	if (aegis_crypto_ok != rc) {
        AEGIS_ERROR("%s: aegis_crypto_encrypt failed (%s)", __func__, 
					aegis_crypto_last_error_str());
        return 1;
	}
	return 0;
}


static int
sha1_by_openssl(int8_t *data, size_t len, int8_t *result)
{
	EVP_MD_CTX mdctx;
	unsigned int mdlen;
    int rc;

	if (EVPOK == EVP_DigestInit(&mdctx, EVP_sha1())) {
        rc = EVP_DigestUpdate(&mdctx, data, len);
        if (EVPOK != rc) {
            AEGIS_ERROR("%s: EVP_DigestUpdate returns %d (%d)", __func__, 
                        rc, errno);
            return 1;
        }
        rc = EVP_DigestFinal(&mdctx, (unsigned char*)result, &mdlen);
        if (rc != EVPOK) {
            AEGIS_ERROR("EVP_DigestFinal returns %d (%d)", rc, errno);
            return 1;
        }
        EVP_MD_CTX_cleanup(&mdctx);
        return 0;
    }
    AEGIS_ERROR("%s: failed", __func__);
    return 1;
}


static int
aes_by_openssl(int8_t *data, size_t len, int8_t *result) 
{
    int rc;
    AES_KEY ctx;
    static unsigned char livec[AES_BLOCK_SIZE];
    if (0 != (rc = AES_set_encrypt_key(symkey, sizeof(symkey) << 3, &ctx))) {
        AEGIS_ERROR("%s: AES_set_encrypt_key (%d)", __func__, rc);
        return 1;
    }
    memcpy(livec, ivec, sizeof(livec));
    AES_cbc_encrypt((const unsigned char*)data, aes_sink, len, 
                    &ctx, livec, AES_ENCRYPT);
    memcpy(result, &aes_sink[len - AES_BLOCK_SIZE], AES_BLOCK_SIZE);
    memset(&ctx, '\0', sizeof(ctx));
    return 0;
}

static int
aes_by_openssl_lesskeyset(int8_t *data, size_t len, int8_t *result) 
{
    static unsigned char livec[AES_BLOCK_SIZE];
    memcpy(livec, ivec, sizeof(livec));
    AES_cbc_encrypt((const unsigned char*)data, aes_sink, len, 
                    &gctx2, livec, AES_ENCRYPT);
    memcpy(result, &aes_sink[len - AES_BLOCK_SIZE], AES_BLOCK_SIZE);
    return 0;
}


static int
run_one_crypto_test(int tst_nbr,
					int rnd_nbr,
					int verify,
                    long rounds, 
                    int8_t *data, 
                    size_t len, 
                    int8_t *result,
                    crypto_test_cb_t callback)
{
    long i;
    struct cresult tmp;
    int8_t *tres;
    if (verify) {
        tres = tmp.md;
    } else {
        tres = result;
        memset(tres, '\0', sizeof(struct cresult));
    }

    int64_t started = unow();
    for (i = 0; i < rounds; i++) {
        if (0 != callback(data, len, tres)) {
            printf("failed\n");
            return 1;
        }
    }
    double msecs = (double)(unow() - started)/1e3;
    double kilob = ((double)rounds * (double)len)/1e3;
    double throughput = kilob/msecs;
	if (tst_nbr < NBROF_TESTS && rnd_nbr < NBROF_LENS)
		tput[tst_nbr * NBROF_LENS + rnd_nbr] = throughput;

    if (!quiet) {
        if (0 != msecs) {
            printf(" %10.3f ms (%8.4f MB/s) %s\n", msecs, throughput, dynhex(tres, 16));
        } else {
            printf(" %10.3f ms\n", msecs);
        }
        if (verify) {
            if (0 != memcmp(tres, result, 16)) {
                AEGIS_ERROR("expected result: %s\n", dynhex(result, 16));
            }
        }
    }
    return 0;
}


static void
run_crypto_tests(int test_setup, long rounds, size_t buflens)
{
    int8_t *data = (int8_t*)malloc(MAX_BUF);
    size_t len = 0, min_buf, max_buf;
    int rc;

    if (0 == buflens) {
        min_buf = MIN_BUF;
        max_buf = MAX_BUF;
    } else {
        min_buf = max_buf = buflens;
		if (inc_factor)
			max_buf = MAX_BUF;
    }

	memset(tput, '\0', sizeof(tput));
    get_random_data(data, MAX_BUF);
    get_random_data((int8_t*)symkey, sizeof(symkey));
    get_random_data((int8_t*)ivec, sizeof(ivec));
    memset(aes_sink, '\0', sizeof(aes_sink));

    if (!quiet) {
        printf("Running crypto tests (%lu rounds)\n  Key:  %s\n  Ivec: %s\n", 
               rounds,
               dynhex(symkey, sizeof(symkey)),
               dynhex(ivec, sizeof(ivec)));
    } 

    if (0x10 > test_setup) {
        /* Run all tests.
         */
        if (test_setup & 0x01) {
            CRYPTO_TEST(HMAC_BY_FPROT, 0, hmac_by_fprot);
            CRYPTO_TEST(SHA1_BY_OPENSSL, 0, sha1_by_openssl);
#ifdef USE_BB5
            CRYPTO_TEST(SHA1_BY_OMAP3, 1, sha1_by_omap3);
#endif
        }
        if (test_setup & 0x02) {
            CRYPTO_TEST(AES_CBC_BY_FPROT, 0, aes_by_fprot);
            CRYPTO_TEST(AES_CBC_BY_OPENSSL, 0, aes_by_openssl);
#ifdef USE_BB5
            CRYPTO_TEST(AES_CBC_BY_OMAP3, 1, aes_by_omap3);
#endif
        }
        if (test_setup & 0x04) {
            rc = AES_set_encrypt_key(symkey, sizeof(symkey) << 3, &gctx2);
            if (0 != rc) {
                AEGIS_ERROR("%s: AES_set_encrypt_key (%d)", __func__, rc);
            } else {
				CRYPTO_TEST(AES_CBC_BY_OPENSSL_BARE, 0, aes_by_openssl_lesskeyset);
			}
#ifdef USE_BB5
            gctx1 = bb5_aes_cbc_init(symkey, sizeof(symkey));
            if (NULL == gctx1) {
                AEGIS_ERROR("%s: omap3_aes_cbc_init failed", __func__);
            } else {
				CRYPTO_TEST(AES_CBC_BY_OMAP3_BARE, 1, aes_by_omap3_lesskeyset);
				bb5_aes_cleanup(gctx1);
			}
#endif
        }
    } else if (0x20 > test_setup) {
        /* Just run reference tests
         */
        if (test_setup & 0x01) 
            CRYPTO_TEST(SHA1_BY_OPENSSL, 0, sha1_by_openssl);
        if (test_setup & 0x02) 
			CRYPTO_TEST(AES_CBC_BY_OPENSSL, 0, aes_by_openssl);
        if (test_setup & 0x04) {
            rc = AES_set_encrypt_key(symkey, sizeof(symkey) << 3, &gctx2);
            if (0 != rc) {
                AEGIS_ERROR("%s: AES_set_encrypt_key (%d)", __func__, rc);
            } else {
				CRYPTO_TEST(AES_CBC_BY_OPENSSL_BARE, 1, aes_by_openssl_lesskeyset);
			}
        }
    } else if (0x30 > test_setup) {
        /* Just run chipset tests
         */
#ifdef USE_BB5
        if (test_setup & 0x01) 
            CRYPTO_TEST(SHA1_BY_OMAP3, 0, sha1_by_omap3);
        if (test_setup & 0x02) 
            CRYPTO_TEST(AES_CBC_BY_OMAP3, 0, aes_by_omap3);
        if (test_setup & 0x04) {
            gctx1 = bb5_aes_cbc_init(symkey, sizeof(symkey));
            if (NULL == gctx1) {
                AEGIS_ERROR("%s: omap3_aes_cbc_init failed", __func__);
            } else {
				CRYPTO_TEST(AES_CBC_BY_OMAP3_BARE, 1, aes_by_omap3_lesskeyset);
				bb5_aes_cleanup(gctx1);
			}
        }
#endif
    } else  {
        /* Just run FPROT tests
         */
        if (test_setup & 0x01) 
            CRYPTO_TEST(HMAC_BY_FPROT, 0, hmac_by_fprot);
        if (test_setup & 0x02) 
            CRYPTO_TEST(AES_CBC_BY_FPROT, 0, aes_by_fprot);
    }
    free(data);

	if (quiet) {
		size_t len = 16;
		int i, j;
		printf("{| cellspacing=\"0\" cellpadding=\"5\""
			   " border=\"1\"\n| Buffer/Algorithm\n");
		for (i = 0; i < NBROF_TESTS; i++) {
			printf("| %s\n", test_names[i]);
		}
		for (j = 0; j < NBROF_LENS; j++) {
			printf("|-\n");
			if (len < 1024)
				printf("| %d B\n", len);
			else
				printf("| %d kB\n", len/1024);
			for (i = 0; i < NBROF_TESTS; i++) {
				printf("| %.4f\n", tput[i * NBROF_LENS + j]);
			}
			len <<= 1;
		}
		printf("|}\n");
	}
}

/* File handling
 */
static void
write_file(int fh, const char* data)
{
    size_t len = strlen(data);
    ssize_t written = write(fh, data, len);
    if (written < (ssize_t)len) {
        AEGIS_ERROR("cannot write to fh(%d): %s (%ld written, %lu tried)", fh, 
                    strerror(errno), (long)written, (unsigned long)len);
    }
}


static int 
open_file(const char *path, int mode)
{
    int fh = -1;

    AEGIS_DEBUG(1, "%s: %s mode %o", __func__, path, mode);
    if (O_RDONLY == mode) {
        fh = open(path, mode);
        if (-1 == fh) {
            AEGIS_ERROR("Cannot open '%s' for reading (%s)", path, strerror(errno));
        }
    } else if (O_WRONLY == mode) {
        fh = creat(path, 0644);
        if (-1 == fh) {
            AEGIS_ERROR("Cannot open '%s' for writing (%s)", path, strerror(errno));
        }
    }
    return fh;
}


static size_t
read_file_into_buffer(int ih, RAWDATA_RPTR data)
{
    struct stat fs;
    ssize_t len = 0;
    size_t olen = 0;

	if (fstat(ih, &fs) == -1) {
		AEGIS_ERROR("fstat (%s)", strerror(errno));
        return 0;
	}
    olen = fs.st_size;
    if (0 == olen) {
        /* Cannot get file size, the handle is a pipe or a socket
         */
        unsigned char lbuf[512];
        size_t bsize = sizeof(lbuf);
        unsigned char *tgt = (unsigned char*)malloc(bsize);

        AEGIS_DEBUG(1, "%s: reading data in 1/2 kB pieces", __func__);

        *data = tgt;
        do {
            len = read(ih, lbuf, sizeof(lbuf));
			if (0 > len && EINTR == errno) {
				AEGIS_DEBUG(1, "Got EINTR");
				continue;
			}
            if (0 < len) {
                olen += len;
                if (bsize < olen) {
                    unsigned char* nbuf;
                    AEGIS_DEBUG(3, "%s: increase buffer at %p(%lu)",
                                __func__, *data, (unsigned long)bsize);
                    bsize <<= 1;
                    nbuf = (unsigned char*)realloc(*data, bsize);
                    AEGIS_DEBUG(3, "%s: new buffer at %p(%lu)",
                                __func__, nbuf, (unsigned long)bsize);
					if (NULL == nbuf) {
						AEGIS_ERROR("%s: realloc failed (%s)", __func__, 
									strerror(errno));
						free(*data);
						return 0;
					}
                    if (nbuf != *data) {
                        tgt = nbuf + (tgt - (unsigned char*)*data);
                        *data = nbuf;
                    }
                }
                memcpy(tgt, lbuf, len);
                tgt += len;
            }
        } while (0 < len);
        len = olen;

    } else {
        *data = (unsigned char*)malloc(olen);
        AEGIS_DEBUG(1, "%s: reading %lu bytes to %p", __func__, 
                    (unsigned long)olen, *data);
        len = read(ih, *data, olen);
    }
    if (len < (ssize_t)olen)
		AEGIS_ERROR("read truncated (%s)", strerror(errno));
    else
        AEGIS_DEBUG(1, "%s: got %lu bytes in %p", __func__, 
                    (unsigned long)len, *data);

    return len;
}


#define DIGESTTYP   EVP_sha1
#define DIGESTNAME "SHA1"

static int
compute_digest(int ih, unsigned char* digest, ssize_t maxdigestlen)
{
	EVP_MD_CTX mdctx;
	unsigned int mdlen;
	unsigned char data[512];
	int rc;
    ssize_t len;

    if (maxdigestlen < DIGESTLEN)
        return -EINVAL;

	rc = EVP_DigestInit(&mdctx, DIGESTTYP());
	if (EVPOK != rc) {
		AEGIS_ERROR("EVP_DigestInit returns %d", rc);
		return 0;
	}

    while (0 < (len = read(ih, data, sizeof(data)))) {
        rc = EVP_DigestUpdate(&mdctx, data, len);
        if (EVPOK != rc) {
            AEGIS_ERROR("EVP_DigestUpdate returns %d (%d)", rc, errno);
            return 0;
        }
    }

	rc = EVP_DigestFinal(&mdctx, digest, &mdlen);
	if (rc != EVPOK) {
		AEGIS_ERROR("EVP_DigestFinal returns %d (%d)", rc, errno);
		return 0;
	}

	EVP_MD_CTX_cleanup(&mdctx);

    return mdlen;
}


/*
 * Making a verifying a signature over a file. 
 * SHA1 hash over the data is signed.
 */

static void
create_signature(int ih, int oh, const char* resource_id, signature_format sfmt)
{
    struct aegis_signature_t signature;
    aegis_format_t msformat = aegis_as_base64;
    aegis_crypto_result res;

    AEGIS_DEBUG(1, "%s: enter", __func__);

    if (!direct) {
        int mdlen;
        unsigned char digest[DIGESTLEN];
        mdlen = compute_digest(ih, digest, sizeof(digest));
        if (0 >= mdlen) {
            return;
        } else {
            AEGIS_DEBUG(1, "%s: digest %s", __func__, dynhex(digest, mdlen));
        }
        res = aegis_crypto_sign(digest, mdlen, resource_id, &signature);
        if (aegis_crypto_ok != res) {
            AEGIS_ERROR("%s: %s", __func__, aegis_crypto_last_error_str());
            return;
        }
    } else {
        RAWDATA_PTR data = NULL;
        size_t len;

        len = read_file_into_buffer(ih, &data);
        if (NULL == data) {
            AEGIS_ERROR("could not read data (%s)", strerror(errno));
            return;
        }

        AEGIS_DEBUG(1, "%s: sign directly %lu bytes at %p", __func__,
                    (unsigned long)len, data);

        res = aegis_crypto_sign(data, len, resource_id, &signature);
        if (aegis_crypto_ok != res) {
            AEGIS_ERROR("%s: %s", __func__, aegis_crypto_last_error_str());
        }

        if (data)
            aegis_crypto_free(data);
    }

    if (aegis_crypto_ok != res)
        return;

    if (fmt_raw == sfmt) {
        if (sizeof(signature) > (size_t)write(oh, &signature, sizeof(signature)))
            AEGIS_ERROR("cannot write to fh(%d): %s", oh, strerror(errno));
        return;
    } else if (fmt_hex == sfmt)
        msformat = aegis_as_hexstring;

    char* str_sig = NULL;
    if (0 < aegis_crypto_signature_to_string(&signature, 
                                             msformat, 
                                             resource_id,
                                             &str_sig))
        write_file(oh, str_sig);

    write_file(oh, "\n");
    if (NULL != str_sig)
        aegis_crypto_free(str_sig);
}


static int
verify_signature(int ih, int sh, const char* resource_id)
{
    aegis_system_mode_t cmode;
    aegis_crypto_result erc;
    struct aegis_signature_t signature;
    char txt_signature [512];
    char *rid = NULL;
    ssize_t len;
    int res;

    len = read(sh, txt_signature, sizeof(txt_signature));
    if (0 >= len) {
        AEGIS_ERROR("invalid signature");
        return 0;
    } else if (sizeof(txt_signature) > (size_t)len) {
        txt_signature[len] = '\0';
    } else {
        AEGIS_DEBUG(1, "%s: signature truncated to %u bytes",
                       sizeof(txt_signature) - 1);
        txt_signature[sizeof(txt_signature) - 1] = '\0';
    }

    if (sizeof(signature) == len) {
        /*
         * Raw format signature
         */
        memcpy(&signature, txt_signature, sizeof(signature));
        rid = (char*)resource_id;

    } else {
        erc = aegis_crypto_string_to_signature(txt_signature,
                                               &signature,
                                               &rid);
        if (aegis_crypto_ok != erc) {
            AEGIS_ERROR("Invalid signature: '%s'", txt_signature);
            return 0;
        }
        AEGIS_DEBUG(1, "%s: signature made by '%s'", __func__, rid);
    }

    if (!direct) {
        int mdlen;
        unsigned char digest[DIGESTLEN];

        mdlen = compute_digest(ih, digest, sizeof(digest));
        if (0 >= mdlen) {
            return 0;
        } else {
            AEGIS_DEBUG(1, "%s: digest %s", __func__, dynhex(digest, mdlen));
            erc = aegis_crypto_verify(&signature, rid, digest, mdlen, &cmode);
        }
    
    } else {
        RAWDATA_PTR data = NULL;
        len = read_file_into_buffer(ih, &data);
        if (NULL == data) {
            AEGIS_ERROR("could not read data (%s)", strerror(errno));
            return 0;

        } else {
            AEGIS_DEBUG(1, "%s: verify directly %lu bytes at %p", __func__,
                        (unsigned long)len, data);
            erc = aegis_crypto_verify(&signature, rid, data, len, &cmode);
            aegis_crypto_free(data);
        }
    }

    switch (erc)
        {
        case aegis_crypto_ok:
			switch (cmode) {
			case aegis_system_protected:
                AEGIS_DEBUG(1, "%s: ok in normal mode", __func__);
				break;
			case aegis_system_open:
                AEGIS_DEBUG(1, "%s: ok in open mode", __func__);
				break;
			case aegis_system_emulated:
                AEGIS_DEBUG(1, "%s: ok with TEE emulator", __func__);
				break;
			case aegis_system_plain:
                AEGIS_DEBUG(1, "%s: ok without security framework", __func__);
				break;
			}
            res = 1;
            if (NULL != rid)
                printf("Signed by %s\n", rid);
            break;
        default:
            AEGIS_DEBUG(1, "%s: verification failed", __func__);
            res = 0;
        }

    if (rid != resource_id && NULL != rid)
        aegis_crypto_free(rid);

    return res;
}


static void
crypto_op(int ih, int oh, int enc, const char* resource_id)
{
    RAWDATA_PTR data = NULL;
    RAWDATA_PTR odata = NULL;
    size_t len = 0, olen = 0;
    aegis_crypto_result erc = aegis_crypto_ok;

    len = read_file_into_buffer(ih, &data);

    if (NULL == data) {
        AEGIS_ERROR("could not read data (%s)", strerror(errno));
        return;
    }

    if (0 < len) {
        if (enc)
            erc = aegis_crypto_encrypt(data, len, resource_id, (void**)&odata, &olen);
        else
            erc = aegis_crypto_decrypt(data, len, resource_id, (void**)&odata, &olen);
    }

    if (aegis_crypto_ok == erc && NULL != odata) {
        len = write(oh, odata, olen);
        if (len < olen)
            AEGIS_ERROR("write truncated (%s)", strerror(errno));
    }
    if (data)
        aegis_crypto_free(data);
    if (odata)
        aegis_crypto_free(odata);
}

/* TCB signing
 */
const char* tcb_files[] = {
	"/var/lib/aegis/domains",
	"/var/lib/aegis/refhashlist",
	"/var/lib/aegis/restok/restok.conf",
	"/var/lib/aegis/restok/restok.conf.backup"
};

struct sign_ctx {
	aegis::storage::visibility_t vis;
};


static int
tcb_sign_pstore_index(int nbr, void *name, void *context)
{
	struct sign_ctx *ctx = (struct sign_ctx*)context;
	aegis::storage pstore((const char*)name, ctx->vis, aegis::storage::prot_signed);
	return (int)aegis_crypto_sign_file(pstore.filename(), NULL, 0, "tcb");
}


static int
tcb_sign_all(void)
{
	size_t i;
	int rc;
	aegis_crypto_result res;

	if (aegis_system_plain != aegis_current_mode()) {
		AEGIS_ERROR("'tcb_sign_all' command can be used only in the build environment");
		return 1;
	}
	/* Sign fixed tcb files
	 */
	for (i = 0; i < sizeof(tcb_files)/sizeof(char*); i++) {
		res = aegis_crypto_sign_file(tcb_files[i], NULL, 0, "tcb");
		if (aegis_crypto_ok != res) {
			AEGIS_ERROR("failed to sign '%s' (%s)", tcb_files[i], strerror(errno));
			return (int)res;
		}
	}
	/* Sign global and shared pstore indices
	 */
	struct sign_ctx ctx;
	ctx.vis = aegis::storage::vis_global;
	rc = aegis::storage::iterate_storage_names(aegis::storage::vis_global, 
											   aegis::storage::prot_signed,
											   NULL, tcb_sign_pstore_index, 
											   &ctx);
	if (0 > rc)
		return 0 - rc;
	ctx.vis = aegis::storage::vis_shared;
	rc = aegis::storage::iterate_storage_names(aegis::storage::vis_shared, 
											   aegis::storage::prot_signed,
											   NULL, tcb_sign_pstore_index, 
											   &ctx);
	if (0 > rc)
		return 0 - rc;
	return 0;
}


#ifdef USE_CREDS
static creds_t
creds_of_a_binary(const char *path)
{
	creds_t child_creds = creds_gettask(0);
	int fd[2] = {-1, -1};
	pid_t cpid = (pid_t)-1;
	char c = 'a';

	/* Check if I have tcb
	 */
	creds_type_t c_type;
	creds_value_t c_value;
	c_type = creds_str2creds("tcb", &c_value);
	if (CREDS_BAD != c_type && 1 != creds_have_p(child_creds, c_type, c_value)) {
		printf("Warning: I don't have 'tcb' so the credentials list most likely is not complete.\n");
	}
	creds_free(child_creds);
	child_creds = NULL;
	
	AEGIS_DEBUG(1, "%s: '%s'", __func__, path);
	if (0 > pipe(fd)) {
		AEGIS_ERROR("%s: failed to pipe (%s)", __func__, strerror(errno));
		goto end;
	}
	cpid = fork();
	if (0 == cpid) {
		int pol_flags = 0, pol_mask = 0;
		pol_flags = credp_str2flags("set", &pol_mask);
		if (0 > pol_flags) {
			AEGIS_ERROR("%s: cannot convert policy flags (%s)", 
						__func__, strerror(errno));
		} else {
			AEGIS_DEBUG(1, "%s: pid %d policy %d", __func__, (int)getpid(), pol_flags);
		}
		long res = creds_confine2(path, pol_flags, NULL);
		AEGIS_DEBUG(1, "%s: pid %d creds_confine2 returned %ld (%s)", 
					__func__, getpid(), res, strerror(errno));
		if (0 > write(fd[1], &c, 1)) {
			AEGIS_DEBUG(1, "%s: failed to write to %d (%s)", 
						__func__, fd[1], strerror(errno));
		} else {
			pause();
		}
		AEGIS_DEBUG(1, "%s: exiting", __func__);
		exit(0);

	} else if (0 < cpid) {

		if (1 != read(fd[0], &c, 1)) {
			AEGIS_ERROR("%s: failed to read (%s)", __func__, strerror(errno));
			goto end;
		} else {
			AEGIS_DEBUG(1, "%s: got '%c'", __func__, c);
		}
		child_creds = creds_gettask(cpid);
		if (NULL == child_creds)
			AEGIS_ERROR("%s: failed to get child creds (%s)", __func__, strerror(errno));
		else
			AEGIS_DEBUG(1, "%s: got child creds (%d)", __func__, (int)child_creds);

		if (0 > kill(cpid, SIGTERM)) {
			/* The child switched uid and I don't have cap_kill. Never mind,
			 * it will die when I die.
			 */
			goto end;
		} else {
			int status;
			if (0 > wait(&status)) {
				AEGIS_ERROR("%s: failed to wait (%s)", __func__, strerror(errno));
			} else {
				if (WIFEXITED(status))
					AEGIS_DEBUG(1, "%s: child exited with %d", __func__, WEXITSTATUS(status));
				else 
					AEGIS_DEBUG(1, "%s: child died (%d)", __func__, status);
			}
		}
	} else {
		AEGIS_ERROR("%s: failed to fork (%s)", __func__, strerror(errno));
	}
end:
	if (-1 != fd[0])
		close(fd[0]);
	if (-1 != fd[1])
		close(fd[1]);
	return child_creds;
}
#endif

static void
usage(void)
{
	printf(
        "Usage:\n"
		"accli [-c <cmd>] [-r <token>] [-i <filename>]  [-o <filename>] "
        "[-f <x|b>] [-s <filename>]\n"
        "[-p <pid>] [-I] [-t <credential>]\n"
        " -c <cmd> for command: e=encrypt, d=decrypt, s=sign v=verify\n"
        " -i <filename> for input. If omitted, stdin is used.\n"
        " -s <filename> for signature input. If omitted, stdin is used.\n"
        " -o <filename> for output. If omitted, stdout is used\n"
        " -r <token> for the resource token name\n"
        "\t(must be owned by the process, use -I to check available tokens)\n"
        " -f <x|b|r> for signature-format: x=hex string, b=base64, r=raw 20 bytes binary\n"
        " -I show system info and current/given process's credentials\n"
        " -t <credential> to test if the current/given process has this credential\n"
        " -p <pid> to show or test the credentials of another process\n"
        " -b <filename> to show or test the credentials of a binary (requires tcb)\n"
        " -d to sign the input data in stead of a hash of it (direct mode)\n"
        " -x <path> to test of the given directory or file is in an AegisFS filesystem\n"
        " -u to list credentials at one line, suitable as input for aegis-su\n"
        "\tNote: when verifying, either -i or -s must be used,\n"
        "\tas both input data and signature cannot be read from stdin.\n"
		);
}


/**
 * \brief The main program
 * Execute the command without any parameters to get the help
 */

enum aecli_cmd_t {
    cmd_encrypt,
    cmd_decrypt,
    cmd_sign,
    cmd_verify,
    cmd_verify_aegisfs,
	cmd_tcb_sign,
	cmd_tcb_sign_all,
	cmd_tcb_verify,
	cmd_show_creds,
	cmd_run_tests
} aecli_cmd = cmd_sign;

int
main(int argc, char* argv[])
{
#ifdef USE_CREDS
    int rc, i;
    creds_t creds = NULL;
#endif
	int a, ih = STDIN_FILENO, sh = STDIN_FILENO, oh = STDOUT_FILENO, 
        fh = -1, return_status = 0;
    signature_format sfmt = fmt_hex;
    pid_t pid = -1;
    const char *resourceid = NULL;
	const char *tcb_file = NULL;
    size_t testbuflen = 0;
    long rounds = 1000;
	int test_setup = 0;
	int require_io = 0;
	int list_short = 0;
	RAWDATA_PTR data = NULL;
	size_t len = 0;

	if (1 == argc) {
		usage();
		return 2;
	}

    aegis_crypto_init();

#ifdef USE_CREDS
	/* By default show parent process's 
	 * credentials.
	 */
    creds = creds_gettask(getppid());
#endif

    while (1) {
		a = getopt(argc, argv, "c:r:i:o:s:f:hp:U:x:t:I?dqT:l:R:F:b:u");
		if (a < 0) {
			break;
		}
		switch(a) {
        case 'c':
            {
                char cmd = tolower(*optarg);
                if ('e' == cmd) {
                    aecli_cmd = cmd_encrypt;
					require_io = 1;
                } else if ('d' == cmd) {
                    aecli_cmd = cmd_decrypt;
					require_io = 1;
                } else if ('s' == cmd) {
                    aecli_cmd = cmd_sign;
					require_io = 1;
                } else if ('v' == cmd) {
                    aecli_cmd = cmd_verify;
					require_io = 1;
                } else if (0 == strcmp("tcb-sign", optarg)) {
					aecli_cmd = cmd_tcb_sign;
                } else if (0 == strcmp("tcb-sign-all", optarg)) {
					aecli_cmd = cmd_tcb_sign_all;
                } else if (0 == strcmp("tcb-verify", optarg)) {
					aecli_cmd = cmd_tcb_verify;
				} else {
                    AEGIS_ERROR("Invalid command: '%s'", optarg);
                    return 1;
                }
            }
            break;

        case 'i':
        case 's':
        case 'o':
            {
                const char* fname = optarg;
				int fh = -1;
                switch (a) {
                case 'i':
                    fh = ih = open_file(fname, O_RDONLY);
                    break;
                case 's':
                    if (cmd_sign == aecli_cmd)
                        fh = sh = open_file(fname, O_WRONLY);
                    else
                        fh = sh = open_file(fname, O_RDONLY);
                    break;
                case 'o':
                    fh = oh = open_file(fname, O_WRONLY);
                    break;
                }
				if (-1 == fh) {
					return 1;
				}
            }
            break;

        case 'r':
            resourceid = optarg;
            break;

        case 'f':
            {
                char fmt = tolower(*optarg);
                if ('x' == fmt) {
                    sfmt = fmt_hex;
                } else if ('b' == fmt) {
                    sfmt = fmt_base64;
                } else if ('r' == fmt) {
                    sfmt = fmt_raw;
                } else {
                    AEGIS_ERROR("Unknown signature format: '%c'", fmt);
                    return 1;
                }
            }
            break;

#ifdef USE_CREDS
        case 'p':
			pid = (pid_t)atol(optarg);
			creds = creds_gettask(pid);
			quiet = 1;
            break;

		case 'b':
			creds = creds_of_a_binary(optarg);
			quiet = 1;
			break;
#endif

        case 'I':
			aecli_cmd = cmd_show_creds;
            break;

        case 'd':
            direct = 1;
            break;

        case 'x':
            aecli_cmd = cmd_verify_aegisfs;
            resourceid = optarg;
            break;

        case 'q':
            quiet = 1;
            break;

		case 'u':
			list_short = 1;
			break;

#ifdef USE_CREDS 
        case 't':
            {
                creds_value_t cv = 0;
                long ct = creds_str2creds(optarg, &cv);
                if (CREDS_BAD != ct) {
                    return_status = 1 - creds_have_p(creds, ct, cv);
                    goto end;
                } else {
                    AEGIS_ERROR("unknown credential '%s'", optarg);
                    return_status = 2;
                    goto end;
                }
            }
            break;
#endif
		case 'F':
			tcb_file = optarg;
			break;

        case 'l':
            testbuflen = (size_t)atol(optarg);
            break;

        case 'R':
            rounds = atol(optarg);
            break;

        case 'T':
			aecli_cmd = cmd_run_tests;
			if (*optarg == 'r') {
				test_setup = 0x10;
				optarg++;
			} else if (*optarg == 'c') {
				test_setup = 0x20;
				optarg++;
			} else if (*optarg == 'f') {
				test_setup = 0x30;
				optarg++;
			}
			switch (*optarg) {
			case '1':
				test_setup += 1;
				break;
			case '2':
				test_setup += 2;
				break;
			case '3':
				test_setup += 4;
				break;
			default:
				test_setup += 7;
			}
			if (0 < testbuflen) {
				size_t mask = 1;
				int bits = 0;
				while (mask) {
					if (testbuflen & mask)
						bits++;
					mask <<= 1;
				}
				if (1 <= bits)
					inc_factor = 0x10;
			}
            break;

		default:
			usage();
			return 2;
		}
	}

    if (require_io && (-1 == ih || -1 == oh)) {
		AEGIS_ERROR("Not given input and output files");
        return_status = 1;
		goto end;
    }

    switch (aecli_cmd) {
	case cmd_verify:
		if (ih == sh) {
			AEGIS_ERROR("Input data and signature cannot both be read from stdin");
			return 1;
		}
		if (-1 != sh) {
			if (verify_signature(ih, sh, resourceid)) {
				if (!quiet)
					printf("Signature verified\n");
				return_status = 0;
			} else {
				if (!quiet)
					printf("Invalid signature\n");
				return_status = 1;
			}
		} else
			return_status = 1;
		break;

	case cmd_sign:
		if (STDIN_FILENO < sh) {
			if (STDOUT_FILENO < oh)
				close(oh);
			oh = sh;
		}
		create_signature(ih, oh, resourceid, sfmt);
		break;

	case cmd_encrypt:
	case cmd_decrypt:
		crypto_op(ih, oh, cmd_encrypt==aecli_cmd, resourceid);
		break;
		
	case cmd_verify_aegisfs: 
	    {
			int vres;
			aegis_system_mode_t cmode;
		
			vres = aegis_crypto_verify_aegisfs(resourceid, &cmode);
			if (0 == vres) {
				printf("'%s' is in AegisFS, system is in %s mode\n", 
					   resourceid, (aegis_system_open == cmode) ? "open" : "protected");
				return_status = 0;
			} else {
				printf("'%s' is NOT in AegisFS (%s)\n", 
					   resourceid, aegis_crypto_last_error_str());
				return_status = 1;
			}
		}
		break;

	case cmd_show_creds:            
		if (!quiet && !list_short) {
			printf("Current mode: ");
			switch (aegis_current_mode()) 
			{
			case aegis_system_protected:
				printf("%s\n", "normal");
				break;
			case aegis_system_open:
				printf("%s\n", "open");
				break;
			case aegis_system_plain:
				printf("%s\n", "no security framework");
				break;
			case aegis_system_emulated:
				printf("%s\n", "TEE emulator");
				break;
			}
			printf("IMEI: %s\n", aegis_system_invariant(sysinvariant_imei));
		}
#ifdef USE_CREDS
		/*
		 * Show credentials
		 */
		if (!creds) {
			AEGIS_ERROR("could not read credentials, quit");
			goto end;
		}
		if (!list_short) {
			printf("Credentials:\n");
			if (NULL != creds) {
				for (i = 0; i < 1000; i++) {
					creds_value_t cval;
					creds_type_t cred = creds_list(creds, i, &cval);
					if (CREDS_BAD != cred) {
						char cred_name[256];
						rc = creds_creds2str(cred, cval, cred_name, sizeof(cred_name));
						if (0 <= rc)
							printf("\t%s\n", cred_name);
					} else
						break;
				}
			}
		} else {
			char prefix = ' ';
			for (i = 0; i < 1000; i++) {
				creds_value_t cval;
				creds_type_t cred = creds_list(creds, i, &cval);
				if (CREDS_BAD != cred) {
					char cred_name[256];
					rc = creds_creds2str(cred, cval, cred_name, sizeof(cred_name));
					if (0 <= rc) {
						char use_prefix = ' ';
						char *arg = NULL;
#define has_prefix(s,p) (strlen(s) >= strlen(p) && (0 == memcmp(s,p,strlen(p))))
						if (has_prefix(cred_name, "UID::")) {
							use_prefix = 'u';
							arg = cred_name + 5;
						} else if (has_prefix(cred_name, "GRP::")) {
							use_prefix = 'g';
							arg = cred_name + 5;
						} else if (has_prefix(cred_name, "CAP::")) {
							use_prefix = 'c';
							arg = cred_name + 5;
						} else {
							use_prefix = 'r';
							arg = cred_name;
						}
						if (use_prefix != prefix) {
							printf(" -%c %s", use_prefix, arg);
							prefix = use_prefix;
						} else {
							printf(",%s", arg);
						}
					} else
						break;
				}
			}
		}
#endif
		break;
		
	case cmd_run_tests:
		run_crypto_tests(test_setup, rounds, testbuflen);
		break;

	case cmd_tcb_sign: 
		if (NULL == tcb_file) {
			AEGIS_ERROR("Must give filename with -F");
			return_status = 1;
			goto end;
		}
		if (-1 != ih) {
			len = read_file_into_buffer(ih, &data);
		}
		if (NULL == resourceid)
			resourceid = "tcb";
		return_status = (int)aegis_crypto_sign_file(tcb_file, 
													data, 
													len,
													resourceid);
		if (NULL != data)
			aegis_crypto_free(data);
		if ((int)aegis_crypto_ok != return_status) {
			AEGIS_ERROR("cannot sign '%s' (%s)", tcb_file, strerror(errno));
		}
		break;

	case cmd_tcb_sign_all:
		return_status = tcb_sign_all();
		break;

	case cmd_tcb_verify:
		if (NULL == tcb_file) {
			AEGIS_ERROR("Must give filename with -F");
			return_status = 1;
			goto end;
		}
		if (NULL == resourceid)
			resourceid = "tcb";

		fh = open_file(tcb_file, O_RDONLY);
		if (-1 != fh) {
			len = read_file_into_buffer(fh, &data);
			close(fh);
		} else {
			AEGIS_ERROR("cannot open '%s' (%s)", tcb_file, strerror(errno));
		}
		return_status = (int)aegis_crypto_verify_file(tcb_file, 
													  data, 
													  len,
													  resourceid);
		if ((int)aegis_crypto_ok == return_status) {
			if (!quiet) {
				ssize_t plen = 0;
				char *tmp = (char*)data;
				while (0 < len) {
					plen = write(oh, tmp, len);
					if (0 < plen) {
						if (plen < len)
							AEGIS_DEBUG(1, "Partial write (%lu/%lu)", (unsigned long)plen, 
										(unsigned long)len);
						len -= plen;
						tmp += plen;
					} else if (EINTR != errno) {
						AEGIS_ERROR("cannot write (%s)", strerror(errno));
						return_status = aegis_crypto_error;
						break;
					} else {
						AEGIS_DEBUG(1, "Got EINTR");
					}
				}
			}
		}
		if (NULL != data)
			aegis_crypto_free(data);
		break;
		
	default:
		;
    }

end:
#ifdef USE_CREDS
	if (creds)
		creds_free(creds);
#endif

    if (STDIN_FILENO < sh) close (sh);
    if (STDIN_FILENO < ih) close (ih);
    if (STDOUT_FILENO < oh) close (oh);

    aegis_crypto_finish();

	return return_status;
}
