/* -*- mode:c++; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of Aegis crypto services
 *
 * Copyright (C) 2010-2011 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <aegis_crypto.h>

#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <malloc.h>
#include <elf.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <utime.h>
#include <sys/fcntl.h>
#include <sys/xattr.h>

#ifdef USE_BB5
#define LBB5_MAC_WITH_PROGRAMID
#include <libbb5.h>
#endif

#include <openssl/sha.h>
#include <openssl/rand.h>
#include <openssl/md5.h>
#include <openssl/aes.h>
#include <openssl/bio.h>
#include <openssl/pem.h>
#include <openssl/err.h>
#include <openssl/ssl.h>

#include <aegis_storage.h>

/* Use this private method from aegis_storage 
 */
bool has_token(const char *token, bool *valid_token);

using namespace std;
using namespace aegis;

#ifdef USE_CREDS
#include <sys/creds.h>
#endif

#ifdef USE_BB5
/* Libbb5.h should define itself how large
 * pieces of data it can handle. These are
 * based on experiments.
 */
#define EBUF_ERROR EMSGSIZE

#define BB5_ERROR do {											\
	uint32_t elib = 0, erom = 0, epa = 0;						\
	if (0 > bb5_read_error(&elib, &erom, &epa))					\
		AEGIS_ERROR("%s: BB5 ERROR(-1) lib=%d rom=%d pa=%d",	\
					__func__, elib, erom, epa);					\
	else														\
		AEGIS_ERROR("%s: BB5 ERROR lib=%d rom=%d pa=%d",		\
					__func__, elib, erom, epa);					\
	} while (0)
	
#else
/*
 * A mock implementation of the used BB5/FPROT functions.
 */
#ifndef uint64_t
#define uint64_t unsigned long long
#endif

#define BB5_ERROR

enum bb5_fprot_mac_result {
        BB5_FPROT_MAC_INVALID = 0,
        BB5_FPROT_MAC_VALID_TRUSTED,
        BB5_FPROT_MAC_SPARE,
        BB5_FPROT_MAC_VALID_UNTRUSTED,
};

static int bb5_open(char *pa_name);
int bb5_get_random(unsigned char *buf, size_t len);
static int  bb5_set_programid(const unsigned char *new_appl_id);
static ssize_t bb5_fprot_mac(const unsigned char *msg,
							 size_t len,
							 unsigned char **authcode);
static ssize_t bb5_fprot_check_mac(const unsigned char *prog_id,
                                   const unsigned char *msg,
                                   size_t len,
                                   const unsigned char *authcode,
                                   size_t authlen);
static ssize_t bb5_fprot_encrypt(const unsigned char *msg,
								 size_t len,
								 unsigned char **ciphertxt);
static ssize_t bb5_fprot_decrypt(const unsigned char *msg,
								 size_t len,
								 unsigned char **plaintxt);
typedef uint8_t bb5_imei_t[15];
static ssize_t bb5_imei_read(bb5_imei_t* tobuf);
static int bb5_close(void);

int bb5_rsakp_sign(EVP_MD_CTX* ctx, unsigned char* md, size_t maxlen);
ssize_t bb5_rsakp_decrypt(int set, int key,
                          const unsigned char *msg, size_t len,
                          unsigned char **plain);

static char bb5_filename[256] = "";

#endif

/**
 * \def SYMKEYLEN
 * \brief The length of the symmetric crypto key of the AES256
 * crypto algorithm
 */

#define SYMKEYLEN 32

/*
 * System invariant cache
 */
static map<string,string> ivar_cache;

/*
 * Improved error handling
 */
extern char* last_dlog_error_message;

/*
 * Data file root
 */
static const char libroot [] = "/var/lib/aegis";

#define DIGESTTYP EVP_sha1
// ASSERT(DIGESTLEN==SHA_DIGEST_LENGTH)

#ifndef EVPOK
#define EVPOK 1
#endif

/* Crypto subsystem initialization and finalization.
 * Do autoinit to be usable also from libraries and
 * such that do not have explicit start point.
 */

static pid_t init_pid = -1;
static bool bb5_usable = true;
static char* my_appid = NULL;

/* From aegis_common 
 */
extern int resolve_symlinks;

/* Initialize the cryptosystem separately for each
 * new forked process. The BB5 driver wants that 
 * bb5_open is done for each process separately,
 * otherwise it will start mixing their requests.
 */
#define AUTOINIT do {							\
		if (getpid() != init_pid) {				\
			if (-1 != init_pid)					\
				aegis_crypto_finish();			\
			aegis_crypto_init();				\
		}										\
	} while (0)

/* Local prototypes */
static bool
compute_digest(const unsigned char* data, 
			   const size_t bytes,
               struct aegis_digest_t *digest);

static unsigned char*
map_file(const char* pathname, int mode, int* fd, size_t* len);

static void
munmap_file(void* data, size_t len, int fd);

static bool
digest_of_file(const char *pathname, 
			   struct aegis_digest_t *digest);

/*
 * BB5 emulator
 * ============
 */
#ifndef USE_BB5

static const char root_crt_name [] = "$HOME/.maemosec-secure/root.ca";
static const char root_key_name [] = "$HOME/.maemosec-secure/root.key";

#if 0
static X509*      root_crt = NULL;
static EVP_PKEY*  root_key = NULL;
#endif

static RAWDATA_PTR g_aes_plainkey = NULL;
static struct aegis_digest_t g_progdigest;

static c_xmlnode* bb5_emu = NULL;
static c_xmldoc   bb5_file;

static int
find_or_add(const char* path,
             const char* default_value,
             char* to_buf,
             size_t maxlen)
{
    c_xmlnode *val;

    if (NULL == bb5_emu) {
        AEGIS_ERROR("bb5_open not called yet");
        return (ssize_t)-1;
    }
    val = bb5_emu->navigate(path, default_value);
    if (NULL != val) {
        strncpy(to_buf, val->content(), maxlen);
        *(to_buf + maxlen - 1) = '\0';
        return 0;
    }
    AEGIS_ERROR("what hell?");
    return 0;
}


#if 0
static void
load_root_certificate()
{
	X509_LOOKUP *lookup = NULL;
    X509_STORE* to_this;
	int rc;

    to_this = X509_STORE_new();
	lookup = X509_STORE_add_lookup(to_this, X509_LOOKUP_file());
	if (lookup == NULL) {
		AEGIS_ERROR("cannot add lookup");
		// print_openssl_errors();
		goto end;
	}

	rc = X509_LOOKUP_load_file(
		lookup,
		root_crt_name,
		X509_FILETYPE_PEM);

	if (0 == rc) {
		AEGIS_DEBUG(1, "cannot load root certificate from '%s'", root_crt_name);
	} else {
		X509_OBJECT* obj;
		AEGIS_DEBUG(1, "loaded root ca from '%s'", root_crt_name);
		obj = sk_X509_OBJECT_value(to_this->objs, 0);
		if (obj && obj->type == X509_LU_X509)
			root_crt = X509_dup(obj->data.x509);
		else
			AEGIS_ERROR("cannot find root certificate");
	}
    X509_STORE_free(to_this);
  end:
	;
}

static void
load_root_key(void)
{
	BIO* keyfile = NULL;

	keyfile = BIO_new(BIO_s_file());

	if (!keyfile) {
		AEGIS_ERROR("cannot create BIO");
		return;
	}

	// TODO: there are many different formats for keys
	if (BIO_read_filename(keyfile, root_key_name) <= 0) {
		AEGIS_ERROR("cannot load root CA key from '%s' (%s)",
					   root_key_name, strerror(errno));
		// print_openssl_errors();
		return;
	}
	root_key = PEM_read_bio_PrivateKey(keyfile, NULL, NULL, NULL);
	if (!root_key) {
		AEGIS_ERROR("Cannot load private key from '%s'", root_key_name);
	} else
		AEGIS_DEBUG(1, "loaded root key from '%s'", root_key_name);
	BIO_free(keyfile);
}
#endif

#define MASTERKEY_XPATH "/encryption/master-key/"

static int
bb5_open(char *pa_name)
{
    c_xmlnode* mkey = NULL;

    AEGIS_DEBUG(1, "%s: enter", __func__);

    memset(&g_progdigest, '\0', sizeof(g_progdigest));
    snprintf(bb5_filename, sizeof(bb5_filename),
             "%s/.bb5-emu.xml", "/var/lib/aegis");

    if (file_exists(bb5_filename)) {
        bb5_file.parse_file(bb5_filename);
        bb5_emu = bb5_file.root();
        if (bb5_emu)
            mkey = bb5_emu->navigate(MASTERKEY_XPATH, false);
    }

    if (NULL == mkey) {
        /* Do NOT generate a new master key every time, but use
         * a fixed value instead. The purpose is to emulate BB5,
         * and the device keys do not change either in every flash.
         */
#if 0
        unsigned char new_aes_key[SYMKEYLEN];
#endif
        char* tmp;
		bb5_imei_t imei;

        bb5_emu = bb5_file.create("BB5-Emulator");
		bb5_imei_read(&imei);
#if 0
        bb5_get_random(new_aes_key, SYMKEYLEN);
        AEGIS_DEBUG(1, "%s: created new encryption key", __func__);
        tmp = base64_encode(new_aes_key, SYMKEYLEN);
        memset(new_aes_key, '\0', SYMKEYLEN);
#else
        /* Use the same master key as in the DAYPD release.
         */
        tmp = strdup("k+qm/Xn6c93MNY1ae2PZGaEXdhb4yqSJQ1qAdPRojyQ=");
#endif
        mkey = bb5_emu->navigate(MASTERKEY_XPATH, tmp);
        free(tmp);
        if (NULL == mkey || 0 == bb5_file.save(bb5_filename)) {
            AEGIS_DEBUG(1, "%s: cannot save '%s' (%s)", __func__,
                           bb5_filename, strerror(errno));
            return -1;
        }
    }

    size_t res = base64_decode(mkey->content(), (RAWDATA_RPTR)&g_aes_plainkey);
    if (SYMKEYLEN == res) {
        AEGIS_DEBUG(1, "%s: got master encryption key", __func__);
		return 0;
    } else {
        AEGIS_ERROR("%s: could not read master encryption key,"
                    " corrupted '%s'?", __func__, bb5_filename);
		return -1;
    }
}


static int
bb5_close(void)
{
    if (0 < strlen(bb5_filename)) {
        if (0 == access(bb5_filename, W_OK))
            bb5_file.save(bb5_filename);
        bb5_file.release_content();
        bb5_emu = NULL;
    }
    if (g_aes_plainkey) {
        free(g_aes_plainkey);
        g_aes_plainkey = NULL;
    }
    return 0;
}


static int
set_programid(const unsigned char *new_appl_id)
{
    compute_digest(new_appl_id, strlen((char*)new_appl_id), &g_progdigest);
    AEGIS_DEBUG(3, "%s: progid '%s'", __func__,
                   dynhex(g_progdigest.d, sizeof(g_progdigest)));
    return 0;
}


static int
bb5_set_programid(const unsigned char *new_appl_id)
{
#if USE_CREDS
	creds_t creds = NULL;
	creds_type_t c_type;
	creds_value_t c_value;

	if (NULL != new_appl_id && 0 == strcmp((const char*)new_appl_id, UNKNOWN_APP_ID))
		goto allowed;

	c_type = creds_str2creds((const char*)new_appl_id, &c_value);
	if (CREDS_BAD == c_type) {
		AEGIS_ERROR("Invalid credential '%s'", new_appl_id);
		return EACCES;
	}
	creds = creds_gettask(0);
	if (1 != creds_have_p(creds, c_type, c_value)) {
		AEGIS_ERROR("Not having '%s'", new_appl_id);
        creds_free(creds);
		return EACCES;
	}
    creds_free(creds);
  allowed:
#endif
	return set_programid(new_appl_id);
}


static void
set_encryption_key(AES_KEY* to_this)
{
    unsigned char localkey[SYMKEYLEN];
    size_t pos = 0;

    if (NULL == g_aes_plainkey) {
        AEGIS_ERROR("Crypto not initialized");
        return;
    }
    memcpy(localkey, g_aes_plainkey, SYMKEYLEN);

    AEGIS_DEBUG(5, "%s: key '%s'", __func__, dynhex(localkey, sizeof(localkey)));
    AEGIS_DEBUG(5, "%s: g_progdigest '%s'", __func__,
				dynhex(g_progdigest.d, sizeof(g_progdigest)));

    for (size_t i = 0; i < sizeof(g_progdigest); i++) {
        localkey[pos++] ^= g_progdigest.d[i];
        if (SYMKEYLEN == pos)
            pos = 0;
    }
    AEGIS_DEBUG(5, "%s: key '%s'", __func__, dynhex(localkey, sizeof(localkey)));
    AES_set_encrypt_key(localkey, SYMKEYLEN << 3, to_this);
}


int
bb5_get_random(unsigned char *buf, size_t len)
{
	int fd;
	ssize_t res = 0;

	/*
	 * /dev/random blocks if there is not enough
	 * randomness available
	 */
	fd = open("/dev/urandom", O_RDONLY);
	if (fd != -1) {
		res = read(fd, buf, len);
		if (0 < res) {
			len -= res;
		} else
			AEGIS_ERROR("cannot read /dev/urandom");
		close(fd);
	}
	/*
	 * backup method
	 */
	while (len--) {
		*buf++ = rand() % 256;
		res++;
	}
	return(0);
}


static ssize_t
bb5_fprot_mac(const unsigned char *msg, size_t len, unsigned char **authcode)
{
    struct aegis_digest_t digest;
    unsigned char salt[AES_BLOCK_SIZE];
    AES_KEY key;

    if (NULL == msg || NULL == authcode || 0 == len) {
        errno = EINVAL;
        return -1;
    }
    compute_digest(msg, len, &digest);
    AEGIS_DEBUG(3, "%s: digest '%s'", __func__, dynhex(&digest, 
													   sizeof(digest)));
    set_encryption_key(&key);
    *authcode = (unsigned char*)malloc(sizeof(digest));
    memcpy(*authcode, &digest, sizeof(digest));
    if (NULL == *authcode)
        return 0;
	if (sizeof(digest) >= sizeof(salt)) {
		AES_encrypt(digest.d, salt, &key);
		memcpy(*authcode, salt, sizeof(salt));
	} else {
		AEGIS_ERROR("Digest is too small (%u < %u)",
					(unsigned)sizeof(digest), (unsigned)sizeof(salt));
	}
    memset(&key, '\0', sizeof(key));
    return(sizeof(struct aegis_signature_t));
}


static ssize_t
bb5_fprot_check_mac(const unsigned char *prog_id,
                    const unsigned char *msg,
                    size_t len,
                    const unsigned char *authcode,
                    size_t authlen)
{
    size_t res;
    unsigned char *ref;

    if (NULL == prog_id || NULL == msg || NULL == authcode || 0 == len) {
        errno = EINVAL;
        return -1;
    }

    set_programid(prog_id);
    res = bb5_fprot_mac(msg, len, &ref);
    if (authlen == res && 0 == memcmp(authcode, ref, res))
        res = BB5_FPROT_MAC_VALID_UNTRUSTED;
    else
        res = BB5_FPROT_MAC_INVALID;
    free(ref);
    return res;
}

/*
 * TODO: Change to work more closely the same way
 * as real FPROT PA.
 */
static ssize_t
local_cryptop(unsigned char *iv,
			  const unsigned char *msg,
			  size_t len,
			  unsigned char *tgt)
{
	size_t i;
	AES_KEY key;
    set_encryption_key(&key);
    while (AES_BLOCK_SIZE <= len) {
        AES_encrypt(iv, iv, &key);
        for (i = 0; i < AES_BLOCK_SIZE; i++) {
            *tgt++ = iv[i] ^ *msg++;
        }
        len -= AES_BLOCK_SIZE;
    }
    AES_encrypt(iv, iv, &key);
    memset(&key, '\0', sizeof(key));
    for (i = 0; i < len; i++)
        *tgt++ = iv[i] ^ *msg++;
	return len;
}


static ssize_t
bb5_fprot_encrypt(const unsigned char *msg,
				  size_t len,
				  unsigned char **ciphertxt)
{
    unsigned char *tgt;
	unsigned char iv[AES_BLOCK_SIZE];
    size_t totlen = len + AES_BLOCK_SIZE;

    if (NULL == msg || NULL == ciphertxt || 0 == len) {
        errno = EINVAL;
        return -1;
    }

    AEGIS_DEBUG(3, "%s: enter (%p,%u)", __func__, msg, len);
	/*
	 * Random initialization vector at the beginning
	 * of the message.
	 */
    *ciphertxt = tgt = (unsigned char*)malloc(totlen);
    AEGIS_DEBUG(3, "%s: malloc (%p,%d)", __func__, tgt, totlen);
	bb5_get_random(iv, AES_BLOCK_SIZE);
	memcpy(tgt, iv, AES_BLOCK_SIZE);
	local_cryptop(iv, msg, len, tgt + AES_BLOCK_SIZE);
    AEGIS_DEBUG(3, "%s: encrypted %u bytes into %u bytes",
				   __func__, len, totlen);
    return totlen;
}


static ssize_t
bb5_fprot_decrypt(const unsigned char *msg,
				  size_t len,
				  unsigned char **plaintxt)
{
    unsigned char *tgt;
	unsigned char iv[AES_BLOCK_SIZE];
    size_t totlen;

    if (NULL == msg || NULL == plaintxt || 0 == len) {
        errno = EINVAL;
        return -1;
    }

    AEGIS_DEBUG(3, "%s: enter (%p,%u)", __func__, msg, len);
	if (AES_BLOCK_SIZE > len)
		return 0;
	else
		totlen = len - AES_BLOCK_SIZE;
    *plaintxt = tgt = (unsigned char*)malloc(totlen);
	memcpy(iv, msg, AES_BLOCK_SIZE);
	local_cryptop(iv, msg + AES_BLOCK_SIZE, totlen, tgt);
    AEGIS_DEBUG(3, "%s: decrypted %u bytes", __func__, totlen);
    return totlen;
}


#if 0
/*
 * These two are not used at the moment
 */
int
bb5_rsakp_sign(EVP_MD_CTX* ctx, unsigned char* md, size_t maxlen)
{
    int rc;
    unsigned int signlen = 0;
    unsigned char lmd[1024];

    if (!root_key) {
        AEGIS_ERROR("cannot sign: no private key");
        return(0);
    }

    rc = EVP_SignFinal(ctx, lmd, &signlen, root_key);
    if (rc != 1) {
        AEGIS_ERROR("signing failed");
        // print_openssl_errors();
        return(0);
    }
    if (signlen <= maxlen) {
        memcpy(md, lmd, signlen);
        return(signlen);
    } else {
        AEGIS_ERROR("signature buffer overflow (%d > %d)", signlen, maxlen);
        return(-ENOMEM);
    }
}

ssize_t
bb5_rsakp_decrypt(int set,
                  int key,
                  const unsigned char *msg,
                  size_t len,
                  unsigned char **plain)
{
    ssize_t res;
    RSA *rsakey = EVP_PKEY_get1_RSA(root_key);

    if (!rsakey) {
        AEGIS_ERROR("No RSA key available");
        return(-1);
    }
    *plain = (unsigned char*)malloc(RSA_size(rsakey));
    if (!*plain) {
        AEGIS_ERROR("cannot malloc");
        return(-1);
    }
    res = RSA_private_decrypt(len, msg, *plain, rsakey, RSA_PKCS1_PADDING);
    RSA_free(rsakey);
    return(res);
}
#endif

/*
 * XML file to emulate secure storage
 */

#define IMEI_PATH "/system-invariants/IMEI/"
#define IMEI_FALLBACK "12345678901234"
#define IMEI_LEN strlen(IMEI_FALLBACK)

static ssize_t
bb5_imei_read(bb5_imei_t *tobuf)
{
	char txtbuf[15];
	uint8_t *to = tobuf[0];

    find_or_add(IMEI_PATH, IMEI_FALLBACK, txtbuf, sizeof(txtbuf));
	AEGIS_DEBUG(3, "%s: %s", __func__, txtbuf);
    if (strlen(txtbuf) < sizeof(bb5_imei_t)) {
        memset(to, '0', sizeof(bb5_imei_t) - strlen(txtbuf));
        memcpy(to + (sizeof(bb5_imei_t) - strlen(txtbuf)), txtbuf, strlen(txtbuf));
    } else {
        memcpy(to, txtbuf, sizeof(bb5_imei_t));
    }
    for (unsigned i = 0; i < sizeof(bb5_imei_t); i++)
        *to++ -= (uint8_t)'0';
    return 0;
}
#else

#ifdef USE_HW_CRYPTO_ACCEL
/* BB5 does not implement all the same functions as OpenSSL
 */
static void
SHA1(void *data, size_t len, unsigned char *md)
{
    bb5_sha_ctx *shactx;

    if (NULL != (shactx = bb5_sha1_init())) {
        bb5_sha1_update(shactx, (unsigned char*)data, len);
        bb5_sha1_final(shactx, md);
    } else {
        AEGIS_ERROR("bb5_sha1_init from BB5 failed");
    }
}
#endif
#endif // ifndef USE_BB5 - BB5 emulator

#ifdef USE_MD5_LOOKUP

/*
 * This is intermediate code to provide the application id
 * functionality by help of md5sums-files.
 */

typedef struct md5_t {
	uint8_t d[16];
} md5_t;

class pitree
{
public:
	typedef void callback_t(string& filename,
							md5_t *csum,
							off_t pkgi,
							void* ctx);
	pitree();
	~pitree();
	void add(string *filename, md5_t *csum, off_t pkgi);
	size_t size();
	void iterate(callback_t cb, void* ctx);
private:
	struct procinfo_t {
		md5_t             md5sum;
		string            *binname;
		off_t             pkgi;
		struct procinfo_t *left;
		struct procinfo_t *right;
	} *root;
	void local_iterate(struct procinfo_t* node, callback_t cb, void* ctx);
	void local_cleanup(struct procinfo_t* node);
	size_t m_count;
};

/*
 * A lookup table for finding out the package and
 * signer of an executable. Backed up by a file.
 */
class bin_lookup
{
public:
	bin_lookup();
	~bin_lookup();
	bool init();
	bool create(const char* filename);
	void finish();
	void print();
	bool lookup(pid_t pid, string& appid);
	bool lookup(const char* pathname, string& appid);

	typedef struct lin_md5i_t {
		md5_t  csum;
		off_t  pkgname_off;
	} md5i_t;

private:
#define MAGIC "Appid lookup table-1.0"
	size_t parse_hex(const char* str, size_t maxlen, md5_t *to);
	bool is_elf_executable(const char* filename);
	size_t read_md5sums(string& packagename, off_t pkg_off);
	void test_prefix(char *in_data, const char *prefix, string& valto);
	off_t make_packagelist(void);
	bool md5sum_of(const char* pathname, md5_t* to_this);
	bool make_appid(size_t sum_idx, string& to_this);
	bool lookup(md5_t* csum, string& appid);

	/*
	 * Data structures suitable for storing the
	 * data in disk with a single write.
	 */
	struct lookup_hdr_t {
		char   magic[32];
		size_t nrof_pkg;
		off_t  pkg_size;
		size_t nrof_csum;
	} *m_hdr;
	/*
	 * Package names and origins as strings
	 */
	char *m_pkginfo;
	/*
	 * A sorted array of md5sums for binary search
	 */
	md5i_t *m_csums;
	int m_fd;
	size_t m_len;
	pitree *m_stree;
    string m_filename;
};

static bin_lookup md_lookup;
#endif // USE_MD5_LOOKUP


/*
 * Utilities.
 */
static int
report_openssl_error(const char* str, size_t len, void* u)
{
#if 0
	char* tmp = (char*)strrchr(str, '\n');
	if (tmp && ((tmp - str) == (int)strlen(str)))
		*tmp = '\0';
#endif
	AEGIS_DEBUG(0, "OpenSSL error '%s'", str);
	ERR_clear_error();
	return(0);
}


static bool
compute_digest(const unsigned char* data, 
			   const size_t bytes,
               struct aegis_digest_t *digest)
{
	EVP_MD_CTX mdctx;
	unsigned char md[DIGESTLEN];
	unsigned int mdlen;
	int rc;

	memset(digest, '\0', sizeof(struct aegis_digest_t));
	rc = EVP_DigestInit(&mdctx, DIGESTTYP());
	if (EVPOK != rc) {
		AEGIS_ERROR("EVP_DigestInit returns %d (%s)", rc, strerror(errno));
		return false;
	}

	// AEGIS_DEBUG(1, "computing digest over %d bytes", bytes);

	rc = EVP_DigestUpdate(&mdctx, data, bytes);
	if (EVPOK != rc) {
		AEGIS_ERROR("EVP_DigestUpdate returns %d (%d)", rc, errno);
        return false;
	}

	rc = EVP_DigestFinal(&mdctx, md, &mdlen);
	if (rc != EVPOK) {
		AEGIS_ERROR("EVP_DigestFinal returns %d (%d)", rc, errno);
		return false;
	}
	EVP_MD_CTX_cleanup(&mdctx);

	if ((int)mdlen != DIGESTLEN) {
		AEGIS_ERROR("Digestlen mismatch (%d != %d)", mdlen, DIGESTLEN);
		return false;
	}

#if 0
    AEGIS_DEBUG(1, "%s: %d bytes digest %s", __func__,
                   sizeof(struct aegis_signature_t),
                   dynhex(md, mdlen));
#endif

	memcpy(digest, md, sizeof(struct aegis_signature_t));
	return true;
}


static unsigned char*
map_file(const char* pathname, int mode, int* fd, size_t* len)
{
	int lfd = -1, mflags, mprot;
	struct stat fs;
	unsigned char* res;
	ssize_t llen = 0;

	if (NULL != pathname) {
		lfd = open(pathname, mode, 0600);
		if (0 > lfd) {
			AEGIS_DEBUG(3, "%s: cannot open '%s' (%s)",
						   __func__, pathname, strerror(errno));
			return NULL;
		} else {
			if (fstat(lfd, &fs) == -1) {
				close(lfd);
				AEGIS_ERROR("cannot stat '%s' (%s)", pathname,
							   strerror(errno));
				return NULL;
			}
			llen = fs.st_size;
		}
		if (O_RDONLY == mode) {
			if (0 == llen) {
				close(lfd);
				AEGIS_ERROR("'%s' is empty", pathname);
				return NULL;
			}
			mflags = MAP_PRIVATE;
			mprot = PROT_READ;
		} else {
			/* Notice: max 64k can be locked by the default
			 * ulimit. Don't bother to lock as these are no
			 * secrets.
			 */
			mflags = MAP_PRIVATE;
			mprot = PROT_READ | PROT_WRITE;
			if (llen < (ssize_t)*len) {
				/* Increase the file size to match the mapping.
				 */
#ifndef _XOPEN_UNIX
				char buf [4096];
				memset(buf, '\0', sizeof(buf));
				llen = lseek(lfd, llen, SEEK_SET);
				AEGIS_DEBUG(3, "%s: writing %d bytes", __func__,
							   *len - llen);
				while (llen < *len) {
					size_t wlen = sizeof(buf);
					if (llen + wlen > *len)
						wlen = *len - llen;
					llen += write(lfd, buf, wlen);
				}
				lseek(lfd, 0, SEEK_SET);
#else
				if (0 > ftruncate(lfd, *len)) 
					AEGIS_DEBUG(1, "%s: ftruncate to %ld failed (%s)", __func__,
								(long)*len, strerror(errno));
#endif
			}
		}
	} else {
		mflags = MAP_PRIVATE | MAP_ANONYMOUS;
		mprot = PROT_READ | PROT_WRITE;
		llen = *len;
		if (0 == llen) {
			AEGIS_ERROR("Cannot create anonymous map with no data");
			return NULL;
		}
	}

	AEGIS_DEBUG(5, "%s: '%s' is %d bytes long", __func__, pathname, llen);
	res = (unsigned char*)mmap(NULL, llen, mprot, mflags, lfd, 0);
	AEGIS_DEBUG(5, "%s: mapped at %p", __func__, res);

	if (MAP_FAILED == res) {
		if (-1 != lfd)
			if (0 > close(lfd))
				AEGIS_DEBUG(1, "%s: close failed (%s)", __func__, strerror(errno));
		AEGIS_ERROR("cannot mmap '%s' of %ld bytes (%s)", pathname, (long)llen,
					   strerror(errno));
		return NULL;
	}
	*len = llen;
	*fd = lfd;
	return res;
}


static void
munmap_file(void* data, size_t len, int fd)
{
	if (MAP_FAILED != data && 0 < len && 0 > munmap(data, len))
		AEGIS_DEBUG(1, "%s: munmap error (%s)", __func__, strerror(errno));
	if (-1 != fd && 0 > close(fd))
		AEGIS_DEBUG(1, "%s: close error (%s)", __func__, strerror(errno));
}


static bool
digest_of_file(const char *pathname, 
			   struct aegis_digest_t *digest)
{
	unsigned char *data = NULL;
	size_t len = 0;
	int fd = -1;
	bool res = false;

	if (NULL == pathname || NULL == digest) {
		AEGIS_ERROR("%s: invalid parameter", __func__);
		errno = EINVAL;
		return false;
	}
	data = map_file(pathname, O_RDONLY, &fd, &len);
	if (NULL != data) {
		res = compute_digest(data, len, digest);
		munmap_file(data, len, fd);
	} else {
		AEGIS_DEBUG(1, "%s: failed to open '%s' (%s)", __func__, pathname, 
					strerror(errno));
		return false;
	}
	return res;
}


/* Globally signed and distinct files
 */

typedef map<string, string> vdata_t;
const char aegis_vdata_file[] = "/etc/aegis_vdata";
static aegis_digest_t vdata_digest;
static bool vdata_digest_valid = false;

static void
get_vdata_digest(void)
{
#ifdef USE_BB5
	if (bb5_usable && !vdata_digest_valid) {
		struct bb5_sw_cert_v2 *swcert = NULL;
		size_t swlen = 0;
		int rc = bb5_swcert_read(&swcert, &swlen);
		AEGIS_DEBUG(1, "%s: bb5_swcert_read returned %d", __func__, rc);
		if (0 != rc || NULL == swcert) {
			AEGIS_ERROR("%s: failed to read the SW cert (%d)", 
						__func__, rc);
			BB5_ERROR;
		} else {
			for (size_t i = 0; i < swcert->entries; i++) {
				if (0 == strcmp("vdata", swcert->images[i].name)) {
					memcpy(&vdata_digest, swcert->images[i].sha, sizeof(vdata_digest));
					vdata_digest_valid = true;
					AEGIS_DEBUG(1, "%s: %s => %s", __func__, 
								swcert->images[i].name, 
								dynhex(&vdata_digest, sizeof(vdata_digest)));
				}
			}
		}
		if (NULL != swcert)
			free(swcert);
	}
#else
	digest_of_file(aegis_vdata_file, &vdata_digest);
	vdata_digest_valid = true;
#endif			
}

static aegis_crypto_result
read_aegis_vdata(vdata_t **to_this)
{
	char *data = NULL;
	int fd = -1;
	size_t len = 0;
	vdata_t *res = new vdata_t;
	aegis_digest_t vdigest;
	aegis_crypto_result erc;

	AEGIS_ENTER;
	data = (char*)map_file(aegis_vdata_file, O_RDONLY, &fd, &len);
	if (NULL == data) {
		AEGIS_DEBUG(1, "%s: cannot open '%s' (%s)", __func__, aegis_vdata_file,
					strerror(errno));
		*to_this = res;
		return aegis_crypto_ok;
	}
	get_vdata_digest();
	erc = aegis_crypto_error_signature_missing;
	if (bb5_usable && vdata_digest_valid) {
		if (compute_digest((unsigned char*)data, len, &vdigest)) {
			if (0 == memcmp(&vdigest, &vdata_digest, sizeof(vdata_digest))) {
				erc = aegis_crypto_ok;
			} else {
				AEGIS_ERROR("%s: hash of '%s' (%s) does not match hash in swcert (%s)",
							__func__, 
							aegis_vdata_file,
							dynhex(&vdigest, sizeof(vdigest)), 
							dynhex(&vdata_digest, sizeof(vdata_digest)));
				erc = aegis_crypto_error_wrong_signature;
			} 
		} else {
			AEGIS_DEBUG(1, "%s: cannot compute digest of '%s' (%s)", __func__, 
						aegis_vdata_file, strerror(errno));
			munmap_file(data, len, fd);
			return aegis_crypto_error_signature_missing;
		}
	}

	if (aegis_crypto_ok != erc) {
		/* SW Cert could not be read or it didn't match. Check for
		 * a local signature before giving up.
		 */
		erc = aegis_crypto_verify_file(aegis_vdata_file, data, len, "tcb");
		if (aegis_crypto_ok != erc) {
			munmap_file(data, len, fd);
			return erc;
		}
	}

#ifdef AEGIS_DEBUG_ENABLED
	int line = 1;
#endif
	char *c = data, *end = data + len;
	string aname;
	string digest;

	while (c && c < end) {
		char *eol, *sep, *str;
		
		sep = strchr(c, ' ');
		if (!sep) {
			AEGIS_DEBUG(1, "%s: broken vdata file at line %d", __func__, line);
			goto end;
		}
		for (str = sep + 1; (' ' == *str || '*' == *str) && str < end; str++);
		if (str == end || isspace(*str)) {
			AEGIS_DEBUG(1, "%s: broken vdata file at line %d", __func__, line);
			goto end;
		}
		eol = strchr(str + 1, '\n');
		if (!eol) {
			if (2 * DIGESTLEN != strlen(str)) {
				AEGIS_DEBUG(1, "%s: broken vdata file at line %d", __func__, line);
				goto end;
			} else {
				eol = str + 2 * DIGESTLEN;
			}
		}
		aname.assign(str, eol - str);
		digest.assign(c, sep - c);
		(*res)[aname.c_str()] = digest.c_str();
		AEGIS_DEBUG(1, "%s: '%s' => '%s'", __func__, aname.c_str(), digest.c_str());
		if ('\n' == *eol)
			c = eol + 1;
		else
			break;
	}
	munmap_file(data, len, fd);
	
  end:
	*to_this = res;
	return aegis_crypto_ok;
}


static bool
save_aegis_vdata(vdata_t *vdata)
{
	AEGIS_DEBUG(1, "%s: enter", __func__);
	int count = 0;
	FILE* vdata_f = fopen(aegis_vdata_file, "w");
	if (NULL == vdata_f) {
		AEGIS_ERROR("%s: cannot open '%s' for writing (%s)", __func__,
					aegis_vdata_file, strerror(errno));
		return false;
	}
	for (
		vdata_t::const_iterator ii = vdata->begin();
		ii != vdata->end();
		ii++
	 ) {
		fprintf(vdata_f, "%s  %s\n", ii->second.c_str(), ii->first.c_str());
		count++;
	}
	fclose(vdata_f);
	AEGIS_DEBUG(1, "%s: saved %d hashes", __func__, count);
	return true;
}


aegis_crypto_result
verify_global_signature(const char *filename, struct aegis_digest_t *digest)
{
	aegis_crypto_result res = aegis_crypto_error;
	string hdigest, rdigest;
	struct aegis_digest_t fdigest;

	AEGIS_DEBUG(1, "%s: %s", __func__, filename);
	if (NULL == digest) {
		if (digest_of_file(filename, &fdigest)) {
			append_hex(hdigest, fdigest.d, sizeof(struct aegis_digest_t));
		} else {
			AEGIS_ERROR("%s: cannot compute digest of '%s' (%s)",
						__func__, filename, strerror(errno));
			res = aegis_crypto_error;
		}
	} else {
		append_hex(hdigest, digest->d, sizeof(struct aegis_digest_t));
		memcpy(&fdigest, digest, sizeof(struct aegis_digest_t));
	}

	AEGIS_DEBUG(1, "%s: computed hash %s", __func__, hdigest.c_str());

	if (0 == strcmp(filename, aegis_vdata_file)) {
		/* Verify the actual master hash file by comparing it's hash to the
		 * digest in the SW certificate.
		 */
		if (bb5_usable) {
			get_vdata_digest();
			if (vdata_digest_valid && 
				0 == memcmp(fdigest.d, &vdata_digest, sizeof(struct aegis_digest_t))) 
			{
				AEGIS_DEBUG(1, "%s: %s matches swcert", __func__, aegis_vdata_file);
				/* Make a local signature if possible
				 */
				bool is_valid = false;
				if (has_token("tcb", &is_valid) && is_valid) {
					if (aegis_crypto_ok == aegis_crypto_sign_file(
							aegis_vdata_file, 
							NULL, 
							0, 
							"tcb")) 
					{
						printf("Aegis: signed '%s' locally\n", aegis_vdata_file);
					} else {
						printf("Aegis: warning, failed to sign '%s' locally (%s)\n", 
							   aegis_vdata_file, 
							   aegis_crypto_last_error_str());
					}
				}
				return aegis_crypto_ok;
			} else if (vdata_digest_valid) {
				AEGIS_ERROR("%s: hash of '%s' (%s) does not match hash in swcert (%s)",
							__func__, 
							aegis_vdata_file,
							dynhex(fdigest.d, sizeof(struct aegis_digest_t)), 
							dynhex(&vdata_digest, sizeof(vdata_digest)));
				res = aegis_crypto_error_wrong_signature;
			} else {
				AEGIS_ERROR("%s: cannot verify '%s', cannot read SW cert", __func__, 
							aegis_vdata_file);
				return aegis_crypto_ok;
			}
		} else {
			AEGIS_ERROR("%s: cannot verify '%s', BB5 not usable", __func__, 
						aegis_vdata_file);
			return aegis_crypto_ok;
		}
	} else {
		/* Verify by comparing hash in the master hash file. 
		 * Function read_aegis_vdata verifies the hash file separately.
		 */
		vdata_t *vdata = NULL;
		res = read_aegis_vdata(&vdata);
		if (aegis_crypto_ok != res || NULL == vdata) {
			AEGIS_ERROR("%s: failed to load '%s'", __func__, aegis_vdata_file);
		} else {
			vdata_t::iterator ii = vdata->find(filename);
			if (vdata->end() != ii) {
				if (0 == strcmp(hdigest.c_str(), ii->second.c_str())) {
					AEGIS_DEBUG(1, "%s: match", __func__, aegis_vdata_file);
					res = aegis_crypto_ok;
				} else {
					AEGIS_DEBUG(1, "%s: mismatch, refhash %s", __func__, 
								ii->second.c_str());
					res = aegis_crypto_error_wrong_signature;
					rdigest.assign(ii->second.c_str());
				}
			} else {
				AEGIS_DEBUG(1, "%s: no hash found", __func__);
				res = aegis_crypto_error_signature_missing;
			}
			if (vdata)
				delete vdata;
		}
	}

	if (aegis_crypto_ok != res) {
		if (aegis_crypto_error_signature_missing == res)
			AEGIS_ERROR("%s: no signature for '%s' (%s)", __func__, filename, 
						hdigest.c_str());
		else if (aegis_crypto_error_wrong_signature == res)
			AEGIS_ERROR("%s: wrong signature (%s) for '%s' (%s)", __func__, 
						rdigest.c_str(), filename, hdigest.c_str());
		else if (0 != errno)
			AEGIS_ERROR("%s: generic error with '%s' (%s)", __func__, filename, 
						strerror(errno));
		else
			AEGIS_ERROR("%s: generic error with '%s'", __func__, filename);
	}
	return res;
}


bool
save_global_signature(const char *filename, struct aegis_digest_t *digest)
{
	string hdigest;
	bool res = false;

	AEGIS_DEBUG(1, "%s: %s", __func__, filename);
	if (NULL == digest) {
		struct aegis_digest_t fdigest;
		if (digest_of_file(filename, &fdigest)) {
			append_hex(hdigest, fdigest.d, sizeof(struct aegis_digest_t));
		} else {
			AEGIS_ERROR("%s: cannot compute digest of '%s' (%s)",
						__func__, filename, strerror(errno));
			return false;
		}
	} else {
		append_hex(hdigest, digest->d, sizeof(struct aegis_digest_t));
	}
    mode_t old_umask = umask(S_IWGRP | S_IWOTH);
    AEGIS_DEBUG(1, "%s: %s  %s", __func__, hdigest.c_str(), filename);
    vdata_t *vdata = NULL;
	if (aegis_crypto_ok == read_aegis_vdata(&vdata) && NULL != vdata) {
		(*vdata)[filename] = hdigest.c_str();
		res = save_aegis_vdata(vdata);
	}
    umask(old_umask);
	if (vdata)
		delete vdata;
	return res;
}


/* Visible functions */
extern "C" {

	int
	aegis_crypto_init(void)
	{
		int rc = 1;
		pid_t lpid = getpid();
#ifdef USE_BB5
#ifdef AEGIS_DEBUG_ENABLED		
		const char using_bb5[] = "libbb5";
#endif
#else
#ifdef AEGIS_DEBUG_ENABLED		
		const char using_bb5[] = "BB5 emulator";
#endif
		bb5_usable = true;
#endif
		if (lpid != init_pid) {
			AEGIS_DEBUG(1, "%s: use %s int size %d long size %d, long long size %d", 
						__func__, using_bb5, (int)sizeof(int), (int)sizeof(long), 
						(int)sizeof(long long));

			/* This should be done in a postinstall script, but to
			 * make it easier to test in non-debian environments
			 * let's do it here for a while.
			 */
			if (!directory_exists(libroot)) {
				if (0 > mkdir(libroot, 0755)) {
					AEGIS_DEBUG(1, "%s: cannot create '%s' (%s)", __func__,
								libroot, strerror(rc));
					return 0;
				}
			}

			ERR_load_crypto_strings();
			OpenSSL_add_all_algorithms();
			ERR_print_errors_cb(report_openssl_error, NULL);
			init_pid = lpid;

			rc = bb5_open(NULL);
			AEGIS_DEBUG(1, "%s: bb5_open returned %d", __func__, rc);

			if (0 == rc) {
				bb5_usable = true;
				atexit(aegis_crypto_finish);

#ifdef USE_MD5_LOOKUP
				/* Use MD5 sums as a backup method
				 */
				if (!md_lookup.init()) {
					AEGIS_ERROR("%s: couldn't init hash lookup", __func__);
					return 0;
				}
#endif
				aegis_application_id(0, &my_appid);
				AEGIS_DEBUG(1, "%s: I am '%s'", __func__, my_appid);
				/*
				 * TODO: Inconsistent return values, when "int" 0 is OK
				 * and others are error codes. Change this.
				 */
				rc = 1;
			} else {
#ifdef USE_BB5
				AEGIS_DEBUG(1, "%s: No BB5 available. Only global signing is possible", 
							__func__);
				bb5_usable = false;
				BB5_ERROR;
				rc = 1;
#else
				/* BB5 emulator open returned error? 
				 */
				rc = 0;
#endif
			}
		}
		if (!bb5_usable)
			resolve_symlinks = 0;
		return rc;
	}


	void
	aegis_crypto_finish(void)
	{
		if (-1 != init_pid) {
			AEGIS_DEBUG(1, "%s: enter", __func__);
			bb5_close();
			EVP_cleanup();
			X509_TRUST_cleanup();
			CRYPTO_cleanup_all_ex_data();
			ERR_remove_state(0);
			ERR_free_strings();
			if (NULL != my_appid) {
				aegis_crypto_free(my_appid);
				my_appid = NULL;
			}
			init_pid = -1;
		}
	}


    size_t
    aegis_crypto_signature_to_string(struct aegis_signature_t *from,
									 const aegis_format_t use_format,
									 const char* token_name,
									 char **to)
    {
        AUTOINIT;
        char *rid = (char*)token_name;
        string buf;

        switch(use_format)
            {
            case aegis_as_base64:
                {
                    char* b64string = base64_encode(from->d,
                                                    sizeof(struct aegis_signature_t));
                    buf = "b-";
                    buf.append(b64string);
                    free(b64string);
                }
                break;

            case aegis_as_hexstring:
                {
                    buf = "x-";
					append_hex(buf, from->d, sizeof(struct aegis_signature_t)); 
                }
                break;
            }

        if (NULL == rid) {
            aegis_application_id(getpid(), &rid);
        }
        buf.append("-");
        buf.append(rid);
        AEGIS_DEBUG(3, "%s: append '%s'", __func__, token_name);
        *to = (char*)malloc(buf.size() + 1);
        strcpy(*to, buf.c_str());
        if (rid != token_name)
            aegis_crypto_free(rid);
        return buf.size();
    }


    aegis_crypto_result
    aegis_crypto_string_to_signature(const char* from,
                                     struct aegis_signature_t *to,
                                     char **token_name)
    {
        const char *txt_signature = from;
        aegis_format_t use_format;
        const char* end_mark = NULL;

        AUTOINIT;

        AEGIS_DEBUG(3, "%s: parse '%s'", __func__, txt_signature);

        if (NULL == from)
            goto malformed_signature;
        if ('b' == *from)
            use_format = aegis_as_base64;
        else if ('x' == *from)
            use_format = aegis_as_hexstring;
        else
            goto malformed_signature;

        from++;
        if ('-' != *from)
            goto malformed_signature;
        from++;

        switch(use_format)
            {
            case aegis_as_base64:
                {
                    string *b64_string;
                    RAWDATA_PTR tmp = NULL;
                    size_t len;

                    end_mark = strchr(from, '-');
                    if (NULL == end_mark)
                        b64_string = new string(from);
                    else
                        b64_string = new string(from, end_mark - from);
                    len = base64_decode(b64_string->c_str(), &tmp);
                    delete b64_string;
                    if (tmp) {
                        if (sizeof(struct aegis_signature_t) == len) {
                            memcpy(to->d, tmp, len);
                        } else {
                            len = 0;
                        }
                        free(tmp);
                    } else {
                        len = 0;
                    }
                    if (0 == len)
                        goto malformed_signature;
                    if (end_mark) {
                        from = end_mark;
                    } else
                        from += strlen(from);
                }
                break;

            case aegis_as_hexstring:
                {
                    size_t i;
                    int c;
                    for (i = 0; i < sizeof(struct aegis_signature_t); i++) {
                        if (1 == sscanf(from, "%02x", &c)) {
                            to->d[i] = (unsigned char)c;
                            from += 2;
                        } else {
                            goto malformed_signature;
                        }
                    }
                }
                break;
            }

        if ('-' == *from) {
            from++;
            end_mark = strchr(from, '\n');
            if (end_mark) {
                size_t len = end_mark - from;
                *token_name = (char*)malloc(len + 1);
                memcpy(*token_name, from, len);
                *(*token_name + len) = '\0';
            } else {
                *token_name = (char*)malloc(strlen(from) + 1);
                strcpy(*token_name, from);
            }
            AEGIS_DEBUG(3, "%s: mac made by '%s'", __func__, *token_name);
        } else if (*from) {
            AEGIS_DEBUG(3, "%s: extra stuff at '%s'", __func__, from);
            goto malformed_signature;
        }
        return aegis_crypto_ok;

    malformed_signature:
        AEGIS_ERROR("malformed signature '%s'", txt_signature);
        return aegis_crypto_error;
    }


    aegis_system_mode_t
    aegis_current_mode(void)
    {
         AUTOINIT;
 #ifdef USE_BB5
		 if (!bb5_usable) {
			 return aegis_system_plain;
		 } else {
			 if (KERNEL_MODE_TRUSTED == bb5_get_kernel_mode())
				 return aegis_system_protected;
			 else
				 return aegis_system_open;
		 }
#else
        return aegis_system_emulated;
#endif
    }


    aegis_crypto_result
    aegis_crypto_sign(const RAWDATA_PTR data,
                      const size_t nbrof_bytes,
                      const char *with_token,
                      struct aegis_signature_t *signature)
    {
        unsigned char *use_token = (unsigned char*)my_appid;
        int rc;
        ssize_t res;
        unsigned char *tmp = NULL;
        aegis_crypto_result retval = aegis_crypto_error;

        AUTOINIT;

        if (NULL == data || NULL == signature) {
            AEGIS_ERROR("%s: invalid argument", __func__);
            return aegis_crypto_error;
        }
        if (NULL != with_token) {
            use_token = (unsigned char*)with_token;
        }
        rc = bb5_set_programid(use_token);
        if (0 != rc) {
            AEGIS_ERROR("%s: bb5_set_programid('%s') returned %d",
                           __func__, use_token, rc);
            BB5_ERROR;
            return aegis_crypto_error;
        } else {
            AEGIS_DEBUG(3, "%s: bb5_set_programid('%s') OK", __func__, use_token);
        }

        if (MAX_CRYPTO_INPUT_SIZE < nbrof_bytes) {
            unsigned char digest[DIGESTLEN];
            SHA1((unsigned char*)data, nbrof_bytes, digest);
            AEGIS_DEBUG(3, "%s: calling bb5_fprot_mac('%s',%u,%p)", __func__,
                        dynhex(digest, sizeof(digest)), sizeof(digest), &tmp);
            res = bb5_fprot_mac(digest, sizeof(digest), &tmp);

        } else {
            AEGIS_DEBUG(3, "%s: calling bb5_fprot_mac('%s',%u,%p)", __func__,
                        dynhex(data, nbrof_bytes), nbrof_bytes, &tmp);
            res = bb5_fprot_mac((unsigned char*)data, nbrof_bytes, &tmp);
        }

        AEGIS_DEBUG(3, "%s: bb5_fprot_mac returned %d", __func__, res);

        if (sizeof(struct aegis_signature_t) == res) {
            AEGIS_DEBUG(3, "%s: bb5_fprot_mac OK '%s'", __func__, dynhex(tmp, res));
            memcpy(signature->d, tmp, sizeof(struct aegis_signature_t));
            retval = aegis_crypto_ok;
        } else {
            AEGIS_ERROR("%s: bb5_fprot_mac with '%s' returned %d", __func__,
                        use_token, res);
			errno = EACCES;
            BB5_ERROR;
        }
        if (NULL != tmp)
            free(tmp);
        return retval;
	}


    aegis_crypto_result
    aegis_crypto_verify(struct aegis_signature_t *signature,
                        const char *with_token,
                        const RAWDATA_PTR data,
                        const size_t nbrof_bytes,
                        aegis_system_mode_t* made_in_mode)
    {
        int res = 0;
        unsigned char* use_token = (unsigned char*)my_appid;

        AUTOINIT;

        if (NULL != with_token) {
            use_token = (unsigned char*)with_token;
        }

        if (MAX_CRYPTO_INPUT_SIZE < nbrof_bytes) {
            unsigned char digest[DIGESTLEN];
            SHA1((unsigned char*)data, nbrof_bytes, digest);
            AEGIS_DEBUG(3, "%s: calling bb5_fprot_check_mac('%s','#%s',%u,'#%s',%u)",
                        __func__, use_token,
                        dynhex(digest, sizeof(digest)),
                        sizeof(digest),
                        dynhex(signature->d, sizeof(struct aegis_signature_t)),
                        sizeof(struct aegis_signature_t));
            res = bb5_fprot_check_mac(use_token, digest,
                                      sizeof(digest),
                                      signature->d,
                                      sizeof(struct aegis_signature_t));
        } else {
            AEGIS_DEBUG(3, "%s: calling bb5_fprot_check_mac('%s','#%s',%u,'#%s',%u)",
                        __func__, use_token, dynhex(data, nbrof_bytes), nbrof_bytes,
                        dynhex(signature->d, sizeof(struct aegis_signature_t)),
                        sizeof(struct aegis_signature_t));
            res = bb5_fprot_check_mac(use_token, (unsigned char*)data,
                                      nbrof_bytes, signature->d,
                                      sizeof(struct aegis_signature_t));
        }

        AEGIS_DEBUG(1, "%s: bb5_fprot_check_mac returned %d", __func__, res);

        if (BB5_FPROT_MAC_VALID_TRUSTED == res) {
            AEGIS_DEBUG(3, "%s: verified with trusted key", __func__);
            if (made_in_mode)
                *made_in_mode = aegis_system_protected;
            return aegis_crypto_ok;

        } else if (BB5_FPROT_MAC_VALID_UNTRUSTED == res) {
            AEGIS_DEBUG(3, "%s: verified with untrusted key", __func__);
            if (made_in_mode)
                *made_in_mode = aegis_system_open;
            return aegis_crypto_ok;
        } else {
            AEGIS_ERROR("verification fails (%d)", res);
            BB5_ERROR;
            return aegis_crypto_error_wrong_signature;
        }
    }

#if USE_XATTR
#define SIGNATURE_XATTR "security.aegis.signature."
#endif

    aegis_crypto_result
    aegis_crypto_sign_file(const char *pathname,
						   const void* data,
						   const size_t len,
                           const char *with_token)
    {
		AUTOINIT;
		if (NULL == pathname) {
			AEGIS_ERROR("%s: pathname cannot be null", __func__);
			errno = EINVAL;
			return aegis_crypto_error;
		}
		if (NULL == with_token)
			with_token = my_appid;

		AEGIS_DEBUG(1, "%s: '%s' data %p len %u token %s", __func__, 
					pathname, data, (unsigned)len, with_token);

		if (!bb5_usable) { 
			/* In build environment sign single files by adding them
			 * to /etc/aegis_vdata. Expect for /etc/aegis_vdata itself of course.
			 */
			if (0 == strcmp(pathname, aegis_vdata_file)) {
				AEGIS_DEBUG(1, "%s: cannot sign globally '%s'", __func__, pathname);
				return aegis_crypto_ok;
			}
			struct aegis_digest_t digest;
			bool dres;
			if (NULL == data) 
				dres = digest_of_file(pathname, &digest);
			else
				dres = compute_digest((unsigned char*)data, len, &digest);
			if (dres && save_global_signature(pathname, &digest))
				return aegis_crypto_ok;
			else
				return aegis_crypto_error;

		} else {
			/* In a healthy environment sign by using protected store.
			 */
			string store_name("files.");
			store_name.append(with_token);
			storage pstore(store_name.c_str(), with_token, 
						   storage::vis_global, storage::prot_signed);
			p_file *pf = NULL;
			aegis_crypto_result erc = aegis_crypto_error;

			if (0 != strcmp(pstore.token(), with_token)) {
				AEGIS_ERROR("access denied, no '%s'", with_token);
				errno = EACCES;
				return aegis_crypto_error;
			}
			
			if (NULL != data) {
				string tmpname = pathname;
				unsigned char random_data[6];
				
				if (0 > aegis_crypto_random(random_data, sizeof(random_data))) {
					tmpname.append(".temp");
				} else {
					tmpname.append(".");
					append_hex(tmpname, random_data, sizeof(random_data));
				}
				pf = pstore.member(tmpname.c_str());
				if (NULL == pf) {
					AEGIS_ERROR("%s: failed to create '%s' (%s)", 
								__func__, pathname, strerror(errno));
					return aegis_crypto_error;
				}
				if (pf->p_open(O_CREAT | O_WRONLY | O_TRUNC)) {
					if ((ssize_t)len != pf->p_write(0, data, len)) {
						AEGIS_ERROR("%s: failed to write to '%s' (%s)", 
									__func__, pathname, strerror(errno));
						delete pf;
						return aegis_crypto_error;
					}
				} else {
					AEGIS_ERROR("%s: failed to open '%s' (%s)", 
								__func__, pathname, strerror(errno));
					delete pf;
					return aegis_crypto_error;
				}
				pf->p_trunc(len);
				pf->p_close();
				if (0 > rename(tmpname.c_str(), pathname)) {
					AEGIS_ERROR("%s: failed to rename '%s' to '%s'(%s)", 
								__func__, tmpname.c_str(), pathname, strerror(errno));
					delete pf;
					return aegis_crypto_error;
				} else {
					pstore.rename(tmpname.c_str(), pathname);
				}
			} else {
				pstore.add_file(pathname);
			}
			if (!pstore.commit()) {
				AEGIS_ERROR("%s: failed to commit '%s' (%s)", 
							__func__, store_name.c_str(), strerror(errno));
				erc = aegis_crypto_error;
			} else {
				erc = aegis_crypto_ok;
			}
			if (pf)
				delete pf;
			return erc;
		}
	}


	aegis_crypto_result
    aegis_crypto_verify_file(const char *pathname,
							 const void *data,
							 const size_t len,
							 const char *with_token)
	{
        AUTOINIT;
		if (NULL == pathname || NULL == with_token) {
			AEGIS_ERROR("%s: pathname or token cannot be null", __func__);
			errno = EINVAL;
			return aegis_crypto_error;
		}
		AEGIS_DEBUG(1, "%s: '%s' data %p len %u token %s", __func__, 
					pathname, data, (unsigned)len, with_token);
		string store_name("files.");
		store_name.append(with_token);
		storage pstore(store_name.c_str(), NULL,
					   storage::vis_global, storage::prot_signed);

		if (pstore.contains_file(pathname)) {
			if (0 != strcmp(pstore.token(), with_token)) {
				AEGIS_ERROR("'%s' not signed by '%s' but '%s'", pstore.name(), 
							with_token, pstore.token());
				return aegis_crypto_error_signature_missing;
			}
			if (NULL != data) {
				if (pstore.verify_content(pathname, (unsigned char*)data, len))
					return aegis_crypto_ok;
				else
					return aegis_crypto_error_wrong_signature;
			} else {
				if (pstore.verify_file(pathname))
					return aegis_crypto_ok;
				else
					return aegis_crypto_error_wrong_signature;
			}
		} else {
			struct aegis_digest_t digest;
			bool dres;
			if (NULL == data) 
				dres = digest_of_file(pathname, &digest);
			else
				dres = compute_digest((unsigned char*)data, len, &digest);
			return verify_global_signature(pathname, &digest);
		}
	}


    aegis_crypto_result
    aegis_crypto_encrypt(const RAWDATA_PTR plaintext,
                         const size_t nbrof_bytes,
                         const char *token_name,
                         RAWDATA_RPTR ciphertext,
                         size_t* result_size)
    {
        unsigned char *use_token = (unsigned char*)my_appid;
        int rc;
        ssize_t res;

        AUTOINIT;
        AEGIS_DEBUG(3, "%s: enter", __func__);

        if ((NULL == plaintext) || (NULL == ciphertext) || (NULL == result_size) 
			|| MAX_CRYPTO_INPUT_SIZE < nbrof_bytes) 
		{
            AEGIS_ERROR("%s: invalid argument", __func__);
            if (NULL == plaintext)
                AEGIS_ERROR("%s: 'plaintext' is NULL", __func__);
            else if (NULL == ciphertext)
                AEGIS_ERROR("%s: 'ciphertext' is NULL", __func__);
            else if (NULL == result_size)
                AEGIS_ERROR("%s: 'result_size' is NULL", __func__);
			else 
                AEGIS_ERROR("%s: too long buffer %lu (max %lu)", __func__, 
							(unsigned long)nbrof_bytes, 
							(unsigned long)MAX_CRYPTO_INPUT_SIZE);
			errno = EINVAL;
            return aegis_crypto_error;
        }

        *ciphertext = NULL;

        if (NULL != token_name) {
            use_token = (unsigned char*)token_name;
        }
        rc = bb5_set_programid(use_token);
        if (0 != rc) {
            AEGIS_ERROR("%s: bb5_set_programid('%s') returned %d",
                           __func__, use_token, rc);
            BB5_ERROR;
            return aegis_crypto_error;
        } else {
            AEGIS_DEBUG(3, "%s: bb5_set_programid('%s') OK", __func__, use_token);
        }
        res = bb5_fprot_encrypt((unsigned char*)plaintext,
                                nbrof_bytes,
                                (unsigned char**)ciphertext);

        AEGIS_DEBUG(3, "%s: bb5_fprot_encrypt returned %d", __func__, res);

        if (0 > res) {
            *result_size = 0;
            AEGIS_ERROR("%s: bb5_fprot_encrypt('%s') returned %d",
                           __func__, use_token, rc);
            BB5_ERROR;
            return aegis_crypto_error;
        } else {
            *result_size = res;
            return aegis_crypto_ok;
        }
    }


    aegis_crypto_result
    aegis_crypto_decrypt(const RAWDATA_PTR ciphertext,
                         const size_t nbrof_bytes,
                         const char* token_name,
                         RAWDATA_RPTR plaintext,
                         size_t* result_size)
    {
        unsigned char *use_token = (unsigned char*)my_appid;
        int rc;
        ssize_t res;

        AUTOINIT;

        AEGIS_DEBUG(3, "%s: enter", __func__);

        if (NULL == plaintext || NULL == ciphertext || NULL == result_size
			|| MAX_CRYPTO_INPUT_SIZE < nbrof_bytes) 
		{
            if (NULL == plaintext)
                AEGIS_ERROR("%s: 'plaintext' is NULL", __func__);
            else if (NULL == ciphertext)
                AEGIS_ERROR("%s: 'ciphertext' is NULL", __func__);
            else if (NULL == result_size)
                AEGIS_ERROR("%s: 'result_size' is NULL", __func__);
			else 
                AEGIS_ERROR("%s: too long buffer %lu (max %lu)", __func__, 
							(unsigned long)nbrof_bytes, 
							(unsigned long)MAX_CRYPTO_INPUT_SIZE);
			errno = EINVAL;
            return aegis_crypto_error;
        }

        *plaintext = NULL;

        if (NULL != token_name) {
            use_token = (unsigned char*)token_name;
        }
        rc = bb5_set_programid(use_token);
        if (0 != rc) {
            AEGIS_ERROR("%s: bb5_set_programid('%s') returned %d",
                           __func__, use_token, rc);
            BB5_ERROR;
            return aegis_crypto_error;
        } else {
            AEGIS_DEBUG(3, "%s: bb5_set_programid('%s') OK", __func__, use_token);
        }
        res = bb5_fprot_decrypt((unsigned char*)ciphertext,
                                nbrof_bytes,
                                (unsigned char**)plaintext);

        AEGIS_DEBUG(3, "%s: bb5_fprot_decrypt returned %d", __func__, res);

        if (0 > res) {
            *result_size = 0;
            AEGIS_ERROR("%s: bb5_fprot_decrypt('%s') returned %d",
                           __func__, use_token, res);
            BB5_ERROR;
            return aegis_crypto_error;
        } else {
            *result_size = res;
            return aegis_crypto_ok;
        }
    }


    aegis_crypto_result
    aegis_crypto_free(RAWDATA_PTR ptr)
    {
		if (ptr) {
			free(ptr);
		}
        return aegis_crypto_ok;
    }


    const char*
    aegis_crypto_last_error_str(void)
    {
        AUTOINIT;
		if (last_dlog_error_message) {
			return last_dlog_error_message;
		} else if (0 != errno) {
			return strerror(errno);
		} else {
			return "";
		}
    }


	const char*
	aegis_system_invariant(aegis_sysinvariant_t invariant)
	{
        const char* ivar_name;
        bb5_imei_t imei;

        AUTOINIT;
#ifdef USE_BB5
		/* This is still needed here. In the principle it
		 * should be necessary only once after boot, but
		 * according to Vesa there is some bug still.
		 */
		static bool esn_initialized = false; 
		if (!esn_initialized) {
			/* Requires CRP::cal */
			int erc = bb5_esn_init(NULL, 0);
			AEGIS_DEBUG(1, "%s: bb5_esn_init returned %d", __func__, erc);
			if (0 == erc) {
				esn_initialized = true;
			}
		}
#endif
        AEGIS_DEBUG(3, "%s: read %d", __func__, (int)invariant);
		switch (invariant)
		{
		case sysinvariant_imei:
			ivar_name = "IMEI";
            if (ivar_cache.end() == ivar_cache.find(ivar_name)) {
                size_t rc = bb5_imei_read(&imei);
                AEGIS_DEBUG(1, "%s: IMEI=%s", __func__, dynhex(imei, sizeof(imei)));
                for (unsigned i = 0; i < sizeof(imei); i++)
                    imei[i] += '0';
                if (0 == rc) {
                    ivar_cache[ivar_name].assign((char*)imei, sizeof(imei));
                } else {
                    AEGIS_DEBUG(3, "%s: bb5_imei_read returned %d, using fallback",
                                   __func__, rc);
                    return "12345678901234";
                }
            }
            break;

		default:
			AEGIS_ERROR("Not yet implemented: sysinvariant %d",
						   (int)invariant);
			return "";
		}

        return ivar_cache[ivar_name].c_str();
    }


    ssize_t
    aegis_crypto_random(RAWDATA_PTR to_buf, size_t bytes)
    {
        AUTOINIT;
        int res = bb5_get_random((unsigned char*)to_buf, bytes);
        if (0 == res)
            return bytes;
        else {
			AEGIS_ERROR("%s: bb5_get_random returned %d", __func__, res);
            BB5_ERROR;
            return -1;
		}
    }


    size_t
    aegis_crypto_new_symkey(RAWDATA_RPTR to_buf)
    {
		RAWDATA_PTR tmp = malloc(SYMKEYLEN);
        int res;

		AUTOINIT;

		mlock(tmp, SYMKEYLEN);
        *to_buf = tmp;
        res = bb5_get_random((unsigned char*)tmp, SYMKEYLEN);
        if (0 == res) {
            return SYMKEYLEN;
        } else {
            return -1;
        }
    }

    size_t
    aegis_crypto_symkeylen(void)
    {
        return SYMKEYLEN;
    }

    void
    aegis_application_id(pid_t of_pid, char **to_this)
	{
        *to_this = NULL;

#ifdef USE_CREDS
        creds_t creds;
        AEGIS_DEBUG(1, "%s: using libcreds to retrieve application id of %d",
                    __func__, (int)of_pid);
        creds = creds_gettask(of_pid);
        if (NULL != creds) {
            char cred_name[512];
            int res = creds_find(creds, "AID::*", cred_name, sizeof(cred_name));
            if (0 <= res) {
                if (sizeof(cred_name) <= (unsigned)res) {
                    AEGIS_DEBUG(1, "%s: Warning: name truncated, needs %d bytes",
                                __func__, res);
					cred_name[sizeof(cred_name) - 1] = '\0';
				}
                *to_this = strdup(cred_name);
				/* Replace slash by a point to enable using appid
				 * in filenames.
				 */
				for (char *c = *to_this; *c; c++)
					if ('/' == *c)
						*c = '.';
                AEGIS_DEBUG(1, "%s: %d: '%s'", __func__, (int)of_pid, *to_this);
            }
            creds_free(creds);
            if (NULL != *to_this)
                return;
        }
#endif

#ifdef USE_MD5_LOOKUP
        if (0 == of_pid)
            of_pid = getpid();
        AEGIS_DEBUG(1, "%s: using hash lookup to retrieve application id of %d",
                    __func__, (int)of_pid);
		AUTOINIT;
		string tmp;
		md_lookup.lookup(of_pid, tmp);
		*to_this = strdup(tmp.c_str());
#endif
        if (NULL == *to_this)
            *to_this = strdup(UNKNOWN_APP_ID);

        AEGIS_DEBUG(1, "%s: pid %d is '%s'", __func__, (int)of_pid, *to_this);
	}

    void
    aegis_application_id_of_bin(const char *pathname, char **to_this)
	{
		string tmp;
        string apathname;

		AUTOINIT;
#ifdef USE_MD5_LOOKUP
        absolute_pathname(pathname, apathname);
		md_lookup.lookup(apathname.c_str(), tmp);
		*to_this = strdup(tmp.c_str());
#else
        *to_this = strdup(UNKNOWN_APP_ID);
#endif
	}


    aegis_crypto_result
    aegis_crypto_verify_aegisfs(const char *dir,
                                aegis_system_mode_t *cmode)
    {
        string real_dir;
        char *challenge, *ch_ptr = NULL;
        aegis_signature_t sgn;
        size_t tot_len;
        int rc = 0;
        aegis_crypto_result res = aegis_crypto_error;

        AEGIS_ENTER;

        if (NULL == dir) {
            errno = EINVAL;
            return aegis_crypto_error;
        }

        if (!absolute_pathname(dir, real_dir)) {
            errno = EINVAL;
            return aegis_crypto_error;
        }

        tot_len = strlen(real_dir.c_str()) + AEGISFS_VFY_DATALEN;
        challenge = (char*)malloc(tot_len);
        if (NULL == challenge) {
            return aegis_crypto_error;
        }

        strcpy(challenge, real_dir.c_str());
        ch_ptr = challenge + strlen(challenge);
        aegis_crypto_random(ch_ptr, AEGISFS_VFY_DATALEN);

        rc = setxattr(dir, AEGISFS_SGN_XATTR, challenge, tot_len, 0);
        if (0 > rc) {
            AEGIS_ERROR("Cannot set challenge (%s)", strerror(errno));
            goto finish;
        }

        /* The response should be a raw signature.
         */
        rc = getxattr(dir, AEGISFS_SGN_XATTR, (char*)&sgn, sizeof(sgn));
        if (sizeof(sgn) != rc) {
            if (0 > rc) {
                AEGIS_ERROR("Cannot get response (%s)", strerror(errno));
                goto finish;
            } else {
                AEGIS_ERROR("Invalid response (wrong size %d)", rc);
                goto finish;
            }
        }

        /* Now verify the signature.
		 */
        AEGIS_DEBUG(1, "%s: verifying %lu bytes by '%s'", __func__,
                    (unsigned long)tot_len, AEGISFS_VFY_TOKEN);
        if (aegis_crypto_ok == aegis_crypto_verify(&sgn, AEGISFS_VFY_TOKEN,
                                                   challenge, tot_len,
                                                   cmode))
        {
            AEGIS_DEBUG(1, "%s: verified ok in %s mode",
                        __func__,
                        aegis_system_open == *cmode ? "open" : "protected");
            res = aegis_crypto_ok;
        } else {
            AEGIS_ERROR("Signature cannot be verified (%s)",
                        aegis_crypto_last_error_str());
            errno = EACCES;
        }

    finish:
        if (challenge)
            free(challenge);
        return res;
    }

} /* extern "C" */

#ifdef USE_MD5_LOOKUP
/*
 * This is intermediate code to provide the application id
 * functionality by help of md5sums-files.
 */

const char dpkg_status_file[] = "/var/lib/dpkg/status";
const char dpkg_info_dir[]    = "/var/lib/dpkg/info/";

/*
 * Constructor & destructor
 */
pitree::pitree():root(NULL),m_count(0) {};

pitree::~pitree()
{
	local_cleanup(root);
	root = NULL;
}

void
pitree::local_cleanup(struct procinfo_t* node)
{
	if (NULL != node) {
		local_cleanup(node->left);
		local_cleanup(node->right);
		delete node->binname;
		delete node;
	}
}

size_t
pitree::size()
{
	return m_count;
}

/*
 * Use the MD5 as sorting key, as it probably has
 * a pretty even distribution and produces a nice
 * balanced search tree.
 */
void
pitree::add(string *filename, md5_t *csum, off_t pkgi)
{
	struct procinfo_t *pi = new struct procinfo_t;

	memset(pi, '\0', sizeof(struct procinfo_t));
	memcpy(&pi->md5sum, csum, sizeof(md5_t));
	pi->binname = filename;
	pi->pkgi = pkgi;

	if (NULL == root) {
		root = pi;
		m_count++;
		return;
	}

	struct procinfo_t *tmp = root;
	while (tmp) {

		int cmp = memcmp(&tmp->md5sum, csum, sizeof(md5_t));
		if (0 == cmp) {
			AEGIS_DEBUG(3, "%s: %s,%s md5sum collision (%s)",
						   __func__, filename->c_str(),
						   tmp->binname->c_str(),
						   dynhex(csum, sizeof(md5_t)));
            delete filename;
			delete pi;
			return;
		} else if (0 > cmp) {
			if (NULL != tmp->right)
				tmp = tmp->right;
			else {
				tmp->right = pi;
				m_count++;
				return;
			}
		} else {
			if (NULL != tmp->left)
				tmp = tmp->left;
			else {
				tmp->left = pi;
				m_count++;
				return;
			}
		}
	}
}

void
pitree::local_iterate(struct procinfo_t* node, callback_t cb, void* ctx)
{
	if (NULL != node) {
		local_iterate(node->left, cb, ctx);
		cb(*node->binname, &node->md5sum, node->pkgi, ctx);
		local_iterate(node->right, cb, ctx);
	}
}

void
pitree::iterate(callback_t cb, void* ctx)
{
	local_iterate(root, cb, ctx);
}


bin_lookup::bin_lookup()
: m_hdr(NULL),
	m_pkginfo(NULL),
	m_csums(NULL),
	m_fd(-1),
	m_len(0),
	m_stree(NULL)
{
	;
}

bin_lookup::~bin_lookup()
{
	finish();
}


bool
bin_lookup::make_appid(size_t sum_idx, string& to_this)
{
	if (NULL == m_hdr || sum_idx >= m_hdr->nrof_csum)
		return false;

	char *pkgname = m_pkginfo + m_csums[sum_idx].pkgname_off;
	char *signer  = pkgname + strlen(pkgname) + 1;
    char *sep, *end;
    /*
     * Normalize the application id a little bit to make it
     * look nice in a demo.
     */
    sep = strchr(signer, '@');
    if (sep) {
        sep++;
        end = strchr(sep, '>');
        if (end) {
            to_this.assign(string(sep, end - sep));
        } else
            to_this.assign("unknown");
    } else
        to_this.assign("unknown");
	to_this.append("@");
	to_this.append(pkgname);
	return true;
}


bool
bin_lookup::lookup(md5_t* csum, string& appid)
{
	ssize_t low = -1, high, middle = 0;
	int cmp = -1;

    if (NULL == m_hdr)
        return false;

    high = (ssize_t) m_hdr->nrof_csum;
	while (1 < high - low) {
		middle = low + (high - low)/2;
		cmp = memcmp(csum, &m_csums[middle].csum, sizeof(md5_t));
		if (0 == cmp)
			break;
		else if (0 > cmp)
			high = middle;
		else
			low = middle;
	}
	if (0 == cmp) {
		make_appid(middle, appid);
		return true;
	} else {
		return false;
	}
}


bool
bin_lookup::lookup(const char* pathname, string& appid)
{
	md5_t csum;
	if (md5sum_of(pathname, &csum)) {
		if (lookup(&csum, appid)) {
            AEGIS_DEBUG(3, "%s: %s belongs to %s",
                        __func__, pathname, appid.c_str());
			return true;
        }
	}
	appid.assign("unknown@unknown");
	return true;
}


bool
bin_lookup::lookup(pid_t pid, string& appid)
{
	string tmp;
	if (0 < process_name_of_pid(pid, tmp))
		return lookup(tmp.c_str(), appid);
	appid.assign("unknown@unknown");
	return true;
}


bool
bin_lookup::md5sum_of(const char* pathname, md5_t* to_this)
{
	MD5_CTX ctx;
	RAWDATA_PTR data = NULL;
	int fd = -1;
	size_t len = 0;
	bool res = false;

	if (NULL == (data = map_file(pathname, O_RDONLY, &fd, &len)))
		return false;
	if (0 == MD5_Init(&ctx)) {
		AEGIS_ERROR("OpenSSL failure (%s)", strerror(errno));
		goto error;
	}
	if (0 == MD5_Update(&ctx, data, len)) {
		AEGIS_ERROR("OpenSSL failure (%s)", strerror(errno));
		goto error;
	}
	if (0 == MD5_Final((unsigned char*)to_this, &ctx)) {
		AEGIS_ERROR("OpenSSL failure (%s)", strerror(errno));
		goto error;
	}
	res = true;
	// AEGIS_DEBUG(1, "%s: %s md5 is %s", __func__, pathname, dynhex(to_this, sizeof(md5_t)));

error:
	munmap_file(data, len, fd);
	return res;
}


/*
 * Callback function for packing the data
 */
static void
pack_to(string& filename, md5_t *csum, off_t pkgi, void* ctx)
{

	bin_lookup::md5i_t *cpyto = *(bin_lookup::md5i_t**)ctx;
#if 0
	AEGIS_DEBUG(1, "%s: pack %s %s %d to %p",
				   __func__,
				   filename.c_str(),
				   dynhex(csum, sizeof(md5_t)),
				   pkgi, cpyto);
#endif
	memcpy(&cpyto->csum, csum, sizeof(md5_t));
	cpyto->pkgname_off = pkgi;
	*(bin_lookup::md5i_t**)ctx = cpyto + 1;
#if 0
	if (listbin)
		printf("%s\n", filename.c_str());
#endif
}


void
bin_lookup::print(void)
{
	AEGIS_DEBUG(1, "%s: %d packages, %d executables (%s) %p",
				   __func__,
				   m_hdr->nrof_pkg,
				   m_hdr->nrof_csum,
				   m_pkginfo,
				   m_csums);
	for (size_t i = 0; i < m_hdr->nrof_csum; i++) {
		char* pkgname = m_pkginfo + m_csums[i].pkgname_off;
		char* signer  = pkgname + strlen(pkgname) + 1;
		// AEGIS_DEBUG(1, "%s: %d %u %p", __func__, i, m_csums[i].pkgname_off, pkgname);
		printf("%s::%s::%s\n",
			   dynhex(&m_csums[i].csum, sizeof(md5_t)),
			   pkgname, signer);
	}
}


size_t
bin_lookup::parse_hex(const char* str, size_t maxlen, md5_t *to)
{
	size_t  len = 0;
	uint8_t acc = 0;

	while (isxdigit(*str) && len < 2 * maxlen) {
		char c = tolower(*str);
		if (len % 2)
			to->d[len/2] = (acc << 4) + (c<'a'?c-'0':10+c-'a');
		else
			acc = c<'a'?c-'0':10+c-'a';
		len++;
		str++;
	}
	return len;
}


/*
 * There is a big tradeoff between time and space here.
 * If we check the file type at loadtime the search
 * tree contains only binaries. But it takes a long time.
 */
bool
bin_lookup::is_elf_executable(const char* filename)
{
	struct stat fs;
	bool res = false;

	if (0 <= stat(filename, &fs)
		&& S_ISREG(fs.st_mode)
		&& 0 != (fs.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH)))
	{
		int fd = open(filename, O_RDONLY);
		if (0 <= fd) {
			size_t len;
			Elf32_Ehdr elfhdr;
			len = read(fd, &elfhdr, sizeof(elfhdr));
			if (len == sizeof(elfhdr)
				&& 0 == memcmp(elfhdr.e_ident, ELFMAG, SELFMAG)
				&& ELFCLASS32 == elfhdr.e_ident[EI_CLASS]
                /*
                 * A binary can also be a shared object? For instance
                 * /usr/bin/ssh in Ubuntu 9.04 with openssh-client.
                 */
				&& (ET_EXEC == elfhdr.e_type || ET_DYN == elfhdr.e_type))
				res = true;
			close(fd);
		}
	}

#if 0
	if (res)
		AEGIS_DEBUG(1, "%s is executable", filename);
	else
		AEGIS_DEBUG(1, "%s is NOT executable", filename);
#endif
	return res;
}


size_t
bin_lookup::read_md5sums(string& packagename, off_t pkg_off)
{
	int fd = -1;
	size_t len = 0, count = 0;
	size_t left, lc = 1;
	char *odata, *data = NULL;

	string md5sumfilename = dpkg_info_dir;
	md5sumfilename.append(packagename);
	md5sumfilename.append(".md5sums");

	if (!file_exists(md5sumfilename.c_str())
		|| NULL == (data = (char*)map_file(md5sumfilename.c_str(),
										   O_RDONLY, &fd, &len)))
	{
		// AEGIS_DEBUG(1, "%s cannot be mapped", md5sumfilename.c_str());
		return 0;
	}
    odata = data;

#define SUMANDSEPLEN 34

	left = len;
	while (left > SUMANDSEPLEN) {
		md5_t  csum;
		char   *newline;
		size_t hlen;

		hlen = parse_hex(data, sizeof(csum), &csum);
		if (hlen != 2 * sizeof(md5_t)) {
			AEGIS_DEBUG(1, "%s: wrong size of md5 (%d)", __func__, left);
			goto error;
		}

		data += hlen;
		left -= hlen;

		if (2 < left && 0 == memcmp(data, "  ", 2)) {
			newline = strchr(data, '\n');
			if (newline) {
				string *binname = new string(data + 2, newline - data - 2);
				binname->insert(0, "/");
				if (is_elf_executable(binname->c_str())) {
					m_stree->add(binname, &csum, pkg_off);
					count++;
				} else
					delete binname;
				left -= (newline - data + 1);
				data = newline + 1;
				// AEGIS_DEBUG(1, "%s  %s", dynhex(&pi.md5sum, sizeof(pi.md5sum)),
				//			   pi.binname->c_str());

			} else {
				AEGIS_DEBUG(1, "%s: no ending newline", __func__);
				goto error;
			}
		} else {
			AEGIS_DEBUG(1, "%s: separator missing", __func__);
			goto error;
		}

		lc++;
	}
	goto final;

  error:
	AEGIS_DEBUG(1, "%s: corrupted md5sum file '%s' at %d", __func__,
				   md5sumfilename.c_str(), lc);
  final:
	munmap_file(odata, len, fd);
	return count;
}


void
bin_lookup::test_prefix(char *in_data, const char *prefix, string& valto)
{
	size_t nlen = strlen(prefix);
	if (0 == memcmp(in_data, prefix, nlen)) {
		valto = "";
		in_data += nlen;
		char* newline = strchr(in_data, '\n');
		if (newline)
			valto.append(in_data, newline - in_data);
	}
}


/*
 * Read /var/lib/dpkg/status to get a list of installed packages
 */
off_t
bin_lookup::make_packagelist(void)
{
	int fd;
	size_t len, lc = 1;
	char *data, *pkginfo;

	pkginfo = m_pkginfo;

	data = (char*)map_file(dpkg_status_file, O_RDONLY, &fd, &len);
	if (NULL == data) {
		AEGIS_ERROR("Cannot open '%s' (%s)", dpkg_status_file, strerror(errno));
		return 0;
	}

	AEGIS_DEBUG(1, "%s: reading %s(%d)", __func__, dpkg_status_file, len);

	char *c = data;
	string packagename, status, maintainer;
	do {
		/*
		 * The status file contains the following Name: Value lines:
		 * Package, Status, Priority, Section, Installed-Size, Maintainer,
		 * Architecture, Source, Version, Replaces, Depends, Conflicts,
		 * Description.
		 */
		switch (*c)
			{
			case 'P':
				test_prefix(c, "Package: ", packagename);
				// AEGIS_DEBUG(1, "%s: package '%s'", __func__, packagename.c_str());
				break;
			case 'S':
				test_prefix(c, "Status: ", status);
				// AEGIS_DEBUG(1, "%s: status '%s'", __func__, packagename.c_str());
				break;
			case 'M':
				/*
				 * Trust that items are in the right order, Maintainer:
				 * as last and that "Maintainer" is the only keyword
				 * that starts with a capital M.
				 */
				test_prefix(c, "Maintainer: ", maintainer);
				// AEGIS_DEBUG(1, "%s: maintainer '%s'", __func__, packagename.c_str());
				if (packagename.size() && status.size() && maintainer.size())
				{
					if ("install ok installed" == status) {
						off_t off = pkginfo - m_pkginfo;

						AEGIS_DEBUG(1, "%s: %s", __func__, packagename.c_str());

						strcpy(pkginfo, packagename.c_str());
						pkginfo += strlen(pkginfo) + 1;
						strcpy(pkginfo, maintainer.c_str());
						pkginfo += strlen(pkginfo) + 1;

						m_hdr->nrof_pkg++;
						m_hdr->nrof_csum += read_md5sums(packagename, off);
					} else {
						AEGIS_DEBUG(1, "%s: %s not installed\n", __func__,
								   packagename.c_str());
					}

				} else {
					AEGIS_DEBUG(1, "%s: odd line at %s(%u)\n%s::%s::%s", __func__,
								   dpkg_status_file, lc, packagename.c_str(),
								   status.c_str(), maintainer.c_str());
				}
				packagename = status = maintainer = "";
				break;

			case '\n':
				// packagename = status = maintainer = "";
				break;
			}
		lc++;
		c = strchr(c, '\n');
		if (c)
			c++;
		else
			break;
	} while ((len + data) > c);

	munmap_file(data, len, fd);

	return pkginfo - m_pkginfo;
}


bool
bin_lookup::create(const char *filename)
{
    AEGIS_DEBUG(1, "%s: '%s'", __func__, filename);
	/*
	 * File didn't exist, so create it.
	 * Currently in Fremantle the space consumption is about 800k
	 * so this should be ample for a long time. TODO: the package
	 * names and signer ids could be stored in a much denser format.
	 */
	m_len = 10 * 1024 * 1024;

	/*
	 * Use an anonymous mapping and then flush its contents into the
	 * file. A RDWR mapping would require to write base data into the
	 * file for the whole map size first.
	 */
	m_hdr = (struct lookup_hdr_t*) map_file(NULL, O_RDWR | O_CREAT,
											&m_fd, &m_len);
	if (NULL == m_hdr) {
		AEGIS_ERROR("Cannot create anonymous map (%s)", strerror(errno));
		return false;
	}

	AEGIS_DEBUG(3, "hdr at %p", m_hdr);
	strcpy(m_hdr->magic, MAGIC);
	m_hdr->nrof_pkg = 0;
	m_hdr->nrof_csum = 0;
	m_pkginfo = (char*)(m_hdr + 1);
	m_stree = new pitree();

	m_hdr->pkg_size = make_packagelist();

	m_csums = (struct lin_md5i_t*)(m_pkginfo + m_hdr->pkg_size);
	AEGIS_DEBUG(3, "%s: csums at %p", __func__, m_csums);

	size_t realsize, nrof_colls = 0;
	void* ctx = m_csums;

	m_stree->iterate(pack_to, &ctx);

    nrof_colls = m_hdr->nrof_csum - m_stree->size();
	if (0 < nrof_colls) {
		AEGIS_DEBUG(1, "%s: %d MD5 collisions",
					   __func__,
					   m_hdr->nrof_csum - m_stree->size());
		m_hdr->nrof_csum = m_stree->size();
	}

	delete m_stree;

	/*
	 * Shrink and flush to file.
	 * TODO: Add signature checking to make the
	 * file immutable by others.
	 */
	realsize = sizeof(struct lookup_hdr_t)
		+ m_hdr->pkg_size
		+ m_hdr->nrof_csum * sizeof(struct lin_md5i_t);

	m_hdr = (struct lookup_hdr_t*) mremap(m_hdr, m_len, realsize, 0);

	m_fd = open(m_filename.c_str(), O_RDWR | O_CREAT, 0644);
	if (-1 == m_fd) {
		AEGIS_DEBUG(1, "%s: cannot create lookup file '%s', (%s)",
                    __func__, m_filename.c_str(), strerror(errno));
        munmap(m_hdr, m_len);
        return false;
	}
	m_len = write(m_fd, m_hdr, realsize);
	if (m_len == realsize) {
		AEGIS_DEBUG(1, "Created the lookup table, %d packages, %d executables,"
                    " %d collisions, %d bytes.\n",
					m_hdr->nrof_pkg,
					m_hdr->nrof_csum,
					nrof_colls,
					m_len);
	} else {
		AEGIS_DEBUG(1, "%s: cannot store lookup table to '%s' (%s)",
                    __func__, m_filename.c_str(), strerror(errno));
        munmap(m_hdr, m_len);
        close(m_fd);
        return false;
    }
	munmap_file(m_hdr, m_len, m_fd);

    /* Copy the timestamp from the reference file
     * in case its timestamp is in the future.
     */
	struct stat sfs;
    struct utimbuf utim;
    if (0 <= stat(dpkg_status_file, &sfs)) {
        utim.actime = sfs.st_atime;
        utim.modtime = sfs.st_mtime;
        if (0 > utime(m_filename.c_str(), &utim)) {
            AEGIS_DEBUG(1, "%s: cannot set times for '%s' (%s)",
                        __func__, m_filename.c_str(), strerror(errno));
        }
    }
    return true;
}


bool
bin_lookup::init()
{
	struct stat ofs;
	struct stat sfs;
    const char *lookup_names[] = {
        "/var/lib/aegis/.md5sums.tbl",
        "/tmp/.md5sums.tbl",
        NULL
    };
    bool can_init = false;
    bool need_init = false;

    if (0 > stat(dpkg_status_file, &sfs)) {
        AEGIS_DEBUG(1, "%s: dpkg status file '%s' not found, quit",
                    __func__, dpkg_status_file);
        return false;
    }

    for (int i = 0; NULL != lookup_names[i]; i++) {
        m_filename.assign(lookup_names[i]);
        if (0 > stat(m_filename.c_str(), &ofs)) {
            need_init = true;
            int fd = open(m_filename.c_str(), O_RDWR | O_CREAT, 0644);
            if (0 <= fd) {
                AEGIS_DEBUG(1, "%s: create lookup table '%s'",
                    __func__, m_filename.c_str());
                can_init = true;
                close(fd);
                unlink(m_filename.c_str());
                break;
            } else {
                can_init = false;
            }
        } else {
            if (sfs.st_mtime > ofs.st_mtime) {
                need_init = true;
                if (0 == access(m_filename.c_str(), R_OK | W_OK)) {
                    can_init = true;
                    break;
                }
            } else {
                need_init = false;
                break;
            }
        }
    }

    if (need_init && !can_init) {
        AEGIS_ERROR("Lookup file '%s' does not exist or is expired"
                    " and cannot be created (%s)",
                    m_filename.c_str(), strerror(errno));
        return false;

    } else {
        if (need_init) {
            AEGIS_DEBUG(1, "%s: make new lookup table '%s'",
                        __func__, m_filename.c_str());
            if (!this->create(m_filename.c_str()))
                return false;
        } else
            AEGIS_DEBUG(1, "%s: use existing lookup table '%s'",
                        __func__, m_filename.c_str());
    }

	m_hdr = (struct lookup_hdr_t*) map_file(m_filename.c_str(),
											O_RDONLY,
											&m_fd,
											&m_len);

	if ((NULL != m_hdr) && (sizeof(struct lookup_hdr_t) <= m_len))
	{
		m_pkginfo = (char*)(m_hdr + 1);
		m_csums   = (struct lin_md5i_t*)(m_pkginfo + m_hdr->pkg_size);
		AEGIS_DEBUG(3, "%s: pkg at %p (%s), csums at %p", __func__,
					   m_pkginfo, m_pkginfo, m_csums);
		return true;
	}
	AEGIS_DEBUG(1, "%s: corrupted lookup file", __func__);
	return false;
}


void
bin_lookup::finish()
{
    AEGIS_ENTER;
	munmap_file(m_hdr, m_len, m_fd);
}
#endif
