/* -*- mode:c; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of Aegis crypto services
 *
 * Copyright (C) 2010-2011 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
 \file aegis_storage.h
 \brief Protected storage.

 This library provides functions for protecting files against
 unauthorized reading by encryption and off-line tampering by
 signing. Files are grouped into stores, each store is collection
 of files with common security attributes.

 Signed stores are just lists of filenames and hashes. The files
 exist where they are, but their integrity can be controlled 
 by the functions provided by the classes. Globally signed stores
 can be created and populated at build time by using external 
 PKI signing (to be added).

 Encrypted stores implement a private storage area, where the 
 contents of the files are stored in an encrypted form. The files
 do not necessarily exist with the filenames they are referred 
 to in this API. An encrypted store hence implements a small 
 virtual filesystem. Encrypted stores cannot be created at 
 build time but they must be created by the calling applications.
 Hence, encrypted stores are always initially empty.

 Each encrypted store uses an own key, which is generated from
 random data when the store is created. The secrets are protected
 by the BB5-backed functions in aegis_crypto.h.

*/


#ifndef AEGIS_STORAGE_H
#define AEGIS_STORAGE_H

#include <string.h>
#include <unistd.h>
#include <sys/fcntl.h>
#include <sys/types.h>
#include <utime.h>

/**
 * \def foff_t
 * \brief A macro to encapsulate off_t
 * 
 * The size of the off_t type defined in sys/stat.h varies depending 
 * on how _FILE_OFFSET_BITS is defined. But in this library it must
 * always be 64 bits, so this macro is used to make sure that it is
 * right even if the client would not have 
 * `pkg-config --cflags aegis-crypto`
 * as a compiler switch.
 */
#if defined(_FILE_OFFSET_BITS) && (_FILE_OFFSET_BITS == 64)
#define foff_t off_t
#else
#include <stdint.h>
#define foff_t uint64_t
#endif

#include <openssl/aes.h>
#include <string>
#include <vector>
#include <map>

#include "aegis_common.h"

/**
 * \def O_RECOVER
 * \brief An additional open mode flag for p_open
 * 
 * Allow opening an encrypted file even if it fails integrity
 * check and restore the store integrity with the new content.
 */
#define O_RECOVER     010000000

namespace aegis {

	class storage;
	class pe_file;
    // class p_file_lock;
    class storage_lock;

    /**
     * \brief The storage directory
     * 
     * Returns the path to the directory where all protected storage
     * data is stored into.
     */
    const char* storage_root();

	/**
	 * \brief Set alternative encrypted storage directory
	 * 
	 * Store encrypted files in the given path. Must be set before any
	 * storage objects are created and will make any existing stores 
	 * created by different settings inaccessible. Use cautiously.
	 * \param to_this (in) an absolute path of a directory. If the directory
	 * doesn't exist it will be created on demand given that the caller has 
	 * credentials. If not, subsequent attempts to 
	 * \return true if the call was succesful, false otherwise check errno
	 */
	bool set_alternative_storage_root(const char *to_this);

    /**
     * \class p_file
     * \brief A single file in a protected store. 
     * 
     * Implements the usual POSIX semantics of open, close, read and write.
     */
	class p_file {
		friend class storage;
		friend class pe_file;
		friend class storage_lock;

	public:
        /**
         * \brief Destructor. 
         * 
         * Notice that the constructor of this class is private, i.e.
         * a p_file instance can only be created by function 
         * \ref storage::member.
         */
		virtual ~p_file();

        /**
         * \brief Open file
         * \param flags (in) as in man(2) open, flags plus O_RECOVER
         * \return true if the file could be opened/created
		 * \credential * For opening a file for writing
		 * in a signed store one must have the storage token. For opening
		 * a file in any way in an encrypted store one must have the 
		 * storage token.
         * 
         * Opening of a file may fail because of the protection was violated.
         * For instance the file contents didn't match its stored hash (it
         * has been tampered) or the caller doesn't have the required 
         * credentials to open the file for writing or to decrypt and 
         * encrypted file. In all these cases the errno will be set to 
         * EACCES.
         */
		virtual bool p_open(int flags);

        /**
         * \brief Read data from a file
         * \param at (in) The offset from which to read
         * \param data (in) The buffer to copy read data
         * \param len (in) The number of bytes to read
         * \return The number of bytes actually read. Can be 
         * smaller than the given len if an EOF is detected.
         */
		virtual ssize_t p_read(foff_t at, RAWDATA_PTR data, size_t len);

        /**
         * \brief Write data to a file
         * \param at (in) The offset to which to write
         * \param data (in) A pointer to the buffer to write from
         * \param len (in) Number of bytes to write
         * \return The number of bytes actually written. Can be
         * smaller than the given len if the filesystem is full.
         */
		virtual ssize_t p_write(foff_t at, const RAWDATA_PTR data, size_t len);

        /**
         * \brief Truncate the file
         * \param at (in) The new size of the file
         * \return 0 on success, otherwise -1. Use errno to
         * find out the actual error condition.
         */
		virtual int p_trunc(foff_t at);

        /**
         * \brief Close the file
         * 
         * The contents of the file are flushed to the disk and 
         * if it was opened with O_CREAT or O_TRUNC, its size is
         * set to the last written byte offset. Also a O_RDWR file
         * grows if data was written beoynd its original size.
         */
		virtual void p_close();

        /**
         * \brief Is the file currently open
         * \return true if it is, false otherwise
         * 
         * Attempts to read or write into a non-open file return and
         * error and errno is set to EBADF.
         */
		bool is_open();

        /**
         * \brief Get file status
         * \param st (in,out) A pointer to the status buffer
         * \return 0, on success, -1 otherwise. Errno is set.
		 * \credential * For reading the attributes
		 * of a file in an encrypted store one must have the 
		 * storage token.
         */
		virtual int p_stat(struct stat *st);

        /**
         * \brief Get the digest of the file
         * \return A base64 encoded hash computed of the 
         * current contents of the file, terminated with a NUL.
         */
		virtual const char* digest();

		/**
		 * \brief The name shown to outside
		 * \return The public name, not necessarily the place
		 * where the actual contents are stored
		 */
		const char* name();

		/**
		 * \brief Return a pointer to the owning pstore
		 * \return A pointer to the pstore this file belongs to
		 */
        storage* owner();

        /**
         * \brief Rename the file
         * \param new_name (in) The new name
         * \return 0 on success, -1 otherwise. Errno is set.
         */
        virtual int p_rename(const char* new_name);

        /**
         * \brief Change mode flags 
         * \param flags (in) new value of mode flags
         * \return 0 on success, -1 otherwise
         */
        virtual int p_chmod(mode_t flags);

        /** 
         * \brief Change ownership
         * \param uid (in) new file owner uid
         * \param gid (in) new group owner gid
         * \return 0 on success, -1 otherwise
         */
        virtual int p_chown (uid_t uid, gid_t gid);

        /**
         * \brief Change access times
         * \param ntime (in) new access and modification times
         * \return 0 on success, -1 otherwise
         */
        virtual int p_utime(struct utimbuf *ntime);

		/**
		 * \brief Forget pending changes 
		 * 
		 * This function only has effect if the called instance
		 * is opened for writing. It changes the open mode as
		 * read only to prevent committing any changes in the 
		 * protected store when the file is closed. Any changes
		 * already made will not be rolled back in the current
		 * implementation, however. The main use of this function 
		 * is to discard pending changes when a file is removed
		 * from the storage while it was still open.
		 */
		void p_rollback(void);

	private:
		p_file(storage* owner, const char* pathname);
        virtual int p_cleanup();
		bool int_open(int flags, bool lock = false);
		bool check_integrity();
		bool roundup(size_t len);
		virtual size_t datasize();
		void fill_in(size_t len);
		virtual const char* p_name();
		void p_flush(bool do_trunc);
		void update_metadata(struct stat *fs);

		std::string m_name;
		storage*    m_owner;
		int         m_fd;
		size_t      m_size;
		std::string m_digest;
		RAWDATA_PTR m_data;
		size_t      m_mapsize;
		bool        m_new_file;

		enum openmode_t {
			om_readonly,
			om_readwrite,
			om_writeonly,
			om_closed
		} m_omode;

		// p_file_lock *m_lock;
		std::string m_semname;
	};

	struct metadata_t;
	typedef std::map<std::string, metadata_t> contents_map_t;

	/**
	 * \class storage
	 * \brief A secure file container.
	 * 
	 * This class is used to ensure file integrity by local 
	 * signing and prevent unauthorized reading by encryption.
	 */

	class storage
	{
		friend class p_file;
		friend class pe_file;
		friend class storage_lock;

	public:

		/**
         * \enum visibility_t
		 * \brief The visibility of a store. Defined
		 * when a storage is created and cannot be changed
		 * afterwards.
		 */
		typedef enum {
            vis_global, /**< Store is globally signed and 
                         * cannot be modified locally 
                         */

            vis_shared, /**< The caller must have a resource
                         * token to modify a signed store 
                         * or read an encrypted store 
                         */

            vis_private /**< The store is owned by a single
                         * application and can only be modified
                         * by it or read by it. 
                         */
        } visibility_t;
		
		/**
         * \enum protection_t
		 * \brief The protection method of a storage. Defined
		 * when a storage is created and cannot be changed
		 * afterwards.
		 */
		typedef enum {
            prot_signed,   /**< Integrity is protected by signatures 
                            */

            prot_encrypted /**< Confidentiality is protected by encryption 
                            */
        } protection_t;

		/**
		 * \enum status_t
		 * \brief Access status of a store.
		 */
		typedef enum {
			writable,    /**< Store can be read and written to */
			readable,    /**< Store is read only */
			no_access    /**< Access denied, check errno and last_error_str */
		} status_t;

		/**
		 * \brief Create a new store or open an existing one
		 * \param name (in) The logical name of the store
		 * \param owner (in) The resource token to protect the store 
		 *                   or NULL for the application id
		 * \param visibility (in) The visibility of the store
		 * \param protection (in) The protection level of the store
		 * \credential * Caller must have the given owner token to 
		 * create a store, to open an existing signed store for writing
		 * or to open an existing encrypted store for any purpose
		 */
		storage(const char* name, 
				const char* owner, 
				visibility_t visibility, 
				protection_t protection);

		/**
		 * \brief Create a new store or open an existing one with
		 *        the application id.
		 * \param name (in) The logical name of the store
		 * \param visibility (in) The visibility of the store
		 * \param protection (in) The protection level of the store
         * 
         * This function is retained by compatibility, and it cannot
         * be used to create new shared stores which do not yet exist.
		 * 
		 * \credential AID::* Caller must have an application id to 
		 * create a store. Only stores created by the same application 
		 * can be modified if they are signed or read if they are 
		 * encrypted.
		 */
		storage(const char* name, 
				visibility_t visibility, 
				protection_t protection);

		/**
		 * \brief Destructor. Release memory allocations.
		 */
		~storage();

        /**
         * \brief Erase the storage permanently. If the
         * storage is encrypted, also the member files fill
         * be erase. Use cautiously.
         * \return true if the command could be completed.
         * If false, errno tells why.
		 * 
		 * \credential * Caller must have the resource token of the store
         */
        bool remove_all_files();

		/**
         * \typedef stringlist
		 * \brief A list of strings
		 */
		typedef std::vector<const char*> stringlist;

		/**
		 * \brief Get a list of the files in the store
         * \param names (out) The names of the files in the store
         * \return The number of elements in the returned list
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		size_t get_files(stringlist &names);

		/**
		 * \brief Get a list of the actual files that contain the
         * content in the store, including the index file.
         * \param names (out) The names of the files in the store
         * \return The number of elements in the returned list or
		 *    -1 if the caller doesn't have read access.
		 * 
		 * \credential CAP::dac_override Caller must either have the 
		 * resource token of the store or CAP::dac_override in order
		 * to call this function of the store is encrypted. If the
		 * store is signed, no credentials are needed.
		 */
		ssize_t get_ufiles(stringlist &names);

        /**
         * \brief Release a stringlist after use.
         * \param list The list retirned by get_files or get_ufiles
         */
        void release(stringlist &list);

		/**
		 * \brief Check if the store contains the given file
         * \param pathname The name of the file to test
         * \return true, if the file exists in the store
		 *
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		bool contains_file(const char *pathname);

		/**
		 * \brief Check if the store contains the given link
         * \param pathname The name of a symbolic link
         * \return true if the link exists in the store
         * 
         * Links are an internal feature of a store and do not
         * map into actual symbolic links in the filesystem.
         * A link can be added into a store with \ref add_link,
         * adding a file that is a symbolic link will result
         * into a regular file member.
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		bool contains_link(const char *pathname);

		/**
		 * \brief Return the current hash of a file
		 * \param pathname The name of the file
		 * \param hash (out) The current hash as a base64 string
		 * \return true if the file exists in the store
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		bool hash_of_file(const char *pathname, std::string &hash);

		/**
		 * \brief Add an existing file into the store.
		 * \param pathname (in) The name of the file. Relative pathnames
		 * are automatically converted to absolute.
         * 
         * Existing files can only be added to signed stores.
         * New files are added to encrypted stores by \ref put_file
		 * 
		 * \credential * Caller must have the resource token of the store
		 */
		void add_file(const char *pathname);

		/**
		 * \brief Remove a file from the store
		 * \param pathname The name of the file
         * 
         * In a signed store this does not affect the actual file,
         * which will remain where it is. Only its hash is removed
         * from the store.
         * 
         * NOP if the file does not exist in the store.
		 * 
		 * \credential * Caller must have the resource token of the store
		 */
		void remove_file(const char *pathname);

		/**
		 * \brief Add a link to an existing file into the store
		 * \param pathname (in) The name of the link.
		 * \param to_file (in) The file where the link points to.
         * 
		 * The file identified by to_file must already be a member 
         * in the store, otherwise no link is added.
		 * 
		 * \credential * Caller must have the resource token of the store
		 */
		void add_link(const char *pathname, const char *to_file);

		/**
		 * \brief Remove a link from the store
		 * \param pathname The name of the link
         * 
         * NOP if the link doesn't exist in the store.
		 * 
		 * \credential * Caller must have the resource token of the store
		 */
		void remove_link(const char *pathname);

        /**
         * \brief Rename a file or link to something else
         * \param pathname (in) the name of the file or link
         * \param to_this (in) the new name
         *
         * NOP if the given file or link doesn't exist
		 * 
		 * \credential * Caller must have the resource token of the store
         */
        void rename(const char *pathname, const char *to_this);

		/**
		 * \brief Get the name of the file a link
		 * points to
		 * \param pathname (in) The name of the link
		 * \param points_to (out) The full pathname of the file
         * 
         * If no such link exists, the content of the points_to
         * variable remains unchanged.
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		void read_link(const char* pathname, std::string& points_to);

		/**
		 * \brief Check that a file matches the checksum stored
		 * in \see commit
		 * \param pathname (in) The name of the file
		 * \returns true, if the checksum matches
         * 
         * Using this function will easily result in a race condition, 
         * if the file is opened and read later. It is safer
         * to use \ref get_file or the \ref member function to 
         * return a p_open instance which allows reading the 
         * file contents without a race.
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		bool verify_file(const char* pathname);

		/**
		 * \brief Check if the content in a buffer matches the
         * hash recorded for a file.
		 * \param data (in) The data to verify. If the store 
         * is encrypted this should be the plaintext data.
		 * \param of_len (in) Length of data in bytes
		 * \param pathname (in) The name of the file
		 * \returns true, if the contents match
		 *
		 * It is also possible to verify a piece of data at the start
		 * of the file that is smaller than the real file contents.
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		bool verify_content(const char* pathname, 
                            unsigned char* data, 
                            size_t of_len);

		/**
		 * \brief Read an entire file into memory. Verification
		 * and decryption are performed automatically.
		 * \param pathname (in) The name of the file
		 * \param to_buf (out) The buffer where the 
		 * file contents are copied. Decryption is done
		 * automatically if needed. The parameter
		 * needs not to have any value at entry.
		 * \param bytes (out) The number of bytes read,
		 * should equal the file size.
		 * \returns 0 on success, otherwise an error code
		 * Use \see release_buffer to release the returned
		 * buffer after use.
         * 
         * This function is safe to use but requires the 
         * while file to be in memory at once. Use the \ref
         * member function and \ref p_file::p_read to read data
         * in smaller pieces if necessary.
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		int get_file(const char* pathname, 
					 RAWDATA_RPTR to_buf, 
					 size_t* bytes);
	
		/**
		 * \brief Release a buffer
		 * \param buf The buffer to be released, returned 
		 * by \see get_file
		 */
		void release_buffer(RAWDATA_PTR buf);
	
		/**
		 * \brief Write a file to the filesystem. Encrypt if needed.
		 * \param pathname (in) The name of the file to write. If the file
		 * does not yet exist in the store, it's added.
		 * \param data (in) The data to be written and optionally
		 * encrypted
		 * \param bytes (in) The number of bytes to be written
		 * \returns 0 on success, otherwise and error code
		 * \credential * Caller must have the resource token of the store
		 */
		int put_file(const char* pathname, RAWDATA_PTR data, size_t bytes);

		/**
		 * \brief Seal a store
         * \returns true on success. If returns false, check errno 
		 *
		 * Save a new copy of the store index file. Unless this
         * call is made the changes will not be recorded.
		 * 
		 * \credential * Caller must have the resource token of the store
		 */
		bool commit();

		/**
		 * \brief Check for changes made by other processes
         * \returns true if the store was changed by some other
		 *   process after the last commit
		 *
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		bool refresh();


		/**
		 * \brief How many files the storage contains
		 * \return The number of files and links in the store.
		 * If the function returns -1 it means that the caller
		 * does not have read access to the store.
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		int nbrof_files();

		/**
		 * \brief How many links the storage contains
		 * \return The number of links in the store.
		 * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
		 */
		int nbrof_links();

		/**
		 * \brief Logical name of the storage
         * \return The name that was used when the store was opened.
		 */
		const char* name();

		/**
		 * \brief Location of the store
         * \return The actual pathname of the store index file.
         * 
         * The pathname of the index file is subject to changes
         * and no assumptions should be made about it.
		 */
		const char* filename();

		/**
		 * \brief Owner token of the store
		 * \return The token with which the store is protected
		 */
		const char* token();
		  
		/**
		 * \brief List stores of the given visibility and
		 * protection
		 * \param of_visibility (in) vis_shared or vis_private
		 * \param of_protection (in) prot_signed or prot_encrypted
		 * \param matching_names (in) a regular expression for the storage name
		 * \param cb_func (in) an \ref aegis_callback function, the logical 
         * store name as the payload parameter.
		 * \param ctx (in) the context pointer for the callback
		 * \return if < 0, 0 - error code
		 *         if >= 0, the value returned by the callback function
		 */
		static int iterate_storage_names(storage::visibility_t of_visibility, 
                                         storage::protection_t of_protection, 
                                         const char* matching_names,
                                         aegis_callback* cb_func,
                                         void* ctx);

        /**
         * \brief Return the visibility of the store
         * \return The value given in the constructor
         */
		visibility_t visibility() { return m_vis; };

        /**
         * \brief Return the protection type of the store
         * \return The value given in the constructor
         */
		protection_t protection() { return m_prot; };

		/**
		 * \brief Return the status of the store, can it be
		 * written to or just read or is there access at all.
		 * Use this function after constructor to check the
		 * status of the created or opened store.
		 */
		status_t status();

        /**
         * \brief Get the status of a member file
         * \param pathname (in) The name of the file
         * \param stbuf (out) status buffer
         * \return as with man(2) stat
		 *
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
         */
		int stat_file(const char* pathname, struct stat *stbuf);

        /**
         * \brief Create a protected handle to a file in the store
         * \param pathname (in) The name of the file
         * \return A protected handle of type \ref p_file or NULL
         * if some error occured. 
         * 
         * With the protected handle the file can be read from and
         * written to in pieces like a regular file, with no race
         * conditions.
         * 
         * If the file didn't exist, it is created automatically, 
         * so this function cannot be used to test the existence
         * of a file, use the contains_file or contains_link functions
         * for that in stead.
         * 
		 * \credential * Caller must have the resource token of the store
		 * if the store is encrypted
         */
		p_file* member(const char *pathname);

		/**
		 * \brief Test the encryption algorithm with reference data
		 * \param do_encrypt if true encrypt, else decrypt
		 * \param key A 256 bits long AES-key
		 * \param ivec A 16 bytes long initialization vector
		 * \param block_nr Ordering number of the 16 byte block
		 * \param idata 16 bytes of input data
		 * \param odata 16 bytes of output data
		 */
		bool test_xts_aes(bool do_encrypt, 
						  int8_t key[32],
						  int8_t ivec[16],
						  size_t block_nr,
						  int8_t idata[16],
						  int8_t odata[16]);

	private:
		protection_t m_prot;
		visibility_t m_vis;
        std::string  m_token;
		std::string  m_name;
		std::string  m_filename;
		RAWDATA_PTR  m_symkey;
		contents_map_t m_contents;
		std::map<std::string, std::string> m_links;
		size_t m_symkey_len;

		void init_storage(const char* name, 
                          const char* token);
		void reinitialize();
		void create_store_sem();
		const char *store_sem_name();
		bool compute_digest(unsigned char* data, 
							size_t bytes, 
                            char format,
							std::string& digest);
        bool set_aes_keys(bool is_protected, AES_KEY *to_this);
		void clear_aes_keys();
		bool internal_hash(struct aegis_digest_t *bhash, std::string& to_this);
        void normalize_filename(const char *name, std::string& result);
        bool generate_new_symkey();
        void rename(const char *pathname, const char *to_this, p_file *use_member);
		bool storage_changed();
		void lock_store();
		void unlock_store();
		bool convert_store(const char *new_filename);
		bool move_file(const char *old_name, const char *new_name);

        storage_lock *m_lock;
	};
};

#endif
