/* -*- mode:c; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of Aegis crypto services
 *
 * Copyright (C) 2010-2011 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
 * \file aegis_crypto.h
 * \brief Keyless cryptographic primitives
 *
 * The underlying algorithms are subject to change. The scope of the
 * operations is limited in one device: a signature made in one device
 * cannot be verified in another device and data encrypted in one device
 * cannot be decrypted in another device.
 *
 * Also the mode in which the device is limits the scope. Data encrypted
 * in the closed mode cannot be decrypted in the open mode. Other than
 * that the operations work across modes. There is a special status code
 * for a signature made in the other mode.
 */

#ifndef AEGIS_CRYPTO_H
#define AEGIS_CRYPTO_H
#endif

#include "aegis_common.h"

#ifdef	__cplusplus
extern "C" {
#endif

    /**
     * \def MAX_CRYPTO_INPUT_SIZE
     * \brief The crypto operations are guaranteed to work in all environments
	 * at most with this long input buffer in functions aegis_crypto_sign,
	 * aegis_crypto_verify, aegis_crypto_encrypt and aegis_crypto_decrypt.
	 * The signing functions work with longer buffers, but they change the
	 * strategy by first computing a hash over the input data and then 
	 * signing the hash. The encryption functions will return aegis_crypto_error
	 * with errno set as EINVAL, so you will just have to encrypt your data in 
	 * smaller chunks than this.
     */
#define MAX_CRYPTO_INPUT_SIZE 4050

    /**
     * \def SIGNATURE_LENGTH
     * \brief The length of a binary signature in bytes
     */
#define SIGNATURE_LENGTH 20

    /**
     * \struct aegis_signature_t
     * \brief A binary signature
     * \typedef AEGIS_SIGNATURE_T
     * \brief A synonym for "struct aegis_signature_t"
     */
    typedef struct aegis_signature_t {
        unsigned char d[SIGNATURE_LENGTH]; /**< The actual signature bytes */
    } AEGIS_SIGNATURE_T;

    /**
     * \def SIGNATURE_STRING_LENGTH_HEX
     * \brief The length of a signature string in hexadecimal,
     * less the terminating NUL
     */
#define SIGNATURE_STRING_LENGTH_HEX 40

    /**
     * \def SIGNATURE_STRING_LENGTH_BASE64
     * \brief The length of a signature string in base64
     * less the terminating NUL
     */
#define SIGNATURE_STRING_LENGTH_BASE64 30

    /**
     * \def APPLICATION_ID
     * \brief Use this constant to make a signature with
     * the application id
     */
#define APPLICATION_ID NULL

    /**
     * \def AEGISFS_SGN_XATTR
     * \brief Any aegisfs mountpoint can be verified by
     * help of this extended attribute. It is done by writing
     * "token-name+newline+at-least-ten-bytes-of-random-data" to
     * the attribute and then reading its value, which should be
     * a valid signature made by the given token. If it is not,
     * something funny is going on.
     * Function aegis_crypto_verify_aegisfs implements this
     * challenge/response check in one function.
     */

#define AEGISFS_SGN_XATTR ".aegisfs-signature"

    /**
     * \def AEGISFS_VFY_TOKEN
     * \brief The AegisFS verification is made by this resource
     * token
     */
#define AEGISFS_VFY_TOKEN "aegisfs::aegisfs-verify"

    /**
     * \def AEGISFS_VFY_DATALEN
     * \brief When verifying aegisfs, exactly this many bytes
     * of random data must be given to get the signature
     */
#define AEGISFS_VFY_DATALEN 32

    /**
     * \enum aegis_system_mode_t
     * \brief The mode the system is in.
     *
     * "Open" means that that the integrity cannot be guaranteed,
     * as for instance the kernel may have been modified.
     * "Protected" means that the system integrity is guaranteed.
	 * "Plain" means that the security frameowrk is not active, no
	 * integrity is guaranteed and not all functions can be used.
     */
    typedef enum {
        aegis_system_open,      /**< Integrity cannot be guaranteed  */
        aegis_system_protected, /**< Integrity should be guaranteed  */
		aegis_system_emulated,  /**< Using the TEE emulator          */
		aegis_system_plain      /**< No security framework available */
    } aegis_system_mode_t;

    /**
     * \brief In what mode the system seems to be in
     * \return The current mode.
     *
     * Notice that this function cannot be fully trusted, as it
     * is possible to fool the system to believe that it is in
     * protected mode even if it is not by a custom kernel and
     * by emulating BB5 functions, or simply be replacing this
     * very function. It should be used only as a guideline.
     */
    aegis_system_mode_t
    aegis_current_mode(void);

    /**
     * \enum aegis_crypto_result
     * \brief Error codes from various functions
     *
     * Use function \ref aegis_crypto_last_error_str
     * to get detailed information about the error.
     */
    typedef enum {
        aegis_crypto_ok = 0,                  /**< No errors detected           */
        aegis_crypto_error,                   /**< Miscellaneous, use errno and 
									               aegis_crypto_last_error_str 
												   to get details               */
		aegis_crypto_error_signature_missing, /**< Signature could not be found */
		aegis_crypto_error_wrong_signature    /**< Signature mismatch           */
    } aegis_crypto_result;

    /**
     * \enum aegis_sysinvariant_t
     * \brief Supported system invariants
     * These are system wide constant values that are not supposed
     * to change uncontrollably.
     */
	typedef enum {
		sysinvariant_imei /** The IMEI code of the device */
	} aegis_sysinvariant_t;

    /**
     * \brief Query the value of a given system invariant
     * \return The value of the invariant as a NUL-terminated string
     */
	const char*
	aegis_system_invariant(aegis_sysinvariant_t invariant);


	/**
	 * \brief Return the application id of a given process
     * \param of_pid (in) the process id
     * \param to_this (out) the application id as a NUL-terminated
     * string. Use \ref aegis_crypto_free to release when no longer
     * needed.
	 */
	void aegis_application_id(pid_t of_pid, char **to_this);

    /**
     * \def UNKNOWN_APP_ID
     * \brief The application id if there is no manifest for
     * the package from which the binary comes from.
     */
     #define UNKNOWN_APP_ID "unknown.unknown."

	/**
	 * \brief Return the application id of a given binary
     * \param pathname (in) the pathname of a binary executable
     * \param to_this (out) the application id as a NUL-terminated
     * string. Use \ref aegis_crypto_free to release when no longer
     * needed.
	 */
    void aegis_application_id_of_bin(const char* pathname, char **to_this);

    /**
     * \brief Make a signature of a given data with a given resource token
     * \param data (in) A pointer to the data to be signed
     * \param nbrof_bytes (in) The number of bytes to be signed
     * \param with_token (in) The name (a NUL-terminated string) to make
     * the signature with, or APPLICATION_ID (i.e. NULL) for the calling 
	 * process's application id. To use a token one must possess it.
     * \param signature (out) The signature in the binary form
     * \return OK or an error
	 * \credential * Caller must have the given with_token
     */
    aegis_crypto_result
    aegis_crypto_sign(const RAWDATA_PTR data,
					  const size_t nbrof_bytes,
					  const char *with_token,
					  struct aegis_signature_t *signature);

    /**
     * \brief Verify a signature
     * \param signature (in) The signature to be verified in the binary form
     * \param with_token (in) The resource token the signature was made with.
     * One does not need to possess a token to be able to verify.
     * \param data (in) The data the signature was made of
     * \param nbrof_bytes (in) The number of bytes in the data
     * \param made_in_mode (out) In which mode the signature was made in
     * \return OK or an error
     */
    aegis_crypto_result
    aegis_crypto_verify(struct aegis_signature_t *signature,
						const char *with_token,
						const RAWDATA_PTR data,
						const size_t nbrof_bytes,
						aegis_system_mode_t* made_in_mode);

    /**
     * \enum aegis_format_t
     * \brief Various textual forms for a signature
     */
    typedef enum {
        aegis_as_hexstring, /**< As a hex-string, using characters 0-9 and a-f */
        aegis_as_base64     /**< As base64 */
    } aegis_format_t;

    /**
     * \brief Convert a binary signature into a human-readable string
     * \param from (in) The signature in the binary form
     * \param use_format (in) What encoding to use
     * \param token_name (in) If given, will be embedded in the signature.
     * This makes the result string variable length
     * \param to (out) The result string as a NUL terminated string.
     * Use \ref aegis_crypto_free to release
     * \return The length of the result string, less the terminating NUL
     */
    size_t
    aegis_crypto_signature_to_string(struct aegis_signature_t *from,
									 const aegis_format_t use_format,
									 const char* token_name,
									 char **to);

    /**
     * \brief Convert a signature string into the binary format
     * \param from (in) The NUL-terminated string to be converted
     * \param to (out) The signature in the binary form
     * \param token_name (out) The token name, if it was included in
     * the signature string. A NUL-terminated string, use aegis_crypto_free
     * to release.
     * \return OK or an error
     */
    aegis_crypto_result
    aegis_crypto_string_to_signature(const char *from,
									 struct aegis_signature_t *to,
	                                     char **token_name);

    /**
     * \brief Release a piece of memory allocated by some of the
     * other functions
     * \param ptr (in) A pointer to the piece of memory
     */
    aegis_crypto_result
    aegis_crypto_free(RAWDATA_PTR ptr);

    /**
     * \brief Encrypt a piece of memory
     * \param plaintext (in) The data to be encrypted
     * \param nbrof_bytes (in) The number of bytes to encrypt
     * \param token_name (in) The resource token to encrypt with.
     * The calling process must possess this token.
     * \param ciphertext (out) A pointer to the encrypted data.
     * Use \ref aegis_crypto_free to release when no longer needed.
     * \param result_size (out) The number of bytes in the ciphertext
     * \return OK or an error
	 * \credential * Caller must have the given token_name
     */
    aegis_crypto_result
    aegis_crypto_encrypt(const RAWDATA_PTR plaintext,
						 const size_t nbrof_bytes,
						 const char *token_name,
						 RAWDATA_RPTR ciphertext,
						 size_t *result_size);

    /**
     * \brief Decrypt a piece of memory
     * \param ciphertext (in) The encrypted data to be decrypted
     * \param nbrof_bytes (in) The length of the encrypted data
     * \param token_name (in) The resource token to decrypt with.
     * The calling process must possess this token.
     * \param plaintext (out) A pointer to the plaintext data.
     * Use \ref aegis_crypto_free to release when no longer needed.
     * \param result_size (out) The number of bytes in the plaintext.
     * \return OK or an error
	 * \credential * Caller must have the given token_name
     */
    aegis_crypto_result
    aegis_crypto_decrypt(const RAWDATA_PTR ciphertext,
						 const size_t nbrof_bytes,
						 const char *token_name,
						 RAWDATA_RPTR plaintext,
						 size_t *result_size);

    /**
     * \brief Return a description of the last error detected
     * \return A text describing the last error, including source
     * file name and a line number.
     *
     * Call this function immediately after an error. Calling any
     * other crypto function before this function may change the
     * information.
     */
    const char*
    aegis_crypto_last_error_str(void);

	/**
     * \brief Initialize the cryptofunctions
	 * \return 1 when success, 0 when failure.
	 * \credential application-id The calling process must
	 * have an application id, i.e. it must come from a signed
	 * package with a security manifest and mentioned in the
	 * manifest or inherit the application id from a parent
	 * process.
     *
     * It is not absolutely necessary to call this function,
     * as calling any other function will cause it being called
     * automatically. If your application has a well-defined entry
     * point (main or similar), it is good to call this explicitly
     * to catch possible errors.
	 * 
	 * \credential AID::* Must have an application id
	 */
    int aegis_crypto_init(void);

    /**
     * \brief Finish the cryptofunctions.
     *
     * Again, it is not necessary to call this function, as it is
     * installed as an atexit callback when aegis_crypto_init is
     * called. You can call it for symmetry if you have also called
     * aegis_crypto_init explicitly or to release some resources
     * early. If other crypto functions are called after this
     * the cryptofunctions will get re-initialized automatically.
     */
    void aegis_crypto_finish(void);

    /**
     * \brief Generate random bytes
     * \param to_buf (in) A pointer to the buffer that must hold
     * at least the given number of bytes
     * \param bytes (in) How many bytes to create
     * \return The number of random bytes returned. Should be equal
     * to the given amount, -1 in case of error.
     */
    ssize_t aegis_crypto_random(RAWDATA_PTR to_buf, size_t bytes);

    /**
     * \brief Generate a new symmetric encryption key
     * \param to_buf (out) The new symmetric key
     * \return The length of the new key
     *
     * This function is for internal use only. Basically it just
     * returns random bytes.
     */
    size_t aegis_crypto_new_symkey(RAWDATA_RPTR to_buf);

    /**
     * \brief Return the length of a symmetric encryption key
     * \return Number of bytes in the key
     *
     * This function is for internal use only.
     */
    size_t aegis_crypto_symkeylen(void);

    /**
     * \brief Make a signature of a given file with a given resource token
     * \param pathname (in) The filename to be signed
     * \param data (in) The data to be written into the file. If NULL, the
	 * current file contents are signed.
	 * \param len (in) If data is not NULL, how many bytes are written
     * \param with_token (in) The name (a NULL-terminated string) to make
     * the signature with, or APPLICATION_ID for the calling process's
     * application id. To use a token one must possess it.
     * \return OK or an error
	 * \credential * Caller must have the given with_token
     */
    aegis_crypto_result
    aegis_crypto_sign_file(const char *pathname,
						   const void *data,
						   const size_t len,
						   const char *with_token);
    /**
     * \brief Verify a signature of a distinct file
     * \param pathname (in) The filename the signature was made of
     * \param data (in) The data supposed to be in the file. If NULL, the
	 *  current file contents are verified.
	 * \param len (in) If data is not NULL, how many bytes are verified.
     * \param with_token (in) The resource with which  the signature must have
	 *  made with.
     * \return OK or an error
     */
    aegis_crypto_result
    aegis_crypto_verify_file(const char *pathname,
							 const void *data,
							 const size_t len,
							 const char *with_token);

    /**
     * \brief Verify that the given directory is an AegisFS mountpoint
     * \param dir (in) Name of the directory
     * \param cmode (out) On ouput, the variable tells in which mode
     *                    the security framework is on. If the mode
     *                    is open, take the result with a grain of salt.
     * \return 0 if the result is positive, an error number otherwise
     *
     * Use this function to verify that a directory is mounted by
     * the AegisFS cryptofilesystem. If the result is positive and
     * the returned mode is protected, you should be good to go.
     * Otherwise you might want to reconsider whether you want to
     * store sensitive data in this directory or rely on the
     * integrity of its contents.
     *
     * This function can be used for all types of aegisfs mountpoints
     * regardless of their type and access rights.
     */
    aegis_crypto_result
    aegis_crypto_verify_aegisfs(const char *dir,
                                aegis_system_mode_t *cmode);

#ifdef	__cplusplus
};
#endif
