/* -*- mode:c; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of Aegis crypto services
 *
 * Copyright (C) 2010-2011 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
  
  \file aegis_common.h
  \brief Common definitions and functions
  
  \mainpage Aegis Crypto
  
  The aegis-crypto package provides the basic cryptographic 
  functions for creating and verifying signatures and for 
  encryption and decryption. The key management is based on 
  the resource token model, where generating signatures and
  the encryption and decryption require that the calling process
  possesses the given resource token. The security is based on
  access control.

  In stead of named resource tokens an application can use its
  unique application id in stead. Every binary has such an id,
  which is derived from the name of the package it is installed
  from. When application id is used, only binaries installed from
  the same package can decrypt any encrypted data.

  The functionality is divided into three modules:
  
  - \ref aegis_crypto.h
  Contains functions for handling a single, contiguous piece of 
  memory. This is most useful in signing and/or encrypting messages
  to implement secure IPC. API provided for both C and C++ applications.

  - \ref aegis_storage.h
  Implements a type of a virtual file system API with the usual 
  POSIX semantics. Creating or opening a file via this API automatically
  adds the needed protection to the operation, including transparent
  encryption. API provided only for C++.

  The low-level crypto operations are based on the BB5 chipset assisted
  functionality. If BB5 is not available an emulation is provided, which
  is not really secure but can be used for testing.

  Copyright (c) Nokia Devices 2010

  For licencing details please consult the COPYING file

*/

#ifndef AEGIS_COMMON_H
#define AEGIS_COMMON_H

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <syslog.h>

#if 0
/**
 * \def AEGIS_DEBUG_ENABLED
 * \brief If enabled, all kinds of debug messages are emitted
 * as UDP messages. Use the aclogmon utility to capture and 
 * display and/or log the messages.
 */

#define AEGIS_DEBUG_ENABLED
#endif

/**
 * \def RAWDATA_PTR
 * \brief A pointer to typeless binary data
 * 
 * This data type is used as a parameter type with such
 * arguments that refer to a given piece of memory, for instance
 * data buffers given as input.
 */
#define RAWDATA_PTR void*

/**
 * \def RAWDATA_RPTR
 * \brief A pointer to returned typeless binary data
 * 
 * This data type is used as a parameter type with such
 * arguments that the called function allocates a buffer for
 * to return variable length output. All such values should
 * be release by the caller with \ref aegis_crypto_free when
 * no longer needed.
 */
#define RAWDATA_RPTR void**

#ifdef	__cplusplus
#include <expat.h>
#include <string>
#include <vector>
#include <map>

extern "C" {

	/**
	 * \brief Find out the absolute pathname of a file or directory
	 * \param name (in) The name of the file or directory
	 * \param to_this (out) The absolute pathname of the file or directory
	 * \returns true if the file or directory was found and was accessible,
	 * otherwise false
	 */
	bool absolute_pathname(const char* name, std::string& to_this);

	/**
	 * \brief Return the name of the executable running a process
	 * \param of_pid (in) The process identifier (pid)
	 * \param to_this (out) The full pathname of the process
	 * \returns 1 if the operations succeeds or 0 if it fails
	 */
	int process_name_of_pid(pid_t of_pid, std::string& to_this);

	/**
	 * \brief Return the name of the executable running this process
	 * \param to_this (out) The full pathname of the process
	 * \returns true if the operations succeeds or false if it fails
	 */
	bool process_name(std::string& to_this);

	/**
	 * \brief Append the given raw data as a hexdump to the given string
	 * \param to_this (in out) The string where the data is appended to
	 * \param dta (in) The raw data to be appended in printable string
	 * \param len (in) The number of bytes in the raw data
	 */
	void append_hex(std::string& to_this, unsigned char* dta, unsigned len);

#else // Not C++
	/**
	 * \def bool
	 * \brief In C-environment, define 'bool' as 'int'
	 */
	 #define bool int
#endif

	/**
	 * \def PATH_SEP
	 * \brief The path separator used in pathnames
	 */
	#define PATH_SEP "/"

    /**
	 * \brief Test if a file exists
	 * \param name (in) filename
	 * \returns true, if the file exists and is a regular file
	 */
	bool file_exists(const char* name);

    /**
	 * \brief Test if a directory exists
	 * \param name (in) directory name
	 * \returns true, if the file exists and is a directory
	 */
	bool directory_exists(const char* name);

	/**
	 * \brief Create a new directory. Create also all missing
	 * intermediate directories in the path, if they do not 
	 * exist already
	 * \param path (in) pathname of the directory
	 * \param mode (in) access control bits of the directory
	 * \returns 0, if the directory could be created or an error
	 *          code otherwise
	 */
	int create_directory(const char* path, int mode);

	/**
	 * \brief A generic callback function type. A function of this
	 * type is called in the various iterate_*-functions in this library.
	 * \param nbr (in) The order number of the item, starting from 0
	 * \param item (in) An item in the collection. The actual type
	 * depends on the type of the collection and is documentd for each
	 * iterate-function separately.
	 * \param context (in) A generic context pointer, relayed from
     * the iterate-function call.
	 */
	typedef int aegis_callback(int nbr, void* item, void* context);

	/**
	 * \brief Iterate through the files in a directory
	 * \param in_directory (in) the name of the directory
	 * \param matching_names (in) a regular expression matched against
	 * the filenames
	 * \param cb_func (in) a callback function which is called for each
	 * file in the given directory that has a name matching with the
	 * given expression. The item parameter is a NUL-terminated filename.
	 * \param ctx (in) the context pointer
	 */
	int	iterate_files(const char* in_directory,
					  const char* matching_names,
					  aegis_callback* cb_func,
					  void* ctx);

    /**
	 * \def DLOG_TO_CONSOLE
	 * \brief A special value to give to init_dlogger to make
	 *   all debug output in stdout
	 */
	#define DLOG_TO_CONSOLE 17

	/**
	 * \brief Init debug logging
	 * \param enabled (in) if zero or more, dispatch debug messages 
	 *   up to this level (LOG_ERR, LOG_WARN, LOG_DEBUG, LOG_INFO etc.). 
	 *   -1 means that all debug output is disabled. DLOG_TO_CONSOLE
	 *   makes all debug output to be printed to stdout.
	 * \param use_signal (in) if non-zero, install a handler for
	 *   the given signal to turn on debugging info on the fly. 
	 *   Usually SIGUSR1.
	 * \param other_signal (in) if use_signal was non-zero, this
	 *   is the signal to turn off debugging info. SIGUSR2.
	 */
	void init_dlogger(int enabled, int use_signal, int other_signal);

	/**
	 * \brief Send a debug or error message to the dlog
	 */
	void dlog_message(const char* format, ...);

	/**
	 * \brief Return a hexdump of the given data. No need
	 * to allocate, but there can be at most 10 such strings
	 * in use at any point of time. They are released when
	 * to calling process terminates.
	 * \param d (in) the data
	 * \param len (in) number of bytes
	 * \return a dynamically allocated string that will be
	 * automatically deallocated after 10 subsequent calls
	 * to this function. Use strdup to make a permanent 
	 * copy if needed.
	 */
	const char* dynhex(const void* d, size_t len);

	/**
	 * \brief Base-64 encode
	 * \param data (in) the data to be encoded
	 * \param len (in) length of data in bytes
	 * \return a dynamically allocated string. Use
	 * free to release when no longer needed.
	 */
	char* base64_encode(const RAWDATA_PTR data, size_t len);

	/**
	 * \brief Base-64 decode
	 * \param string (in) base64 encoded NULL-terminated string
	 * \param to_buf (out) a pointer to a pointer to store the data
	 * use free to release when no longer needed.
	 * \return the length of returned data in bytes
	 */
	size_t base64_decode(const char* string, RAWDATA_RPTR to_buf);

#ifdef	__cplusplus
} // extern "C"


/**
 * \namespace aegis
 * \brief All the C++ classes and functions belong to this namespace.
 */
namespace aegis {

    /** 
     * \class c_xmlnode
     * \brief A simple XML DOM implementation
     * 
     *  This class is used together with the c_xmldoc-class to manage
     *  XML as in-memory DOM-tree
     */
    class c_xmlnode 
    {
    public:
        /**
         * \brief Constructor
         *  \param of_parent The node's parent (NULL, if this is root node)
         *  \param of_name The name of the element to create
         */
        c_xmlnode (c_xmlnode* of_parent, const char* of_name);
            
        /**
         *  \brief Destructor
         */
        ~c_xmlnode ();
        
        /**
         * \brief Return the parent of the node
         *  \return The c_xmlnode-pointer that was given in
         * the constructor
         */
        c_xmlnode* parent();
        
        /**
         * \brief Return the element's name
         *  \return The name of the element
         */
        const char* name ();

        /**
         * \brief Return the contents of a subtree as a string
         * \param add_linebreaks Is the string pretty-printed or not
         * \param indent_width With how many spaces to indent
         * \param indent_level At what level to start the indentation
         * \return a string containing the node as XML-text
         */
        std::string as_string (bool add_linebreaks, 
                               int indent_width, 
                               int indent_level);

        /**
         *  \brief Append text into the content
         * \param data The text to be appended
         * \param len How many bytes of data
         */
        void append_content (const char* data, int len);

        /**
         *  \brief Append text into the content
         * \param data A NULL-terminated string
         */
        void append_content (const char* data);

        /**
         * \brief Append content from an external file
         *  \param file_name The name of the file where the
         *  data is read
         */
        void append_file (const char* file_name);

        /**
         * \brief Return the content
         *  \return A pointer to the (raw) content
         * 
         * When using this function, the data is in raw-format,
         * i.e. no special characters are escaped. Use the 
         * as_string function to escape the data properly.
         */
        const char* content ();

        /**
         *  \brief Add an attribute to the node with a text value
         *  \param name The name of the attribute
         *  \param value The value of the attribute as a NULL-terminated
         *  string
         * 
         *  If the node already contains an attribute with the same name,
         *  its current value is overwritten
         */
        void append_attribute (const char* name, const char* value);

        /**
         *  \brief Add an attribute to the node with an integer value
         *  \param name The name of the attribute
         *  \param value A numeric integer value of the attribute
         * 
         *  If the node already contains an attribute with the same name,
         *  its current value is overwritten
         */
        void append_attribute (const char* name, long value);
            
        /**
         *  \brief How many attributes the node contains
         *  \return The number of attributes
         */
        int nbrof_attributes ();

        /**
         *  \brief Return the name of the nth attribute
         *  \param of_pos attribute order number, starting from 0
         *  \return The attribute name
         */
        const char* attribute_name (unsigned of_pos);

        /**
         * \brief Return the value of the nth attribute
         *  \param of_pos attribute order number, starting from 0
         *  \return The attribute (raw) value
         */
        const char* attribute_value (unsigned of_pos);

        /**
         * \brief Return an attribute value identified by its name
         *  \param name The name of the attribute
         *  \param required If true and the attribute does not exist,
         *  an exception is thrown
         *  \param defval If not required and the attribute does not
         *  exist, this value is returned
         */
        const char* attribute(const char* name, 
                              bool required, 
                              const char* defval);

        /**
         * \brief Remove an attribute
         * \param name The name of the attribute
         * 
         * NOP if the node does not contain the given attribute
         */
        void remove_attribute (const char* name);

        /**
         * \brief Return the selected subnode
         * \param to_xpath An xpath-like expression, see c_xmldoc::navigate
         * for further details
         * \param required If true and the given path is not found,
         * an exception is thrown, otherwise NULL is returned
         */
        c_xmlnode* navigate(const char* to_xpath, bool required);

        /**
         * \brief Return the selected subnode, create if it does not exist
         * \param to_xpath An xpath-like expression, see c_xmldoc::navigate
         *  for further details
         * \param default_content If the searched node does not yet exist and
         *  is created, use this as the default value
         */
        c_xmlnode* navigate(const char* to_xpath, const char* default_content);

        /**
         * \brief Add a child-element to the node
         * \param element_name The name of the child-element to add
         * \return A pointer to the newly added element
         * 
         *  The new child is always appended as the last element
         *  regardless whether the node already contains a 
         *  child-element with the same name or not
         */
        c_xmlnode* append_child (const char* element_name);

        /**
         * \brief How many child-elements the node has
         * \return The number of child-elements
         */
        int nbrof_children ();

        /**
         * \brief Return a specific child element
         * \param of_pos The order number of the child, starting from 0
         * \return A pointer to the child-element or NULL of the given
         *  position is out-of-bounds
         */
        c_xmlnode* child (unsigned of_pos);

        /**
         * \brief Return a child element specified by its name
         * \param of_name The name of the child element
         * \param required If true and there is no child element
         * with the given name, an exception is thrown, otherwise
         * NULL is returned
         * 
         * If the node contains several child elements with the
         * given name, the first one is returned
         */
        c_xmlnode* child (const char* of_name, bool required);

        /**
         * \brief Remove a child element
         * \param of_pos The order number of the child, starting from 0
         */
        void remove_child (unsigned of_pos);

        /**
         * \brief Tag the content as CDATA
         * \param to_this if true, the content is tagged as CDATA
         * 
         *  In a CDATA-node the input data is not parsed for XML-escape
         *  sequences and when requesting an XML-string, the content
         *  is enclosed in a <![CDATA[...]]>-section
         */
        void set_cdata(bool to_this);

        /**
         * \brief Remove leading and trailing whitespace from the content
         */
        void trim_whitespace();

        /**
         * \brief Return the xpath of the node starting from root
         * \return A string that contains the names of the parent
         * nodes separated by '/'
         */
        std::string xpath();

        /**
         * \brief Release the node's contents and all subnodes
         */
        void reset();

    private:
        c_xmlnode* navigate(const char* to_xpath, bool required, const char* default_content);
        void xml_escaped(const char *from, std::string& to);
        
        std::string m_tagname;
        std::string m_content;
        c_xmlnode* m_parent;
        std::map<std::string,std::string> m_attributes;
        std::vector<c_xmlnode*> m_children;
        bool is_cdata;
        bool cdata_ended;
    };

    /**
     * \class c_xmldoc
     * \brief XML file i/o
     * 
     * Functions for parsing a file into a c_xmlnode DOM tree and writing
     * a file out of one.
     */
    class c_xmldoc
    {
    public:
        /**
         * \brief Constructor
         */
        c_xmldoc ();
            
        /**
         * \brief Destructor
         */
        ~c_xmldoc ();

        /**
         * \brief Parse xml from a file
         * \param file_name The name of the file
         */
        void parse_file (const char* file_name);

        /**
         * \brief Parse xml from a string
         * \param xml_as_string The text of the xml
         * \param length The length of the given string. If zero, 
         * strlen(xml_as_string) is assumed
         */
        void parse_string (const char* xml_as_string, int length);

        /**
         * \brief Release the parser. Use this method to release 
         * the resources reserved by the expat parser, 
         * if it probably is not needed any more.
         * 
         * It is safe to make this call and continue parsing,
         * since the parser is re-initialized automatically
         */
        void release_parser();

        /**
         * \brief Release the string buffer allocated for 
         * converting the DOM-tree into a string.
         * 
         * Use this function to minimize runtime memory usage. 
         * It is not mandatory to call this function, since 
         * the destructor will release the buffer eventually 
         * anyway
         */
        void release_string_buffer ();

        /**
         * \brief Create a new empty xml DOM tree with the 
         * given node as root
         * \param root_node_name The name of the root tag
         * \returns a pointer to the root node
         */
        c_xmlnode* create (const char* root_node_name);
	
        /**
         * \brief Return the current root node
         * \returns A pointer to the root node, or NULL, 
         * if the document is empty
         */
        c_xmlnode* root();

        /**
         * \brief Return the contents of the document
         * \param pretty_printing if true, add newlines and
         *  indentation
         * \returns A string that contains the whole document
         * in a parsable form.
         */
        std::string as_string (bool pretty_printing);

        /**
         * \brief Release the whole DOM-tree
         */
        void release_content(void);

        /**
         * \brief Save the contents of the document into a file
         * \param to_file The name of the file where 
         * the contents are written to
         * returns true when success
         */
        bool save(const char* to_file);

        
        // Expat-parser's hook routines
        // ----------------------------
        // These need to be public to be callable from the
        // actual, static hook routines. Do not use them directly.
	
        /**
         * \brief Receive an element start from the expat-parser
         * \param element_name the name of the element
         * \param attributes possible attributes, as a NULL-terminated
         *  string list of name=value pairs
         * 
         * Only for internal use
         */
        void xml_element_start(const char* element_name, const char** attributes);

        /**
         * \brief Receive an element end from the expat-parser
         * \param element_name the name of the element
         * 
         * Only for internal use
         */
        void xml_element_end(const char* element_name);

        /**
         * \brief Receive a piece of node content data
         * \param data The data
         * \param len The length of data
         * 
         * Only for internal use
         */
        void xml_character_data(const char* data, const int len);
            
        /**
         * \brief Indicate that what follows is CDATA
         * 
         * Only for internal use
         */
        void xml_cdata_start();
            
        /**
         * \brief CDATA ends
         * 
         * Only for internal use
         */
        void xml_cdata_end();

        /**
         * \var trim_whitespace
         * \brief Set this trigger to make the parser to discard leading
         * and trailing whitespace from XML content data
         * 
         * Only for internal use
         */
        bool trim_whitespace;

    private:
        void init_parser();
        void xml_parsing_error();
        
        XML_Parser expat_parser;
        c_xmlnode* root_node;
        c_xmlnode* cur_node;
        char* xml_str_buf;
        std::string m_filename;
    };

    /**
	 * \def DIGESTLEN
	 * \brief The length of the digest checksum used in integrity
	 * protection, currently SHA_DIGEST_LENGTH from openssl/sha.h
	 */

#define DIGESTLEN 20

    /**
     * \struct aegis_digest_t
     * \brief A binary hash
     * \typedef AEGIS_DIGEST_T
     * \brief A synonym for "struct aegis_digest_t"
     */
	typedef struct aegis_digest_t {
		unsigned char d[DIGESTLEN]; /**< The actual digest bytes */
	} AEGIS_DIGEST_T;

} /* namespace aegis */

#endif

#ifndef _STRING_H
#include <string.h>
#endif

/**
 * \def bare_file_name
 * \brief Return the file name part of a pathname
 * \param s a path name
 */
#define bare_file_name(s) strrchr(s,'/')?strrchr(s,'/')+1:s

/**
 * def AEGIS_SHOW_ERRORS
 * \brief If defined, all error messages are printed to the
 * stderr in stead of being logged.
 *
 * This macro is undefined by default. You can define it in
 * command line utilities.
 */

/**
 * \def AEGIS_ERROR
 * \brief Report an error to the dlog server.
 * \param format,args Format string and a list of optional arguments
 * as in "printf".
 */
#ifdef AEGIS_SHOW_ERRORS
#define AEGIS_ERROR(format,args...)                      \
	do {												 \
		fprintf(stderr, "%s(%d): ERROR: " format "\n",	 \
				bare_file_name(__FILE__), __LINE__		 \
				,##args);								 \
		fflush(stderr);									 \
	} while (0)
#else
#ifdef AEGIS_DEBUG_ENABLED
#define AEGIS_ERROR(format,args...)                          \
	do {                                                     \
        dlog_message("<0>%s(%d)[%d]: ERROR " format,         \
                 bare_file_name(__FILE__), __LINE__,         \
                 getpid() ,##args);                          \
        syslog(LOG_ERR, "%s(%d): ERROR " format,             \
           bare_file_name(__FILE__), __LINE__                \
           ,##args);                                         \
	} while (0)
#else
#define AEGIS_ERROR(format,args...)                          \
	do {                                                     \
        syslog(LOG_ERR, "%s(%d): ERROR " format,             \
           bare_file_name(__FILE__), __LINE__                \
           ,##args);                                         \
	} while (0)
#endif
#endif

/**
 * \def AEGIS_DEBUG
 * \brief Send a debug message to the dlog server.
 * \param level (in) The detail level. Only those messages are actually
 * printed thatb have the detail level less than or equal than the
 * current value of the debug_level variable.
 * \param format,args (in) Format string and a list of optional arguments
 * as in "printf".
 */
#ifdef AEGIS_DEBUG_ENABLED
#define AEGIS_DEBUG(level,format,args...)	\
	do { \
	  dlog_message("<%d>%s(%d)[%d]: " format, level, bare_file_name(__FILE__), __LINE__, \
					 getpid() ,##args);							\
    } while (0)
/**
 * \def AEGIS_ENTER
 * \brief Send a debug message when entering a function.
 * Uses gcc's __PRETTY_FUNCTION__ macro to show the function name,
 * useful with overloaded and virtual functions.
 */
#define AEGIS_ENTER	\
	do { \
		dlog_message("<2>%s(%d)[%d]: => %s", bare_file_name(__FILE__), __LINE__, getpid(), \
					 __PRETTY_FUNCTION__);								\
    } while (0)
/**
 * \def AEGIS_EXIT
 * \brief Send a debug message when exiting a function.
 * Uses gcc's __PRETTY_FUNCTION__ macro to show the function name,
 * useful with overloaded and virtual functions.
 */
#define AEGIS_EXIT	\
	do { \
		dlog_message("<2>%s(%d)[%d]: <= %s", bare_file_name(__FILE__), __LINE__, getpid(), \
					 __PRETTY_FUNCTION__);								\
    } while (0)
#else
#define AEGIS_DEBUG(level,format,args...)
#define AEGIS_ENTER
#define AEGIS_EXIT
#endif

/**
 * \def GETENV
 * \brief Get environment value or the default if not found
 * \param name of the environment variable 
 * \param deflt of the environment variable 
 */
#define GETENV(name,deflt) ({char* c = getenv(name); c?c:deflt;})

#endif
