/* -*- mode:c; tab-width:4; c-basic-offset:4; -*- */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#ifdef USE_AEGIS
#include <aegis_crypto.h>
#endif

#define RNG_BLOCK_SZ 16
   
#define FIPS_NUMBYTES     2500
#define FIPS_MONO_LOBOUND 9654
#define FIPS_MONO_HIBOUND 10346
   
/* For each of the 256 possible bit values, how many 1 bits are set? */
static char nb_tbl[256] = { 
  0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3,
  4, 3, 4, 4, 5, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4,
  4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 2,
  3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5,
  4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 1, 2, 2, 3, 2, 3, 3,
  4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 2, 3,
  3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5,
  6, 6, 7, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6,
  4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
  6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8 
};
   
static int 
spc_fips_monobit(unsigned char data[FIPS_NUMBYTES]) 
{
  int i, result;
   
  for (i = result = 0;  i < FIPS_NUMBYTES;  i++)
    result += nb_tbl[data[i]];
  printf("%8d ", result);
  return (result > FIPS_MONO_LOBOUND && result < FIPS_MONO_HIBOUND);
}

#define FIPS_POKER_LOBOUND 1.03
#define FIPS_POKER_HIBOUND 57.4
   
static int 
spc_fips_poker(unsigned char data[FIPS_NUMBYTES]) 
{
  int    i;
  long   counts[16] = {0,}, sum = 0;
  double result;
   
  for (i = 0;  i < FIPS_NUMBYTES;  i++) {
    counts[data[i] & 0xf]++;
    counts[data[i] >> 4]++;
  }
  for (i = 0;  i < 16;  i++)
    sum += (counts[i] * counts[i]);
  result = (16.0 / 5000) * (double)sum - 5000.0;
  printf("%8.2lf ", result);
  return (result > FIPS_POKER_LOBOUND && result < FIPS_POKER_HIBOUND);
}


#define FIPS_LONGRUN   34
#define FIPS_RUNS_1_LO 2267
#define FIPS_RUNS_1_HI 2733
#define FIPS_RUNS_2_LO 1079
#define FIPS_RUNS_2_HI 1421
#define FIPS_RUNS_3_LO 502
#define FIPS_RUNS_3_HI 748
#define FIPS_RUNS_4_LO 223
#define FIPS_RUNS_4_HI 402
#define FIPS_RUNS_5_LO 90
#define FIPS_RUNS_5_HI 223
#define FIPS_RUNS_6_LO 90
#define FIPS_RUNS_6_HI 223
   
/* Perform both the "Runs" test and the "Long Run" test */
static int 
spc_fips_runs(unsigned char data[FIPS_NUMBYTES]) 
{
  /* We allow a zero-length run size, mainly just to keep the array indexing less
   * confusing.  It also allows us to set cur_val arbitrarily below (if the first
   * bit of the stream is a 1, then runs[0] will be 1; otherwise, it will be 0).
   */
  int           runs[2][7] = {{0,},{0,}}; 
  int           cur_val, i, j, runsz;
  unsigned char curr;
   
  for (cur_val = i = runsz = 0;  i < FIPS_NUMBYTES;  i++) {
    curr = data[i];
    for (j = 0;  j < 8;  j++) {
      /* Check to see if the current bit is the same as the last one */
      if ((curr & 0x01) ^ cur_val) {
        /* The bits are different. A run is over, and a new run of 1 has begun */
        if (runsz >= FIPS_LONGRUN) return 0;
        if (runsz > 6) runsz = 6;
        runs[cur_val][runsz]++;
        runsz = 1;
        cur_val = (cur_val + 1) & 1; /* Switch the value. */
      } else runsz++;
      curr >>= 1;
    }
  }

  printf("%8s ", "");
   
  return (runs[0][1] > FIPS_RUNS_1_LO && runs[0][1] < FIPS_RUNS_1_HI &&
          runs[0][2] > FIPS_RUNS_2_LO && runs[0][2] < FIPS_RUNS_2_HI &&
          runs[0][3] > FIPS_RUNS_3_LO && runs[0][3] < FIPS_RUNS_3_HI &&
          runs[0][4] > FIPS_RUNS_4_LO && runs[0][4] < FIPS_RUNS_4_HI &&
          runs[0][5] > FIPS_RUNS_5_LO && runs[0][5] < FIPS_RUNS_5_HI &&
          runs[0][6] > FIPS_RUNS_6_LO && runs[0][6] < FIPS_RUNS_6_HI &&
          runs[1][1] > FIPS_RUNS_1_LO && runs[1][1] < FIPS_RUNS_1_HI &&
          runs[1][2] > FIPS_RUNS_2_LO && runs[1][2] < FIPS_RUNS_2_HI &&
          runs[1][3] > FIPS_RUNS_3_LO && runs[1][3] < FIPS_RUNS_3_HI &&
          runs[1][4] > FIPS_RUNS_4_LO && runs[1][4] < FIPS_RUNS_4_HI &&
          runs[1][5] > FIPS_RUNS_5_LO && runs[1][5] < FIPS_RUNS_5_HI &&
          runs[1][6] > FIPS_RUNS_6_LO && runs[1][6] < FIPS_RUNS_6_HI);
}

typedef int fips_1401_test(unsigned char*);

struct {
	const char *name;
	int (*func)(unsigned char*);
} fips_1401_tests [] = {
	{ "monobit", &spc_fips_monobit },
	{ "poker",   &spc_fips_poker },
	{ "runs",    &spc_fips_runs }
};


int 
main(int argc, char *argv[])
{
	unsigned char data[FIPS_NUMBYTES];
	struct timeval tv;
	int i, res = 0;

	if (2 > argc) {
#ifdef USE_AEGIS
		if (sizeof(data) != aegis_crypto_random(data, sizeof(data))) {
			fprintf(stderr, "ERROR: failed to get random data (%s)", aegis_crypto_last_error_str());
			return 1;
		}
#else
		gettimeofday(&tv, NULL);
		srandom((unsigned int)(tv.tv_usec));
		for (i = 0; i < FIPS_NUMBYTES; i++)
			data[i] = (unsigned char)(0xff & random());
#endif
	} else {
		printf("%s:\n", argv[1]);
		int fd = open(argv[1], O_RDONLY);
		if (0 <= fd) {
			size_t len = read(fd, data, sizeof(data));
			if (sizeof(data) > len) {
				/* If there is less data than needs for the test
				 * fill the rest with pseudorandom data that should
				 * pass the test.
				 */
				size_t p;
				gettimeofday(&tv, NULL);
				srandom((unsigned int)(tv.tv_usec));
				for (p = len; p < FIPS_NUMBYTES; p++)
					data[p] = (unsigned char)(0xff & random());
			}
			close(fd);
		} else {
			fprintf(stderr, "ERROR: cannot open '%s' (%s)", argv[1], strerror(errno));
			return 2;
		}
	}

	for (i = 0; i < 3; i++) {
		printf("%-20s ", fips_1401_tests[i].name);
		if (fips_1401_tests[i].func(data)) {
			printf("PASS\n");
		} else {
			printf("FAILED\n");
			res = 1;
		}
	}
	return res;
}
