#
# Copyright 2010 Ruediger Gad <r.c.g@gmx.de>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import cairo
from datetime import datetime
import dbus
import dbus.mainloop.glib
import gconf
import gobject
import gtk
import hildon
import hildondesktop
import math
import pango
import pangocairo
import os
import re
import sys

PLUGIN_GCONF_DIR = "/apps/Maemo/advanced_clock_plugin"

sys.path.append("/usr/lib/advanced-clock-plugin")
import clock

clocks_path = "/usr/lib/advanced-clock-plugin/clocks"
sys.path.append(clocks_path)

clock_files = os.listdir(clocks_path)
py_files = re.compile("\.py$", re.IGNORECASE)
clock_files = filter(py_files.search, clock_files)
file_to_module = lambda f: os.path.splitext(f)[0]
clock_names = map(file_to_module, clock_files)
clocks = map(__import__, clock_names)


class AdvancedClockPlugin(hildondesktop.StatusMenuItem):
    def __init__(self):
        hildondesktop.StatusMenuItem.__init__(self)

        self.init_gconf()

        self.clock_name = self.gconf_client.get_string(PLUGIN_GCONF_DIR + "/clock_name")
        self.active = True
        self.time = None
        self.old_time = None

#        button = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL, "Advanced Clock")
#        image = gtk.Image()
#        image.set_from_file("/usr/share/icons/hicolor/64x64/apps/advanced-clock-plugin.png")
#        button.set_image(image)
#        button.set_alignment(0, 0.5, 0, 0)
#        button.connect("clicked", self.settings_cb)
#        self.add(button)
#        button.show()

        self.drawing_area = gtk.DrawingArea()
      	self.drawing_area.connect("expose-event", self.expose_cb)
        self.set_status_area_widget(self.drawing_area)
      	self.drawing_area.show()

        self.clock = getattr(clocks[clock_names.index(self.clock_name)], self.clock_name)(self.drawing_area)
        self.clock.request_resize()

 #       self.show()

        gobject.timeout_add(1000, self.update_cb)

        dbus.mainloop.glib.DBusGMainLoop(set_as_default = True)
        self.dbus = dbus.SystemBus()
        self.dbus.add_signal_receiver(self.dbus_signal_cb, "display_status_ind", "com.nokia.mce.signal", None, "/com/nokia/mce/signal")
        self.dbus.add_signal_receiver(self.dbus_alarm_cb, "queue_status_ind", "com.nokia.alarmd", None, "/com/nokia/alarmd")

    def dbus_alarm_cb(self, alarms, desktop, actdead, noboot):
        if desktop != actdead :
            self.clock.set_alarm_enabled(True)
            self.clock.set_alarm_time(datetime.fromtimestamp(actdead))
        else :
            self.clock.set_alarm_enabled(False)
        self.drawing_area.queue_draw()

    def dbus_signal_cb(self, msg):
        if msg == "off" :
            self.active = False
        elif msg == "on" :
            self.update_cb()
            self.active = True
            gobject.timeout_add(1000, self.update_cb)

    def expose_cb(self, widget, event):
        self.clock.draw()
        return False

    def gconf_clock_name_cb(self, client, conn_id, entry, args):
        if entry.get_key() == PLUGIN_GCONF_DIR + "/clock_name" :
            old_clock = self.clock
            self.clock_name = entry.get_value().get_string()
            self.clock = getattr(clocks[clock_names.index(self.clock_name)], self.clock_name)(self.drawing_area)
        
            self.clock.set_alarm_enabled(old_clock.get_alarm_enabled())
            self.clock.set_alarm_time(old_clock.get_alarm_time())

            self.clock.request_resize()
            self.drawing_area.queue_draw()

    def init_gconf(self):
        self.gconf_client = gconf.client_get_default()
        if not self.gconf_client.dir_exists(PLUGIN_GCONF_DIR) :
            self.gconf_client.set_string(PLUGIN_GCONF_DIR + "/clock_name", "DecimalClock")
        self.gconf_client.add_dir(PLUGIN_GCONF_DIR, gconf.CLIENT_PRELOAD_NONE)
        self.gconf_client.notify_add(PLUGIN_GCONF_DIR + "/clock_name", self.gconf_clock_name_cb)

    def settings_cb(self, widget, data=None):
        dialog = hildon.Dialog()
        dialog.set_title("Advanced Clock Plugin Settings")
        pa = hildon.PannableArea()
        vb = gtk.VBox()

        selector = hildon.TouchSelector(text = True)
        for c in clocks :
            selector.append_text(c.get_name())

        self.style_button = hildon.PickerButton(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.style_button.set_title("Clock Style")
        self.style_button.set_selector(selector)
        self.style_button.set_active(clock_names.index(self.clock_name))
        vb.pack_start(self.style_button, True, True, 0)
        self.style_button.show()

        self.show_seconds_button = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.show_seconds_button.set_active(self.clock.get_show_seconds())
        self.show_seconds_button.set_label("Show seconds")
        vb.pack_start(self.show_seconds_button, True, True, 0)
        self.show_seconds_button.show()

        self.twelve_hour_mode_button = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.twelve_hour_mode_button.set_active(self.clock.get_twelve_hour_mode())
        self.twelve_hour_mode_button.set_label("Twelve Hour Mode")
        vb.pack_start(self.twelve_hour_mode_button, True, True, 0)
        self.twelve_hour_mode_button.show()

        self.fill_button = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.fill_button.set_active(self.clock.get_fill())
        self.fill_button.set_label("Fill")
        vb.pack_start(self.fill_button, True, True, 0)
        self.fill_button.show()

        self.alarm_indicator_button = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.alarm_indicator_button.set_active(self.clock.get_alarm_indicator_enabled())
        self.alarm_indicator_button.set_label("Show Alarm Indicator")
        vb.pack_start(self.alarm_indicator_button, True, True, 0)
        self.alarm_indicator_button.show()

        self.color_button = gtk.ColorButton()
        self.color_button.set_color(self.clock.get_gdk_color())
        self.color_button.set_label("Color")
        vb.pack_start(self.color_button, True, True, 0)
        self.color_button.show()

        fonts = pangocairo.cairo_font_map_get_default().list_families()
        self.font_names = map(pango.FontFamily.get_name, fonts)
        selector = hildon.TouchSelector(text = True)
        for f in self.font_names :
            selector.append_text(f)

        self.font_button = hildon.PickerButton(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.font_button.set_title("Font")
        self.font_button.set_selector(selector)
        self.font_button.set_active(self.font_names.index(self.clock.get_font_name()))
        vb.pack_start(self.font_button, True, True, 0)
        self.font_button.show() 

        hb = gtk.HBox()
        label = gtk.Label("Text Scaling")
        hb.pack_start(label, True, True, 0)
        self.text_scale_editor = hildon.NumberEditor(1, 200)
        self.text_scale_editor.set_value(int(self.clock.get_text_scaling() * 100))
        hb.pack_start(self.text_scale_editor, True, True, 0)
        vb.pack_start(hb, True, True, 0)

        ok_button = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL, "Ok")
        ok_button.connect("clicked", self.settings_ok_cb, dialog)
        dialog.action_area.pack_start(ok_button, True, True, 0)
        ok_button.show()

        pa.add_with_viewport(vb)
        pa.set_size_request_policy("children")
        dialog.vbox.pack_start(pa, True, True, 0)
        dialog.show_all()

    def settings_ok_cb(self, widget, dialog):
        self.gconf_client.set_bool(clock.GCONF_DIR + "/alarm_indicator_enabled", self.alarm_indicator_button.get_active())

        self.gconf_client.set_bool(clock.GCONF_DIR + "/fill", self.fill_button.get_active())
        self.gconf_client.set_bool(clock.GCONF_DIR + "/show_seconds", self.show_seconds_button.get_active())
  
        self.gconf_client.set_string(clock.GCONF_DIR + "/font_name", self.font_names[self.font_button.get_active()])

        color = self.color_button.get_color()
        self.gconf_client.set_float(clock.GCONF_DIR + "/color_r", color.red / 65535.0)
        self.gconf_client.set_float(clock.GCONF_DIR + "/color_g", color.green / 65535.0)
        self.gconf_client.set_float(clock.GCONF_DIR + "/color_b", color.blue / 65535.0)

        self.gconf_client.set_float(clock.GCONF_DIR + "/text_scaling", self.text_scale_editor.get_value() / 100.0)
        self.gconf_client.set_bool(clock.GCONF_DIR + "/twelve_hour_mode", self.twelve_hour_mode_button.get_active())

        self.gconf_client.set_string(PLUGIN_GCONF_DIR + "/clock_name", clock_names[self.style_button.get_active()])

        dialog.destroy()

    def update_cb(self):
        if self.time != None:
            self.old_time = self.time

        self.time = datetime.today()

        if self.clock.get_show_seconds() or (self.old_time == None) or (self.old_time.minute != self.time.minute):
            self.drawing_area.queue_draw()

        return self.active


hd_plugin_type = AdvancedClockPlugin


