/*
 * Copyright (c) 2010 Kaj-Michael Lang
 * 
 * Addressbook menu extension to export current contact as a QR Code image.
 *
 * Based on example code provided by:
 * Copyright (C) 2010 Nokia Corporation
 *   @author Mathias Hasselmann <mathias.hasselmann@maemo.org>
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <libosso-abook/osso-abook.h>
#include <hildon/hildon.h>
#include <hildon/hildon-fm.h>
#include <cairo.h>

#include "qrcode_export.h"

#define BLOCK_SIZE (12)
#define PADDING (10)
#define MIN_SIZE (480)
#define MAX_SIZE (1024)

typedef struct {
	OssoABookMenuExtension parent_instance;
} QRcodeMenuExtension;

typedef struct {
	OssoABookMenuExtensionClass parent_class;
} QRcodeMenuExtensionClass;

G_GNUC_INTERNAL GType
qrcode_menu_extension_get_type (void) G_GNUC_CONST;

/* Defines the new plugin object */
OSSO_ABOOK_DEFINE_MENU_PLUGIN(QRcodeMenuExtension, qrcode_menu_extension, OSSO_ABOOK_TYPE_MENU_EXTENSION,,);

static void
qrcode_menu_extension_init(QRcodeMenuExtension *ext)
{
}

static void
contact_to_qrcode_cb(GtkWidget *widget, gpointer user_data)
{
OssoABookMenuExtension *ext=user_data;
OssoABookContact *contact;
GtkWidget *dialog;
const gchar *contact_name;
gchar *vcard, *file, *basefile, *dir;
cairo_surface_t *cs;

contact=osso_abook_menu_extension_get_contact(ext);

if (!contact) {
	hildon_banner_show_information(NULL, NULL, "Failed to get contact.");
	return;
}

vcard=osso_abook_contact_to_string(contact, EVC_FORMAT_VCARD_30, FALSE);
if (!vcard) {
	hildon_banner_show_information(NULL, NULL, "Failed to get contact vCard.");
	return;
}

basefile=osso_abook_contact_get_basename(contact, TRUE);
file=g_strdup_printf("qrcode_%s.png", basefile ? basefile : "unknown_contact");
dir=g_build_filename(g_get_home_dir(), "MyDocs", NULL);

dialog=hildon_file_chooser_dialog_new(NULL, GTK_FILE_CHOOSER_ACTION_SAVE);
gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), file);
gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), dir);

if (gtk_dialog_run(GTK_DIALOG(dialog))==GTK_RESPONSE_OK) {
	gchar *fname;
	
	fname=gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
	contact_name=osso_abook_contact_get_display_name(contact);
	cs=gqre_char_to_qrcode_to_cairo(vcard, MIN_SIZE, MAX_SIZE, PADDING);

	if (cairo_surface_write_to_png(cs, fname)==CAIRO_STATUS_SUCCESS)
		hildon_banner_show_informationf(NULL, NULL, "QR Code exported for contact:\n %s", contact_name);
	else
		hildon_banner_show_information(NULL, NULL, "QR Code export failed!");

	g_free(fname);
	cairo_surface_destroy(cs);
}
gtk_widget_destroy(dialog);
g_free(vcard);
g_free(file);
g_free(dir);
}

/* Entries to show in the menu, the values are:
 * - label: a label for the menu item;
 * - accel_mods:  modifier keys to use for the menu item;
 * - callback: a function to call when the menu item is activated;
 * - name: the name of the menu item.
 */
static OssoABookMenuEntry entries[] = {
	{ "QR Code contact card", 0, 0, G_CALLBACK(contact_to_qrcode_cb), NULL },
};

static const OssoABookMenuEntry *
get_menu_entries(OssoABookMenuExtension *menu_extension)
{
return entries;
}

static int
get_n_menu_entries(OssoABookMenuExtension *menu_extension)
{
return G_N_ELEMENTS (entries);
}

static void
qrcode_menu_extension_class_init(QRcodeMenuExtensionClass *cls)
{
OssoABookMenuExtensionClass *menu_extension_class;

menu_extension_class = OSSO_ABOOK_MENU_EXTENSION_CLASS (cls);
menu_extension_class->get_n_menu_entries = get_n_menu_entries;
menu_extension_class->get_menu_entries = get_menu_entries;
menu_extension_class->name = OSSO_ABOOK_MENU_NAME_CONTACT_VIEW ";" OSSO_ABOOK_MENU_NAME_MECARD_VIEW;
}

static void
qrcode_menu_extension_class_finalize(QRcodeMenuExtensionClass *cls)
{
}

