/*
 * Copyright (c) 2010 Kaj-Michael Lang
 * 
 * Addressbook menu extension to export current contact as a QR Code image.
 *
 * Based on example code provided by:
 * Copyright (C) 2010 Nokia Corporation
 *   @author Mathias Hasselmann <mathias.hasselmann@maemo.org>
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <libosso-abook/osso-abook.h>
#include <qrencode.h>
#include <cairo.h>

#define BLOCK_SIZE (12)
#define PADDING (10)
#define MIN_SIZE (480)
#define MAX_SIZE (1024)

typedef struct {
	OssoABookMenuExtension parent_instance;
} QRcodeMenuExtension;

typedef struct {
	OssoABookMenuExtensionClass parent_class;
} QRcodeMenuExtensionClass;

G_GNUC_INTERNAL GType
qrcode_menu_extension_get_type (void) G_GNUC_CONST;

/* Defines the new plugin object */
OSSO_ABOOK_DEFINE_MENU_PLUGIN(QRcodeMenuExtension, qrcode_menu_extension, OSSO_ABOOK_TYPE_MENU_EXTENSION,,);

static void
qrcode_menu_extension_init(QRcodeMenuExtension *ext)
{
}

static cairo_surface_t *
gqre_qrcode_to_cairo(QRcode *q, gint isize, gdouble padding)
{
cairo_surface_t *surface;
cairo_t *cr;
gint x,y;
gdouble rs;
guchar *qb;

surface=cairo_image_surface_create(CAIRO_FORMAT_RGB24, isize, isize);
cr=cairo_create(surface);

cairo_set_antialias(cr, CAIRO_ANTIALIAS_NONE);

cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
cairo_paint(cr);

qb=q->data;
rs=(((gdouble)isize-2.0*padding)/(gdouble)q->width);

for(y=0; y<q->width; y++) {
	gdouble rx,ry;

	ry=padding+y*rs;
	for(x=0; x<q->width; x++) {
		if (*qb & 1)
			cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
		else 
			cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);

		rx=padding+x*rs;

		cairo_rectangle(cr, rx, ry, rs, rs);
#if 1
		cairo_fill(cr);
#else
		cairo_fill_preserve(cr);
		cairo_stroke(cr);
#endif
		qb++;
	}
}

cairo_destroy(cr);
cairo_surface_flush(surface);
return surface;
}

static void
contact_to_qrcode_cb(GtkWidget *widget, gpointer user_data)
{
OssoABookMenuExtension *ext=user_data;
OssoABookContact *contact;
const gchar *contact_name;
gchar *vcard, *file, *basefile, *qrfile;
cairo_surface_t *cs;
gint isize=MIN_SIZE, ts;
QRcode *q;

contact=osso_abook_menu_extension_get_contact(ext);

if (!contact)
	return;

vcard=osso_abook_contact_to_string(contact, EVC_FORMAT_VCARD_30, FALSE);
if (!vcard)
	return;

basefile=osso_abook_contact_get_basename(contact, TRUE);
if (!basefile) {
	g_free(vcard);
	return;
}

file=g_strdup_printf("qrcode_%s.png", basefile);
qrfile=g_build_filename(g_get_home_dir(), "MyDocs", file, NULL);
g_debug("QREF: %s", qrfile);

contact_name=osso_abook_contact_get_display_name (contact);
q=QRcode_encodeString(vcard, 0, QR_ECLEVEL_M, QR_MODE_8, 1);

ts=q->width*BLOCK_SIZE;
isize=CLAMP(ts, MIN_SIZE, MAX_SIZE);

cs=gqre_qrcode_to_cairo(q, isize, PADDING);
QRcode_free(q);
if (cairo_surface_write_to_png(cs, qrfile)==CAIRO_STATUS_SUCCESS)
	hildon_banner_show_informationf(NULL, NULL, "QR Code exported for contact:\n %s", contact_name);
else
	hildon_banner_show_information(NULL, NULL, "QR Code export failed!");

cairo_surface_destroy(cs);

g_free(vcard);
g_free(file);
g_free(qrfile);
}

/* Entries to show in the menu, the values are:
 * - label: a label for the menu item;
 * - accel_mods:  modifier keys to use for the menu item;
 * - callback: a function to call when the menu item is activated;
 * - name: the name of the menu item.
 */
static OssoABookMenuEntry entries[] = {
	{ "QR Code contact card", 0, 0, G_CALLBACK(contact_to_qrcode_cb), NULL },
};

static const OssoABookMenuEntry *
get_menu_entries(OssoABookMenuExtension *menu_extension)
{
return entries;
}

static int
get_n_menu_entries(OssoABookMenuExtension *menu_extension)
{
return G_N_ELEMENTS (entries);
}

static void
qrcode_menu_extension_class_init(QRcodeMenuExtensionClass *cls)
{
OssoABookMenuExtensionClass *menu_extension_class;

menu_extension_class = OSSO_ABOOK_MENU_EXTENSION_CLASS (cls);
menu_extension_class->get_n_menu_entries = get_n_menu_entries;
menu_extension_class->get_menu_entries = get_menu_entries;
menu_extension_class->name = OSSO_ABOOK_MENU_NAME_CONTACT_VIEW;
}

static void
qrcode_menu_extension_class_finalize(QRcodeMenuExtensionClass *cls)
{
}

