/*
 *  Copyright (C) 1999-2001 Harri Porten (porten@kde.org)
 *  Copyright (C) 2004, 2005, 2006, 2007, 2008 Apple Inc. All rights reserved.
 *  Copyright (C) 2007 Samuel Weinig <sam@webkit.org>
 *  Copyright (C) 2008 Luke Kenneth Casson Leighton <lkcl@lkcl.net>
 *  Copyright (C) 2008 Martin Soto <soto@freedesktop.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

// gcc 3.x can't handle including the HashMap pointer specialization
// in this file
#if defined __GNUC__ && !defined __GLIBCXX__ // less than gcc 3.4
#define HASH_MAP_PTR_SPEC_WORKAROUND 1
#endif

#include "config.h"
#include <glib-object.h>
#include "webkit/webkitdom.h"
#include "webkitbinding.h"
#include "webkithtmlelementwrapperfactory.h"

#include <wtf/PassRefPtr.h>
#include <wtf/RefPtr.h>

#include "config.h"

#include "AtomicString.h"
#include "Attr.h"
#include "CDATASection.h"
#include "Clipboard.h"
#include "Comment.h"
#include "CSSValue.h"
#include "CString.h"
#include "DocumentFragment.h"
#include "Document.h"
#include "DocumentType.h"
#include "Entity.h"
#include "EntityReference.h"
#include "EventException.h"
#include "Event.h"
#include "ExceptionCode.h"
#include "Frame.h"
#include "HTMLCollection.h"
#include "HTMLElement.h"
#include "HTMLImageElement.h"
#include "HTMLNames.h"
#include "HTMLOptionsCollection.h"
#include "Notation.h"
#include "ProcessingInstruction.h"
#include "Text.h"

#include "CSSCharsetRule.h"
#include "CSSFontFaceRule.h"
#include "CSSImportRule.h"
#include "CSSMediaRule.h"
#include "CSSPageRule.h"
#include "CSSStyleRule.h"
#include "CSSStyleSheet.h"
#include "CSSVariablesRule.h"

#include "WebKitAttrPrivate.h"
#include "WebKitCDATASectionPrivate.h"
#include "WebKitClipboardPrivate.h"
#include "WebKitCommentPrivate.h"
#include "WebKitCSSCharsetRulePrivate.h"
#include "WebKitCSSFontFaceRulePrivate.h"
#include "WebKitCSSImportRulePrivate.h"
#include "WebKitCSSMediaRulePrivate.h"
#include "WebKitCSSPageRulePrivate.h"
#include "WebKitCSSPrimitiveValuePrivate.h"
#include "WebKitCSSRulePrivate.h"
#include "WebKitCSSStyleRulePrivate.h"
#include "WebKitCSSStyleSheetPrivate.h"
#include "WebKitCSSValueListPrivate.h"
#include "WebKitCSSValuePrivate.h"
#include "WebKitCSSVariablesRulePrivate.h"
#include "WebKitDocumentFragmentPrivate.h"
#include "WebKitDocumentPrivate.h"
#include "WebKitDocumentTypePrivate.h"
#include "WebKitElementPrivate.h"
#include "WebKitEntityPrivate.h"
#include "WebKitEntityReferencePrivate.h"
#include "WebKitHTMLCollectionPrivate.h"
#include "WebKitHTMLDocumentPrivate.h"
#include "WebKitHTMLOptionsCollectionPrivate.h"
#include "WebKitNodePrivate.h"
#include "WebKitNotationPrivate.h"
#include "WebKitProcessingInstructionPrivate.h"
#include "WebKitStyleSheetPrivate.h"
#include "WebKitTextPrivate.h"

namespace WebKit {

using namespace WebCore;
using namespace WebCore::HTMLNames;

typedef HashMap<void*, gpointer> DOMObjectMap;

static DOMObjectMap& domObjects()
{
    static DOMObjectMap staticDOMObjects;
    return staticDOMObjects;
}

gpointer GDOMObjectCache::getDOMObject(void* objectHandle)
{
    return domObjects().get(objectHandle);
}

gpointer GDOMObjectCache::putDOMObject(void* objectHandle, gpointer wrapper)
{
    domObjects().set(objectHandle, wrapper);
    return wrapper;
}

void GDOMObjectCache::forgetDOMObject(void* objectHandle)
{
    domObjects().take(objectHandle);
}

gpointer toGDOM(Document* doc)
{
    if (!doc)
        return NULL;

    gpointer gobj = GDOMObjectCache::getDOMObject(doc);
    if (gobj)
        return gobj;

    gpointer ret;

    if (doc->isHTMLDocument())
        ret = wrapHTMLDocument(static_cast<HTMLDocument*>(doc));
    else
        ret = wrapDocument(doc);

    return GDOMObjectCache::putDOMObject(doc, ret);
}

static gpointer createWrapper(Node* node);

gpointer toGDOMNewlyCreated(Node* node)
{
    if (!node)
        return NULL;

    return createWrapper(node);
}

gpointer toGDOM(Node* node)
{
    if (!node)
        return NULL;

    gpointer ret = GDOMObjectCache::getDOMObject(node);
    if (ret)
        return ret;

    return createWrapper(node);
}

gpointer toGDOM(CSSValue* value)
{
    if (!value)
        return NULL;

    gpointer gobj = GDOMObjectCache::getDOMObject(value);

    if (gobj)
        return gobj;

    gpointer ret;

    if (value->isValueList())
        ret = wrapCSSValueList(static_cast<CSSValueList*>(value));
    else if (value->isPrimitiveValue())
        ret = wrapCSSPrimitiveValue(static_cast<CSSPrimitiveValue*>(value));
    else
        ret = wrapCSSValue(value);

    return GDOMObjectCache::putDOMObject(value, ret);
}

static ALWAYS_INLINE gpointer createWrapper(Node* node)
{
    ASSERT(node);
    ASSERT(!ScriptInterpreter::getDOMObject(node));

    gpointer ret = NULL;

    switch (node->nodeType()) {
        case Node::ELEMENT_NODE:
            if (node->isHTMLElement())
                ret = createGDOMHTMLWrapper(static_cast<HTMLElement*>(node));
            else
                ret = wrapElement(static_cast<Element*>(node));
            break;
        case Node::ATTRIBUTE_NODE:
            ret = wrapAttr(static_cast<Attr*>(node));
            break;
        case Node::TEXT_NODE:
            ret = wrapText(static_cast<Text*>(node));
            break;
        case Node::CDATA_SECTION_NODE:
            ret = wrapCDATASection(static_cast<CDATASection*>(node));
            break;
        case Node::ENTITY_NODE:
            ret = wrapEntity(static_cast<Entity*>(node));
            break;
        case Node::PROCESSING_INSTRUCTION_NODE:
            ret = wrapProcessingInstruction(static_cast<ProcessingInstruction*>(node));
            break;
        case Node::COMMENT_NODE:
            ret = wrapComment(static_cast<Comment*>(node));
            break;
        case Node::DOCUMENT_NODE:
            // we don't want to cache the document itself in the per-document dictionary
            return toGDOM(static_cast<Document*>(node));
        case Node::DOCUMENT_TYPE_NODE:
            ret = wrapDocumentType(static_cast<DocumentType*>(node));
            break;
        case Node::NOTATION_NODE:
            ret = wrapNotation(static_cast<Notation*>(node));
            break;
        case Node::DOCUMENT_FRAGMENT_NODE:
            ret = wrapDocumentFragment(static_cast<DocumentFragment*>(node));
            break;
        case Node::ENTITY_REFERENCE_NODE:
            ret = wrapEntityReference(static_cast<EntityReference*>(node));
            break;
        default:
            ret = wrapNode(node);
    }

    return GDOMObjectCache::putDOMObject(node, ret);
}

gpointer toGDOM(HTMLCollection* collection)
{
    if (!collection)
        return NULL;

    gpointer gobj = GDOMObjectCache::getDOMObject(collection);

    if (gobj)
        return gobj;

    gpointer ret;
    switch (collection->type()) {
        case HTMLCollection::SelectOptions:
            ret = wrapHTMLOptionsCollection(static_cast<HTMLOptionsCollection*>(collection));
            break;
        default:
            ret = wrapHTMLCollection(static_cast<HTMLCollection*>(collection));
            break;
    }

    return GDOMObjectCache::putDOMObject(collection, ret);
}

gpointer toGDOM(Text* text)
{
    if (!text)
        return NULL;

    gpointer gobj = GDOMObjectCache::getDOMObject(text);
    if (gobj)
        return gobj;

    gpointer ret = wrapText(text);
    return GDOMObjectCache::putDOMObject(text, ret);
}

gpointer toGDOM(CSSRule* rule)
{
    if (!rule)
        return NULL;

    gpointer gobj = GDOMObjectCache::getDOMObject(rule);

    if (gobj)
        return gobj;

    gpointer ret;
    switch (rule->type()) {
        case CSSRule::STYLE_RULE:
            ret = wrapCSSStyleRule(static_cast<CSSStyleRule*>(rule));
            break;
        case CSSRule::MEDIA_RULE:
            ret = wrapCSSMediaRule(static_cast<CSSMediaRule*>(rule));
            break;
        case CSSRule::FONT_FACE_RULE:
            ret = wrapCSSFontFaceRule(static_cast<CSSFontFaceRule*>(rule));
            break;
        case CSSRule::PAGE_RULE:
            ret = wrapCSSPageRule(static_cast<CSSPageRule*>(rule));
            break;
        case CSSRule::IMPORT_RULE:
            ret = wrapCSSImportRule(static_cast<CSSImportRule*>(rule));
            break;
        case CSSRule::CHARSET_RULE:
            ret = wrapCSSCharsetRule(static_cast<CSSCharsetRule*>(rule));
            break;
        case CSSRule::VARIABLES_RULE:
            ret = wrapCSSVariablesRule(static_cast<CSSVariablesRule*>(rule));
            break;
        default:
            ret = wrapCSSRule(rule);
            break;
    }

    GDOMObjectCache::putDOMObject(rule, ret);
    return ret;
}

gpointer toGDOM(Element* element)
{
    if (!element)
        return NULL;

    gpointer gobj = GDOMObjectCache::getDOMObject(element);
    if (gobj)
        return gobj;

    gpointer ret;

    if (element->isHTMLElement())
        ret = createGDOMHTMLWrapper(static_cast<HTMLElement*>(element));
    else
        ret = wrapElement(element);

    return GDOMObjectCache::putDOMObject(element, ret);
}

gpointer toGDOM(StyleSheet* styleSheet)
{
    if (!styleSheet)
        return NULL;

    gpointer gobj = GDOMObjectCache::getDOMObject(styleSheet);
    if (gobj)
        return gobj;

    gpointer ret;
    if (styleSheet->isCSSStyleSheet())
        ret = wrapCSSStyleSheet(static_cast<CSSStyleSheet*>(styleSheet));
    else
        ret = wrapStyleSheet(styleSheet);

    return GDOMObjectCache::putDOMObject(styleSheet, ret);
}

} // namespace WebKit

