from gnome import gconf
from wifi_assistant.daemon.daemon import Daemon
from wifi_assistant.settings.network_setting import NetworkSetting

import unittest
from unit.pie import *

default_url_to_open = 'http://wifi-assistant.garage.maemo.org'
default_browser_name = "default"
default_browser_options = {'url':default_url_to_open}
default_settings = NetworkSetting()
default_settings.setNameOfBrowserToLaunch(default_browser_name)
default_settings.setBrowserOptions(default_browser_options)

ssid = 'A Network Name'
settings_without_options = NetworkSetting()
settings_without_options.setNetworkName(ssid)
settings_without_options.setLaunchingOfBrowserEnabled(True)

second_ssid = 'Another Network SSID'
second_url = 'http://some.specific.site/'
second_browser_name = 'special browser'
second_settings = NetworkSetting()
second_settings.setNetworkName(second_ssid)
second_settings.setNameOfBrowserToLaunch(second_browser_name)
second_settings.setLaunchingOfBrowserEnabled(True)
second_browser_options = {'url': second_url}
second_settings.setBrowserOptions(second_browser_options)

class DaemonTest(unittest.TestCase):

    def setUp(self):
        self.launcher = Mock()
        self.application_settings = Mock()
        self.network_settings = Mock()
        self.parent_window = None
        
        
    def test_withANewNetwork_whenUserClicksYes_browserIsLaunched(self):
        # GIVEN
        self._createTesteeWithPopupClickingYes()
        given(self.network_settings).get(ssid).willReturn(None)
        given(self.network_settings).getDefaultSettings().willReturn(default_settings)
        
        # WHEN
        self._replayMocks()
        self.testee.connectionEstablished(ssid)
        
        # THEN
        verify(self.network_settings).getDefaultSettings()
        verify(self.launcher).launchBrowser(default_browser_name, default_browser_options)
        
        
    def test_withANewNetwork_whenUserClicksNo_browserIsNotLaunched(self):
        # GIVEN
        self._createTesteeWithPopupClickingNo()
        given(self.network_settings).get(ssid).willReturn(None)

        # WHEN
        self._replayMocks()
        self.testee.connectionEstablished(ssid)
        
        # THEN
        verify(self.launcher, never()).launchBrowser(default_browser_name, default_browser_options)
        
    
    def test_popupIsNotOpenedIfPopupIsDisabledInApplicationSettings(self):
        # GIVEN
        given(self.application_settings).getUsePopup().willReturn(False)
        self.testee = Daemon(self.launcher, self.application_settings, self.network_settings, self.parent_window)
        
        self._replayMocks()
        self.testee.connectionEstablished(ssid)
        
        verify(self.launcher, never()).launchBrowser()
        # TODO: refactor out gui part (popup) and test/mock that
        
    
    def test_knownSsidWithLaunchSetToTrueUsesDefaultBrowserAndUrl(self):
        given(self.network_settings).get(ssid).willReturn(settings_without_options)
        given(self.network_settings).getDefaultSettings().willReturn(default_settings)
        given(self.application_settings).getUsePopup().willReturn(True)
        self.testee = self._createTestee()
        
        self._replayMocks()
        self.testee.connectionEstablished(ssid)
        
        verify(self.launcher, once()).launchBrowser(default_browser_name, default_browser_options)
    
    
    def test_browserIsLaunchedWithSsidSpecificUrl_insteadOfDefaultUrl(self):
        given(self.network_settings).get(second_ssid).willReturn(second_settings)
        given(self.application_settings).getUsePopup().willReturn(True)
        self.testee = self._createTestee()
        
        self._replayMocks()
        self.testee.connectionEstablished(second_ssid)
        
        verify(self.launcher, once()).launchBrowser(second_browser_name, second_browser_options)
    
    
    def test_daemon_start_registersListenerWithApplicationSettings(self):
        given(self.application_settings).getUsePopup().willReturn(True)
        self.testee = self._createTestee()
        given(self.application_settings).registerUsePopupListener(self.testee._usePopupEventCallback).willReturn(1234)
        
        self._replayMocks()
        
        verify(self.application_settings, never()).registerUsePopupListener()
        self.testee.start()
        verify(self.application_settings, once()).registerUsePopupListener(self.testee._usePopupEventCallback)
        
        verify(self.application_settings, never()).unregisterUsePopupListener()
        self.testee.stop()
        verify(self.application_settings, once()).unregisterUsePopupListener(self.testee._usePopupEventCallback)


    def _createTestee(self):
        def _empty(*args, **kwargs):
            pass
        daemon = Daemon(self.launcher, self.application_settings, self.network_settings, self.parent_window)
        daemon._startLoop = _empty
        return daemon
    

    def _createTesteeWithPopupClickingYes(self):
        given(self.application_settings).getUsePopup().willReturn(True)
        self.testee = self._createTestee()
        
        def yes(*args, **kwargs):
            return True
        self.testee.showDecisionDialog = yes
        

    def _createTesteeWithPopupClickingNo(self):
        given(self.application_settings).getUsePopup().willReturn(True)
        self.testee = self._createTestee()
        
        def no(*args, **kwargs):
            return False
        self.testee.showDecisionDialog = no
        

    def _replayMocks(self):
        self.application_settings.replay()
        self.launcher.replay()
        self.network_settings.replay()
        

if __name__ == '__main__':
    unittest.main()
