/*
 * AuthWindow.cpp - captcha window class implementation.
 *
 * Copyright (C) 2011 - Nikita S. Artemov
 *
 * Webauth is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Webauth is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Webauth; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include "authwindow.h"
#include "authdbusadaptor.h"


/**
 * @brief Captcha result.
 */
namespace CaptchaResult {
    /** User closed window. */
    const QString ClosedByUser = "closed";

}



/** 
 * @brief Constructor.
 */
AuthWindow::AuthWindow(const int ID) : windowID (ID)
{
    this->setAttribute(Qt::WA_DeleteOnClose, true);
    isClosedByUser = true;

    window = new QWidget(this);
    vbox = new QVBoxLayout;
    lblLogin = new QLabel("Email or Username:");
    lblPass = new QLabel("Password:");

    txtLogin = new QLineEdit;
    connect(txtLogin, SIGNAL(returnPressed()), SLOT(enter()));

    txtPass = new QLineEdit;
    txtPass->setEchoMode(QLineEdit::Password);
    connect(txtPass, SIGNAL(returnPressed()), SLOT(enter()));

    button = new QPushButton("Log In");
    connect(button, SIGNAL(clicked()), SLOT(enter()));

    vbox->addWidget(lblLogin);
    vbox->addWidget(txtLogin);
    vbox->addWidget(lblPass);
    vbox->addWidget(txtPass);
    vbox->addWidget(button);
    window->setLayout(vbox);

    setCentralWidget(window);
}


/**
 * @brief Destructor
 */
AuthWindow::~AuthWindow()
{
    qDebug() << "AuthWindow destructor START";
    authParams.clear();
    qDebug() << "AuthWindow destructor END";
}



/********************************************/
/************ Public functions **************/
/**
 * @brief Start authwindow
 *
 * @param clientID for identificate a client
 */
void AuthWindow::startLogin(const QString clientID)
{
    if(!clientID.isEmpty()){
        this->clientID = clientID;
    }
    this->show();
}


/**
 * @brief Getter for wndows ID.
 *
 * @return windows identifier.
 */
int AuthWindow::getWindowID()
{
    return windowID;
}



/********************************************/
/********** Protected functios *************/
/**
 * a window close request for a top-level widget from the window system.
 *
 * @param event close event.
 */
void AuthWindow::closeEvent(QCloseEvent *)
{
//    if (isClosedByUser == true) {
//       /sendLoginResult(CaptchaResult::ClosedByUser);
//    }

    emit loginClosed(getWindowID());
}

/**
 * @brief button was pressed.
 */
void AuthWindow::enter()
{
    if(txtLogin->text().isEmpty() || txtPass->text().isEmpty())
        return;
    loginResult << txtLogin->text() << txtPass->text();
    sendLoginResult(loginResult);
    isClosedByUser = false;
    this->close();
}

/**
 * @brief Send signal with login and pass.
 *  Prepare parameters and send signal.
 *
 * @param loginResult list with login and pass .
 */

void AuthWindow::sendLoginResult(const QStringList &loginResult)
{
    QList<QVariant> signalParams = authParams.prepareLogInSignalParams(loginResult);

    if (AuthDBusAdaptor::sendLoginSignal(clientID, signalParams) == false) {
        qDebug() << "\nMessage not send.";
    } else {
        qDebug() << "\nMessage send (" << signalParams << ").";
    }
}
