/*
 * CaptchaWindow.cpp - captcha window class implementation.
 *
 * Copyright (C) 2011 - Nikita S. Artemov
 *
 * Webauth is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Webauth is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Webauth; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include "captchawindow.h"
#include "authdbusadaptor.h"


/**
 * @brief Captcha result.
 */
namespace CaptchaResult {
    /** User closed window. */
    const QString ClosedByUser = "closed";

}



/** 
 * @brief Constructor.
 */
CaptchaWindow::CaptchaWindow(const int ID) : windowID (ID)
{
    this->setAttribute(Qt::WA_DeleteOnClose, true);
    isClosedByUser = true;

    window = new QWidget(this);
    vbox = new QVBoxLayout;
    view = new QLabel;
    locationEdit = new QLineEdit;
    button = new QPushButton("Ok");

    vbox->addWidget(view);
    vbox->addWidget(locationEdit);
    vbox->addWidget(button);
    window->setLayout(vbox);

    connect(button, SIGNAL(clicked()), SLOT(enter()));

    setCentralWidget(window);
}


/**
 * @brief Destructor
 */
CaptchaWindow::~CaptchaWindow()
{
    qDebug() << "CaptchaWindow destructor START";
    authParams.clear();
    qDebug() << "CaptchaWindow destructor END";
}



/********************************************/
/************ Public functions **************/
/**
 * @brief Set new image.
 *
 * @param strBase64 image decoded in base64
 */
void CaptchaWindow::setImage(const QString &strBase64)
{
    QByteArray ba = QByteArray::fromBase64(strBase64.toAscii());
    pix.loadFromData(ba);
    view->setPixmap(pix);
    view->setAlignment(Qt::AlignCenter);
}


/**
 * @brief Start captcha test.
 *
 * @param strBase64 for show image.
 * @param clientID for identificate a client
 */
void CaptchaWindow::startCaptcha(const QString &strBase64, const QString clientID)
{
    if(!clientID.isEmpty()){
        this->clientID = clientID;
    }
    setImage(strBase64);
    this->show();
}


/**
 * @brief Getter for wndows ID.
 *
 * @return windows identifier.
 */
int CaptchaWindow::getWindowID()
{
    return windowID;
}



/********************************************/
/********** Protected functios *************/
/**
 * a window close request for a top-level widget from the window system.
 *
 * @param event close event.
 */
void CaptchaWindow::closeEvent(QCloseEvent *)
{
    if (isClosedByUser == true) {
        sendCaptchaResult(CaptchaResult::ClosedByUser);
    }

    emit captchaClosed(getWindowID());
}

/**
 * @brief button was pressed.
 */
void CaptchaWindow::enter()
{
    sendCaptchaResult(locationEdit->text());
    isClosedByUser = false;
    this->close();
}

/**
 * @brief Send signal with captcha result.
 *  Prepare parameters and send signal.
 *
 * @param captchaResult result of captcha test.
 * @param withReturnParams include return parameters to signal.
 */

void CaptchaWindow::sendCaptchaResult(const QString &captchaResult)
{
    QList<QVariant> signalParams = authParams.prepareCaptchaSignalParams(captchaResult, false);
    //const QString &clientID = authParams.getClientID();

    if (AuthDBusAdaptor::sendSignal(clientID, signalParams, false) == false) {
        qDebug() << "\nMessage not send.";
    } else {
        qDebug() << "\nMessage send (" << signalParams << ").";
    }
}
