/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "listpicker.hpp"

// Global includes
#include <QtCore/QAbstractItemModel>
#include <QtCore/QPointer>
#include <QtGui/QHBoxLayout>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5/QMaemo5ValueButton>
#include <QtMaemo5/QMaemo5ListPickSelector>
#else // Q_WS_MAEMO_5
#include <QtGui/QLabel>
#include <QtGui/QComboBox>
#endif // Q_WS_MAEMO_5

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class ListPickerPrivate : public QObject
{
  Q_OBJECT

public:
  // Properties
  ListPicker* self;
  QHBoxLayout layout;

  QPointer<QAbstractItemModel> model;

#ifdef Q_WS_MAEMO_5
  QMaemo5ListPickSelector selector;
  QMaemo5ValueButton button;
#else // Q_WS_MAEMO_5
  QLabel label;
  QComboBox comboBox;
#endif // Q_WS_MAEMO_5

  // Functions
  void init();

public slots:
  void onModelReset();
  void onSelectionChanged();
};

//////////////////////////////////////////////////////////////////////////

void ListPickerPrivate::init()
{
  // Remove the margins
  layout.setContentsMargins(0, 0, 0, 0);

#ifdef Q_WS_MAEMO_5
  button.setPickSelector(&selector);
  layout.addWidget(&button);
  connect(&selector, SIGNAL(selected(QString)), this, SLOT(onSelectionChanged()));
#else // Q_WS_MAEMO_5
  layout.addWidget(&label);
  layout.addWidget(&comboBox);
  connect(&comboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(onSelectionChanged()));
#endif // Q_WS_MAEMO_5

  self->setLayout(&layout);
}

//////////////////////////////////////////////////////////////////////////

void ListPickerPrivate::onModelReset()
{
#ifdef Q_WS_MAEMO_5
  int index = selector.currentIndex();

  QModelIndex currentIndex = model ? model->index(index, 0) : QModelIndex();
  button.setIcon( qvariant_cast<QIcon>(currentIndex.data(Qt::DecorationRole)) );
  button.setValueText( currentIndex.data(Qt::DisplayRole).toString() );
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

void ListPickerPrivate::onSelectionChanged()
{
#ifdef Q_WS_MAEMO_5
  int index = selector.currentIndex();
#else // Q_WS_MAEMO_5
  int index = comboBox.currentIndex();
#endif // Q_WS_MAEMO_5

  // Get the current index
  QModelIndex currentIndex = model ? model->index(index, 0) : QModelIndex();

  // Give the button a shiny icon
#ifdef Q_WS_MAEMO_5
  button.setIcon( qvariant_cast<QIcon>(currentIndex.data(Qt::DecorationRole)) );
#endif // Q_WS_MAEMO_5

  // Emit the changed signal
  emit self->currentIndexChanged(currentIndex.row());
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

ListPicker::ListPicker(QWidget* parent /* = 0 */)
: QWidget(parent)
, d(new ListPickerPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

ListPicker::~ListPicker()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void ListPicker::setText(const QString& text)
{
#ifdef Q_WS_MAEMO_5
  d->button.setText(text);
#else // Q_WS_MAEMO_5
  d->label.setText(text);
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

QString ListPicker::text() const
{
#ifdef Q_WS_MAEMO_5
  return d->button.text();
#else // Q_WS_MAEMO_5
  return d->label.text();
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

void ListPicker::setModel(QAbstractItemModel* model)
{
  if ( d->model == model )
    return;

  if ( d->model )
    disconnect(d->model, SIGNAL(modelReset()), d, SLOT(onModelReset()));

  // Take the model
  d->model = model;

#ifdef Q_WS_MAEMO_5
  d->selector.setModel(model);
#else // Q_WS_MAEMO_5
  d->comboBox.setModel(model);
#endif // Q_WS_MAEMO_5

  if ( d->model )
    connect(d->model, SIGNAL(modelReset()), d, SLOT(onModelReset()));
}

//////////////////////////////////////////////////////////////////////////

QAbstractItemModel* ListPicker::model() const
{
  return d->model;
}

//////////////////////////////////////////////////////////////////////////

void ListPicker::setCurrentIndex(int index)
{
#ifdef Q_WS_MAEMO_5
  d->selector.setCurrentIndex(index);
#else // Q_WS_MAEMO_5
  d->comboBox.setCurrentIndex(index);
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

int ListPicker::currentIndex() const
{
#ifdef Q_WS_MAEMO_5
  return d->selector.currentIndex();
#else // Q_WS_MAEMO_5
  return d->comboBox.currentIndex();
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "listpicker.moc"
