/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "mainwindow.hpp"
#include "messagewidget.hpp"
#include "providers/providerinterface.hpp"
#include "providers/accountsdialog.hpp"
#include "providers/accounts.hpp"
#include "providers/accountsmodel.hpp"

// Global includes
#include <QtGui/QProgressDialog>
#include <QtGui/QMessageBox>
#include <QtGui/QMenuBar>
#include <QtGui/QCloseEvent>

#include <QtGui/QListView>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class MainWindowPrivate : public QObject
{
  Q_OBJECT

public:
  MainWindowPrivate()
  : accounts(new Accounts)
  {}

  MainWindow* self;

  // Properties
  AccountsPtr accounts;
  MessageWidget* messageWidget;

  ProviderInterfacePtr currentProvider;

  QListView listWidget;

  // Functions
  void init();

  void login();
  void logout();

  void setCurrentProvider(const ProviderInterfacePtr& provider);

public slots:
  // Provider callbacks
  void onLoginReply(bool success);
  void onError(const QString& message);

  // Ui callbacks
  void onEditAccounts();
  void onClicked(const QModelIndex& index);
};

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::init()
{
  // ### TODO: Remove this hackish solution for maemo window flags
  Qt::WidgetAttribute WA_Maemo5StackedWindow = static_cast<Qt::WidgetAttribute>(127);

  // Update the window title
  self->setWindowTitle( tr("Web2SMS") );
  self->setAttribute(/*Qt::*/WA_Maemo5StackedWindow);

  messageWidget = new MessageWidget(self, Qt::Window);
  messageWidget->setAttribute(/*Qt::*/WA_Maemo5StackedWindow);

  AccountsModel* model = new AccountsModel(this);
  model->setAccounts(accounts);

  listWidget.setModel(model);
  listWidget.setSelectionMode(QAbstractItemView::NoSelection);

  // Set the message widget as central widget
  self->setCentralWidget(&listWidget);

  // Connect signals and slots
  connect(&listWidget, SIGNAL(clicked(QModelIndex)), this, SLOT(onClicked(QModelIndex)));

  accounts->loadPlugins();
  accounts->loadSettings();

  // Setup the menu
  QMenu* menu = self->menuBar()->addMenu(QString());
  menu->addAction(tr("Accounts ..."), this, SLOT(onEditAccounts()));
  //menu->addAction(tr("About ..."), this, SLOT(onAbout()));

#if !defined(Q_WS_S60) && !defined(Q_WS_MAEMO_5)
  // We have to set a window title, otherwise Windows won't show the menu
  menu->setTitle("menu");
#endif
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::login()
{
  QProgressDialog* progressDialog = new QProgressDialog(messageWidget);
  progressDialog->setAttribute(Qt::WA_DeleteOnClose);
  progressDialog->setWindowTitle( tr("Logging in ...") );
  progressDialog->setRange(0, 0);
  progressDialog->show();

  connect(currentProvider.data(), SIGNAL(loginReply(bool)), progressDialog, SLOT(accept()));

  currentProvider->login();
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onLoginReply(bool success)
{
  if ( success )
    return;

  // Show a message that tells the user that's something wrong
  QMessageBox::critical(self, self->windowTitle(), tr("Login failed, please check your username and password"));
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onError(const QString& message)
{
  QMessageBox::critical(self, self->windowTitle(), message);
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onEditAccounts()
{
  // Show the accounts dialog
  AccountsDialog dialog(self);
  dialog.setAccounts(accounts);

  // Run the dialog (we don't care about the exit code)
  dialog.exec();

  // Always save the account settings when the dialog is finished
  accounts->saveSettings();
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onClicked(const QModelIndex& index)
{
  QString alias = index.data(AccountsModel::AccountAliasRole).toString();

  if ( alias.isEmpty() )
    return;

  ProviderInterfacePtr ptr = accounts->providerForAlias(alias);
  if ( !ptr )
    return;

  // Update the message widget
  messageWidget->setProviderInfo( accounts->providerInfoForAlias(alias) );
  messageWidget->setProvider(ptr);
  messageWidget->show();

  // Change the current provider
  setCurrentProvider(ptr);
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::setCurrentProvider(const ProviderInterfacePtr& provider)
{
  // Disconnect the old provider
  if ( currentProvider )
  {
    if ( currentProvider->isLoggedIn() )
      currentProvider->logout();
  }

  // Accept the new provider
  currentProvider = provider;

  // Connect the new provider
  if ( currentProvider )
  {
    /*currentProvider->*/login();
  }
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

MainWindow::MainWindow(QWidget *parent /* = 0 */)
: QMainWindow(parent)
, d(new MainWindowPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

MainWindow::~MainWindow()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void MainWindow::init()
{
  /*if ( !d->provider )
    return;

  if ( !d->provider->hasLoginData() )
  {
    QMessageBox::information(this, tr("First start notice"), tr("This program only works with o2 Germany. Use at your own risk, bugs may be reported on the project website."));
  }
  else
    d->login();*/
}

//////////////////////////////////////////////////////////////////////////

void MainWindow::closeEvent(QCloseEvent* closeEvent)
{
  // Logout first?
  if ( d->currentProvider && d->currentProvider->isLoggedIn() )
  {
    // Ignore the close event
    closeEvent->ignore();

    // Show a dialog that we're logging out first
    QProgressDialog* progressDialog = new QProgressDialog(this);
    progressDialog->setAttribute(Qt::WA_DeleteOnClose);
    progressDialog->setWindowTitle( tr("Logging out ...") );
    progressDialog->setRange(0, 0);
    progressDialog->show();

    // Close the information dialog and this window
    connect(d->currentProvider.data(), SIGNAL(logoutReply(bool)), progressDialog, SLOT(accept()));
    connect(d->currentProvider.data(), SIGNAL(logoutReply(bool)), this, SLOT(close()));

    // Log out now
    d->currentProvider->logout();
  }
  else
  {
    QWidget::closeEvent(closeEvent);
  }
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "mainwindow.moc"
