/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Global includes
#include <libosso-abook/osso-abook.h>
#include <rtcom-eventlogger/eventlogger.h>

// Local includes
#include "qt_c_api.h"

// Defines
#define PROGRAMNAME    "web2sms"
#define PROGRAMVERSION "0.1"

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

void init_contacts()
{
  osso_context_t* osso_ctx;

  // Init osso (wheat means "osso"?)
  osso_ctx = osso_initialize(PROGRAMNAME, PROGRAMVERSION, FALSE, NULL);

  // Init the phonebook
  osso_abook_init_with_name(PROGRAMNAME, osso_ctx);
}

//////////////////////////////////////////////////////////////////////////

void deinit_contacts()
{
  // Deinitialize osso
  osso_deinitialize( osso_abook_get_osso_context() );
}

//////////////////////////////////////////////////////////////////////////

int get_contact_for_sms(const struct QString* dialogTitle, struct QString* name, struct QString* number, struct QString* uid)
{
  int success = FALSE;
  GtkWidget* chooser;
  GList* selection;
  struct QByteArray* utf8Title;

  utf8Title = QString_toUtf8(dialogTitle);

  // Create a new contact chooser
  chooser = osso_abook_contact_chooser_new(NULL, QByteArray_getValue(utf8Title));

  // Run the contact chooser
  gtk_dialog_run(GTK_DIALOG(chooser));
  gtk_widget_hide(chooser);

  // Get the selection
  selection = osso_abook_contact_chooser_get_selection(OSSO_ABOOK_CONTACT_CHOOSER(chooser));

  if ( selection )
  {
    EVCardAttribute* selectedAttribute;

    // Show a new dialog where the user chooses a phone number
    GtkWidget* numberChooser;
    numberChooser = osso_abook_contact_detail_selector_new_for_contact(NULL, selection->data, OSSO_ABOOK_CONTACT_DETAIL_PHONE);

    gtk_dialog_run(GTK_DIALOG(numberChooser));
    gtk_widget_hide(numberChooser);

    // Get the selected attribute
    selectedAttribute = osso_abook_contact_detail_selector_get_detail(OSSO_ABOOK_CONTACT_DETAIL_SELECTOR(numberChooser));
    if ( selectedAttribute )
    {
      // Get the display name of the contact
      QString_setValue(name, osso_abook_contact_get_display_name(selection->data));
      // Get the number of the contact
      QString_setValue(number, e_vcard_attribute_get_value(selectedAttribute));
      // Get the uid of the contact
      QString_setValue(uid, e_contact_get_const(E_CONTACT(selection->data), E_CONTACT_UID));

      success = TRUE;
    }

    // Destroy the number chooser
    gtk_widget_destroy(numberChooser);
  }

  // Free some memory
  g_list_free(selection);
  gtk_widget_destroy(chooser);
  QByteArray_delete(utf8Title);

  return success;
}

//////////////////////////////////////////////////////////////////////////

int add_to_sent_sms(const struct QString* name, const struct QString* number, const struct QString* uid, const struct QString* message)
{
  RTComEl* el;
  RTComElEvent* event;

  // Init RTCom
  el = rtcom_el_new();
  if ( el )
    return FALSE;

  struct QByteArray* nameBytes = QString_toUtf8(name);
  struct QByteArray* numberBytes = QString_toUtf8(number);
  struct QByteArray* uidBytes = QString_toUtf8(uid);
  struct QByteArray* messageBytes = QString_toUtf8(message);

  // Create a new event
  event = rtcom_el_event_new();

  // Fill some fields
  RTCOM_EL_EVENT_SET_FIELD(event, service, g_strdup("RTCOM_EL_SERVICE_SMS"));
  RTCOM_EL_EVENT_SET_FIELD(event, event_type, g_strdup("RTCOM_EL_EVENTTYPE_SMS_OUTBOUND"));
  RTCOM_EL_EVENT_SET_FIELD(event, storage_time, time(NULL));
  RTCOM_EL_EVENT_SET_FIELD(event, start_time, time(NULL));
  RTCOM_EL_EVENT_SET_FIELD(event, end_time, time(NULL));
  RTCOM_EL_EVENT_SET_FIELD(event, is_read, TRUE);
  RTCOM_EL_EVENT_SET_FIELD(event, local_uid, g_strdup("ring/tel/ring"));
  RTCOM_EL_EVENT_SET_FIELD(event, local_name, g_strdup("<SelfHandle>"));
  RTCOM_EL_EVENT_SET_FIELD(event, remote_uid, g_strdup(QByteArray_getValue(numberBytes)));
  RTCOM_EL_EVENT_SET_FIELD(event, remote_name, g_strdup(QByteArray_getValue(nameBytes)));
  RTCOM_EL_EVENT_SET_FIELD(event, remote_ebook_uid, g_strdup(QByteArray_getValue(uidBytes)));
  //RTCOM_EL_EVENT_SET_FIELD(event, group_uid, message->group_uid);
  RTCOM_EL_EVENT_SET_FIELD(event, free_text, g_strdup(QByteArray_getValue(messageBytes)));

  // Add the event to the event logger
  rtcom_el_add_event(el, event, NULL);

  // Free the event
  rtcom_el_event_free(event);

  // Free the QByteArrays
  QByteArray_delete(nameBytes);
  QByteArray_delete(numberBytes);
  QByteArray_delete(uidBytes);
  QByteArray_delete(messageBytes);

  return FALSE;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
