/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of vumeter.
 *
 *  vumeter is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  vumeter is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with vumeter.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "mainwindow.h"

#include <math.h>

#include <QDate>
#include <QLineEdit>
#include <QSettings>
#include <QString>
#include <QTime>
#include <QTextStream>
#include <QApplication>
#include <QMessageBox>

#include "analogmeterwidget.h"
#include "horizontalvolumebarwidget.h"
#include "settingsdialog.h"
#include "volumelinechartwidget.h"

MainWindow::MainWindow (QWidget *parent)
    : QMainWindow(parent)
    , logFile(NULL)
    , logIndex(0)
    , triggerHigh(true, triggerLow)
    , triggerLow(false, triggerHigh)
    , volumeMeterLeft(NULL)
    , volumeMeterRight(NULL)
{
    ui.setupUi(this);

    QApplication::setApplicationName("VU Meter");
    QApplication::setApplicationVersion("0.12.0");
    QApplication::setOrganizationName(QApplication::applicationName());
    QApplication::setOrganizationDomain("https://garage.maemo.org/projects/vumeter");

#if defined(Q_WS_MAEMO_5) || defined(Q_WS_MAEMO_6)
    setWindowIcon(QIcon("/usr/share/icons/hicolor/48x48/apps/vumeter.png"));
#else
    setWindowIcon(QIcon("/usr/share/icons/hicolor/64x64/apps/vumeter.png"));

    while (ui.sourcesComboBox->count())
        ui.sourcesComboBox->removeItem(0);
    sources.append(*adapter.getAvailableSources());
    ui.sourcesComboBox->addItems(sources);
#endif

    connect(ui.sourcesComboBox, SIGNAL(activated(int)), this, SLOT(setSource(int)));
    connect(ui.monoStereoComboBox, SIGNAL(activated(int)), this, SLOT(setStereo(int)));
    connect(ui.dBLinearComboBox, SIGNAL(activated(int)), this, SLOT(setScale(int)));
    connect(ui.meterComboBox, SIGNAL(activated(int)), this, SLOT(setMeter(int)));

    connect(&adapter, SIGNAL(audioDataReceived(const short*, int)), this, SLOT(updateValues(const short*, int)), Qt::QueuedConnection);
    connect(ui.minimumLabel, SIGNAL(linkActivated(QString)), this, SLOT(resetMinValue()));
    connect(ui.maximumLabel, SIGNAL(linkActivated(QString)), this, SLOT(resetMaxValue()));

    connect(ui.actionSettings, SIGNAL(triggered()), this, SLOT(settingsDialog()));
    connect(ui.actionHelp, SIGNAL(triggered()), this, SLOT(helpDialog()));
    connect(ui.actionAbout, SIGNAL(triggered()), this, SLOT(aboutDialog()));
    connect(ui.actionAboutQt, SIGNAL(triggered()), qApp, SLOT(aboutQt()));

    applySettings(true);
}

MainWindow::~MainWindow ()
{
    delete volumeMeterLeft;
    delete volumeMeterRight;
    closeFile(logFile);
}

void MainWindow::closeEvent (QCloseEvent *event)
{
    QSettings settings;
    settings.setValue(SettingsDialog::SOURCE, ui.sourcesComboBox->currentIndex());
    settings.setValue(SettingsDialog::STEREO, stereo);
    settings.setValue(SettingsDialog::SCALE, scale);
    settings.setValue(SettingsDialog::METER, ui.meterComboBox->currentIndex());
    settings.setValue(SettingsDialog::FULL_SCREEN, !ui.meterComboBox->isVisible());
    settings.sync();
    event->accept();
}

static void initVolumeMeterWidget (VolumeMeterWidget *widget, QSettings &settings)
{
    widget->setTriggerValues(settings.value(SettingsDialog::TRIGGER_VALUE_HIGH, SettingsDialog::TRIGGER_VALUE_DEFAULT).toInt(),
                             settings.value(SettingsDialog::TRIGGER_VALUE_LOW, SettingsDialog::TRIGGER_VALUE_DEFAULT).toInt());

    widget->setUseColoredScale(settings.value(SettingsDialog::COLOR_SCALE, SettingsDialog::COLOR_SCALE_DEFAULT).toBool());
    widget->setColors(settings.value(SettingsDialog::COLOR_LOWER_COLOR, SettingsDialog::COLOR_LOWER_COLOR_DEFAULT).value<QColor>(),
                      settings.value(SettingsDialog::COLOR_MID_COLOR, SettingsDialog::COLOR_MID_COLOR_DEFAULT).value<QColor>(),
                      settings.value(SettingsDialog::COLOR_UPPER_COLOR, SettingsDialog::COLOR_UPPER_COLOR_DEFAULT).value<QColor>());
    widget->setColorThresholds(settings.value(SettingsDialog::COLOR_LOWER_THRESHOLD, SettingsDialog::COLOR_LOWER_THRESHOLD_DEFAULT).toInt(),
                               settings.value(SettingsDialog::COLOR_UPPER_THRESHOLD, SettingsDialog::COLOR_UPPER_THRESHOLD_DEFAULT).toInt());
}

static void upgradeSettings (QSettings *settings)
{
    static const struct mapping {
        const QString oldName;
        const QString newName;
    } mapping[] = {
        { SettingsDialog::TRIGGER_VALUE,   SettingsDialog::TRIGGER_VALUE_HIGH   },
        { SettingsDialog::TRIGGER_COMMAND, SettingsDialog::TRIGGER_COMMAND_HIGH },
        { SettingsDialog::TRIGGER_COUNT,   SettingsDialog::TRIGGER_COUNT_HIGH   }
    };

    for (size_t i = 0; i < sizeof mapping / sizeof mapping[0]; ++i)
    {
        if (settings->contains(mapping[i].oldName))
        {
            QVariant const value = settings->value(mapping[i].oldName);
            settings->setValue(mapping[i].newName, value);
            settings->remove(mapping[i].oldName);
        }
    }
}

void MainWindow::applySettings (bool startup)
{
    QSettings settings;
    if (startup)
        upgradeSettings(&settings);

    triggerHigh.value = settings.value(SettingsDialog::TRIGGER_VALUE_HIGH, SettingsDialog::TRIGGER_VALUE_DEFAULT).toInt();
    triggerHigh.command = settings.value(SettingsDialog::TRIGGER_COMMAND_HIGH, SettingsDialog::TRIGGER_COMMAND_DEFAULT).toString();
    triggerHigh.countSetting = settings.value(SettingsDialog::TRIGGER_COUNT_HIGH, SettingsDialog::TRIGGER_COUNT_DEFAULT).toInt();

    triggerLow.value = settings.value(SettingsDialog::TRIGGER_VALUE_LOW, SettingsDialog::TRIGGER_VALUE_DEFAULT).toInt();
    triggerLow.command = settings.value(SettingsDialog::TRIGGER_COMMAND_LOW, SettingsDialog::TRIGGER_COMMAND_DEFAULT).toString();
    triggerLow.countSetting = settings.value(SettingsDialog::TRIGGER_COUNT_LOW, SettingsDialog::TRIGGER_COUNT_DEFAULT).toInt();

    switch (settings.value(SettingsDialog::TRIGGER_ORDER, SettingsDialog::TRIGGER_ORDER_DEFAULT).toInt())
    {
    case 1: // High trigger first
        triggerHigh.waitForBrother = false;
        triggerLow.waitForBrother = true;
        break;
    case 2: // Low trigger first
        triggerHigh.waitForBrother = true;
        triggerLow.waitForBrother = false;
        break;
    default: // Triggers allowed in any order
        triggerHigh.waitForBrother = false;
        triggerLow.waitForBrother = false;
        break;
    }

    Trigger::onEdge = settings.value(SettingsDialog::TRIGGER_EDGE_MODE, SettingsDialog::TRIGGER_EDGE_MODE_DEFAULT).toBool();
    Trigger::stereoMode = Trigger::StereoMode(settings.value(SettingsDialog::TRIGGER_STEREO_MODE, SettingsDialog::TRIGGER_STEREO_MODE_DEFAULT).toInt());

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, settings.value(SettingsDialog::AUTO_ROTATE, SettingsDialog::AUTO_ROTATE_DEFAULT).toBool());
#endif

    loggingEnabled = settings.value(SettingsDialog::LOG_LOGGING_ENABLED, SettingsDialog::LOG_LOGGING_ENABLED_DEFAULT).toBool();
    logFileName = settings.value(SettingsDialog::LOG_FILE_NAME, SettingsDialog::LOG_FILE_NAME_DEFAULT).toString();

    if (loggingEnabled)
    {
        if (logFile && logFile->fileName().compare(logFileName) != 0)
            closeFile(logFile);

        if (!logFile && logFileName.compare(SettingsDialog::LOG_FILE_NAME_DEFAULT) != 0)
        {
            logIndex = 0;
            logFile = new QFile(logFileName);
            logFile->open(QIODevice::WriteOnly | QIODevice::Append | QIODevice::Text);
        }
    }

    interpolationLevel = settings.value(SettingsDialog::INTERPOLATION_LEVEL, SettingsDialog::INTERPOLATION_LEVEL_DEFAULT).toInt();

    if (startup)
    {
        int const source = settings.value(SettingsDialog::SOURCE, SettingsDialog::SOURCE_DEFAULT).toInt();
        stereo = settings.value(SettingsDialog::STEREO, SettingsDialog::STEREO_DEFAULT).toBool();
        scale = settings.value(SettingsDialog::SCALE, SettingsDialog::SCALE_DEFAULT).toInt();
        int const meter = settings.value(SettingsDialog::METER, SettingsDialog::METER_DEFAULT).toInt();

        ui.sourcesComboBox->setCurrentIndex(source);
        ui.dBLinearComboBox->setCurrentIndex(scale);
        ui.meterComboBox->setCurrentIndex(meter);

        setSourceVirgin(source);

        if (settings.value(SettingsDialog::FULL_SCREEN, SettingsDialog::FULL_SCREEN_DEFAULT).toBool())
            setFullScreen(true);
    }

    if (volumeMeterLeft)
        initVolumeMeterWidget(volumeMeterLeft, settings);
    if (volumeMeterRight)
        initVolumeMeterWidget(volumeMeterRight, settings);
}

void MainWindow::closeFile (QFile *file)
{
    if (file && file->isOpen())
    {
        qDebug("Closing file %s.", file->fileName().toLocal8Bit().data());
        file->flush();
        file->close();
        delete file;
        file = NULL;
        qDebug("File closed.");
    }
}

void MainWindow::resetMinValue ()
{
    minVol = INT_MAX;
    ui.minimumVolume->setText("");
}

void MainWindow::resetMaxValue ()
{
    maxVol = -INT_MAX;
    ui.maximumVolume->setText("");
}

void MainWindow::resetLabels ()
{
    ui.currentVolume->setText("");
    resetMinValue();
    resetMaxValue();
}

void MainWindow::settingsDialog ()
{
    SettingsDialog dialog(this);
    if (dialog.exec())
        applySettings(false);
}

void MainWindow::helpDialog () {}

void MainWindow::aboutDialog ()
{
    QString title("About");
    title += " ";
    title += QApplication::applicationName();

    QString body("<h3>");
    body += title;
    body += "</h3> Version ";
    body += QApplication::applicationVersion();
    body += "<p>A simple Qt based program to measure and visualize the current sound volume level.</p>"
            "<p>&copy; 2010 Ruediger Gad (original design)<br>"
            "&copy; 2014 Peter Pichler (minor improvements)</p>"
            "<p>This program is distributed under <a href=\"http://www.gnu.org/licenses/\">GPL3.</a></p>";

    QMessageBox::about(this, title, body);
}

void MainWindow::setStereo (int index)
{
    stereo = bool(index); // 0(false)=mono, 1(true)=stereo
    setSource(ui.sourcesComboBox->currentIndex());
}

void MainWindow::setSource (int index)
{
    adapter.disconnectAudioStream();
    setSourceVirgin(index);
}

void MainWindow::setSourceVirgin (int index)
{
#if defined(Q_WS_MAEMO_5) || defined(Q_WS_MAEMO_6)
    switch (index)
    {
    case 0: // Microphone
        ui.monoStereoComboBox->setCurrentIndex(stereoReal = false);
        ui.monoStereoComboBox->setEnabled(false);
        adapter.connectAudioStream("source.hw0", stereoReal);
        break;
    case 1: // Bluetooth
        ui.monoStereoComboBox->setCurrentIndex(stereoReal = stereo);
        ui.monoStereoComboBox->setEnabled(true);
        adapter.connectAudioStream("source.hw1", stereoReal);
        break;
    case 2: // Monitor
        ui.monoStereoComboBox->setCurrentIndex(stereoReal = stereo);
        ui.monoStereoComboBox->setEnabled(true);
        adapter.connectAudioStream("sink.hw0.monitor", stereoReal);
        break;
    default:
        qDebug("Unknown source!");
    }
#else
    adapter.connectAudioStream(sources.at(index), stereoReal = stereo);
#endif

    setMeter(ui.meterComboBox->currentIndex());
}

void MainWindow::setMeter (int index)
{
    delete volumeMeterLeft;
    delete volumeMeterRight;
    volumeMeterLeft = volumeMeterRight = NULL;

    switch (index)
    {
    case 0: // Analog
        ui.volumeMeterLeft->addWidget(volumeMeterLeft = new AnalogMeterWidget(this));
        if (stereoReal)
            ui.volumeMeterRight->addWidget(volumeMeterRight = new AnalogMeterWidget(this));
        break;
    case 1: // Bar
        ui.volumeMeterTop->addWidget(volumeMeterLeft = new HorizontalVolumeBarWidget(this));
        if (stereoReal)
            ui.volumeMeterBottom->addWidget(volumeMeterRight = new HorizontalVolumeBarWidget(this));
        break;
    case 2: // Chart
        ui.volumeMeterTop->addWidget(volumeMeterLeft = new VolumeLineChartWidget(this));
        if (stereoReal)
            ui.volumeMeterBottom->addWidget(volumeMeterRight = new VolumeLineChartWidget(this));
        break;
    }

    setScale(scale);
}

void MainWindow::setScale (int index)
{
    static int const maxValue[] = { MAX_LINEAR, MAX_DB };
    int const val = maxValue[scale = index];

    if (volumeMeterLeft)
       volumeMeterLeft->setScale(val);
    if (volumeMeterRight)
       volumeMeterRight->setScale(val);

    applySettings(false);
    resetLabels();
}

static int convertAudioLinear (int value)
{
    return value * MAX_LINEAR / SHRT_MAX;
}

static int convertAudioDB (int value)
{
    return 10.195 * log(value + 1);
}

void MainWindow::updateValues (const short *values, int length)
{
    typedef int (*convert_t)(int);
    static const convert_t convertAudio[] = {
        convertAudioLinear,
        convertAudioDB
    };

    const int volL = convertAudio[scale](values[0]);
    const int volR = (stereoReal && length > 1)
        ? convertAudio[scale](values[1])
        : volL;

    const bool th = triggerHigh.checkAndFire(volL, volR);
    const bool tl = triggerLow.checkAndFire(volL, volR);

    if (loggingEnabled)
    {
        QTextStream out(logFile);
        out << logIndex++ << ";"
            << QDate::currentDate().toString("yyyy-MM-dd") << ";"
            << QTime::currentTime().toString("hh:mm:ss.zzz") << ";"
            << volL << ";";
        if(stereoReal)
            out << volR << ";";

        static char const * const scaleStr[] = { "Linear", "dB" };
        out << scaleStr[scale];

        if (th)
            out << ";trigger(high)";
        if (tl)
            out << ";trigger(low)";
        out << "\n";
    }

    if (volumeMeterLeft)
        volumeMeterLeft->updateValue(volL);
    if (volumeMeterRight)
        volumeMeterRight->updateValue(volR);

    const int mn = std::max(volL, volR);
    if (mn < minVol)
        ui.minimumVolume->setNum(minVol = mn);
    const int mx = std::max(volL, volR);
    ui.currentVolume->setNum(mx);
    if (mx > maxVol)
        ui.maximumVolume->setNum(maxVol = mx);
}

void MainWindow::toggleFullScreen ()
{
    setFullScreen(ui.meterComboBox->isVisible());
}

void MainWindow::setFullScreen (bool fullScreen)
{
    const bool enable = !fullScreen;

    ui.currentLabel->setVisible(enable);
    ui.currentVolume->setVisible(enable);
    ui.minimumLabel->setVisible(enable);
    ui.minimumVolume->setVisible(enable);
    ui.maximumLabel->setVisible(enable);
    ui.maximumVolume->setVisible(enable);

    ui.sourcesComboBox->setVisible(enable);
    ui.monoStereoComboBox->setVisible(enable);
    ui.dBLinearComboBox->setVisible(enable);
    ui.meterComboBox->setVisible(enable);
}

// ---------- Trigger methods and static data ----------

bool Trigger::onEdge;
Trigger::StereoMode Trigger::stereoMode;

Trigger::Trigger (bool hi, Trigger & bro)
    : high(hi)
    , brother(bro)
    , conditionMetBefore(false)
    , count(0)
    , process(NULL)
    , triggered(false)
{}

Trigger::~Trigger ()
{
    if (process)
    {
        process->terminate();
        delete process;
    }
}

bool Trigger::checkAndFire (int volL, int volR)
{
    if (!value || command.isEmpty() || process || (waitForBrother && !brother.triggered))
        return false;

    int currentValue;
    switch (stereoMode)
    {
    case LEFT:
        currentValue = volL;
        break;
    case RIGHT:
        currentValue = volR;
        break;
    case EITHER:
        currentValue = high ? std::max(volL, volR)
                            : std::min(volL, volR);
        break;
    case BOTH:
    default: // not possible, but gcc complains about potentially uninitialized 'currentValue'
        currentValue = high ? std::min(volL, volR)
                            : std::max(volL, volR);
        break;
    }

    bool const conditionMet = high ? (currentValue > value)
                                   : (currentValue < value);

    if (onEdge)
    {
        if (conditionMet && !conditionMetBefore)
            ++count;
        conditionMetBefore = conditionMet;
    }
    else
    {
        if (conditionMet)
            ++count;
        else
            count = 0;
    }

    if (count < countSetting)
        return false;

    qDebug("%s trigger fired (\"%s\")", high ? "High" : "Low", command.toStdString().c_str());
    process = new QProcess();
    connect(process, SIGNAL(finished(int)), this, SLOT(processFinished()));
    process->start(command);

    count = 0;
    brother.triggered = false;
    return triggered = true;
}

void Trigger::processFinished ()
{
    qDebug("%s trigger finished", high ? "High" : "Low");
    if (process)
    {
        process->deleteLater();
        process = NULL;
    }
}
