/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of vumeter.
 *
 *  vumeter is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  vumeter is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with vumeter.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "analogmeterwidget.h"

/*
 * Analog meter is a square with only the top part showing.
 * The base of the square is the screen width (w). The spindle
 * of the indicator needle and the centre of rotation is 3/4 down.
 *
 * The following macros assume the existence of the variable w.
 */

#define XOFF  (w / 2)
#define YOFF  ((3 * w) / 4)
#define ARCR  ((12 * XOFF) / 10)   // Arc radius

void AnalogMeterWidget::draw (QPainter *painter)
{
    int const angle = -startAngle + span * currentValue / maximumValue;
    painter->rotate(angle);

    painter->setPen(QPen(Qt::white, 10));
    painter->drawLine(0, 0, 0, (YOFF * 95) / -100);

    painter->setPen(QPen(Qt::red, 6));
    painter->drawLine(0, 0, 0, (YOFF * 95) / -100);

    painter->rotate(-angle);
}

void AnalogMeterWidget::drawBackground (QPainter *painter)
{
    painter->drawImage(QRect(-XOFF, -YOFF - verticalCorrection(), w, h),
                       *background,
                       QRect(0, 0, w, h));
}

void AnalogMeterWidget::drawTriggerIndicator (QPainter *painter, int triggerValue)
{
    int const angle = -startAngle + span * triggerValue / maximumValue;
    painter->rotate(angle);

    painter->setPen(QPen(Qt::white, 10));
    painter->drawLine(0, 0, 0, (YOFF * 92) / -100);

    painter->setPen(QPen(triggerIndicatorColor, 6));
    painter->drawLine(0, 0, 0, (YOFF * 92) / -100);

    painter->rotate(-angle);
}

void AnalogMeterWidget::prepareBackground ()
{
    QPainter painter(background);
    painter.fillRect(0, 0, w, h, QBrush(Qt::white));
    initPainter(&painter);

    QFont f = painter.font();
#ifdef Q_WS_MAEMO_5
    int const fontSize = f.pointSize() * (w + 300) / 370;
#else
    int const fontSize = f.pointSize() * (w + 300) / 220;
#endif
    f.setPixelSize(fontSize);
    painter.setFont(f);

    if (useColoredScale)
    {
        int start = 16 * (90 + startAngle);
        int angle = 16 * (lowerThreshold * span) / maximumValue;
        painter.setPen(lowerColor);
        painter.setBrush(lowerColor);
        painter.drawPie(-ARCR, -ARCR, 2 * ARCR, 2 * ARCR, start, -angle);

        start -= angle;
        angle = 16 * ((upperThreshold - lowerThreshold) * span) / maximumValue;
        painter.setPen(midColor);
        painter.setBrush(midColor);
        painter.drawPie(-ARCR, -ARCR, 2 * ARCR, 2 * ARCR, start, -angle);

        start -= angle;
        angle = 16 * ((maximumValue - upperThreshold) * span) / maximumValue;
        painter.setPen(upperColor);
        painter.setBrush(upperColor);
        painter.drawPie(-ARCR, -ARCR, 2 * ARCR, 2 * ARCR, start, -angle);

        start -= angle;
        angle = 16 * ((maximumValue - upperThreshold) * span) / maximumValue;
        painter.setPen(QPen(Qt::white, thickPen.width()));
        painter.setBrush(Qt::white);
        int const blank = 4 * ARCR / 5;
        painter.drawPie(-blank, -blank, blank * 2, blank * 2, 16 * (90 - startAngle), 16 * span);
    }

    painter.setPen(thinPen);
    painter.drawArc(-ARCR, -ARCR, 2 * ARCR, 2 * ARCR, 16 * (90 - startAngle), 16 * span);
    painter.rotate(-startAngle);

    int const tag = maximumValue < 60 ? 10 : maximumValue < 150 ? 20 : 50;

    for (int i = 0; i <= maximumValue; i += 10)
    {
        int markLen;
        if (i % tag == 0)
        {
            painter.setPen(thickPen);
            markLen = w / 32;
            painter.drawText(-100, -ARCR - markLen - 10 - fontSize, 200, fontSize, Qt::AlignCenter, QString().setNum(i));
        }
        else
        {
            painter.setPen(thinPen);
            markLen = w / 52;
        }

        int const origin = -ARCR - (painter.pen().width() + 1) / 2;
        painter.drawLine(0, origin, 0, origin - markLen);
        painter.rotate(((float)span) / maximumValue * 10);
    }
}

void AnalogMeterWidget::initPainter(QPainter *painter)
{
    painter->setRenderHint(QPainter::Antialiasing, true);
    painter->translate(XOFF, YOFF + verticalCorrection());
}

int AnalogMeterWidget::verticalCorrection() const
{
    return (h < 2 * w / 3) ? 0 : (h / 8);
}
