/*   VLC-REMOTE for MAEMO 5
*   Copyright (C) 2010 Schutz Sacha <istdasklar@gmail.com>, Dru Moore <usr@dru-id.co.uk>, Yann Nave <yannux@onbebop.net>
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License version 2,
*   or (at your option) any later version, as published by the Free
*   Software Foundation
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details
*
*   You should have received a copy of the GNU General Public
*   License along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/
#include "browsemainwindow.h"
#include "ui_browsemainwindow.h"
#include <QSettings>
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
#include <QMaemo5InformationBox>
#endif
#include "configdialog.h"
#include "aboutdialog.h"
#include "vlcbrowseelement.h"
#include "appsettings.h"

BrowseMainWindow::BrowseMainWindow(QWidget *parent) :
        QMainWindow(parent),
        ui(new Ui::BrowseMainWindow)
{

    ui->setupUi(this);
    mCurrentDir = "~/"; //AppSettings::getHomeDirectory().path; // This works on win as well as linux, would guess mac too.
    setWindowTitle("Vlc remote");


    mNetManager = new QNetworkAccessManager(this);

    mContents = new QList<VlcBrowseElement>();

    ui->listWidget->setTextElideMode(Qt::ElideMiddle);
    ui->listWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    ui->playButton->setIcon(QIcon::fromTheme("camera_playback"));
    ui->addButton->setIcon(QIcon::fromTheme("general_add"));
    ui->browseButton->setIcon(QIcon::fromTheme("filemanager_media_folder"));
    ui->browseButton->setDisabled(true);
    ui->playButton->setDisabled(true);
    ui->addButton->setDisabled(true);

    connect(ui->listWidget,SIGNAL(doubleClicked(QModelIndex)),this,SLOT(onBrowse()));
    connect(ui->browseButton,SIGNAL(clicked()),this,SLOT(onBrowse()));
    connect(ui->addButton,SIGNAL(clicked()),this,SLOT(onAddToPlaylist()));
    connect(ui->playButton,SIGNAL(clicked()),this,SLOT(onPlay()));
    connect(ui->listWidget, SIGNAL(itemSelectionChanged()), this, SLOT(onListSelectionChanged()));

    connect(ui->actionGoUserHome, SIGNAL(triggered()), this, SLOT(showUserHomeFolder()));
    connect(ui->actionGoHome, SIGNAL(triggered()), this, SLOT(showHomeFolder()));
    connect(ui->actionSetHome, SIGNAL(triggered()), this, SLOT(setHomeFolder()));
    connect(ui->actionViewFavourites, SIGNAL(triggered()), this, SLOT(showFavourites()));
    connect(ui->actionSetFavourite, SIGNAL(triggered()), this, SLOT(setFavourite()));

    init();


}
void BrowseMainWindow::init()  // THIS METHOD IS CALLED WHEN CONFIG CHANGED...
{
    mIp = AppSettings::getCurrentIp(); // AccountDialog::currentIp();
    setHomeDirectory();
}
void BrowseMainWindow::setHomeDirectory()
{
    mCurrentDir = AppSettings::getHomeDirectory().path;
}
void BrowseMainWindow::showCurrentDirectory()  // THIS METHOD IS CALLED WHEN WINDOW IS OPENED...
{
    browseDirectory(mCurrentDir);
}

BrowseMainWindow::~BrowseMainWindow()
{
    delete ui;
}

void BrowseMainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void BrowseMainWindow::showHomeFolder() {
    browseDirectory(AppSettings::getHomeDirectory().path);
}
void BrowseMainWindow::showUserHomeFolder() {
    browseDirectory("~/");
}
void BrowseMainWindow::setHomeFolder() {
    if (0 < mCurrentElement.name.length() && (0 == QString::compare("directory", mCurrentElement.type) || 0 == QString::compare("dir", mCurrentElement.type))) {
        VlcDirectory dir;
        dir.name = mCurrentElement.name;
        dir.path = mCurrentElement.path;
        AppSettings::setHomeDirectory(dir);
    }
    else if (0 < mCurrentDir.length()) {
        VlcDirectory dir;
        QString name = mCurrentDir;
        int idx = mCurrentDir.lastIndexOf('/');
        if (0 > idx) idx = mCurrentDir.lastIndexOf('\\');
        if (0 < idx) {
            name = mCurrentDir.right(mCurrentDir.length() - (idx + 1));
        }
        dir.name = name;
        dir.path = mCurrentDir;
        AppSettings::setHomeDirectory(dir);
    }
}
void BrowseMainWindow::showFavourites() {
    emit showFavouritesWindow();
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
    this->hide();
#endif
}
void BrowseMainWindow::setFavourite() {
    if (0 < mCurrentElement.name.length() && (0 == QString::compare("directory", mCurrentElement.type) || 0 == QString::compare("dir", mCurrentElement.type))) {
        VlcDirectory dir;
        dir.name = mCurrentElement.name;
        dir.path = mCurrentElement.path;
        AppSettings::addFavourite(dir);
    }
    else if (0 < mCurrentDir.length()) {
        VlcDirectory dir;
        QString name = mCurrentDir;
        int idx = mCurrentDir.lastIndexOf('/');
        if (0 > idx) idx = mCurrentDir.lastIndexOf('\\');
        if (0 < idx) {
            name = mCurrentDir.right(mCurrentDir.length() - (idx + 1));
        }
        dir.name = name;
        dir.path = mCurrentDir;
        AppSettings::addFavourite(dir);
    }
}

void BrowseMainWindow::onListSelectionChanged() {
    QList<QListWidgetItem *> items = ui->listWidget->selectedItems();
    if (0 < items.count()) {
        mCurrentElement = getElementFromText(items.at(0)->text());
        // are we up dir?
        if (0 == QString::compare("..", mCurrentElement.name)) {
            ui->browseButton->setDisabled(true);
            ui->playButton->setDisabled(true);
            ui->addButton->setDisabled(true);
            mCurrentDir = mCurrentElement.path;
            browseDirectory(mCurrentDir);
        }
        else {
            // can we browse?
            if (0 == QString::compare("directory", mCurrentElement.type) || 0 == QString::compare("dir", mCurrentElement.type)) {
                ui->browseButton->setDisabled(false);
            }
            else {
                ui->browseButton->setDisabled(true);
            }
            // can we play?
            ui->playButton->setDisabled(false);
            // can we playlist?
            ui->addButton->setDisabled(false);
        }
    }
}

VlcBrowseElement BrowseMainWindow::getElementFromText(QString text) {
    for (int idx = 0; idx < mContents->count(); ++idx) {
        if (0 == QString::compare(text, mContents->at(idx).name)) {
            return mContents->at(idx);
        }
    }
    return *(new VlcBrowseElement());
}

void BrowseMainWindow::onBrowse() {
    // check for directory
    if (0 == QString::compare("directory", mCurrentElement.type) || 0 == QString::compare("dir", mCurrentElement.type)) {
        // call browseDirectory
        mCurrentDir = mCurrentElement.path;
        browseDirectory(mCurrentDir);
    }
    else {
        ui->browseButton->setDisabled(true);
    }
}

void BrowseMainWindow::onAddToPlaylist() {
    QUrl url = QUrl("http://"+mIp+"/requests/status.xml?command=in_enqueue");
    url.addEncodedQueryItem(QByteArray("input"), QUrl::toPercentEncoding(mCurrentElement.path.replace("\\", "\\\\").replace("'", "\\'")));
    mNetManager->get(QNetworkRequest(url));
}

void BrowseMainWindow::onPlay() {
    QUrl url = QUrl("http://"+mIp+"/requests/status.xml?command=in_play");
    url.addEncodedQueryItem(QByteArray("input"), QUrl::toPercentEncoding(mCurrentElement.path.replace("\\", "\\\\").replace("'", "\\'")));
    mNetManager->get(QNetworkRequest(url));
}

void BrowseMainWindow::browseDirectory(QString dir) {
    mResponse.clear();
    QUrl url = QUrl("http://"+mIp+"/requests/browse.xml");
    url.addEncodedQueryItem(QByteArray("dir"), QUrl::toPercentEncoding(dir));
    QNetworkReply * reply = mNetManager->get(QNetworkRequest(url));
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
    this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    connect(reply,SIGNAL(readyRead()),this,SLOT(readReady()));
    connect(reply,SIGNAL(error(QNetworkReply::NetworkError)),this,SLOT(error(QNetworkReply::NetworkError)));
    connect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(finished(QNetworkReply *)));
}
void BrowseMainWindow::error(QNetworkReply::NetworkError code) {
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
    this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
    qDebug() << code;
}
void BrowseMainWindow::readReady() {
    QNetworkReply * reply = qobject_cast<QNetworkReply*>(sender());
    // append to buffer
    mResponse += reply->readAll();
}
void BrowseMainWindow::finished(QNetworkReply * reply) {
    // now we can call parseXmlDirectory to process the full buffers
    this->parseXmlDirectory();
    // only interested in finished signals
    disconnect(mNetManager,SIGNAL(finished(QNetworkReply *)),this,SLOT(finished(QNetworkReply *)));
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
    this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
    delete reply;
}
void BrowseMainWindow::parseXmlDirectory() {
    QDomDocument doc;
    doc.setContent(this->mResponse);
    QDomElement docElem = doc.documentElement();
    QDomNodeList elements = docElem.elementsByTagName("element");
    // we can sort by folders then files alphabetically by running to lists and appending them at the end
    // vlc alpha sorts everything in the incoming stream, we just need to seperate files from folders.
    if (0 < elements.count()) {
        QList<VlcBrowseElement>* files = new QList<VlcBrowseElement>();
        int idx = 0;
        mContents->clear();
        do {
            QDomNode node = elements.at(idx);
            VlcBrowseElement* dir = new VlcBrowseElement();
            dir->type = node.attributes().namedItem("type").nodeValue();
            dir->size = node.attributes().namedItem("size").nodeValue().toInt();
            dir->date = QDate::fromString(node.attributes().namedItem("date").nodeValue());
            dir->path = node.attributes().namedItem("path").nodeValue();
            dir->name = node.attributes().namedItem("name").nodeValue();
            dir->extension = getExtension(dir->path, node.attributes().namedItem("extension").nodeValue());
            ++idx;
            if (0 != QString::compare("directory", dir->type) && 0 != QString::compare("dir", dir->type)) {
                files->append(*dir);
            }
            else if (0 == QString::compare("..", dir->name)) {
                this->mContents->prepend(*dir);
            }
            else {
                this->mContents->append(*dir);
            }
            delete dir;
        } while (idx < elements.count());
        if (0 < files->count()) {
            mContents->append(*files);
        }
        delete files;
        // Update UI
        this->updateList();
    }
    else {
        // alert user of error / empty directory
        qDebug() << "can't browse dir: " << mCurrentDir;
#if defined(Q_WS_S60) || defined(Q_WS_MAEMO_5)
        QMaemo5InformationBox::information(this, tr("Directory could not be browsed!"), QMaemo5InformationBox::DefaultTimeout);
#endif
    }
    mResponse.clear();

}

QString BrowseMainWindow::getExtension(QString path, QString extension) {
    // return extension if exists
    if (!extension.isNull() && !extension.isEmpty()) return extension;
    // return blank if no path
    if (path.isNull() || path.isEmpty()) return "";
    // otherwise extract the extension
    int dot_pos = path.lastIndexOf('.');
    if (0 < dot_pos) {
        return path.right(path.length() - (dot_pos + 1));
    }
    else { // no dot
        return "";
    }
}

void BrowseMainWindow::updateList() {
    ui->listWidget->clear();
    int ct = this->mContents->count();
    if (0 < ct) {
        QIcon icon_up     = QIcon::fromTheme("filemanager_folder_up");
        QIcon icon_folder = QIcon::fromTheme("general_folder");
        QIcon icon_audio  = QIcon::fromTheme("general_audio_file");
        QIcon icon_video  = QIcon::fromTheme("general_video_file");
        QIcon icon_image  = QIcon::fromTheme("general_image");
        QIcon icon_flash  = QIcon::fromTheme("filemanager_flash_file");
        QIcon icon_real   = QIcon::fromTheme("filemanager_real_music");
        QIcon icon_playl  = QIcon::fromTheme("filemanager_playlist");
        QIcon icon_unknown= QIcon::fromTheme("filemanager_unknown_file");
        for (int idx = 0; idx < ct; ++idx) {
            VlcBrowseElement dir = mContents->at(idx);
            QListWidgetItem* item;
            bool item_good = false;
            if (0 == QString::compare("directory", dir.type) || 0 == QString::compare("dir", dir.type)) {
                if (0 == QString::compare("..", dir.name)) {
                    item = new QListWidgetItem(icon_up, dir.name, ui->listWidget, 0);
                    item_good = true;
                }
                else {
                    item = new QListWidgetItem(icon_folder, dir.name, ui->listWidget, 0);
                    item_good = true;
                }
            }
            else if (0 == QString::compare("file", dir.type)) {
                if ( 0 == QString::compare(dir.extension, "jpg")  ||
                     0 == QString::compare(dir.extension, "jpeg") ||
                     0 == QString::compare(dir.extension, "gif")  ||
                     0 == QString::compare(dir.extension, "png")  ||
                     0 == QString::compare(dir.extension, "bmp")  ) {
                    item_good = true;
                    item = new QListWidgetItem(icon_image, dir.name, ui->listWidget, 0); // .jpg, .jpeg, .gif, .png, .bmp
                }
                else if ( 0 == QString::compare(dir.extension, "mp3")  ||
                          0 == QString::compare(dir.extension, "m4a")  ||
                          0 == QString::compare(dir.extension, "ogg")  ||
                          0 == QString::compare(dir.extension, "oga")  ||
                          0 == QString::compare(dir.extension, "wav")  ||
                          0 == QString::compare(dir.extension, "bmu")  ||
                          0 == QString::compare(dir.extension, "wma")  ||
                          0 == QString::compare(dir.extension, "flac")  ) {
                    item_good = true;
                    item = new QListWidgetItem(icon_audio, dir.name, ui->listWidget, 0); // .mp3, .m4a, .ogg, .oga, .wav, .bmu, .wma, .flac
                }
                else if ( 0 == QString::compare(dir.extension, "m3u")  ||
                          0 == QString::compare(dir.extension, "wpl")  ||
                          0 == QString::compare(dir.extension, "pls")  ||
                          0 == QString::compare(dir.extension, "asx")  ||
                          0 == QString::compare(dir.extension, "xspf") ||
                          0 == QString::compare(dir.extension, "cmml")  ) {
                    item_good = true;
                    item = new QListWidgetItem(icon_playl, dir.name, ui->listWidget, 0); // .m3u, .wpl, .pls, .asx, .xspf, .cmml
                }
                else if ( 0 == QString::compare(dir.extension, "avi")  ||
                          0 == QString::compare(dir.extension, "mpeg") ||
                          0 == QString::compare(dir.extension, "mpg")  ||
                          0 == QString::compare(dir.extension, "mov")  ||
                          0 == QString::compare(dir.extension, "mp4")  ||
                          0 == QString::compare(dir.extension, "m4v")  ||
                          0 == QString::compare(dir.extension, "wmv")  ||
                          0 == QString::compare(dir.extension, "mkv")  ||
                          0 == QString::compare(dir.extension, "ogv")  ) {
                    item_good = true;
                    item = new QListWidgetItem(icon_video, dir.name, ui->listWidget, 0); // .avi, .mpg, .mpeg, .mov, .mp4, .m4v, .wmv, .mkv, .ogv
                }
                else if ( 0 == QString::compare(dir.extension, "rm")  ||
                          0 == QString::compare(dir.extension, "ra")  ||
                          0 == QString::compare(dir.extension, "ram")  ) {
                    item = new QListWidgetItem(icon_real, dir.name, ui->listWidget, 0); // .ram, 'rm, 'ra
                }
                else if ( 0 == QString::compare(dir.extension, "flv")  ) {
                    item_good = true;
                    item = new QListWidgetItem(icon_flash, dir.name, ui->listWidget, 0); // .flv
                }
                else {
                    if (dir.name.startsWith("Flash")) {
                        item_good = true;
                        item = new QListWidgetItem(icon_flash, dir.name, ui->listWidget, 0);
                    }
                    else {
                        if (AppSettings::getShowUnknownFileTypes()) {
                            item_good = true;
                            item = new QListWidgetItem(icon_unknown, dir.name, ui->listWidget, 0);
                        }
                        else {
                            item_good = false;
                        }
                    }
                }
            }
            if (item_good) {
                ui->listWidget->addItem(item);
            }
            // other types ignored
        }
    }
}


