/*
@version: 0.7
@author: Sudheer K. <scifi1947 at gmail.com>
@license: GNU General Public License
*/

#include "profileswindow.h"
#include "ui_profileswindow.h"
#include "profilesettingsdialog.h"
#include "dbusutility.h"
#include "telepathyutility.h"
#include "databaseutility.h"
#include <QDebug>
#include <QDBusMetaType>

class ProfilesWindowPrivate
{
public:
    ProfilesWindowPrivate(ProfilesWindow * p) :
            dbusUtility(new DbusUtility(p)),
            tpUtility(new TelepathyUtility(p)),
            databaseUtility(new DatabaseUtility(p)),
            parent(p)
    {
        databaseUtility->openDatabase();

    }

    ~ProfilesWindowPrivate()
    {
        databaseUtility->closeDatabase();
        qDebug() << "VICaR: ProfilesWindowPrivate Destructing";
    }

    DbusUtility * dbusUtility;
    TelepathyUtility *tpUtility;
    DatabaseUtility *databaseUtility;
    org::maemo::vicar::Profile *profile;
    ProfilesWindow * const parent;    
};

// ---------------------------------------------------------------------------

ProfilesWindow::ProfilesWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::ProfilesWindow),
    d(new ProfilesWindowPrivate(this))
{
    ui->setupUi(this);
    loadProfilesList();

    ui->actionRoutingEnabled->setChecked(d->tpUtility->accountExists());        

    //bool routeOnDefault = d->gconfUtility->getGconfValueBoolean("route_on_default");
    QString routeOnDefaultSetting = d->databaseUtility->getSetting("route_on_default");
    ui->actionRouteOnDefault->setChecked(routeOnDefaultSetting == "1"?true:false);
}

ProfilesWindow::~ProfilesWindow()
{
    delete ui;
}

void ProfilesWindow::loadProfilesList(){
    org::maemo::vicar::ProfileList profileList;
    bool result = d->databaseUtility->getAllProfiles(&profileList);
    if (!result){
        d->dbusUtility->displayNotification("Unable to get profiles list.");
        qDebug() << "Error fetching profiles list. "<< d->databaseUtility->lastError();
    }
    else{
        org::maemo::vicar::Profile currentProfile;
        QString strProfileDetails;
        QListWidgetItem *item;
        foreach (currentProfile,profileList){            
            if (currentProfile.phoneNumberPattern != "%"){
                strProfileDetails = QString("Number starts with %1, then call via %2")
                                            .arg(currentProfile.phoneNumberPattern)
                                            .arg(currentProfile.gatewayNumber);
            }
            else{
                strProfileDetails = QString("No profile found, then call via %1")
                                            .arg(currentProfile.gatewayNumber);
            }
            item = new QListWidgetItem(strProfileDetails,ui->profilesListWidget);
            item->setData(Qt::UserRole,currentProfile.profileID);
        }
    }
}

void ProfilesWindow::on_addButton_clicked()
{
    org::maemo::vicar::Profile newProfile;
    ProfileSettingsDialog *dialog = new ProfileSettingsDialog(this,&newProfile);
    int result = dialog->exec();
    if (result == QDialog::Accepted){
        qDebug() << "Inserting new profile record";
        int result = d->databaseUtility->insertProfile(newProfile);
        if (result == -1){
            d->dbusUtility->displayNotification("Unable to create new profile.");
            qDebug() << "Error inserting new profile record. "<< d->databaseUtility->lastError();
        }
        else{
            QString strProfileDetails = QString("Number starts with %1, then call via %2")
                                        .arg(newProfile.phoneNumberPattern)
                                        .arg(newProfile.gatewayNumber);
            QListWidgetItem *item = new QListWidgetItem(strProfileDetails,ui->profilesListWidget);
            item->setData(Qt::UserRole,result);
        }
    }
}

void ProfilesWindow::on_editButton_clicked(){
    QListWidgetItem *item = ui->profilesListWidget->currentItem();
    if (item != 0){
        int intProfileID = item->data(Qt::UserRole).toInt();
        org::maemo::vicar::Profile profile;
        bool result = d->databaseUtility->selectProfile(intProfileID,&profile);
        if (!result){
            d->dbusUtility->displayNotification("Unable to fetch profile details.");
            qDebug() << "Error fetching profile record for id "<< intProfileID
                    <<". "<< d->databaseUtility->lastError();
        }
        else{
            ProfileSettingsDialog *dialog = new ProfileSettingsDialog(this,&profile);
            int response = dialog->exec();
            if (response == QDialog::Accepted){
                qDebug() << "Updating profile record";
                bool result = d->databaseUtility->updateProfile(profile);
                if (!result){
                    d->dbusUtility->displayNotification("Unable to update profile.");
                    qDebug() << "Error fetching profile record for id "<< intProfileID
                            <<". "<< d->databaseUtility->lastError();
                }
                else{
                    QString strProfileDetails;
                    if (profile.phoneNumberPattern != "%"){
                        strProfileDetails = QString("Number starts with %1, then call via %2")
                                                    .arg(profile.phoneNumberPattern)
                                                    .arg(profile.gatewayNumber);
                    }
                    else{
                        strProfileDetails = QString("No profile found, then call via %1")
                                                    .arg(profile.gatewayNumber);
                    }
                    item->setText(strProfileDetails);
                    item->setData(Qt::UserRole,profile.profileID);
                }
            }
        }
    }
    else{
        d->dbusUtility->displayNotification("Select a profile to edit.");
    }
}

void ProfilesWindow::on_removeButton_clicked(){
    QListWidgetItem *item = ui->profilesListWidget->currentItem();
    if (item != 0){
        qDebug() << "Selected item data is " << item->data(Qt::UserRole).toInt();
        int intProfileID = item->data(Qt::UserRole).toInt();
        bool result = d->databaseUtility->deleteProfile(intProfileID);
        if (!result){
            d->dbusUtility->displayNotification("Unable to delete profile.");
            qDebug() << "Error deleting profile record for id "<< intProfileID
                    <<". "<< d->databaseUtility->lastError();
        }
        else{
            //ui->profilesListWidget->removeItemWidget(item);
            ui->profilesListWidget->clear();
            this->loadProfilesList();
        }
    }
    else{
        d->dbusUtility->displayNotification("Select a profile to delete.");
    }
}

void ProfilesWindow::on_actionRoutingEnabled_triggered(bool checked){

    qDBusRegisterMetaType<org::freedesktop::Telepathy::SimplePresence>();

    bool result;

    if (checked){
        if (!d->tpUtility->accountExists()){
            result = d->tpUtility->createAccount();
            if (!result){
                d->dbusUtility->displayNotification("Unable to enable call routing.");
                ui->actionRoutingEnabled->setChecked(false);
            }
        }
    }
    else{
        if (d->tpUtility->accountExists()){
            result = d->tpUtility->deleteAccount();
            if (!result){
                d->dbusUtility->displayNotification("Unable to disable call routing.");
                ui->actionRoutingEnabled->setChecked(true);
            }
        }
    }
}

void ProfilesWindow::on_actionRouteOnDefault_triggered(bool checked){
    d->databaseUtility->setSetting("route_on_default",checked?"1":"0");
    //d->gconfUtility->setGconfValueBoolean("route_on_default",checked);
}

void ProfilesWindow::on_actionEditDefaultProfile_triggered(){
    org::maemo::vicar::Profile profile;
    bool result = d->databaseUtility->getDefaultProfile(&profile);
    qDebug() << "Default profile exits? " <<result;
    if (result){
        //Default profile does not exist - edit it
        ProfileSettingsDialog *dialog = new ProfileSettingsDialog(this,&profile);
        int response = dialog->exec();
        if (response == QDialog::Accepted){
            qDebug() << "Updating default profile record";
            bool result = d->databaseUtility->updateProfile(profile);
            if (!result){
                d->dbusUtility->displayNotification("Unable to update default profile.");
                qDebug() << "Error updating default profile."<< d->databaseUtility->lastError();
            }
            else{
                ui->profilesListWidget->clear();
                this->loadProfilesList();
            }
        }
    }
    else{
       //Create new default profile
        profile.phoneNumberPattern = "%";
        ProfileSettingsDialog *dialog = new ProfileSettingsDialog(this,&profile);
        int result = dialog->exec();
        if (result == QDialog::Accepted){
            qDebug() << "Inserting new default profile record";
            int result = d->databaseUtility->insertProfile(profile);
            if (result == -1){
                d->dbusUtility->displayNotification("Unable to create default profile.");
                qDebug() << "Error inserting new default profile record. "<< d->databaseUtility->lastError();
            }
            else{
                QString strProfileDetails = QString("No profile found, then call via %1")
                                            .arg(profile.gatewayNumber);
                QListWidgetItem *item = new QListWidgetItem(strProfileDetails,ui->profilesListWidget);
                item->setData(Qt::UserRole,result);
            }
        }
    }
}
