/*
@version: 0.4
@author: Sudheer K. <scifi1947 at gmail.com>
@license: GNU General Public License
*/

#include "callrouter.h"
#include "vicardbusadaptor.h"
#include <dbusutility.h>
#include <gconfutility.h>
#include <QDebug>
#include <QRegExp>
#include <QDBusConnection>
#include <QDBusMessage>
#include <QStringListIterator>

static QString strLastDialedNumber = QString();

class CallRouterPrivate
{
public:
    CallRouterPrivate(CallRouter * p) :
        dbusAdaptor(new VicarDbusAdaptor(p)),
        gconfUtility(new GConfUtility(p)),
        dbusUtility(new DbusUtility(p)),
        parent(p)
    {
        Q_ASSERT(0 != dbusAdaptor);
    }

    ~CallRouterPrivate()
    {
        qDebug() << "VICaR: Call Router Destructing";
    }

    VicarDbusAdaptor * dbusAdaptor;
    GConfUtility * gconfUtility;
    DbusUtility * dbusUtility;
    CallRouter * const parent;
};

// ---------------------------------------------------------------------------

CallRouter::CallRouter(QObject *parent) :
    QObject(parent),
    d(new CallRouterPrivate(this))
{
        Q_ASSERT(0 != d);
        this->registerDBusService();
        qDebug() << "Registered DBus Service " << APPLICATION_DBUS_SERVICE;
}

CallRouter::~CallRouter(){
}

void CallRouter::registerDBusService(){
    QDBusConnection connection = d->dbusUtility->getConnection();

    if (!connection.interface()->isServiceRegistered(APPLICATION_DBUS_SERVICE)){

        if (!connection.registerService(APPLICATION_DBUS_SERVICE)) {
            qDebug() << d->dbusUtility->getErrorMessage();
            exit(1);
        }
    }

    if (!connection.registerObject(APPLICATION_DBUS_PATH, this, QDBusConnection::ExportAdaptors)) {
        qDebug() << d->dbusUtility->getErrorMessage();
        exit(2);
    }

    //this->connectToDBusSignals();

}


void CallRouter::unregisterDBusService(){

    //this->disconnectFromDBusSignals();

    QDBusConnection connection = d->dbusUtility->getConnection();

    connection.unregisterObject(APPLICATION_DBUS_PATH,QDBusConnection::UnregisterTree);

    if (!connection.unregisterService(APPLICATION_DBUS_SERVICE)) {
        qDebug() << d->dbusUtility->getErrorMessage();
        exit(3);
    }

}

QString CallRouter::callViaCallingCard(){
        //Now call the calling card number. This is generally a local and/or tollfree number

        QString strCallingCardNumber = d->gconfUtility->getGconfValueString("calling_card_number");

        qDebug() << "Initiating call to "<< strCallingCardNumber;

        bool status = this->placeCall(strCallingCardNumber);

        QString strUserMessage;
        QString strErrorMessage;
        if (status){
            qDebug() << "Call initiated successfully. Connecting DBus slot for audio connection monitor";
             startCallStatusMonitors();
        }
        else {
            strUserMessage = QString("Unable to initiate new call to ").append(strCallingCardNumber);
            strErrorMessage = d->dbusUtility->getErrorMessage();
            qDebug() << strErrorMessage;
            strLastDialedNumber.clear();
        }

        d->dbusUtility->displayNotification(strUserMessage);
        return strErrorMessage;
}

bool CallRouter::placeCall(QString number){

    QList<QVariant> argsToSend;
    argsToSend.append(number);
    argsToSend.append(0);

    bool status = d->dbusUtility->sendMethodCall(CSD_SERVICE,
                                             CSD_CALL_PATH,
                                         CSD_CALL_INTERFACE,
                                         QString("CreateWith"),argsToSend);
    return status;

}

void CallRouter::startCallStatusMonitors(){
    /* Declare the slot to be executed when a call is picked up by other party (Audio connection established).
       We need this to confirm whether a call went though successfully.
    */

    QDBusConnection connection = d->dbusUtility->getConnection();

    bool success = connection.connect(QString(""),
                           CSD_CALL_INSTANCE_PATH,
                           CSD_CALL_INSTANCE_INTERFACE,
                           QString("AudioConnect"),this,
                           SLOT(sendNumberAsDTMFCode(const QDBusMessage&)));

    if (success){
        qDebug() << "Successfully connected to Dbus signal AudioConnect in interface "<< CSD_CALL_INSTANCE_INTERFACE;
    }
    else{
        qDebug() << "Failed to connect to Dbus signal AudioConnect in interface "<< CSD_CALL_INSTANCE_INTERFACE;
        qDebug() <<"DBus Error: "<< d->dbusUtility->getErrorMessage();
    }

    /* Declare the slot to be executed when the call is terminated (due to connection errors etc).
       We need this to avoid sending DTMF code on wrong calls.
    */

    success = connection.connect(QString(""),
                               CSD_CALL_INSTANCE_PATH,
                               CSD_CALL_INSTANCE_INTERFACE,
                               QString("Terminated"),this,
                               SLOT(stopCallStatusMonitors()));

    if (success){
        qDebug() << "Successfully connected to Dbus signal Terminated in interface "<< CSD_CALL_INSTANCE_INTERFACE;
    }
    else{
        qDebug() << "Failed to connect to Dbus signal Terminated in interface "<< CSD_CALL_INSTANCE_INTERFACE;
        qDebug() <<"DBus Error: "<< d->dbusUtility->getErrorMessage();
    }

    /* Declare the slot to be executed when a call is received
      (before we can place the call to calling card number).
       It is extremely rare that somebody should get a call within these few seconds.
       In any case, we need this to avoid sending DTMF code on the received call.

       Btw - I don't care for the incoming number here. If anyone is calling the user before we can send DTMF code,
       then we stop sending the DTMF code even if user does not respond to the call.
    */

    success = connection.connect(QString(""),
                               CSD_CALL_PATH,
                               CSD_CALL_INTERFACE,
                               QString("Coming"),this,
                               SLOT(stopCallStatusMonitors()));

    if (success){
        qDebug() << "Successfully connected to Dbus signal Coming in interface" << CSD_CALL_INTERFACE;
    }
    else{
        qDebug() << "Failed to connect to Dbus signal Coming in interface" << CSD_CALL_INTERFACE;
        qDebug() <<"DBus Error: "<< d->dbusUtility->getErrorMessage();
    }
}

void CallRouter::stopCallStatusMonitors(){

    strLastDialedNumber.clear();

    QDBusConnection connection = d->dbusUtility->getConnection();

    // Disconnect the slot for audio connection status
    bool status = connection.disconnect(QString(""),
                                   CSD_CALL_INSTANCE_PATH,
                                   CSD_CALL_INSTANCE_INTERFACE,
                                   QString("AudioConnect"),this,
                                   SLOT(sendNumberAsDTMFCode(const QDBusMessage&)));

    if (status){
        qDebug() << "Successfully disconnected from Dbus signal AudioConnect in interface "<< CSD_CALL_INSTANCE_INTERFACE;
    }
    else{
        qDebug() << "Failed to disconnect from Dbus signal AudioConnect in interface "<< CSD_CALL_INSTANCE_INTERFACE;
        qDebug() <<"DBus Error: "<< d->dbusUtility->getErrorMessage();
    }

    // Disconnect the slot for monitoring terminated calls
    status = connection.disconnect(QString(""),
                                   CSD_CALL_INSTANCE_PATH,
                                   CSD_CALL_INSTANCE_INTERFACE,
                                   QString("Terminated"),this,
                                   SLOT(stopCallStatusMonitors()));

    if (status){
        qDebug() << "Successfully disconnected from Dbus signal Terminated in interface "<< CSD_CALL_INSTANCE_INTERFACE;
    }
    else{
        qDebug() << "Failed to disconnect from Dbus signal Terminated in interface "<< CSD_CALL_INSTANCE_INTERFACE;
        qDebug() <<"DBus Error: "<< d->dbusUtility->getErrorMessage();
    }

    // Disconnect the slot for monitoring incoming calls
    status = connection.disconnect(QString(""),
                                   CSD_CALL_PATH,
                                   CSD_CALL_INTERFACE,
                                   QString("Coming"),this,
                                   SLOT(stopCallStatusMonitors()));

    if (status){
        qDebug() << "Successfully disconnected from Dbus signal Coming in interface" << CSD_CALL_INTERFACE;
    }
    else{
        qDebug() << "Failed to disconnect from Dbus signal Coming in interface" << CSD_CALL_INTERFACE;
        qDebug() <<"DBus Error: "<< d->dbusUtility->getErrorMessage();
    }
}

void CallRouter::sendNumberAsDTMFCode(const QDBusMessage& dbusMessage){

    if (!strLastDialedNumber.isEmpty()){
        //Verify whether we have the last dialed number available

        QList<QVariant> listArguments = dbusMessage.arguments();
        bool audioConnected =  listArguments.first().toBool();

        if (audioConnected){
            // Now that the call to Calling card number is successful. We can send the original number as DTMF tones
            QString strDTMFCode = convertToDTMFCode(strLastDialedNumber);

            qDebug() << "Audio connection established. Sending DTMF code "<< strDTMFCode;

            QList<QVariant> argsToSend;
            argsToSend.append(strDTMFCode);

            bool status = d->dbusUtility->sendMethodCall(CSD_SERVICE,
                                                     CSD_CALL_PATH,
                                                 CSD_CALL_INTERFACE,
                                                 QString("SendDTMF"),argsToSend);

            if (status){
                QString strMessage = strDTMFCode.append(" sent as DTMF code");
                qDebug() << strMessage;
                d->dbusUtility->displayNotification(strMessage);
            }
            else{
                qDebug() << "Unable to send DTMF code.";
            }


            /*
              Connecting and Disconnecting from/to DBus signal for each international call
              may not be the most efficient way of handling this. But we need to make sure
              that the DTMF codes are sent only for the calls placed by this app (i.e calls to Calling card number).
             */

            qDebug() << "Now disconnecting from call status monitors..";
            stopCallStatusMonitors();

        }
        else{
            qDebug() << "Audio not yet connected.";
        }
    }
    else
    {
        qDebug() << "Last dialed number is empty.";
    }
}

QString CallRouter::convertToDTMFCode(QString strNumber){
    QString strDTMFCode;

    if (!strNumber.isEmpty()){

        //int intDTMFDelay = d->gconfUtility->getGconfValueInteger("dtmf_delay");

        //if (intDTMFDelay <1 ) intDTMFDelay = 1;

        int intDTMFDelay = 1;

        //Add the prefix p so that there is some delay after the call is picked up by the automated system to send DTMF tones.
        strDTMFCode = QString("").fill('p',intDTMFDelay);        

        //Now check whether we need a prefix
        QString strDTMFPrefix = d->gconfUtility->getGconfValueString("dtmf_prefix");
        if (!strDTMFPrefix.isEmpty()){
            strDTMFCode = strDTMFCode.append(strDTMFPrefix);
        }

        //Get the format required by calling card from coniguration
        QString qstrDTMFFormat = d->gconfUtility->getGconfValueString("dtmf_format");
        if (qstrDTMFFormat.isEmpty()) qstrDTMFFormat = "<Country Code><Area Code><Phone Number>";

        /* Replace 00 (international dialing code) at the beginning
           and also replace any character other than the numbers 0-9 and p.
           */
        QRegExp regexp = QRegExp("(^0{2})|[^0-9p]");        
        strNumber = strNumber.replace(regexp,"");                

        /* Now we have a clean number with only country code, area code and phone number,
           lets convert it to the calling card friendly format
           */
        if (qstrDTMFFormat.startsWith("+")){
            strDTMFCode = strDTMFCode.append("+");
        }
        else if (qstrDTMFFormat.startsWith("00")){
            strDTMFCode = strDTMFCode.append("00");
        }
        else if (qstrDTMFFormat.startsWith("011")){
            strDTMFCode = strDTMFCode.append("011");
        }

        strDTMFCode = strDTMFCode.append(strNumber);

        //Now check whether we need a suffix
        QString strDTMFSuffix = d->gconfUtility->getGconfValueString("dtmf_suffix");
        if (!strDTMFSuffix.isEmpty() && !strDTMFSuffix.contains("--None--")){
            strDTMFCode = strDTMFCode.append(strDTMFSuffix);
        }
    }

    return strDTMFCode;
}

bool CallRouter::isExcludedNumber(QString strInternationalNumber){

    bool isExcluded = false;

    //Get the list of excluded codes
    QString qstrExcludedNumbers = d->gconfUtility->getGconfValueString("numbers_to_exclude");
    QStringList strExcludedCodeList = qstrExcludedNumbers.split(",");
    QStringListIterator iterator(strExcludedCodeList);

    QRegExp regexp = QRegExp("(^0{2})|[^0-9p]");

    while (iterator.hasNext()){
        QString strCode = iterator.next();
        strCode = strCode.replace(regexp,"");
        strInternationalNumber = strInternationalNumber.replace(regexp,"");
        if (!strCode.isEmpty() && strInternationalNumber.startsWith(strCode)){
            isExcluded = true;
        }
    }
    return isExcluded;
}

//DBus Method used by external applications to check whether VICaR is enabled and running
bool CallRouter::isRunning(){

    //Verify Whether Call Routing is Enabled
    bool isRoutingEnabled = d->gconfUtility->getGconfValueBoolean("routing_enabled");
    if (isRoutingEnabled){
        return true;
    }
    else{
        return false;
    }
}

//DBus Method used by external applications to call via VICaR
QString CallRouter::callInternationalNumber(const QString& strInternationalNumber){

    //QDBusConnection connection = d->dbusUtility->getConnection();
    QString strReturnMessage = "";

    qDebug() << "New call requested by external application. Destination number is " << strInternationalNumber;

     if (strInternationalNumber.startsWith("+") ||
         strInternationalNumber.startsWith("00"))
     {
         qDebug() << "International number "<< strInternationalNumber << " recognized. Starting proceedings..";

         //Check whether this is one of the excluded country codes
         if (!isExcludedNumber(strInternationalNumber)){
             strLastDialedNumber = strInternationalNumber;
             QString strErrorMessage = this->callViaCallingCard();
             if (!strErrorMessage.isEmpty()){
                 strReturnMessage = strErrorMessage;
             }
         }
         else{
             //Country code is excluded, place a call directly
             qDebug() << "Country code is excluded, placing call directly";
             bool status = this->placeCall(strInternationalNumber);
             if (!status){
                 strReturnMessage = d->dbusUtility->getErrorMessage();
             }
         }
    }
    else{
        strReturnMessage = "VICaR Error: Phone number is not in international format. Use +123456789 format.";
    }

    qDebug() << strReturnMessage;

    if (strReturnMessage.isEmpty()){
        return QString("Success");
    }
    else{
        return strReturnMessage;
    }
 }
