/* valascanner.vala
 *
 * Copyright (C) 2008-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gobject/gvaluecollector.h>


#define VALA_TYPE_SCANNER (vala_scanner_get_type ())
#define VALA_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCANNER, ValaScanner))
#define VALA_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCANNER, ValaScannerClass))
#define VALA_IS_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCANNER))
#define VALA_IS_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCANNER))
#define VALA_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCANNER, ValaScannerClass))

typedef struct _ValaScanner ValaScanner;
typedef struct _ValaScannerClass ValaScannerClass;
typedef struct _ValaScannerPrivate ValaScannerPrivate;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;

#define VALA_TYPE_COMMENT (vala_comment_get_type ())
#define VALA_COMMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_COMMENT, ValaComment))
#define VALA_COMMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_COMMENT, ValaCommentClass))
#define VALA_IS_COMMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_COMMENT))
#define VALA_IS_COMMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_COMMENT))
#define VALA_COMMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_COMMENT, ValaCommentClass))

typedef struct _ValaComment ValaComment;
typedef struct _ValaCommentClass ValaCommentClass;

#define VALA_SCANNER_TYPE_CONDITIONAL (vala_scanner_conditional_get_type ())
typedef struct _ValaScannerConditional ValaScannerConditional;
#define _vala_source_file_unref0(var) ((var == NULL) ? NULL : (var = (vala_source_file_unref (var), NULL)))
#define _vala_comment_unref0(var) ((var == NULL) ? NULL : (var = (vala_comment_unref (var), NULL)))

#define VALA_TYPE_TOKEN_TYPE (vala_token_type_get_type ())

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;
#define _vala_source_reference_unref0(var) ((var == NULL) ? NULL : (var = (vala_source_reference_unref (var), NULL)))
#define _g_free0(var) (var = (g_free (var), NULL))

#define VALA_TYPE_SOURCE_LOCATION (vala_source_location_get_type ())
typedef struct _ValaSourceLocation ValaSourceLocation;

#define VALA_TYPE_CODE_CONTEXT (vala_code_context_get_type ())
#define VALA_CODE_CONTEXT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_CONTEXT, ValaCodeContext))
#define VALA_CODE_CONTEXT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_CONTEXT, ValaCodeContextClass))
#define VALA_IS_CODE_CONTEXT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_CONTEXT))
#define VALA_IS_CODE_CONTEXT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_CONTEXT))
#define VALA_CODE_CONTEXT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_CONTEXT, ValaCodeContextClass))

typedef struct _ValaCodeContext ValaCodeContext;
typedef struct _ValaCodeContextClass ValaCodeContextClass;
typedef struct _ValaParamSpecScanner ValaParamSpecScanner;

struct _ValaScanner {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaScannerPrivate * priv;
};

struct _ValaScannerClass {
	GTypeClass parent_class;
	void (*finalize) (ValaScanner *self);
};

struct _ValaScannerConditional {
	gboolean matched;
	gboolean else_found;
	gboolean skip_section;
};

struct _ValaScannerPrivate {
	ValaSourceFile* _source_file;
	gchar* current;
	gchar* end;
	gint line;
	gint column;
	ValaComment* _comment;
	ValaScannerConditional* conditional_stack;
	gint conditional_stack_length1;
	gint conditional_stack_size;
};

typedef enum  {
	VALA_TOKEN_TYPE_NONE,
	VALA_TOKEN_TYPE_ABSTRACT,
	VALA_TOKEN_TYPE_AS,
	VALA_TOKEN_TYPE_ASSIGN,
	VALA_TOKEN_TYPE_ASSIGN_ADD,
	VALA_TOKEN_TYPE_ASSIGN_BITWISE_AND,
	VALA_TOKEN_TYPE_ASSIGN_BITWISE_OR,
	VALA_TOKEN_TYPE_ASSIGN_BITWISE_XOR,
	VALA_TOKEN_TYPE_ASSIGN_DIV,
	VALA_TOKEN_TYPE_ASSIGN_MUL,
	VALA_TOKEN_TYPE_ASSIGN_PERCENT,
	VALA_TOKEN_TYPE_ASSIGN_SHIFT_LEFT,
	VALA_TOKEN_TYPE_ASSIGN_SUB,
	VALA_TOKEN_TYPE_ASYNC,
	VALA_TOKEN_TYPE_BASE,
	VALA_TOKEN_TYPE_BITWISE_AND,
	VALA_TOKEN_TYPE_BITWISE_OR,
	VALA_TOKEN_TYPE_BREAK,
	VALA_TOKEN_TYPE_CARRET,
	VALA_TOKEN_TYPE_CASE,
	VALA_TOKEN_TYPE_CATCH,
	VALA_TOKEN_TYPE_CHARACTER_LITERAL,
	VALA_TOKEN_TYPE_CLASS,
	VALA_TOKEN_TYPE_CLOSE_BRACE,
	VALA_TOKEN_TYPE_CLOSE_BRACKET,
	VALA_TOKEN_TYPE_CLOSE_PARENS,
	VALA_TOKEN_TYPE_COLON,
	VALA_TOKEN_TYPE_COMMA,
	VALA_TOKEN_TYPE_CONST,
	VALA_TOKEN_TYPE_CONSTRUCT,
	VALA_TOKEN_TYPE_CONTINUE,
	VALA_TOKEN_TYPE_DEFAULT,
	VALA_TOKEN_TYPE_DELEGATE,
	VALA_TOKEN_TYPE_DELETE,
	VALA_TOKEN_TYPE_DIV,
	VALA_TOKEN_TYPE_DO,
	VALA_TOKEN_TYPE_DOUBLE_COLON,
	VALA_TOKEN_TYPE_DOT,
	VALA_TOKEN_TYPE_DYNAMIC,
	VALA_TOKEN_TYPE_ELLIPSIS,
	VALA_TOKEN_TYPE_ELSE,
	VALA_TOKEN_TYPE_ENUM,
	VALA_TOKEN_TYPE_ENSURES,
	VALA_TOKEN_TYPE_ERRORDOMAIN,
	VALA_TOKEN_TYPE_EOF,
	VALA_TOKEN_TYPE_EXTERN,
	VALA_TOKEN_TYPE_FALSE,
	VALA_TOKEN_TYPE_FINALLY,
	VALA_TOKEN_TYPE_FOR,
	VALA_TOKEN_TYPE_FOREACH,
	VALA_TOKEN_TYPE_GET,
	VALA_TOKEN_TYPE_HASH,
	VALA_TOKEN_TYPE_IDENTIFIER,
	VALA_TOKEN_TYPE_IF,
	VALA_TOKEN_TYPE_IN,
	VALA_TOKEN_TYPE_INLINE,
	VALA_TOKEN_TYPE_INTEGER_LITERAL,
	VALA_TOKEN_TYPE_INTERFACE,
	VALA_TOKEN_TYPE_INTERNAL,
	VALA_TOKEN_TYPE_INTERR,
	VALA_TOKEN_TYPE_IS,
	VALA_TOKEN_TYPE_LAMBDA,
	VALA_TOKEN_TYPE_LOCK,
	VALA_TOKEN_TYPE_MINUS,
	VALA_TOKEN_TYPE_NAMESPACE,
	VALA_TOKEN_TYPE_NEW,
	VALA_TOKEN_TYPE_NULL,
	VALA_TOKEN_TYPE_OUT,
	VALA_TOKEN_TYPE_OP_AND,
	VALA_TOKEN_TYPE_OP_DEC,
	VALA_TOKEN_TYPE_OP_EQ,
	VALA_TOKEN_TYPE_OP_GE,
	VALA_TOKEN_TYPE_OP_GT,
	VALA_TOKEN_TYPE_OP_INC,
	VALA_TOKEN_TYPE_OP_LE,
	VALA_TOKEN_TYPE_OP_LT,
	VALA_TOKEN_TYPE_OP_NE,
	VALA_TOKEN_TYPE_OP_NEG,
	VALA_TOKEN_TYPE_OP_OR,
	VALA_TOKEN_TYPE_OP_PTR,
	VALA_TOKEN_TYPE_OP_SHIFT_LEFT,
	VALA_TOKEN_TYPE_OPEN_BRACE,
	VALA_TOKEN_TYPE_OPEN_BRACKET,
	VALA_TOKEN_TYPE_OPEN_PARENS,
	VALA_TOKEN_TYPE_OVERRIDE,
	VALA_TOKEN_TYPE_OWNED,
	VALA_TOKEN_TYPE_PARAMS,
	VALA_TOKEN_TYPE_PERCENT,
	VALA_TOKEN_TYPE_PLUS,
	VALA_TOKEN_TYPE_PRIVATE,
	VALA_TOKEN_TYPE_PROTECTED,
	VALA_TOKEN_TYPE_PUBLIC,
	VALA_TOKEN_TYPE_REAL_LITERAL,
	VALA_TOKEN_TYPE_REF,
	VALA_TOKEN_TYPE_REQUIRES,
	VALA_TOKEN_TYPE_RETURN,
	VALA_TOKEN_TYPE_SEMICOLON,
	VALA_TOKEN_TYPE_SET,
	VALA_TOKEN_TYPE_SIGNAL,
	VALA_TOKEN_TYPE_SIZEOF,
	VALA_TOKEN_TYPE_STAR,
	VALA_TOKEN_TYPE_STATIC,
	VALA_TOKEN_TYPE_STRING_LITERAL,
	VALA_TOKEN_TYPE_STRUCT,
	VALA_TOKEN_TYPE_SWITCH,
	VALA_TOKEN_TYPE_THIS,
	VALA_TOKEN_TYPE_THROW,
	VALA_TOKEN_TYPE_THROWS,
	VALA_TOKEN_TYPE_TILDE,
	VALA_TOKEN_TYPE_TRUE,
	VALA_TOKEN_TYPE_TRY,
	VALA_TOKEN_TYPE_TYPEOF,
	VALA_TOKEN_TYPE_UNOWNED,
	VALA_TOKEN_TYPE_USING,
	VALA_TOKEN_TYPE_VAR,
	VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL,
	VALA_TOKEN_TYPE_VIRTUAL,
	VALA_TOKEN_TYPE_VOID,
	VALA_TOKEN_TYPE_VOLATILE,
	VALA_TOKEN_TYPE_WEAK,
	VALA_TOKEN_TYPE_WHILE,
	VALA_TOKEN_TYPE_YIELD
} ValaTokenType;

struct _ValaSourceLocation {
	gchar* pos;
	gint line;
	gint column;
};

struct _ValaParamSpecScanner {
	GParamSpec parent_instance;
};


static gpointer vala_scanner_parent_class = NULL;

gpointer vala_scanner_ref (gpointer instance);
void vala_scanner_unref (gpointer instance);
GParamSpec* vala_param_spec_scanner (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scanner (GValue* value, gpointer v_object);
gpointer vala_value_get_scanner (const GValue* value);
GType vala_scanner_get_type (void);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
gpointer vala_comment_ref (gpointer instance);
void vala_comment_unref (gpointer instance);
GParamSpec* vala_param_spec_comment (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_comment (GValue* value, gpointer v_object);
gpointer vala_value_get_comment (const GValue* value);
GType vala_comment_get_type (void);
static GType vala_scanner_conditional_get_type (void);
static ValaScannerConditional* vala_scanner_conditional_dup (const ValaScannerConditional* self);
static void vala_scanner_conditional_free (ValaScannerConditional* self);
#define VALA_SCANNER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SCANNER, ValaScannerPrivate))
enum  {
	VALA_SCANNER_DUMMY_PROPERTY
};
static void vala_scanner_set_source_file (ValaScanner* self, ValaSourceFile* value);
gchar* vala_source_file_get_mapped_contents (ValaSourceFile* self);
gsize vala_source_file_get_mapped_length (ValaSourceFile* self);
ValaScanner* vala_scanner_new (ValaSourceFile* source_file);
ValaScanner* vala_scanner_construct (GType object_type, ValaSourceFile* source_file);
static gboolean vala_scanner_is_ident_char (ValaScanner* self, gchar c);
static gboolean vala_scanner_matches (gchar* begin, const char* keyword);
GType vala_token_type_get_type (void);
ValaTokenType vala_scanner_get_identifier_or_keyword (gchar* begin, gint len);
static ValaTokenType vala_scanner_read_number (ValaScanner* self);
static void vala_scanner_space (ValaScanner* self);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceFile* vala_scanner_get_source_file (ValaScanner* self);
ValaSourceReference* vala_source_reference_new (ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column);
ValaSourceReference* vala_source_reference_construct (GType object_type, ValaSourceFile* _file, gint _first_line, gint _first_column, gint _last_line, gint _last_column);
GType vala_source_location_get_type (void);
ValaSourceLocation* vala_source_location_dup (const ValaSourceLocation* self);
void vala_source_location_free (ValaSourceLocation* self);
ValaTokenType vala_scanner_read_token (ValaScanner* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end);
static gboolean vala_scanner_pp_whitespace (ValaScanner* self);
static void vala_scanner_parse_pp_if (ValaScanner* self);
static void vala_scanner_parse_pp_elif (ValaScanner* self);
static void vala_scanner_parse_pp_else (ValaScanner* self);
static void vala_scanner_parse_pp_endif (ValaScanner* self);
static void vala_scanner_pp_directive (ValaScanner* self);
static void vala_scanner_pp_eol (ValaScanner* self);
static gboolean vala_scanner_parse_pp_expression (ValaScanner* self);
static void _vala_array_add2 (ValaScannerConditional** array, int* length, int* size, const ValaScannerConditional* value);
gpointer vala_code_context_ref (gpointer instance);
void vala_code_context_unref (gpointer instance);
GParamSpec* vala_param_spec_code_context (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_context (GValue* value, gpointer v_object);
gpointer vala_value_get_code_context (const GValue* value);
GType vala_code_context_get_type (void);
ValaCodeContext* vala_source_file_get_context (ValaSourceFile* self);
gboolean vala_code_context_is_defined (ValaCodeContext* self, const char* define);
static gboolean vala_scanner_parse_pp_symbol (ValaScanner* self);
static gboolean vala_scanner_parse_pp_primary_expression (ValaScanner* self);
static gboolean vala_scanner_parse_pp_unary_expression (ValaScanner* self);
static gboolean vala_scanner_parse_pp_equality_expression (ValaScanner* self);
static gboolean vala_scanner_parse_pp_and_expression (ValaScanner* self);
static gboolean vala_scanner_parse_pp_or_expression (ValaScanner* self);
static gboolean vala_scanner_whitespace (ValaScanner* self);
static void vala_scanner_push_comment (ValaScanner* self, const char* comment_item, ValaSourceReference* source_reference, gboolean file_comment);
static gboolean vala_scanner_comment (ValaScanner* self, gboolean file_comment);
void vala_scanner_parse_file_comments (ValaScanner* self);
ValaComment* vala_comment_new (const char* comment, ValaSourceReference* _source_reference);
ValaComment* vala_comment_construct (GType object_type, const char* comment, ValaSourceReference* _source_reference);
void vala_source_file_add_comment (ValaSourceFile* self, ValaComment* comment);
ValaComment* vala_scanner_pop_comment (ValaScanner* self);
static void vala_scanner_finalize (ValaScanner* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaScanner* vala_scanner_construct (GType object_type, ValaSourceFile* source_file) {
	ValaScanner* self;
	gchar* begin;
	g_return_val_if_fail (source_file != NULL, NULL);
	self = (ValaScanner*) g_type_create_instance (object_type);
	vala_scanner_set_source_file (self, source_file);
	begin = vala_source_file_get_mapped_contents (source_file);
	self->priv->end = begin + vala_source_file_get_mapped_length (source_file);
	self->priv->current = begin;
	self->priv->line = 1;
	self->priv->column = 1;
	return self;
}


ValaScanner* vala_scanner_new (ValaSourceFile* source_file) {
	return vala_scanner_construct (VALA_TYPE_SCANNER, source_file);
}


static gboolean vala_scanner_is_ident_char (ValaScanner* self, gchar c) {
	gboolean result;
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0_ = FALSE;
	if (g_ascii_isalnum (c)) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = c == '_';
	}
	result = _tmp0_;
	return result;
}


ValaTokenType vala_scanner_get_identifier_or_keyword (gchar* begin, gint len) {
	ValaTokenType result;
	switch (len) {
		case 2:
		{
			switch (begin[0]) {
				case 'a':
				{
					if (vala_scanner_matches (begin, "as")) {
						result = VALA_TOKEN_TYPE_AS;
						return result;
					}
					break;
				}
				case 'd':
				{
					if (vala_scanner_matches (begin, "do")) {
						result = VALA_TOKEN_TYPE_DO;
						return result;
					}
					break;
				}
				case 'i':
				{
					switch (begin[1]) {
						case 'f':
						{
							result = VALA_TOKEN_TYPE_IF;
							return result;
						}
						case 'n':
						{
							result = VALA_TOKEN_TYPE_IN;
							return result;
						}
						case 's':
						{
							result = VALA_TOKEN_TYPE_IS;
							return result;
						}
					}
					break;
				}
			}
			break;
		}
		case 3:
		{
			switch (begin[0]) {
				case 'f':
				{
					if (vala_scanner_matches (begin, "for")) {
						result = VALA_TOKEN_TYPE_FOR;
						return result;
					}
					break;
				}
				case 'g':
				{
					if (vala_scanner_matches (begin, "get")) {
						result = VALA_TOKEN_TYPE_GET;
						return result;
					}
					break;
				}
				case 'n':
				{
					if (vala_scanner_matches (begin, "new")) {
						result = VALA_TOKEN_TYPE_NEW;
						return result;
					}
					break;
				}
				case 'o':
				{
					if (vala_scanner_matches (begin, "out")) {
						result = VALA_TOKEN_TYPE_OUT;
						return result;
					}
					break;
				}
				case 'r':
				{
					if (vala_scanner_matches (begin, "ref")) {
						result = VALA_TOKEN_TYPE_REF;
						return result;
					}
					break;
				}
				case 's':
				{
					if (vala_scanner_matches (begin, "set")) {
						result = VALA_TOKEN_TYPE_SET;
						return result;
					}
					break;
				}
				case 't':
				{
					if (vala_scanner_matches (begin, "try")) {
						result = VALA_TOKEN_TYPE_TRY;
						return result;
					}
					break;
				}
				case 'v':
				{
					if (vala_scanner_matches (begin, "var")) {
						result = VALA_TOKEN_TYPE_VAR;
						return result;
					}
					break;
				}
			}
			break;
		}
		case 4:
		{
			switch (begin[0]) {
				case 'b':
				{
					if (vala_scanner_matches (begin, "base")) {
						result = VALA_TOKEN_TYPE_BASE;
						return result;
					}
					break;
				}
				case 'c':
				{
					if (vala_scanner_matches (begin, "case")) {
						result = VALA_TOKEN_TYPE_CASE;
						return result;
					}
					break;
				}
				case 'e':
				{
					switch (begin[1]) {
						case 'l':
						{
							if (vala_scanner_matches (begin, "else")) {
								result = VALA_TOKEN_TYPE_ELSE;
								return result;
							}
							break;
						}
						case 'n':
						{
							if (vala_scanner_matches (begin, "enum")) {
								result = VALA_TOKEN_TYPE_ENUM;
								return result;
							}
							break;
						}
					}
					break;
				}
				case 'l':
				{
					if (vala_scanner_matches (begin, "lock")) {
						result = VALA_TOKEN_TYPE_LOCK;
						return result;
					}
					break;
				}
				case 'n':
				{
					if (vala_scanner_matches (begin, "null")) {
						result = VALA_TOKEN_TYPE_NULL;
						return result;
					}
					break;
				}
				case 't':
				{
					switch (begin[1]) {
						case 'h':
						{
							if (vala_scanner_matches (begin, "this")) {
								result = VALA_TOKEN_TYPE_THIS;
								return result;
							}
							break;
						}
						case 'r':
						{
							if (vala_scanner_matches (begin, "true")) {
								result = VALA_TOKEN_TYPE_TRUE;
								return result;
							}
							break;
						}
					}
					break;
				}
				case 'v':
				{
					if (vala_scanner_matches (begin, "void")) {
						result = VALA_TOKEN_TYPE_VOID;
						return result;
					}
					break;
				}
				case 'w':
				{
					if (vala_scanner_matches (begin, "weak")) {
						result = VALA_TOKEN_TYPE_WEAK;
						return result;
					}
					break;
				}
			}
			break;
		}
		case 5:
		{
			switch (begin[0]) {
				case 'a':
				{
					if (vala_scanner_matches (begin, "async")) {
						result = VALA_TOKEN_TYPE_ASYNC;
						return result;
					}
					break;
				}
				case 'b':
				{
					if (vala_scanner_matches (begin, "break")) {
						result = VALA_TOKEN_TYPE_BREAK;
						return result;
					}
					break;
				}
				case 'c':
				{
					switch (begin[1]) {
						case 'a':
						{
							if (vala_scanner_matches (begin, "catch")) {
								result = VALA_TOKEN_TYPE_CATCH;
								return result;
							}
							break;
						}
						case 'l':
						{
							if (vala_scanner_matches (begin, "class")) {
								result = VALA_TOKEN_TYPE_CLASS;
								return result;
							}
							break;
						}
						case 'o':
						{
							if (vala_scanner_matches (begin, "const")) {
								result = VALA_TOKEN_TYPE_CONST;
								return result;
							}
							break;
						}
					}
					break;
				}
				case 'f':
				{
					if (vala_scanner_matches (begin, "false")) {
						result = VALA_TOKEN_TYPE_FALSE;
						return result;
					}
					break;
				}
				case 'o':
				{
					if (vala_scanner_matches (begin, "owned")) {
						result = VALA_TOKEN_TYPE_OWNED;
						return result;
					}
					break;
				}
				case 't':
				{
					if (vala_scanner_matches (begin, "throw")) {
						result = VALA_TOKEN_TYPE_THROW;
						return result;
					}
					break;
				}
				case 'u':
				{
					if (vala_scanner_matches (begin, "using")) {
						result = VALA_TOKEN_TYPE_USING;
						return result;
					}
					break;
				}
				case 'w':
				{
					if (vala_scanner_matches (begin, "while")) {
						result = VALA_TOKEN_TYPE_WHILE;
						return result;
					}
					break;
				}
				case 'y':
				{
					if (vala_scanner_matches (begin, "yield")) {
						result = VALA_TOKEN_TYPE_YIELD;
						return result;
					}
					break;
				}
			}
			break;
		}
		case 6:
		{
			switch (begin[0]) {
				case 'd':
				{
					if (vala_scanner_matches (begin, "delete")) {
						result = VALA_TOKEN_TYPE_DELETE;
						return result;
					}
					break;
				}
				case 'e':
				{
					if (vala_scanner_matches (begin, "extern")) {
						result = VALA_TOKEN_TYPE_EXTERN;
						return result;
					}
					break;
				}
				case 'i':
				{
					if (vala_scanner_matches (begin, "inline")) {
						result = VALA_TOKEN_TYPE_INLINE;
						return result;
					}
					break;
				}
				case 'p':
				{
					switch (begin[1]) {
						case 'a':
						{
							if (vala_scanner_matches (begin, "params")) {
								result = VALA_TOKEN_TYPE_PARAMS;
								return result;
							}
							break;
						}
						case 'u':
						{
							if (vala_scanner_matches (begin, "public")) {
								result = VALA_TOKEN_TYPE_PUBLIC;
								return result;
							}
							break;
						}
					}
					break;
				}
				case 'r':
				{
					if (vala_scanner_matches (begin, "return")) {
						result = VALA_TOKEN_TYPE_RETURN;
						return result;
					}
					break;
				}
				case 's':
				{
					switch (begin[1]) {
						case 'i':
						{
							switch (begin[2]) {
								case 'g':
								{
									if (vala_scanner_matches (begin, "signal")) {
										result = VALA_TOKEN_TYPE_SIGNAL;
										return result;
									}
									break;
								}
								case 'z':
								{
									if (vala_scanner_matches (begin, "sizeof")) {
										result = VALA_TOKEN_TYPE_SIZEOF;
										return result;
									}
									break;
								}
							}
							break;
						}
						case 't':
						{
							switch (begin[2]) {
								case 'a':
								{
									if (vala_scanner_matches (begin, "static")) {
										result = VALA_TOKEN_TYPE_STATIC;
										return result;
									}
									break;
								}
								case 'r':
								{
									if (vala_scanner_matches (begin, "struct")) {
										result = VALA_TOKEN_TYPE_STRUCT;
										return result;
									}
									break;
								}
							}
							break;
						}
						case 'w':
						{
							if (vala_scanner_matches (begin, "switch")) {
								result = VALA_TOKEN_TYPE_SWITCH;
								return result;
							}
							break;
						}
					}
					break;
				}
				case 't':
				{
					switch (begin[1]) {
						case 'h':
						{
							if (vala_scanner_matches (begin, "throws")) {
								result = VALA_TOKEN_TYPE_THROWS;
								return result;
							}
							break;
						}
						case 'y':
						{
							if (vala_scanner_matches (begin, "typeof")) {
								result = VALA_TOKEN_TYPE_TYPEOF;
								return result;
							}
							break;
						}
					}
					break;
				}
			}
			break;
		}
		case 7:
		{
			switch (begin[0]) {
				case 'd':
				{
					switch (begin[1]) {
						case 'e':
						{
							if (vala_scanner_matches (begin, "default")) {
								result = VALA_TOKEN_TYPE_DEFAULT;
								return result;
							}
							break;
						}
						case 'y':
						{
							if (vala_scanner_matches (begin, "dynamic")) {
								result = VALA_TOKEN_TYPE_DYNAMIC;
								return result;
							}
							break;
						}
					}
					break;
				}
				case 'e':
				{
					if (vala_scanner_matches (begin, "ensures")) {
						result = VALA_TOKEN_TYPE_ENSURES;
						return result;
					}
					break;
				}
				case 'f':
				{
					switch (begin[1]) {
						case 'i':
						{
							if (vala_scanner_matches (begin, "finally")) {
								result = VALA_TOKEN_TYPE_FINALLY;
								return result;
							}
							break;
						}
						case 'o':
						{
							if (vala_scanner_matches (begin, "foreach")) {
								result = VALA_TOKEN_TYPE_FOREACH;
								return result;
							}
							break;
						}
					}
					break;
				}
				case 'p':
				{
					if (vala_scanner_matches (begin, "private")) {
						result = VALA_TOKEN_TYPE_PRIVATE;
						return result;
					}
					break;
				}
				case 'u':
				{
					if (vala_scanner_matches (begin, "unowned")) {
						result = VALA_TOKEN_TYPE_UNOWNED;
						return result;
					}
					break;
				}
				case 'v':
				{
					if (vala_scanner_matches (begin, "virtual")) {
						result = VALA_TOKEN_TYPE_VIRTUAL;
						return result;
					}
					break;
				}
			}
			break;
		}
		case 8:
		{
			switch (begin[0]) {
				case 'a':
				{
					if (vala_scanner_matches (begin, "abstract")) {
						result = VALA_TOKEN_TYPE_ABSTRACT;
						return result;
					}
					break;
				}
				case 'c':
				{
					if (vala_scanner_matches (begin, "continue")) {
						result = VALA_TOKEN_TYPE_CONTINUE;
						return result;
					}
					break;
				}
				case 'd':
				{
					if (vala_scanner_matches (begin, "delegate")) {
						result = VALA_TOKEN_TYPE_DELEGATE;
						return result;
					}
					break;
				}
				case 'i':
				{
					if (vala_scanner_matches (begin, "internal")) {
						result = VALA_TOKEN_TYPE_INTERNAL;
						return result;
					}
					break;
				}
				case 'o':
				{
					if (vala_scanner_matches (begin, "override")) {
						result = VALA_TOKEN_TYPE_OVERRIDE;
						return result;
					}
					break;
				}
				case 'r':
				{
					if (vala_scanner_matches (begin, "requires")) {
						result = VALA_TOKEN_TYPE_REQUIRES;
						return result;
					}
					break;
				}
				case 'v':
				{
					if (vala_scanner_matches (begin, "volatile")) {
						result = VALA_TOKEN_TYPE_VOLATILE;
						return result;
					}
					break;
				}
			}
			break;
		}
		case 9:
		{
			switch (begin[0]) {
				case 'c':
				{
					if (vala_scanner_matches (begin, "construct")) {
						result = VALA_TOKEN_TYPE_CONSTRUCT;
						return result;
					}
					break;
				}
				case 'i':
				{
					if (vala_scanner_matches (begin, "interface")) {
						result = VALA_TOKEN_TYPE_INTERFACE;
						return result;
					}
					break;
				}
				case 'n':
				{
					if (vala_scanner_matches (begin, "namespace")) {
						result = VALA_TOKEN_TYPE_NAMESPACE;
						return result;
					}
					break;
				}
				case 'p':
				{
					if (vala_scanner_matches (begin, "protected")) {
						result = VALA_TOKEN_TYPE_PROTECTED;
						return result;
					}
					break;
				}
			}
			break;
		}
		case 11:
		{
			if (vala_scanner_matches (begin, "errordomain")) {
				result = VALA_TOKEN_TYPE_ERRORDOMAIN;
				return result;
			}
			break;
		}
	}
	result = VALA_TOKEN_TYPE_IDENTIFIER;
	return result;
}


static ValaTokenType vala_scanner_read_number (ValaScanner* self) {
	ValaTokenType result;
	ValaTokenType type;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	gboolean _tmp5_;
	gboolean _tmp6_;
	gboolean _tmp8_;
	g_return_val_if_fail (self != NULL, 0);
	type = VALA_TOKEN_TYPE_INTEGER_LITERAL;
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	if (self->priv->current < (self->priv->end - 2)) {
		_tmp2_ = self->priv->current[0] == '0';
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		_tmp1_ = self->priv->current[1] == 'x';
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = g_ascii_isxdigit (self->priv->current[2]);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		self->priv->current = self->priv->current + 2;
		while (TRUE) {
			gboolean _tmp3_;
			_tmp3_ = FALSE;
			if (self->priv->current < self->priv->end) {
				_tmp3_ = g_ascii_isxdigit (self->priv->current[0]);
			} else {
				_tmp3_ = FALSE;
			}
			if (!_tmp3_) {
				break;
			}
			self->priv->current++;
		}
	} else {
		while (TRUE) {
			gboolean _tmp4_;
			_tmp4_ = FALSE;
			if (self->priv->current < self->priv->end) {
				_tmp4_ = g_ascii_isdigit (self->priv->current[0]);
			} else {
				_tmp4_ = FALSE;
			}
			if (!_tmp4_) {
				break;
			}
			self->priv->current++;
		}
	}
	_tmp5_ = FALSE;
	_tmp6_ = FALSE;
	if (self->priv->current < (self->priv->end - 1)) {
		_tmp6_ = self->priv->current[0] == '.';
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		_tmp5_ = g_ascii_isdigit (self->priv->current[1]);
	} else {
		_tmp5_ = FALSE;
	}
	if (_tmp5_) {
		type = VALA_TOKEN_TYPE_REAL_LITERAL;
		self->priv->current++;
		while (TRUE) {
			gboolean _tmp7_;
			_tmp7_ = FALSE;
			if (self->priv->current < self->priv->end) {
				_tmp7_ = g_ascii_isdigit (self->priv->current[0]);
			} else {
				_tmp7_ = FALSE;
			}
			if (!_tmp7_) {
				break;
			}
			self->priv->current++;
		}
	}
	_tmp8_ = FALSE;
	if (self->priv->current < self->priv->end) {
		_tmp8_ = g_ascii_tolower (self->priv->current[0]) == 'e';
	} else {
		_tmp8_ = FALSE;
	}
	if (_tmp8_) {
		gboolean _tmp9_;
		type = VALA_TOKEN_TYPE_REAL_LITERAL;
		self->priv->current++;
		_tmp9_ = FALSE;
		if (self->priv->current < self->priv->end) {
			gboolean _tmp10_;
			_tmp10_ = FALSE;
			if (self->priv->current[0] == '+') {
				_tmp10_ = TRUE;
			} else {
				_tmp10_ = self->priv->current[0] == '-';
			}
			_tmp9_ = _tmp10_;
		} else {
			_tmp9_ = FALSE;
		}
		if (_tmp9_) {
			self->priv->current++;
		}
		while (TRUE) {
			gboolean _tmp11_;
			_tmp11_ = FALSE;
			if (self->priv->current < self->priv->end) {
				_tmp11_ = g_ascii_isdigit (self->priv->current[0]);
			} else {
				_tmp11_ = FALSE;
			}
			if (!_tmp11_) {
				break;
			}
			self->priv->current++;
		}
	}
	if (self->priv->current < self->priv->end) {
		gboolean real_literal;
		gboolean _tmp15_;
		real_literal = type == VALA_TOKEN_TYPE_REAL_LITERAL;
		switch (self->priv->current[0]) {
			case 'l':
			case 'L':
			{
				if (type == VALA_TOKEN_TYPE_INTEGER_LITERAL) {
					gboolean _tmp12_;
					self->priv->current++;
					_tmp12_ = FALSE;
					if (self->priv->current < self->priv->end) {
						_tmp12_ = g_ascii_tolower (self->priv->current[0]) == 'l';
					} else {
						_tmp12_ = FALSE;
					}
					if (_tmp12_) {
						self->priv->current++;
					}
				}
				break;
			}
			case 'u':
			case 'U':
			{
				if (type == VALA_TOKEN_TYPE_INTEGER_LITERAL) {
					gboolean _tmp13_;
					self->priv->current++;
					_tmp13_ = FALSE;
					if (self->priv->current < self->priv->end) {
						_tmp13_ = g_ascii_tolower (self->priv->current[0]) == 'l';
					} else {
						_tmp13_ = FALSE;
					}
					if (_tmp13_) {
						gboolean _tmp14_;
						self->priv->current++;
						_tmp14_ = FALSE;
						if (self->priv->current < self->priv->end) {
							_tmp14_ = g_ascii_tolower (self->priv->current[0]) == 'l';
						} else {
							_tmp14_ = FALSE;
						}
						if (_tmp14_) {
							self->priv->current++;
						}
					}
				}
				break;
			}
			case 'f':
			case 'F':
			case 'd':
			case 'D':
			{
				type = VALA_TOKEN_TYPE_REAL_LITERAL;
				self->priv->current++;
				break;
			}
		}
		_tmp15_ = FALSE;
		if (!real_literal) {
			_tmp15_ = vala_scanner_is_ident_char (self, self->priv->current[0]);
		} else {
			_tmp15_ = FALSE;
		}
		if (_tmp15_) {
			while (TRUE) {
				gboolean _tmp16_;
				_tmp16_ = FALSE;
				if (self->priv->current < self->priv->end) {
					_tmp16_ = vala_scanner_is_ident_char (self, self->priv->current[0]);
				} else {
					_tmp16_ = FALSE;
				}
				if (!_tmp16_) {
					break;
				}
				self->priv->current++;
			}
			type = VALA_TOKEN_TYPE_IDENTIFIER;
		}
	}
	result = type;
	return result;
}


ValaTokenType vala_scanner_read_token (ValaScanner* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end) {
	ValaTokenType result;
	ValaTokenType type;
	gchar* begin;
	gint token_length_in_chars;
	g_return_val_if_fail (self != NULL, 0);
	vala_scanner_space (self);
	type = 0;
	begin = self->priv->current;
	(*token_begin).pos = begin;
	(*token_begin).line = self->priv->line;
	(*token_begin).column = self->priv->column;
	token_length_in_chars = -1;
	if (self->priv->current >= self->priv->end) {
		type = VALA_TOKEN_TYPE_EOF;
	} else {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (g_ascii_isalpha (self->priv->current[0])) {
			_tmp0_ = TRUE;
		} else {
			_tmp0_ = self->priv->current[0] == '_';
		}
		if (_tmp0_) {
			gint len;
			len = 0;
			while (TRUE) {
				gboolean _tmp1_;
				_tmp1_ = FALSE;
				if (self->priv->current < self->priv->end) {
					_tmp1_ = vala_scanner_is_ident_char (self, self->priv->current[0]);
				} else {
					_tmp1_ = FALSE;
				}
				if (!_tmp1_) {
					break;
				}
				self->priv->current++;
				len++;
			}
			type = vala_scanner_get_identifier_or_keyword (begin, len);
		} else {
			if (self->priv->current[0] == '@') {
				gint len;
				(*token_begin).pos++;
				self->priv->current++;
				len = 0;
				while (TRUE) {
					gboolean _tmp2_;
					_tmp2_ = FALSE;
					if (self->priv->current < self->priv->end) {
						_tmp2_ = vala_scanner_is_ident_char (self, self->priv->current[0]);
					} else {
						_tmp2_ = FALSE;
					}
					if (!_tmp2_) {
						break;
					}
					self->priv->current++;
					len++;
				}
				type = VALA_TOKEN_TYPE_IDENTIFIER;
			} else {
				if (g_ascii_isdigit (self->priv->current[0])) {
					type = vala_scanner_read_number (self);
				} else {
					switch (self->priv->current[0]) {
						case '{':
						{
							type = VALA_TOKEN_TYPE_OPEN_BRACE;
							self->priv->current++;
							break;
						}
						case '}':
						{
							type = VALA_TOKEN_TYPE_CLOSE_BRACE;
							self->priv->current++;
							break;
						}
						case '(':
						{
							type = VALA_TOKEN_TYPE_OPEN_PARENS;
							self->priv->current++;
							break;
						}
						case ')':
						{
							type = VALA_TOKEN_TYPE_CLOSE_PARENS;
							self->priv->current++;
							break;
						}
						case '[':
						{
							type = VALA_TOKEN_TYPE_OPEN_BRACKET;
							self->priv->current++;
							break;
						}
						case ']':
						{
							type = VALA_TOKEN_TYPE_CLOSE_BRACKET;
							self->priv->current++;
							break;
						}
						case '.':
						{
							type = VALA_TOKEN_TYPE_DOT;
							self->priv->current++;
							if (self->priv->current < (self->priv->end - 1)) {
								gboolean _tmp3_;
								_tmp3_ = FALSE;
								if (self->priv->current[0] == '.') {
									_tmp3_ = self->priv->current[1] == '.';
								} else {
									_tmp3_ = FALSE;
								}
								if (_tmp3_) {
									type = VALA_TOKEN_TYPE_ELLIPSIS;
									self->priv->current = self->priv->current + 2;
								}
							}
							break;
						}
						case ':':
						{
							gboolean _tmp4_;
							type = VALA_TOKEN_TYPE_COLON;
							self->priv->current++;
							_tmp4_ = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp4_ = self->priv->current[0] == ':';
							} else {
								_tmp4_ = FALSE;
							}
							if (_tmp4_) {
								type = VALA_TOKEN_TYPE_DOUBLE_COLON;
								self->priv->current++;
							}
							break;
						}
						case ',':
						{
							type = VALA_TOKEN_TYPE_COMMA;
							self->priv->current++;
							break;
						}
						case ';':
						{
							type = VALA_TOKEN_TYPE_SEMICOLON;
							self->priv->current++;
							break;
						}
						case '#':
						{
							type = VALA_TOKEN_TYPE_HASH;
							self->priv->current++;
							break;
						}
						case '?':
						{
							type = VALA_TOKEN_TYPE_INTERR;
							self->priv->current++;
							break;
						}
						case '|':
						{
							type = VALA_TOKEN_TYPE_BITWISE_OR;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_TOKEN_TYPE_ASSIGN_BITWISE_OR;
										self->priv->current++;
										break;
									}
									case '|':
									{
										type = VALA_TOKEN_TYPE_OP_OR;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '&':
						{
							type = VALA_TOKEN_TYPE_BITWISE_AND;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_TOKEN_TYPE_ASSIGN_BITWISE_AND;
										self->priv->current++;
										break;
									}
									case '&':
									{
										type = VALA_TOKEN_TYPE_OP_AND;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '^':
						{
							gboolean _tmp5_;
							type = VALA_TOKEN_TYPE_CARRET;
							self->priv->current++;
							_tmp5_ = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp5_ = self->priv->current[0] == '=';
							} else {
								_tmp5_ = FALSE;
							}
							if (_tmp5_) {
								type = VALA_TOKEN_TYPE_ASSIGN_BITWISE_XOR;
								self->priv->current++;
							}
							break;
						}
						case '~':
						{
							type = VALA_TOKEN_TYPE_TILDE;
							self->priv->current++;
							break;
						}
						case '=':
						{
							type = VALA_TOKEN_TYPE_ASSIGN;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_TOKEN_TYPE_OP_EQ;
										self->priv->current++;
										break;
									}
									case '>':
									{
										type = VALA_TOKEN_TYPE_LAMBDA;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '<':
						{
							type = VALA_TOKEN_TYPE_OP_LT;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_TOKEN_TYPE_OP_LE;
										self->priv->current++;
										break;
									}
									case '<':
									{
										gboolean _tmp6_;
										type = VALA_TOKEN_TYPE_OP_SHIFT_LEFT;
										self->priv->current++;
										_tmp6_ = FALSE;
										if (self->priv->current < self->priv->end) {
											_tmp6_ = self->priv->current[0] == '=';
										} else {
											_tmp6_ = FALSE;
										}
										if (_tmp6_) {
											type = VALA_TOKEN_TYPE_ASSIGN_SHIFT_LEFT;
											self->priv->current++;
										}
										break;
									}
								}
							}
							break;
						}
						case '>':
						{
							gboolean _tmp7_;
							type = VALA_TOKEN_TYPE_OP_GT;
							self->priv->current++;
							_tmp7_ = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp7_ = self->priv->current[0] == '=';
							} else {
								_tmp7_ = FALSE;
							}
							if (_tmp7_) {
								type = VALA_TOKEN_TYPE_OP_GE;
								self->priv->current++;
							}
							break;
						}
						case '!':
						{
							gboolean _tmp8_;
							type = VALA_TOKEN_TYPE_OP_NEG;
							self->priv->current++;
							_tmp8_ = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp8_ = self->priv->current[0] == '=';
							} else {
								_tmp8_ = FALSE;
							}
							if (_tmp8_) {
								type = VALA_TOKEN_TYPE_OP_NE;
								self->priv->current++;
							}
							break;
						}
						case '+':
						{
							type = VALA_TOKEN_TYPE_PLUS;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_TOKEN_TYPE_ASSIGN_ADD;
										self->priv->current++;
										break;
									}
									case '+':
									{
										type = VALA_TOKEN_TYPE_OP_INC;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '-':
						{
							type = VALA_TOKEN_TYPE_MINUS;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_TOKEN_TYPE_ASSIGN_SUB;
										self->priv->current++;
										break;
									}
									case '-':
									{
										type = VALA_TOKEN_TYPE_OP_DEC;
										self->priv->current++;
										break;
									}
									case '>':
									{
										type = VALA_TOKEN_TYPE_OP_PTR;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '*':
						{
							gboolean _tmp9_;
							type = VALA_TOKEN_TYPE_STAR;
							self->priv->current++;
							_tmp9_ = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp9_ = self->priv->current[0] == '=';
							} else {
								_tmp9_ = FALSE;
							}
							if (_tmp9_) {
								type = VALA_TOKEN_TYPE_ASSIGN_MUL;
								self->priv->current++;
							}
							break;
						}
						case '/':
						{
							gboolean _tmp10_;
							type = VALA_TOKEN_TYPE_DIV;
							self->priv->current++;
							_tmp10_ = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp10_ = self->priv->current[0] == '=';
							} else {
								_tmp10_ = FALSE;
							}
							if (_tmp10_) {
								type = VALA_TOKEN_TYPE_ASSIGN_DIV;
								self->priv->current++;
							}
							break;
						}
						case '%':
						{
							gboolean _tmp11_;
							type = VALA_TOKEN_TYPE_PERCENT;
							self->priv->current++;
							_tmp11_ = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp11_ = self->priv->current[0] == '=';
							} else {
								_tmp11_ = FALSE;
							}
							if (_tmp11_) {
								type = VALA_TOKEN_TYPE_ASSIGN_PERCENT;
								self->priv->current++;
							}
							break;
						}
						case '\'':
						case '"':
						{
							gboolean _tmp24_;
							if (begin[0] == '\'') {
								type = VALA_TOKEN_TYPE_CHARACTER_LITERAL;
							} else {
								gboolean _tmp12_;
								gboolean _tmp13_;
								_tmp12_ = FALSE;
								_tmp13_ = FALSE;
								if (self->priv->current < (self->priv->end - 6)) {
									_tmp13_ = begin[1] == '"';
								} else {
									_tmp13_ = FALSE;
								}
								if (_tmp13_) {
									_tmp12_ = begin[2] == '"';
								} else {
									_tmp12_ = FALSE;
								}
								if (_tmp12_) {
									gboolean _tmp17_;
									gboolean _tmp18_;
									type = VALA_TOKEN_TYPE_VERBATIM_STRING_LITERAL;
									token_length_in_chars = 6;
									self->priv->current = self->priv->current + 3;
									while (TRUE) {
										gboolean _tmp14_;
										gboolean _tmp15_;
										if (!(self->priv->current < (self->priv->end - 4))) {
											break;
										}
										_tmp14_ = FALSE;
										_tmp15_ = FALSE;
										if (self->priv->current[0] == '"') {
											_tmp15_ = self->priv->current[1] == '"';
										} else {
											_tmp15_ = FALSE;
										}
										if (_tmp15_) {
											_tmp14_ = self->priv->current[2] == '"';
										} else {
											_tmp14_ = FALSE;
										}
										if (_tmp14_) {
											break;
										} else {
											if (self->priv->current[0] == '\n') {
												self->priv->current++;
												self->priv->line++;
												self->priv->column = 1;
												token_length_in_chars = 3;
											} else {
												gunichar u;
												u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
												if (u != ((gunichar) (-1))) {
													self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
													token_length_in_chars++;
												} else {
													ValaSourceReference* _tmp16_;
													vala_report_error (_tmp16_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars), "invalid UTF-8 character");
													_vala_source_reference_unref0 (_tmp16_);
												}
											}
										}
									}
									_tmp17_ = FALSE;
									_tmp18_ = FALSE;
									if (self->priv->current[0] == '"') {
										_tmp18_ = self->priv->current[1] == '"';
									} else {
										_tmp18_ = FALSE;
									}
									if (_tmp18_) {
										_tmp17_ = self->priv->current[2] == '"';
									} else {
										_tmp17_ = FALSE;
									}
									if (_tmp17_) {
										self->priv->current = self->priv->current + 3;
									} else {
										ValaSourceReference* _tmp19_;
										vala_report_error (_tmp19_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars), "syntax error, expected \"\"\"");
										_vala_source_reference_unref0 (_tmp19_);
									}
									break;
								} else {
									type = VALA_TOKEN_TYPE_STRING_LITERAL;
								}
							}
							token_length_in_chars = 2;
							self->priv->current++;
							while (TRUE) {
								gboolean _tmp20_;
								_tmp20_ = FALSE;
								if (self->priv->current < self->priv->end) {
									_tmp20_ = self->priv->current[0] != begin[0];
								} else {
									_tmp20_ = FALSE;
								}
								if (!_tmp20_) {
									break;
								}
								if (self->priv->current[0] == '\\') {
									self->priv->current++;
									token_length_in_chars++;
									if (self->priv->current >= self->priv->end) {
										break;
									}
									switch (self->priv->current[0]) {
										case '\'':
										case '"':
										case '\\':
										case '0':
										case 'b':
										case 'f':
										case 'n':
										case 'r':
										case 't':
										{
											self->priv->current++;
											token_length_in_chars++;
											break;
										}
										case 'x':
										{
											self->priv->current++;
											token_length_in_chars++;
											while (TRUE) {
												gboolean _tmp21_;
												_tmp21_ = FALSE;
												if (self->priv->current < self->priv->end) {
													_tmp21_ = g_ascii_isxdigit (self->priv->current[0]);
												} else {
													_tmp21_ = FALSE;
												}
												if (!_tmp21_) {
													break;
												}
												self->priv->current++;
												token_length_in_chars++;
											}
											break;
										}
										default:
										{
											ValaSourceReference* _tmp22_;
											vala_report_error (_tmp22_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars), "invalid escape sequence");
											_vala_source_reference_unref0 (_tmp22_);
											break;
										}
									}
								} else {
									if (self->priv->current[0] == '\n') {
										break;
									} else {
										gunichar u;
										u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
										if (u != ((gunichar) (-1))) {
											self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
											token_length_in_chars++;
										} else {
											ValaSourceReference* _tmp23_;
											self->priv->current++;
											vala_report_error (_tmp23_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars), "invalid UTF-8 character");
											_vala_source_reference_unref0 (_tmp23_);
										}
									}
								}
							}
							_tmp24_ = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp24_ = self->priv->current[0] != '\n';
							} else {
								_tmp24_ = FALSE;
							}
							if (_tmp24_) {
								self->priv->current++;
							} else {
								char* _tmp26_;
								ValaSourceReference* _tmp25_;
								vala_report_error (_tmp25_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars), _tmp26_ = g_strdup_printf ("syntax error, expected %c", (gint) begin[0]));
								_g_free0 (_tmp26_);
								_vala_source_reference_unref0 (_tmp25_);
							}
							break;
						}
						default:
						{
							gunichar u;
							u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
							if (u != ((gunichar) (-1))) {
								ValaSourceReference* _tmp27_;
								self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
								vala_report_error (_tmp27_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, unexpected character");
								_vala_source_reference_unref0 (_tmp27_);
							} else {
								ValaSourceReference* _tmp28_;
								self->priv->current++;
								vala_report_error (_tmp28_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "invalid UTF-8 character");
								_vala_source_reference_unref0 (_tmp28_);
							}
							self->priv->column++;
							result = vala_scanner_read_token (self, &(*token_begin), &(*token_end));
							return result;
						}
					}
				}
			}
		}
	}
	if (token_length_in_chars < 0) {
		self->priv->column = self->priv->column + ((gint) (self->priv->current - begin));
	} else {
		self->priv->column = self->priv->column + token_length_in_chars;
	}
	(*token_end).pos = self->priv->current;
	(*token_end).line = self->priv->line;
	(*token_end).column = self->priv->column - 1;
	result = type;
	return result;
}


static gboolean vala_scanner_matches (gchar* begin, const char* keyword) {
	gboolean result;
	gchar* keyword_array;
	glong len;
	g_return_val_if_fail (keyword != NULL, FALSE);
	keyword_array = keyword;
	len = g_utf8_strlen (keyword, -1);
	{
		gint i;
		i = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < len)) {
					break;
				}
				if (begin[i] != keyword_array[i]) {
					result = FALSE;
					return result;
				}
			}
		}
	}
	result = TRUE;
	return result;
}


static gboolean vala_scanner_pp_whitespace (ValaScanner* self) {
	gboolean result;
	gboolean found;
	g_return_val_if_fail (self != NULL, FALSE);
	found = FALSE;
	while (TRUE) {
		gboolean _tmp0_;
		gboolean _tmp1_;
		_tmp0_ = FALSE;
		_tmp1_ = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp1_ = g_ascii_isspace (self->priv->current[0]);
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			_tmp0_ = self->priv->current[0] != '\n';
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		found = TRUE;
		self->priv->current++;
		self->priv->column++;
	}
	result = found;
	return result;
}


static void vala_scanner_pp_directive (ValaScanner* self) {
	gchar* begin;
	gint len;
	gboolean _tmp1_;
	gboolean _tmp6_;
	g_return_if_fail (self != NULL);
	self->priv->current++;
	self->priv->column++;
	vala_scanner_pp_whitespace (self);
	begin = self->priv->current;
	len = 0;
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp0_ = g_ascii_isalnum (self->priv->current[0]);
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->current++;
		self->priv->column++;
		len++;
	}
	_tmp1_ = FALSE;
	if (len == 2) {
		_tmp1_ = vala_scanner_matches (begin, "if");
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		vala_scanner_parse_pp_if (self);
	} else {
		gboolean _tmp2_;
		_tmp2_ = FALSE;
		if (len == 4) {
			_tmp2_ = vala_scanner_matches (begin, "elif");
		} else {
			_tmp2_ = FALSE;
		}
		if (_tmp2_) {
			vala_scanner_parse_pp_elif (self);
		} else {
			gboolean _tmp3_;
			_tmp3_ = FALSE;
			if (len == 4) {
				_tmp3_ = vala_scanner_matches (begin, "else");
			} else {
				_tmp3_ = FALSE;
			}
			if (_tmp3_) {
				vala_scanner_parse_pp_else (self);
			} else {
				gboolean _tmp4_;
				_tmp4_ = FALSE;
				if (len == 5) {
					_tmp4_ = vala_scanner_matches (begin, "endif");
				} else {
					_tmp4_ = FALSE;
				}
				if (_tmp4_) {
					vala_scanner_parse_pp_endif (self);
				} else {
					ValaSourceReference* _tmp5_;
					vala_report_error (_tmp5_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column - len, self->priv->line, self->priv->column), "syntax error, invalid preprocessing directive");
					_vala_source_reference_unref0 (_tmp5_);
				}
			}
		}
	}
	_tmp6_ = FALSE;
	if (self->priv->conditional_stack_length1 > 0) {
		_tmp6_ = self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].skip_section;
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		gboolean bol;
		bol = FALSE;
		while (TRUE) {
			gboolean _tmp7_;
			if (!(self->priv->current < self->priv->end)) {
				break;
			}
			_tmp7_ = FALSE;
			if (bol) {
				_tmp7_ = self->priv->current[0] == '#';
			} else {
				_tmp7_ = FALSE;
			}
			if (_tmp7_) {
				self->priv->current = self->priv->current - (self->priv->column - 1);
				self->priv->column = 1;
				return;
			}
			if (self->priv->current[0] == '\n') {
				self->priv->line++;
				self->priv->column = 0;
				bol = TRUE;
			} else {
				if (!g_ascii_isspace (self->priv->current[0])) {
					bol = FALSE;
				}
			}
			self->priv->current++;
			self->priv->column++;
		}
	}
}


static void vala_scanner_pp_eol (ValaScanner* self) {
	gboolean _tmp0_;
	g_return_if_fail (self != NULL);
	vala_scanner_pp_whitespace (self);
	_tmp0_ = FALSE;
	if (self->priv->current >= self->priv->end) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = self->priv->current[0] != '\n';
	}
	if (_tmp0_) {
		ValaSourceReference* _tmp1_;
		vala_report_error (_tmp1_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, expected newline");
		_vala_source_reference_unref0 (_tmp1_);
	}
}


static void _vala_array_add2 (ValaScannerConditional** array, int* length, int* size, const ValaScannerConditional* value) {
	if ((*length) == (*size)) {
		*size = (*size) ? (2 * (*size)) : 4;
		*array = g_renew (ValaScannerConditional, *array, *size);
	}
	(*array)[(*length)++] = *value;
}


static void vala_scanner_parse_pp_if (ValaScanner* self) {
	gboolean condition;
	ValaScannerConditional _tmp1_;
	ValaScannerConditional _tmp0_ = {0};
	gboolean _tmp2_;
	g_return_if_fail (self != NULL);
	vala_scanner_pp_whitespace (self);
	condition = vala_scanner_parse_pp_expression (self);
	vala_scanner_pp_eol (self);
	_vala_array_add2 (&self->priv->conditional_stack, &self->priv->conditional_stack_length1, &self->priv->conditional_stack_size, (_tmp1_ = (memset (&_tmp0_, 0, sizeof (ValaScannerConditional)), _tmp0_), &_tmp1_));
	_tmp2_ = FALSE;
	if (condition) {
		gboolean _tmp3_;
		_tmp3_ = FALSE;
		if (self->priv->conditional_stack_length1 == 1) {
			_tmp3_ = TRUE;
		} else {
			_tmp3_ = !self->priv->conditional_stack[self->priv->conditional_stack_length1 - 2].skip_section;
		}
		_tmp2_ = _tmp3_;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].matched = TRUE;
	} else {
		self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].skip_section = TRUE;
	}
}


static void vala_scanner_parse_pp_elif (ValaScanner* self) {
	gboolean condition;
	gboolean _tmp0_;
	gboolean _tmp2_;
	gboolean _tmp3_;
	g_return_if_fail (self != NULL);
	vala_scanner_pp_whitespace (self);
	condition = vala_scanner_parse_pp_expression (self);
	vala_scanner_pp_eol (self);
	_tmp0_ = FALSE;
	if (self->priv->conditional_stack_length1 == 0) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].else_found;
	}
	if (_tmp0_) {
		ValaSourceReference* _tmp1_;
		vala_report_error (_tmp1_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, unexpected #elif");
		_vala_source_reference_unref0 (_tmp1_);
		return;
	}
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	if (condition) {
		_tmp3_ = !self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].matched;
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		gboolean _tmp4_;
		_tmp4_ = FALSE;
		if (self->priv->conditional_stack_length1 == 1) {
			_tmp4_ = TRUE;
		} else {
			_tmp4_ = !self->priv->conditional_stack[self->priv->conditional_stack_length1 - 2].skip_section;
		}
		_tmp2_ = _tmp4_;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].matched = TRUE;
		self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].skip_section = FALSE;
	} else {
		self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].skip_section = TRUE;
	}
}


static void vala_scanner_parse_pp_else (ValaScanner* self) {
	gboolean _tmp0_;
	gboolean _tmp2_;
	g_return_if_fail (self != NULL);
	vala_scanner_pp_eol (self);
	_tmp0_ = FALSE;
	if (self->priv->conditional_stack_length1 == 0) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].else_found;
	}
	if (_tmp0_) {
		ValaSourceReference* _tmp1_;
		vala_report_error (_tmp1_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, unexpected #else");
		_vala_source_reference_unref0 (_tmp1_);
		return;
	}
	_tmp2_ = FALSE;
	if (!self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].matched) {
		gboolean _tmp3_;
		_tmp3_ = FALSE;
		if (self->priv->conditional_stack_length1 == 1) {
			_tmp3_ = TRUE;
		} else {
			_tmp3_ = !self->priv->conditional_stack[self->priv->conditional_stack_length1 - 2].skip_section;
		}
		_tmp2_ = _tmp3_;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].matched = TRUE;
		self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].skip_section = FALSE;
	} else {
		self->priv->conditional_stack[self->priv->conditional_stack_length1 - 1].skip_section = TRUE;
	}
}


static void vala_scanner_parse_pp_endif (ValaScanner* self) {
	g_return_if_fail (self != NULL);
	vala_scanner_pp_eol (self);
	if (self->priv->conditional_stack_length1 == 0) {
		ValaSourceReference* _tmp0_;
		vala_report_error (_tmp0_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, unexpected #endif");
		_vala_source_reference_unref0 (_tmp0_);
		return;
	}
	self->priv->conditional_stack_length1--;
}


static gboolean vala_scanner_parse_pp_symbol (ValaScanner* self) {
	gboolean result;
	gint len;
	char* identifier;
	gboolean defined;
	g_return_val_if_fail (self != NULL, FALSE);
	len = 0;
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp0_ = vala_scanner_is_ident_char (self, self->priv->current[0]);
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->current++;
		self->priv->column++;
		len++;
	}
	if (len == 0) {
		ValaSourceReference* _tmp1_;
		vala_report_error (_tmp1_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, expected identifier");
		_vala_source_reference_unref0 (_tmp1_);
		result = FALSE;
		return result;
	}
	identifier = g_strndup ((const char*) (self->priv->current - len), (gsize) len);
	defined = FALSE;
	if (_vala_strcmp0 (identifier, "true") == 0) {
		defined = TRUE;
	} else {
		if (_vala_strcmp0 (identifier, "false") == 0) {
			defined = FALSE;
		} else {
			defined = vala_code_context_is_defined (vala_source_file_get_context (self->priv->_source_file), identifier);
		}
	}
	result = defined;
	_g_free0 (identifier);
	return result;
}


static gboolean vala_scanner_parse_pp_primary_expression (ValaScanner* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	if (self->priv->current >= self->priv->end) {
		ValaSourceReference* _tmp0_;
		vala_report_error (_tmp0_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, expected identifier");
		_vala_source_reference_unref0 (_tmp0_);
	} else {
		if (vala_scanner_is_ident_char (self, self->priv->current[0])) {
			result = vala_scanner_parse_pp_symbol (self);
			return result;
		} else {
			if (self->priv->current[0] == '(') {
				gboolean _result_;
				gboolean _tmp1_;
				self->priv->current++;
				self->priv->column++;
				vala_scanner_pp_whitespace (self);
				_result_ = vala_scanner_parse_pp_expression (self);
				vala_scanner_pp_whitespace (self);
				_tmp1_ = FALSE;
				if (self->priv->current < self->priv->end) {
					_tmp1_ = self->priv->current[0] == ')';
				} else {
					_tmp1_ = FALSE;
				}
				if (_tmp1_) {
					self->priv->current++;
					self->priv->column++;
				} else {
					ValaSourceReference* _tmp2_;
					vala_report_error (_tmp2_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, expected `)'");
					_vala_source_reference_unref0 (_tmp2_);
				}
				result = _result_;
				return result;
			} else {
				ValaSourceReference* _tmp3_;
				vala_report_error (_tmp3_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, expected identifier");
				_vala_source_reference_unref0 (_tmp3_);
			}
		}
	}
	result = FALSE;
	return result;
}


static gboolean vala_scanner_parse_pp_unary_expression (ValaScanner* self) {
	gboolean result;
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0_ = FALSE;
	if (self->priv->current < self->priv->end) {
		_tmp0_ = self->priv->current[0] == '!';
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		self->priv->current++;
		self->priv->column++;
		vala_scanner_pp_whitespace (self);
		result = !vala_scanner_parse_pp_unary_expression (self);
		return result;
	}
	result = vala_scanner_parse_pp_primary_expression (self);
	return result;
}


static gboolean vala_scanner_parse_pp_equality_expression (ValaScanner* self) {
	gboolean result;
	gboolean left;
	g_return_val_if_fail (self != NULL, FALSE);
	left = vala_scanner_parse_pp_unary_expression (self);
	vala_scanner_pp_whitespace (self);
	while (TRUE) {
		gboolean _tmp0_;
		gboolean _tmp1_;
		_tmp0_ = FALSE;
		_tmp1_ = FALSE;
		if (self->priv->current < (self->priv->end - 1)) {
			_tmp1_ = self->priv->current[0] == '=';
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			_tmp0_ = self->priv->current[1] == '=';
		} else {
			_tmp0_ = FALSE;
		}
		if (_tmp0_) {
			gboolean right;
			self->priv->current = self->priv->current + 2;
			self->priv->column = self->priv->column + 2;
			vala_scanner_pp_whitespace (self);
			right = vala_scanner_parse_pp_unary_expression (self);
			left = left == right;
		} else {
			gboolean _tmp2_;
			gboolean _tmp3_;
			_tmp2_ = FALSE;
			_tmp3_ = FALSE;
			if (self->priv->current < (self->priv->end - 1)) {
				_tmp3_ = self->priv->current[0] == '!';
			} else {
				_tmp3_ = FALSE;
			}
			if (_tmp3_) {
				_tmp2_ = self->priv->current[1] == '=';
			} else {
				_tmp2_ = FALSE;
			}
			if (_tmp2_) {
				gboolean right;
				self->priv->current = self->priv->current + 2;
				self->priv->column = self->priv->column + 2;
				vala_scanner_pp_whitespace (self);
				right = vala_scanner_parse_pp_unary_expression (self);
				left = left != right;
			} else {
				break;
			}
		}
	}
	result = left;
	return result;
}


static gboolean vala_scanner_parse_pp_and_expression (ValaScanner* self) {
	gboolean result;
	gboolean left;
	g_return_val_if_fail (self != NULL, FALSE);
	left = vala_scanner_parse_pp_equality_expression (self);
	vala_scanner_pp_whitespace (self);
	while (TRUE) {
		gboolean _tmp0_;
		gboolean _tmp1_;
		gboolean right;
		gboolean _tmp2_;
		_tmp0_ = FALSE;
		_tmp1_ = FALSE;
		if (self->priv->current < (self->priv->end - 1)) {
			_tmp1_ = self->priv->current[0] == '&';
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			_tmp0_ = self->priv->current[1] == '&';
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->current = self->priv->current + 2;
		self->priv->column = self->priv->column + 2;
		vala_scanner_pp_whitespace (self);
		right = vala_scanner_parse_pp_equality_expression (self);
		_tmp2_ = FALSE;
		if (left) {
			_tmp2_ = right;
		} else {
			_tmp2_ = FALSE;
		}
		left = _tmp2_;
	}
	result = left;
	return result;
}


static gboolean vala_scanner_parse_pp_or_expression (ValaScanner* self) {
	gboolean result;
	gboolean left;
	g_return_val_if_fail (self != NULL, FALSE);
	left = vala_scanner_parse_pp_and_expression (self);
	vala_scanner_pp_whitespace (self);
	while (TRUE) {
		gboolean _tmp0_;
		gboolean _tmp1_;
		gboolean right;
		gboolean _tmp2_;
		_tmp0_ = FALSE;
		_tmp1_ = FALSE;
		if (self->priv->current < (self->priv->end - 1)) {
			_tmp1_ = self->priv->current[0] == '|';
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			_tmp0_ = self->priv->current[1] == '|';
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->current = self->priv->current + 2;
		self->priv->column = self->priv->column + 2;
		vala_scanner_pp_whitespace (self);
		right = vala_scanner_parse_pp_and_expression (self);
		_tmp2_ = FALSE;
		if (left) {
			_tmp2_ = TRUE;
		} else {
			_tmp2_ = right;
		}
		left = _tmp2_;
	}
	result = left;
	return result;
}


static gboolean vala_scanner_parse_pp_expression (ValaScanner* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = vala_scanner_parse_pp_or_expression (self);
	return result;
}


static gboolean vala_scanner_whitespace (ValaScanner* self) {
	gboolean result;
	gboolean found;
	gboolean bol;
	gboolean _tmp1_;
	gboolean _tmp2_;
	g_return_val_if_fail (self != NULL, FALSE);
	found = FALSE;
	bol = self->priv->column == 1;
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp0_ = g_ascii_isspace (self->priv->current[0]);
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		if (self->priv->current[0] == '\n') {
			self->priv->line++;
			self->priv->column = 0;
			bol = TRUE;
		}
		found = TRUE;
		self->priv->current++;
		self->priv->column++;
	}
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	if (bol) {
		_tmp2_ = self->priv->current < self->priv->end;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		_tmp1_ = self->priv->current[0] == '#';
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		vala_scanner_pp_directive (self);
		result = TRUE;
		return result;
	}
	result = found;
	return result;
}


static gboolean vala_scanner_comment (ValaScanner* self, gboolean file_comment) {
	gboolean result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (self->priv->current > (self->priv->end - 2)) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = self->priv->current[0] != '/';
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		gboolean _tmp2_;
		_tmp2_ = FALSE;
		if (self->priv->current[1] != '/') {
			_tmp2_ = self->priv->current[1] != '*';
		} else {
			_tmp2_ = FALSE;
		}
		_tmp0_ = _tmp2_;
	}
	if (_tmp0_) {
		result = FALSE;
		return result;
	}
	if (self->priv->current[1] == '/') {
		ValaSourceReference* source_reference;
		gchar* begin;
		source_reference = NULL;
		if (file_comment) {
			ValaSourceReference* _tmp3_;
			source_reference = (_tmp3_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), _vala_source_reference_unref0 (source_reference), _tmp3_);
		}
		self->priv->current = self->priv->current + 2;
		begin = self->priv->current;
		while (TRUE) {
			gboolean _tmp4_;
			_tmp4_ = FALSE;
			if (self->priv->current < self->priv->end) {
				_tmp4_ = self->priv->current[0] != '\n';
			} else {
				_tmp4_ = FALSE;
			}
			if (!_tmp4_) {
				break;
			}
			self->priv->current++;
		}
		if (source_reference != NULL) {
			char* _tmp5_;
			vala_scanner_push_comment (self, _tmp5_ = g_strndup ((const char*) begin, (gsize) ((glong) (self->priv->current - begin))), source_reference, file_comment);
			_g_free0 (_tmp5_);
		}
		_vala_source_reference_unref0 (source_reference);
	} else {
		ValaSourceReference* source_reference;
		gboolean _tmp6_;
		gboolean _tmp7_;
		gchar* begin;
		source_reference = NULL;
		_tmp6_ = FALSE;
		if (file_comment) {
			_tmp6_ = self->priv->current[2] == '*';
		} else {
			_tmp6_ = FALSE;
		}
		if (_tmp6_) {
			result = FALSE;
			_vala_source_reference_unref0 (source_reference);
			return result;
		}
		_tmp7_ = FALSE;
		if (self->priv->current[2] == '*') {
			_tmp7_ = TRUE;
		} else {
			_tmp7_ = file_comment;
		}
		if (_tmp7_) {
			ValaSourceReference* _tmp8_;
			source_reference = (_tmp8_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), _vala_source_reference_unref0 (source_reference), _tmp8_);
		}
		self->priv->current = self->priv->current + 2;
		begin = self->priv->current;
		while (TRUE) {
			gboolean _tmp9_;
			_tmp9_ = FALSE;
			if (self->priv->current < (self->priv->end - 1)) {
				gboolean _tmp10_;
				_tmp10_ = FALSE;
				if (self->priv->current[0] != '*') {
					_tmp10_ = TRUE;
				} else {
					_tmp10_ = self->priv->current[1] != '/';
				}
				_tmp9_ = _tmp10_;
			} else {
				_tmp9_ = FALSE;
			}
			if (!_tmp9_) {
				break;
			}
			if (self->priv->current[0] == '\n') {
				self->priv->line++;
				self->priv->column = 0;
			}
			self->priv->current++;
			self->priv->column++;
		}
		if (self->priv->current == (self->priv->end - 1)) {
			ValaSourceReference* _tmp11_;
			vala_report_error (_tmp11_ = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, expected */");
			_vala_source_reference_unref0 (_tmp11_);
			result = TRUE;
			_vala_source_reference_unref0 (source_reference);
			return result;
		}
		if (source_reference != NULL) {
			char* _tmp12_;
			vala_scanner_push_comment (self, _tmp12_ = g_strndup ((const char*) begin, (gsize) ((glong) (self->priv->current - begin))), source_reference, file_comment);
			_g_free0 (_tmp12_);
		}
		self->priv->current = self->priv->current + 2;
		self->priv->column = self->priv->column + 2;
		_vala_source_reference_unref0 (source_reference);
	}
	result = TRUE;
	return result;
}


static void vala_scanner_space (ValaScanner* self) {
	g_return_if_fail (self != NULL);
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (vala_scanner_whitespace (self)) {
			_tmp0_ = TRUE;
		} else {
			_tmp0_ = vala_scanner_comment (self, FALSE);
		}
		if (!_tmp0_) {
			break;
		}
	}
}


void vala_scanner_parse_file_comments (ValaScanner* self) {
	g_return_if_fail (self != NULL);
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (vala_scanner_whitespace (self)) {
			_tmp0_ = TRUE;
		} else {
			_tmp0_ = vala_scanner_comment (self, TRUE);
		}
		if (!_tmp0_) {
			break;
		}
	}
}


static void vala_scanner_push_comment (ValaScanner* self, const char* comment_item, ValaSourceReference* source_reference, gboolean file_comment) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (comment_item != NULL);
	g_return_if_fail (source_reference != NULL);
	if (g_utf8_get_char (g_utf8_offset_to_pointer (comment_item, 0)) == '*') {
		ValaComment* _tmp0_;
		self->priv->_comment = (_tmp0_ = vala_comment_new (comment_item, source_reference), _vala_comment_unref0 (self->priv->_comment), _tmp0_);
	}
	if (file_comment) {
		ValaComment* _tmp1_;
		ValaComment* _tmp2_;
		vala_source_file_add_comment (self->priv->_source_file, _tmp1_ = vala_comment_new (comment_item, source_reference));
		_vala_comment_unref0 (_tmp1_);
		self->priv->_comment = (_tmp2_ = NULL, _vala_comment_unref0 (self->priv->_comment), _tmp2_);
	}
}


static gpointer _vala_comment_ref0 (gpointer self) {
	return self ? vala_comment_ref (self) : NULL;
}


ValaComment* vala_scanner_pop_comment (ValaScanner* self) {
	ValaComment* result;
	ValaComment* comment;
	ValaComment* _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_comment == NULL) {
		result = NULL;
		return result;
	}
	comment = _vala_comment_ref0 (self->priv->_comment);
	self->priv->_comment = (_tmp0_ = NULL, _vala_comment_unref0 (self->priv->_comment), _tmp0_);
	result = comment;
	return result;
}


ValaSourceFile* vala_scanner_get_source_file (ValaScanner* self) {
	ValaSourceFile* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_source_file;
	return result;
}


static gpointer _vala_source_file_ref0 (gpointer self) {
	return self ? vala_source_file_ref (self) : NULL;
}


static void vala_scanner_set_source_file (ValaScanner* self, ValaSourceFile* value) {
	ValaSourceFile* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_source_file = (_tmp0_ = _vala_source_file_ref0 (value), _vala_source_file_unref0 (self->priv->_source_file), _tmp0_);
}


static ValaScannerConditional* vala_scanner_conditional_dup (const ValaScannerConditional* self) {
	ValaScannerConditional* dup;
	dup = g_new0 (ValaScannerConditional, 1);
	memcpy (dup, self, sizeof (ValaScannerConditional));
	return dup;
}


static void vala_scanner_conditional_free (ValaScannerConditional* self) {
	g_free (self);
}


static GType vala_scanner_conditional_get_type (void) {
	static GType vala_scanner_conditional_type_id = 0;
	if (vala_scanner_conditional_type_id == 0) {
		vala_scanner_conditional_type_id = g_boxed_type_register_static ("ValaScannerConditional", (GBoxedCopyFunc) vala_scanner_conditional_dup, (GBoxedFreeFunc) vala_scanner_conditional_free);
	}
	return vala_scanner_conditional_type_id;
}


static void vala_value_scanner_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_scanner_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_scanner_unref (value->data[0].v_pointer);
	}
}


static void vala_value_scanner_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_scanner_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_scanner_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_scanner_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaScanner* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_scanner_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_scanner_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaScanner** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_scanner_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_scanner (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecScanner* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_SCANNER), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_scanner (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SCANNER), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_scanner (GValue* value, gpointer v_object) {
	ValaScanner* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SCANNER));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_SCANNER));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_scanner_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_scanner_unref (old);
	}
}


static void vala_scanner_class_init (ValaScannerClass * klass) {
	vala_scanner_parent_class = g_type_class_peek_parent (klass);
	VALA_SCANNER_CLASS (klass)->finalize = vala_scanner_finalize;
	g_type_class_add_private (klass, sizeof (ValaScannerPrivate));
}


static void vala_scanner_instance_init (ValaScanner * self) {
	self->priv = VALA_SCANNER_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_scanner_finalize (ValaScanner* obj) {
	ValaScanner * self;
	self = VALA_SCANNER (obj);
	_vala_source_file_unref0 (self->priv->_source_file);
	_vala_comment_unref0 (self->priv->_comment);
	self->priv->conditional_stack = (g_free (self->priv->conditional_stack), NULL);
}


GType vala_scanner_get_type (void) {
	static GType vala_scanner_type_id = 0;
	if (vala_scanner_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_scanner_init, vala_value_scanner_free_value, vala_value_scanner_copy_value, vala_value_scanner_peek_pointer, "p", vala_value_scanner_collect_value, "p", vala_value_scanner_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaScannerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_scanner_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaScanner), 0, (GInstanceInitFunc) vala_scanner_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_scanner_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaScanner", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_scanner_type_id;
}


gpointer vala_scanner_ref (gpointer instance) {
	ValaScanner* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_scanner_unref (gpointer instance) {
	ValaScanner* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_SCANNER_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




