/* valavapigen.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gstdio.h>
#include <gee.h>
#include <config.h>


#define VALA_TYPE_VAPI_GEN (vala_vapi_gen_get_type ())
#define VALA_VAPI_GEN(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_VAPI_GEN, ValaVAPIGen))
#define VALA_VAPI_GEN_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_VAPI_GEN, ValaVAPIGenClass))
#define VALA_IS_VAPI_GEN(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_VAPI_GEN))
#define VALA_IS_VAPI_GEN_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_VAPI_GEN))
#define VALA_VAPI_GEN_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_VAPI_GEN, ValaVAPIGenClass))

typedef struct _ValaVAPIGen ValaVAPIGen;
typedef struct _ValaVAPIGenClass ValaVAPIGenClass;
typedef struct _ValaVAPIGenPrivate ValaVAPIGenPrivate;

#define VALA_TYPE_GIR_PARSER (vala_gir_parser_get_type ())
#define VALA_GIR_PARSER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GIR_PARSER, ValaGirParser))
#define VALA_GIR_PARSER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GIR_PARSER, ValaGirParserClass))
#define VALA_IS_GIR_PARSER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GIR_PARSER))
#define VALA_IS_GIR_PARSER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GIR_PARSER))
#define VALA_GIR_PARSER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GIR_PARSER, ValaGirParserClass))

typedef struct _ValaGirParser ValaGirParser;
typedef struct _ValaGirParserClass ValaGirParserClass;

#define VALA_TYPE_GIDL_PARSER (vala_gidl_parser_get_type ())
#define VALA_GIDL_PARSER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GIDL_PARSER, ValaGIdlParser))
#define VALA_GIDL_PARSER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GIDL_PARSER, ValaGIdlParserClass))
#define VALA_IS_GIDL_PARSER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GIDL_PARSER))
#define VALA_IS_GIDL_PARSER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GIDL_PARSER))
#define VALA_GIDL_PARSER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GIDL_PARSER, ValaGIdlParserClass))

typedef struct _ValaGIdlParser ValaGIdlParser;
typedef struct _ValaGIdlParserClass ValaGIdlParserClass;

struct _ValaVAPIGen {
	GObject parent_instance;
	ValaVAPIGenPrivate * priv;
};

struct _ValaVAPIGenClass {
	GObjectClass parent_class;
};

struct _ValaVAPIGenPrivate {
	ValaCodeContext* context;
};



GType vala_vapi_gen_get_type (void);
#define VALA_VAPI_GEN_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_VAPI_GEN, ValaVAPIGenPrivate))
enum  {
	VALA_VAPI_GEN_DUMMY_PROPERTY
};
static char* vala_vapi_gen_directory;
static char* vala_vapi_gen_directory = NULL;
static gboolean vala_vapi_gen_version;
static gboolean vala_vapi_gen_version = FALSE;
static gboolean vala_vapi_gen_quiet_mode;
static gboolean vala_vapi_gen_quiet_mode = FALSE;
static char** vala_vapi_gen_sources;
static char** vala_vapi_gen_sources = NULL;
static char** vala_vapi_gen_vapi_directories;
static char** vala_vapi_gen_vapi_directories = NULL;
static char* vala_vapi_gen_library;
static char* vala_vapi_gen_library = NULL;
static char** vala_vapi_gen_packages;
static char** vala_vapi_gen_packages = NULL;
static char* vala_vapi_gen_metadata_filename;
static char* vala_vapi_gen_metadata_filename = NULL;
static gint vala_vapi_gen_quit (ValaVAPIGen* self);
static gboolean vala_vapi_gen_add_package (ValaVAPIGen* self, const char* pkg);
static char** vala_vapi_gen_get_packages_from_depsfile (const char* depsfile, int* result_length1);
ValaGirParser* vala_gir_parser_new (void);
ValaGirParser* vala_gir_parser_construct (GType object_type);
GType vala_gir_parser_get_type (void);
void vala_gir_parser_parse_metadata (ValaGirParser* self, const char* metadata_filename);
void vala_gir_parser_parse (ValaGirParser* self, ValaCodeContext* context);
ValaGIdlParser* vala_gidl_parser_new (void);
ValaGIdlParser* vala_gidl_parser_construct (GType object_type);
GType vala_gidl_parser_get_type (void);
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context);
const char* vala_gir_parser_get_package_name (ValaGirParser* self);
static gint vala_vapi_gen_run (ValaVAPIGen* self);
ValaVAPIGen* vala_vapi_gen_new (void);
ValaVAPIGen* vala_vapi_gen_construct (GType object_type);
static gint vala_vapi_gen_main (char** args, int args_length1);
ValaVAPIGen* vala_vapi_gen_new (void);
static gpointer vala_vapi_gen_parent_class = NULL;
static void vala_vapi_gen_finalize (GObject* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);

static const GOptionEntry VALA_VAPI_GEN_options[] = {{"vapidir", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_vapi_gen_vapi_directories, "Look for package bindings in DIRECTORY", "DIRECTORY..."}, {"pkg", (gchar) 0, 0, G_OPTION_ARG_STRING_ARRAY, &vala_vapi_gen_packages, "Include binding for PACKAGE", "PACKAGE..."}, {"library", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_vapi_gen_library, "Library name", "NAME"}, {"metadata", (gchar) 0, 0, G_OPTION_ARG_FILENAME, &vala_vapi_gen_metadata_filename, "Metadata filename", "FILE"}, {"directory", 'd', 0, G_OPTION_ARG_FILENAME, &vala_vapi_gen_directory, "Output directory", "DIRECTORY"}, {"version", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_vapi_gen_version, "Display version number", NULL}, {"quiet", 'q', 0, G_OPTION_ARG_NONE, &vala_vapi_gen_quiet_mode, "Do not print messages to the console", NULL}, {"", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_vapi_gen_sources, NULL, "FILE..."}, {NULL}};


static gint vala_vapi_gen_quit (ValaVAPIGen* self) {
	g_return_val_if_fail (self != NULL, 0);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) == 0) {
		if (!vala_vapi_gen_quiet_mode) {
			fprintf (stdout, "Generation succeeded - %d warning(s)\n", vala_report_get_warnings (vala_code_context_get_report (self->priv->context)));
		}
		return 0;
	} else {
		if (!vala_vapi_gen_quiet_mode) {
			fprintf (stdout, "Generation failed: %d error(s), %d warning(s)\n", vala_report_get_errors (vala_code_context_get_report (self->priv->context)), vala_report_get_warnings (vala_code_context_get_report (self->priv->context)));
		}
		return 1;
	}
}


static gboolean vala_vapi_gen_add_package (ValaVAPIGen* self, const char* pkg) {
	char* package_path;
	ValaSourceFile* _tmp2_;
	gboolean _tmp3_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	if (vala_code_context_has_package (self->priv->context, pkg)) {
		/* ignore multiple occurences of the same package*/
		return TRUE;
	}
	package_path = vala_code_context_get_package_path (self->priv->context, pkg, vala_vapi_gen_vapi_directories, _vala_array_length (vala_vapi_gen_vapi_directories));
	if (package_path == NULL) {
		gboolean _tmp1_;
		return (_tmp1_ = FALSE, package_path = (g_free (package_path), NULL), _tmp1_);
	}
	vala_code_context_add_package (self->priv->context, pkg);
	_tmp2_ = NULL;
	vala_code_context_add_source_file (self->priv->context, _tmp2_ = vala_source_file_new (self->priv->context, package_path, TRUE, NULL));
	(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_source_file_unref (_tmp2_), NULL));
	return (_tmp3_ = TRUE, package_path = (g_free (package_path), NULL), _tmp3_);
}


static char** vala_vapi_gen_get_packages_from_depsfile (const char* depsfile, int* result_length1) {
	GError * _inner_error_;
	g_return_val_if_fail (depsfile != NULL, NULL);
	_inner_error_ = NULL;
	{
		char* contents;
		char* _tmp2_;
		gboolean _tmp1_;
		char* _tmp0_;
		char** _tmp3_;
		char** _tmp4_;
		char** _tmp5_;
		contents = NULL;
		_tmp2_ = NULL;
		_tmp0_ = NULL;
		_tmp1_ = g_file_get_contents (depsfile, &_tmp0_, NULL, &_inner_error_);
		contents = (_tmp2_ = _tmp0_, contents = (g_free (contents), NULL), _tmp2_);
		_tmp1_;
		if (_inner_error_ != NULL) {
			contents = (g_free (contents), NULL);
			if (_inner_error_->domain == G_FILE_ERROR) {
				goto __catch4_g_file_error;
			}
			goto __finally4;
		}
		_tmp3_ = NULL;
		_tmp4_ = NULL;
		_tmp5_ = NULL;
		return (_tmp5_ = (_tmp4_ = _tmp3_ = g_strsplit (g_strstrip (contents), "\n", 0), *result_length1 = _vala_array_length (_tmp3_), _tmp4_), contents = (g_free (contents), NULL), _tmp5_);
	}
	goto __finally4;
	__catch4_g_file_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			gpointer _tmp6_;
			char** _tmp7_;
			/* deps files are optional*/
			_tmp7_ = NULL;
			return (_tmp7_ = (_tmp6_ = NULL, *result_length1 = 0, _tmp6_), (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), _tmp7_);
		}
	}
	__finally4:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
}


static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


static gint vala_vapi_gen_run (ValaVAPIGen* self) {
	ValaCodeContext* _tmp0_;
	char** _tmp15_;
	ValaParser* parser;
	ValaGirParser* girparser;
	ValaGIdlParser* gidlparser;
	ValaSymbolResolver* resolver;
	ValaSemanticAnalyzer* analyzer;
	gboolean _tmp22_;
	gint _tmp29_;
	g_return_val_if_fail (self != NULL, 0);
	_tmp0_ = NULL;
	self->priv->context = (_tmp0_ = vala_code_context_new (), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp0_);
	vala_code_context_push (self->priv->context);
	/* default package */
	if (!vala_vapi_gen_add_package (self, "glib-2.0")) {
		vala_report_error (NULL, "glib-2.0 not found in specified Vala API directories");
	}
	if (!vala_vapi_gen_add_package (self, "gobject-2.0")) {
		vala_report_error (NULL, "gobject-2.0 not found in specified Vala API directories");
	}
	/* load packages from .deps file */
	{
		char** source_collection;
		int source_collection_length1;
		int source_it;
		source_collection = vala_vapi_gen_sources;
		source_collection_length1 = _vala_array_length (vala_vapi_gen_sources);
		for (source_it = 0; source_it < _vala_array_length (vala_vapi_gen_sources); source_it = source_it + 1) {
			const char* _tmp7_;
			char* source;
			_tmp7_ = NULL;
			source = (_tmp7_ = source_collection[source_it], (_tmp7_ == NULL) ? NULL : g_strdup (_tmp7_));
			{
				char* _tmp1_;
				char* _tmp2_;
				char* depsfile;
				char** _tmp4_;
				gint deps_size;
				gint deps_length1;
				gint _tmp3_;
				char** deps;
				if (!g_str_has_suffix (source, ".gi")) {
					source = (g_free (source), NULL);
					continue;
				}
				_tmp1_ = NULL;
				_tmp2_ = NULL;
				depsfile = (_tmp2_ = g_strconcat (_tmp1_ = string_substring (source, (glong) 0, g_utf8_strlen (source, -1) - g_utf8_strlen ("gi", -1)), "deps", NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp2_);
				if (!g_file_test (depsfile, G_FILE_TEST_EXISTS)) {
					source = (g_free (source), NULL);
					depsfile = (g_free (depsfile), NULL);
					continue;
				}
				_tmp4_ = NULL;
				deps = (_tmp4_ = vala_vapi_gen_get_packages_from_depsfile (depsfile, &_tmp3_), deps_length1 = _tmp3_, deps_size = deps_length1, _tmp4_);
				{
					char** dep_collection;
					int dep_collection_length1;
					int dep_it;
					dep_collection = deps;
					dep_collection_length1 = deps_length1;
					for (dep_it = 0; dep_it < deps_length1; dep_it = dep_it + 1) {
						const char* _tmp6_;
						char* dep;
						_tmp6_ = NULL;
						dep = (_tmp6_ = dep_collection[dep_it], (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_));
						{
							if (!vala_vapi_gen_add_package (self, dep)) {
								char* _tmp5_;
								_tmp5_ = NULL;
								vala_report_error (NULL, _tmp5_ = g_strdup_printf ("%s not found in specified Vala API directories", dep));
								_tmp5_ = (g_free (_tmp5_), NULL);
							}
							dep = (g_free (dep), NULL);
						}
					}
				}
				source = (g_free (source), NULL);
				depsfile = (g_free (depsfile), NULL);
				deps = (_vala_array_free (deps, deps_length1, (GDestroyNotify) g_free), NULL);
			}
		}
	}
	if (vala_vapi_gen_packages != NULL) {
		char** _tmp10_;
		{
			char** package_collection;
			int package_collection_length1;
			int package_it;
			package_collection = vala_vapi_gen_packages;
			package_collection_length1 = _vala_array_length (vala_vapi_gen_packages);
			for (package_it = 0; package_it < _vala_array_length (vala_vapi_gen_packages); package_it = package_it + 1) {
				const char* _tmp9_;
				char* package;
				_tmp9_ = NULL;
				package = (_tmp9_ = package_collection[package_it], (_tmp9_ == NULL) ? NULL : g_strdup (_tmp9_));
				{
					if (!vala_vapi_gen_add_package (self, package)) {
						char* _tmp8_;
						_tmp8_ = NULL;
						vala_report_error (NULL, _tmp8_ = g_strdup_printf ("%s not found in specified Vala API directories", package));
						_tmp8_ = (g_free (_tmp8_), NULL);
					}
					package = (g_free (package), NULL);
				}
			}
		}
		_tmp10_ = NULL;
		vala_vapi_gen_packages = (_tmp10_ = NULL, vala_vapi_gen_packages = (_vala_array_free (vala_vapi_gen_packages, _vala_array_length (vala_vapi_gen_packages), (GDestroyNotify) g_free), NULL), _tmp10_);
	}
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		return vala_vapi_gen_quit (self);
	}
	{
		char** source_collection;
		int source_collection_length1;
		int source_it;
		source_collection = vala_vapi_gen_sources;
		source_collection_length1 = _vala_array_length (vala_vapi_gen_sources);
		for (source_it = 0; source_it < _vala_array_length (vala_vapi_gen_sources); source_it = source_it + 1) {
			const char* _tmp14_;
			char* source;
			_tmp14_ = NULL;
			source = (_tmp14_ = source_collection[source_it], (_tmp14_ == NULL) ? NULL : g_strdup (_tmp14_));
			{
				if (g_file_test (source, G_FILE_TEST_EXISTS)) {
					ValaSourceFile* _tmp12_;
					_tmp12_ = NULL;
					vala_code_context_add_source_file (self->priv->context, _tmp12_ = vala_source_file_new (self->priv->context, source, TRUE, NULL));
					(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_source_file_unref (_tmp12_), NULL));
				} else {
					char* _tmp13_;
					_tmp13_ = NULL;
					vala_report_error (NULL, _tmp13_ = g_strdup_printf ("%s not found", source));
					_tmp13_ = (g_free (_tmp13_), NULL);
				}
				source = (g_free (source), NULL);
			}
		}
	}
	_tmp15_ = NULL;
	vala_vapi_gen_sources = (_tmp15_ = NULL, vala_vapi_gen_sources = (_vala_array_free (vala_vapi_gen_sources, _vala_array_length (vala_vapi_gen_sources), (GDestroyNotify) g_free), NULL), _tmp15_);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		return vala_vapi_gen_quit (self);
	}
	parser = vala_parser_new ();
	vala_parser_parse (parser, self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		gint _tmp17_;
		return (_tmp17_ = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), _tmp17_);
	}
	girparser = vala_gir_parser_new ();
	if (vala_vapi_gen_metadata_filename != NULL) {
		vala_gir_parser_parse_metadata (girparser, vala_vapi_gen_metadata_filename);
	}
	vala_gir_parser_parse (girparser, self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		gint _tmp18_;
		return (_tmp18_ = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), _tmp18_);
	}
	gidlparser = vala_gidl_parser_new ();
	vala_gidl_parser_parse (gidlparser, self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		gint _tmp19_;
		return (_tmp19_ = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), (gidlparser == NULL) ? NULL : (gidlparser = (vala_code_visitor_unref (gidlparser), NULL)), _tmp19_);
	}
	resolver = vala_symbol_resolver_new ();
	vala_symbol_resolver_resolve (resolver, self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		gint _tmp20_;
		return (_tmp20_ = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), (gidlparser == NULL) ? NULL : (gidlparser = (vala_code_visitor_unref (gidlparser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), _tmp20_);
	}
	analyzer = vala_semantic_analyzer_new ();
	vala_semantic_analyzer_analyze (analyzer, self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		gint _tmp21_;
		return (_tmp21_ = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), (gidlparser == NULL) ? NULL : (gidlparser = (vala_code_visitor_unref (gidlparser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), _tmp21_);
	}
	_tmp22_ = FALSE;
	if (vala_vapi_gen_library == NULL) {
		_tmp22_ = vala_gir_parser_get_package_name (girparser) != NULL;
	} else {
		_tmp22_ = FALSE;
	}
	if (_tmp22_) {
		char* _tmp24_;
		const char* _tmp23_;
		_tmp24_ = NULL;
		_tmp23_ = NULL;
		vala_vapi_gen_library = (_tmp24_ = (_tmp23_ = vala_gir_parser_get_package_name (girparser), (_tmp23_ == NULL) ? NULL : g_strdup (_tmp23_)), vala_vapi_gen_library = (g_free (vala_vapi_gen_library), NULL), _tmp24_);
	}
	if (vala_vapi_gen_library != NULL) {
		ValaCodeWriter* interface_writer;
		char* _tmp27_;
		char* _tmp28_;
		/* interface writer ignores external packages*/
		{
			GeeList* _tmp25_;
			GeeIterator* _tmp26_;
			GeeIterator* _file_it;
			/* interface writer ignores external packages*/
			_tmp25_ = NULL;
			_tmp26_ = NULL;
			_file_it = (_tmp26_ = gee_iterable_iterator ((GeeIterable*) (_tmp25_ = vala_code_context_get_source_files (self->priv->context))), (_tmp25_ == NULL) ? NULL : (_tmp25_ = (gee_collection_object_unref (_tmp25_), NULL)), _tmp26_);
			/* interface writer ignores external packages*/
			while (gee_iterator_next (_file_it)) {
				ValaSourceFile* file;
				/* interface writer ignores external packages*/
				file = (ValaSourceFile*) gee_iterator_get (_file_it);
				if (!g_str_has_suffix (vala_source_file_get_filename (file), ".vapi")) {
					vala_source_file_set_external_package (file, FALSE);
				}
				(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
			}
			(_file_it == NULL) ? NULL : (_file_it = (gee_collection_object_unref (_file_it), NULL));
		}
		interface_writer = vala_code_writer_new (FALSE, FALSE);
		_tmp27_ = NULL;
		vala_code_writer_write_file (interface_writer, self->priv->context, _tmp27_ = g_strdup_printf ("%s.vapi", vala_vapi_gen_library));
		_tmp27_ = (g_free (_tmp27_), NULL);
		_tmp28_ = NULL;
		vala_vapi_gen_library = (_tmp28_ = NULL, vala_vapi_gen_library = (g_free (vala_vapi_gen_library), NULL), _tmp28_);
		(interface_writer == NULL) ? NULL : (interface_writer = (vala_code_visitor_unref (interface_writer), NULL));
	}
	return (_tmp29_ = vala_vapi_gen_quit (self), (parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL)), (girparser == NULL) ? NULL : (girparser = (vala_code_visitor_unref (girparser), NULL)), (gidlparser == NULL) ? NULL : (gidlparser = (vala_code_visitor_unref (gidlparser), NULL)), (resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL)), (analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL)), _tmp29_);
}


static gint vala_vapi_gen_main (char** args, int args_length1) {
	GError * _inner_error_;
	ValaVAPIGen* vapigen;
	gint _tmp3_;
	_inner_error_ = NULL;
	{
		GOptionContext* opt_context;
		opt_context = g_option_context_new ("- Vala API Generator");
		g_option_context_set_help_enabled (opt_context, TRUE);
		g_option_context_add_main_entries (opt_context, VALA_VAPI_GEN_options, NULL);
		g_option_context_parse (opt_context, &args_length1, &args, &_inner_error_);
		if (_inner_error_ != NULL) {
			(opt_context == NULL) ? NULL : (opt_context = (g_option_context_free (opt_context), NULL));
			if (_inner_error_->domain == G_OPTION_ERROR) {
				goto __catch5_g_option_error;
			}
			goto __finally5;
		}
		(opt_context == NULL) ? NULL : (opt_context = (g_option_context_free (opt_context), NULL));
	}
	goto __finally5;
	__catch5_g_option_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			gint _tmp0_;
			fprintf (stdout, "%s\n", e->message);
			fprintf (stdout, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
			return (_tmp0_ = 1, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), _tmp0_);
		}
	}
	__finally5:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return 0;
	}
	if (vala_vapi_gen_version) {
		fprintf (stdout, "Vala API Generator %s\n", PACKAGE_VERSION);
		return 0;
	}
	if (vala_vapi_gen_sources == NULL) {
		fprintf (stderr, "No source file specified.\n");
		return 1;
	}
	vapigen = vala_vapi_gen_new ();
	return (_tmp3_ = vala_vapi_gen_run (vapigen), (vapigen == NULL) ? NULL : (vapigen = (g_object_unref (vapigen), NULL)), _tmp3_);
}


int main (int argc, char ** argv) {
	g_type_init ();
	return vala_vapi_gen_main (argv, argc);
}


ValaVAPIGen* vala_vapi_gen_construct (GType object_type) {
	ValaVAPIGen * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


ValaVAPIGen* vala_vapi_gen_new (void) {
	return vala_vapi_gen_construct (VALA_TYPE_VAPI_GEN);
}


static void vala_vapi_gen_class_init (ValaVAPIGenClass * klass) {
	vala_vapi_gen_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaVAPIGenPrivate));
	G_OBJECT_CLASS (klass)->finalize = vala_vapi_gen_finalize;
}


static void vala_vapi_gen_instance_init (ValaVAPIGen * self) {
	self->priv = VALA_VAPI_GEN_GET_PRIVATE (self);
}


static void vala_vapi_gen_finalize (GObject* obj) {
	ValaVAPIGen * self;
	self = VALA_VAPI_GEN (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	G_OBJECT_CLASS (vala_vapi_gen_parent_class)->finalize (obj);
}


GType vala_vapi_gen_get_type (void) {
	static GType vala_vapi_gen_type_id = 0;
	if (vala_vapi_gen_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaVAPIGenClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_vapi_gen_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaVAPIGen), 0, (GInstanceInitFunc) vala_vapi_gen_instance_init, NULL };
		vala_vapi_gen_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaVAPIGen", &g_define_type_info, 0);
	}
	return vala_vapi_gen_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}




