#include <QtGui>
#include <QSignalMapper>
#include <iostream>
#include <string.h>
#include <libintl.h>

#define PROCESSING "Processing..."

class QUssd : public QMainWindow {
	Q_OBJECT

	public:
		QUssd(QString message, bool lock = false) {
			initOrientation();

			replyLabel = new QTextEdit(message);
			replyLabel->setMinimumWidth(400);
			replyLabel->setMinimumHeight(200);
			replyLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Maximum);
			replyLabel->setFrameStyle(QFrame::NoFrame);
			QPalette pal = palette();
			pal.setColor(QPalette::Base, Qt::transparent);
			pal.setColor(QPalette::Text, pal.color(QPalette::ButtonText));
			replyLabel->setPalette(pal);
			replyLabel->setTextInteractionFlags(Qt::TextSelectableByMouse | Qt::TextSelectableByKeyboard);
			
			numberLabel = new QLabel();
			numberLabel->setMinimumWidth(400);
			numberLabel->setMaximumHeight(100);
			numberLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

			indicationLayout = new QVBoxLayout;
			indicationLayout->addWidget(numberLabel);

			indicationLayout->addWidget(replyLabel);

			padDel = new QPushButton("C");
			connect(padDel, SIGNAL(clicked()), this, SLOT(del()));
			padDel->setMaximumWidth(150);
			padDel->setMaximumHeight(72);
			padDel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);
			padSend = new QPushButton("SEND");
			connect(padSend, SIGNAL(clicked()), this, SLOT(send()));
			padSend->setMaximumWidth(300);
			padSend->setMaximumHeight(72);
			padSend->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);

			btnLayout = new QHBoxLayout;
			btnLayout->addWidget(padSend);
			btnLayout->addWidget(padDel);

			numberPad = new QGridLayout();
			numberPad->setSpacing(2);

			QSignalMapper *signalMapper = new QSignalMapper(this);

			numberPad->addWidget(createButton("1", signalMapper), 0, 0);
			numberPad->addWidget(createButton("2", signalMapper), 0, 1);
			numberPad->addWidget(createButton("3", signalMapper), 0, 2);
			numberPad->addWidget(createButton("4", signalMapper), 1, 0);
			numberPad->addWidget(createButton("5", signalMapper), 1, 1);
			numberPad->addWidget(createButton("6", signalMapper), 1, 2);
			numberPad->addWidget(createButton("7", signalMapper), 2, 0);
			numberPad->addWidget(createButton("8", signalMapper), 2, 1);
			numberPad->addWidget(createButton("9", signalMapper), 2, 2);
			numberPad->addWidget(createButton("0", signalMapper), 3, 1);
			numberPad->addWidget(createButton("#", signalMapper), 3, 2);

			connect(signalMapper, SIGNAL(mapped(QString)), this, SLOT(appendToNumber(QString)));
									
			padBtnStar = new QPushButton("*+");
			connect(padBtnStar, SIGNAL(clicked()), this, SLOT(addStar()));
			padBtnStar->setMaximumWidth(150);
			padBtnStar->setMaximumHeight(72);
			padBtnStar->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);
			numberPad->addWidget(padBtnStar, 3, 0);

			centralWidget = new QWidget;
			setCentralWidget(centralWidget);
			mainLayout = new QGridLayout(centralWidget);

			connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
			setOrientation();
			orientationChanged();

			startUSSD (lock);
		}

		~QUssd () {
			USSDquery->write("exit\n");
			USSDquery->closeWriteChannel();
			USSDquery->waitForFinished();

			delete USSDquery;
		}

	private slots:
		QPushButton* createButton(QString caption, QSignalMapper *signalMapper) {
			QPushButton *button = new QPushButton(caption);
			signalMapper->setMapping(button, caption);
			connect(button, SIGNAL(clicked()), signalMapper, SLOT (map()));
			button->setMaximumWidth(150);
			button->setMaximumHeight(72);
			button->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding);
			return button;
		}

		void initOrientation() {
			QActionGroup *filterGroup = new QActionGroup(this);
			filterGroup->setExclusive(true);

			actPortrait = new QAction(tr("Portrait"), filterGroup);
			actPortrait->setCheckable(true);

			actLandscape = new QAction(tr("Landscape"), filterGroup);
			actLandscape->setCheckable(true);

			actAuto = new QAction(tr("Auto"), filterGroup);
			actAuto->setCheckable(true);
			actAuto->setChecked(true);
		
			menuBar()->addActions(filterGroup->actions());

			connect(actPortrait, SIGNAL(changed()), this, SLOT(setOrientation()));
			connect(actLandscape, SIGNAL(changed()), this, SLOT(setOrientation()));
			connect(actAuto, SIGNAL(changed()), this, SLOT(setOrientation()));
		}

		void restartUSSD( int, QProcess::ExitStatus) {
			replyLabel->setText(QString("Error: ")+QString::fromUtf8(reply));
			reply = "";
			number = "";
			setNumber();

			delete USSDquery;
			startUSSD ();
		}


		void send() {	
			USSDquery->write(number.toUtf8().data());
			USSDquery->write("\n");
			number = tr(PROCESSING);
			setNumber();
		}

		void readData() {
			QByteArray line;
			for (line = USSDquery->readLine(); !line.isEmpty(); line = USSDquery->readLine()) {
				reply += line;
				if (reply.endsWith("\n==MESSAGE END ==\n")) {
					reply.chop(strlen("\n==MESSAGE END ==\n"));
					if (!reply.isEmpty())
						replyLabel->setText(QString::fromUtf8(reply));
					reply = "";
					number = "";
					setNumber();
					break;
				}
			}
		}

		void setOrientation() {
			if (actLandscape->isChecked())
				setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
			else if (actPortrait->isChecked())
				setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
			else
				setAttribute(Qt::WA_Maemo5AutoOrientation, true);
		}

		void orientationChanged() {
			QRect screenGeometry = QApplication::desktop()->screenGeometry();
			mainLayout->removeItem(indicationLayout);
			mainLayout->removeItem(numberPad);
			mainLayout->removeItem(btnLayout);
			
			if (screenGeometry.width() > screenGeometry.height()) {
				mainLayout->addLayout(indicationLayout, 0, 0);
				mainLayout->addLayout(btnLayout, 1, 0);
				mainLayout->addLayout(numberPad, 0, 1, -1, 1);
			} else {
				mainLayout->addLayout(indicationLayout, 0, 0);
				mainLayout->addLayout(numberPad, 1, 0);
				mainLayout->addLayout(btnLayout, 2, 0);
			}
		}
	
		void appendToNumber (QString str) {
			if (number != tr(PROCESSING)) {
				number += str;
				setNumber();
			}
		}

		void addStar () {
			if (number != tr(PROCESSING)) {
				// Do not change star if it was created more than 3 seconds before.
				if (number[number.length()-1] == '*' && lastStar.elapsed() < 3*1000) {
					number[number.length()-1] = '+';
				} else {
					number += "*";
					lastStar = QTime();
					lastStar.start();
				}
				setNumber();
			}
		}

		void del () {
			if (number != tr(PROCESSING)) {
				number.chop(1);
				setNumber();
			}
		}

	private:
		void startUSSD (bool lock = false) {
			QString program = "ussdquery.py";
			QStringList arguments;
			arguments << "interactive" << "-d" << "\n==MESSAGE END ==\n";
			if (lock)
				arguments << "-m";

			USSDquery = new QProcess();
			USSDquery->setProcessChannelMode(QProcess::MergedChannels);
			connect(USSDquery, SIGNAL(readyReadStandardOutput()), this, SLOT(readData()));
			connect(USSDquery, SIGNAL(readyReadStandardError()), this, SLOT(readData()));
			connect(USSDquery, SIGNAL(finished( int, QProcess::ExitStatus )), this, SLOT(restartUSSD( int, QProcess::ExitStatus )));

			USSDquery->start(program, arguments);
		}

		void setNumber () {
			numberLabel->setText(QString("<p style=\"font-size: 30pt;\" align=\"center\">")+number+"</p>");
		}

		QAction *actPortrait;
		QAction *actLandscape;
		QAction *actAuto;

		QTextEdit *replyLabel;
		QLabel *numberLabel;

		QVBoxLayout *indicationLayout;
		QHBoxLayout *btnLayout;
		QGridLayout *numberPad;
		QGridLayout *mainLayout;
		QWidget* centralWidget;

		QPushButton *padBtnStar;

		QPushButton *padDel;
		QPushButton *padSend;

		QProcess *USSDquery;

		QString number;
		QByteArray reply;

		QTime lastStar;
};

int main(int argc, char *argv[]) {
	QApplication app(argc, argv);

	bool lock = false;
	QString message;
	if (argc >= 2)
		message = message.fromUtf8(argv[1]);
	if (argc >=3 && !strcmp(argv[2], "lock"))
		lock = true;

	// If no reply is expected just show notification box
	if (!lock && !message.isEmpty()) {
		QMessageBox box(QMessageBox::NoIcon, QString::fromUtf8(dgettext("rtcom-call-ui", "call_ti_message")), message, QMessageBox::Ok);
		box.button(QMessageBox::Ok)->setText(QString::fromUtf8(dgettext("hildon-libs", "wdgt_bd_done")));
		return box.exec();
	} else {
		QUssd w(message, lock);
		w.show();
	
		return app.exec();
	}
}

#include "qussd.moc"
