/* 
Copyright 2008 by Danny Milosavjevic <danny.milo@scratchpost.org>.
All rights reserved.
   
Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
   
1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
            
This software is provided by the copyright holders and contributors "as is" and any express of implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed.
In no event shall the copyright owner or contributors be liable for any direct, indirect, incidental, special, exemplary or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <gtk/gtk.h>

#include <libosso.h>
#include <hildon/hildon-program.h>

#ifndef PREFIX
#define PREFIX "/usr/" /* sigh... */
#endif

static osso_context_t* osso_context;
static gboolean cmdline = TRUE;

static void verr(const char* format, va_list arg)
{
	char text[2049];

	if (cmdline) {
		fprintf(stderr, "error: ");
		vfprintf(stderr, format, arg);
		fprintf(stderr, "\n");
	} else {
		if (vsnprintf(text, sizeof (text), format, arg) == -1) {
			strcpy(text, "(error message too long)");
		}
		osso_system_note_dialog (osso_context, text, OSSO_GN_ERROR, NULL);
	}
}

static void err(const char *format, ...)
{
	va_list arg;

	va_start(arg, format);
	verr(format, arg);
	va_end(arg);
}

static void err_with_errno(const char* syscall, const char* affected_file)
{
	err("%s(\"%s\") failed: %s", syscall, affected_file, strerror(errno));
}

static void err_with_GError(const GError* g_error, const char* action, const char* affected_file)
{
	err("%s(\"%s\") failed: %s", action, affected_file, g_error->message);
}

/** wait_until_child_terminated: waits until a process terminated.
  *  returns:
  *    -1   on error
  *     0   terminated normally and with exit code 0
  *    >0   terminated "normally" with that exit code
  *    
  * Side effects:
  *   - prints to stderr
  */
static int wait_until_child_terminated(const char* rar_file_name, pid_t child_pid)
{
	int status;
	int waitpid_result;
	do {
		waitpid_result = waitpid(child_pid, &status, 0);
		if (waitpid_result == -1) {
			break;
		}
		
		/* WIFSTOPPED WIFCONTINUED */
	} while (!WIFEXITED(status) && !WIFSIGNALED(status));
	
	if (waitpid_result == -1) { /* error */
		err_with_errno("waitpid", "<\"unrar\" process>");
		return -1;
	}
	
	if (WIFEXITED(status)) {
		if (WEXITSTATUS(status) != 0) {
			err("could not unrar \"%s\"", rar_file_name);
			return WEXITSTATUS(status);
		}
	} else if (WIFSIGNALED(status)) {
		err("could not unrar \"%s\": unrar received signal %d", rar_file_name, WTERMSIG(status));
		return -1;
	}
	
	/* here, unrar finished successfully */
	return 0;
}

/**
  * Side effects:
  *   - prints to stderr
  */
static char** get_rar_content_file_names(const char* rar_file_name)
{
	const char* argv[] = {
		PREFIX "bin/unrar",
		"v",
		"",
		"--",
		rar_file_name,
		NULL
	};
	gint exit_code;
	gchar* standard_output;
	gchar** file_names;
	GError* error;

	error = NULL;
	standard_output = NULL;
	
	if (!g_spawn_sync(NULL, (char**) argv, NULL, G_SPAWN_STDERR_TO_DEV_NULL, NULL, NULL, &standard_output, NULL, &exit_code, &error)) {
		/* unrar printed reason to stderr */
		err_with_GError(error, "could not read rar file", rar_file_name);
		g_error_free(error);
		return NULL;
	}

	if (standard_output == NULL || 	exit_code != 0) {
		g_free(standard_output);
		err("could not read rar file \"%s\"", rar_file_name);
		return NULL;
	}

	/* here, unrar (-Z -1) finished successfully */

	assert(standard_output != NULL);
	
	file_names = g_strsplit(standard_output, "\n", 0);
	assert(file_names != NULL);
	
	g_free(standard_output);

	return file_names;
}

/** prepare_unrar: prepares to unrar (maybe create subfolder, change there, ...)
 * returns:
 *     NULL        error
 *     otherwise   the (new) directory path where it will be extracted to
 *
 * Side effects:
 *   - creates directory
 *   - changes current directory
 *   - prints to stderr
 */
static char* prepare_unrar(const char* rar_file_name)
{
	const char* file_name;
	char* suggested_folder_name;
	char* suggested_folder_name_and_slash;
	int i;
	gboolean contains_only_obvious_subfolder;
	char* rar_base_name;
	char** file_names;
	char* result;
	char* source_directory;
	
	file_names = get_rar_content_file_names(rar_file_name);
	if (file_names == NULL) {
		return NULL;
	}
		
	rar_base_name = g_path_get_basename(rar_file_name);
	assert(rar_base_name != NULL);

	if (g_str_has_suffix(rar_base_name, ".rar")) {
		suggested_folder_name = g_strdup(rar_base_name);
		suggested_folder_name[strlen(suggested_folder_name) - 4] = 0;
	} else {
		suggested_folder_name = g_strdup_printf("%s.d", rar_base_name);
	}

	g_free(rar_base_name);
	rar_base_name = NULL;
	
	suggested_folder_name_and_slash = g_strdup_printf("%s/", suggested_folder_name);

	/* check whether the rar file only contains files in subfolders */
	
	contains_only_obvious_subfolder = TRUE;
	
	for (i = 0; file_names[i] != NULL; ++i) {
		file_name = file_names[i];
		
		if (!g_str_has_prefix(file_name, suggested_folder_name_and_slash) && !g_str_equal(file_name, suggested_folder_name)) {
			contains_only_obvious_subfolder = FALSE;
			break;
		}
	}
	
	g_strfreev(file_names);
	
	/*previous_directory = g_get_current_dir();*/
	
	source_directory = g_path_get_dirname(rar_file_name);

	result = g_build_filename(source_directory, suggested_folder_name, NULL);
	g_free(suggested_folder_name);
	suggested_folder_name = NULL;
	
	if (!contains_only_obvious_subfolder) {
		/* since it is a messy file, specially create a confinement directory for it */
		
		if (mkdir(result, 0777) == -1) {
			err_with_errno("mkdir", result);
			g_free(result);
			g_free(source_directory);
			return NULL;
		}
		
	} else {
		/* it is an orderly file, so just check whether folder doesn't exist */
		
		if (mkdir(result, 0777) == -1) { /* oops. */
			err_with_errno("mkdir", result);
			g_free(result);
			g_free(source_directory);
			return NULL;
		}
		
		/* orderly file, so return source directory as extraction dir */
		
		g_free(result);
		result = g_strdup(source_directory);
	}
	
	g_free(source_directory);

	if (chdir(result) == -1) {
		err_with_errno("chdir", result);
		g_free(result);
		return NULL;
	}
	
	return result;
}

/* Cleans up a bit */
static void undo_unrar(const char* rar_extraction_directory_name)
{
	const char* argv[] = {
		"/bin/rm",
		"-rf",
		"--",
		rar_extraction_directory_name,
		NULL
	};

	gint exit_code;
	GError* error;
	
	error = NULL;
	
	if (rar_extraction_directory_name[0] == '/' && rar_extraction_directory_name[1] == 0)
	{
		err("tried to remove the root directory \"/\". This is a serious program bug. Please report to <danny.milo@scratchpost.org>.");
		return; /* chicken */
	}
	
	if (!g_spawn_sync(NULL, (char**) argv, NULL, 0, NULL, NULL, NULL, NULL, &exit_code, &error)) {
		err_with_GError(error, "could not delete directory", rar_extraction_directory_name);
		g_error_free(error);
	} else if (exit_code != 0) {
		/* unrar printed reason to stderr */
		err("could not delete directory \"%s\"", rar_extraction_directory_name);
		/* ignore error... */
	}
	
	/*
	if (remove(rar_extraction_directory_name) == -1) {
		err_with_errno("remove");
		ignore error...
	}*/
}

/** unrar_file: unrars a file
 * Preconditions:
 *   - prepare_unrar()
 * Side effects:
 *   - prints to stderr
 */
static int unrar_file(const char* rar_file_name, const char* rar_extraction_directory_name)
{
	const char* argv[] = {
		PREFIX "bin/unrar",
		"-o+",
		"-inul",
		"e",
		rar_file_name,
		rar_extraction_directory_name,
		NULL
	};
	GPid child_pid;
	char text[2049];
	GError* error;
	
	error = NULL;
	
	/* Display a HildonBanner to the user */
	if (snprintf(text, sizeof(text), "Unraring \"%s\"", g_basename (rar_file_name)) != -1) {
		gdk_notify_startup_complete();
		osso_system_note_infoprint(osso_context, text, NULL);
	}
	
	if (!g_spawn_async(NULL, (char**) argv, NULL, G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_DO_NOT_REAP_CHILD, NULL, NULL, &child_pid, &error)) {
		/* unrar printed reason to stderr */
		err_with_GError(error, "Unraring failed", rar_file_name);
		g_error_free(error);
		return -1;
	}

	if (wait_until_child_terminated(rar_file_name, child_pid) != 0) { /* bad */
		return -1;
	}
	
	/* here, unrar finished successfully */
	return 0;
}

/** unrar_frontend: the main unrar entry point
 * Side effects:
 *   - prints to stderr
 * Limitations:
 *   - only works with absolute paths most of the time
 */
static int unrar_frontend(const char* rar_file_name)
{
	char* previous_directory;
	char* source_directory;
	char* destination_new_directory;
	int   status = 0;
	
	previous_directory = g_get_current_dir();
	
	source_directory = g_path_get_dirname(rar_file_name);
	assert(source_directory != NULL);
	
	if (source_directory[0] != 0) {
		if (chdir(source_directory) == -1) {
			err_with_errno("chdir", source_directory);
			return -1; /* whoops. this. is. bad. */
		}
	}
	
	g_free(source_directory);
	
	destination_new_directory = prepare_unrar(rar_file_name);
	if (destination_new_directory != NULL) {
		if (unrar_file(rar_file_name, destination_new_directory) != 0)
			status = -1;
	} else {
		status = -1;
	}
	
	
	if (chdir(previous_directory) == -1) {
		err_with_errno("chdir", previous_directory);
		g_free(previous_directory);
		chdir("/"); /* whoops. */
		return -1;
	}
	g_free(previous_directory);


	if (status != 0) {	
		if (destination_new_directory != NULL) {
			undo_unrar(destination_new_directory);
		}
	}

	g_free(destination_new_directory);
	destination_new_directory = NULL;


	return status;
}


static void document_open_callback(gpointer data, int argc, char* argv[]) 
{ 
	char* document_path; 
	char const* document_url; 
	int i;
	
	for (i = 0; i < argc; ++i) {
		document_url = argv[i]; 
	
		if (document_url == NULL) { 
			g_warning("internal error: document_url[%d] = NULL", i);
			break;
		} 

		document_path = g_filename_from_uri(document_url, NULL /*&document_host*/, NULL); 
		if (document_path == NULL) { 
			g_warning("could not find path to document. Maybe the document is not local?"); 
			continue;
		} 

		if (!g_file_test(document_path, G_FILE_TEST_IS_REGULAR)) {
			continue;
		}

		unrar_frontend(document_path);

		g_free(document_path);
	}
	
	exit(0); /* Commit suicide so that we don't consume RAM doing nothing.
	            DBus should be launching us again when we are needed. */
}

static char* expand_name(const char* name)
{
	char* result;
	char* current_directory;
	
	if (name[0] == '/') {
		return g_strdup(name);
	} else {
		current_directory = g_get_current_dir();
		assert(current_directory != NULL);
		
		result = g_strdup_printf("%s/%s", current_directory, name);
		g_free(current_directory);
		return result;
	}
}


int main(int argc, char* argv[])
{
	const char** argp;
	char* absolute_path;
	
	gtk_init(&argc, &argv);
	
	osso_context = osso_initialize("unrar_fm", "1.0", FALSE, NULL);
	if (osso_context == NULL) {
		err("could not register document handler service");
		exit(1);
	}

	/* if we got a parameter, that probably is a file to be batch-unrarped. */
	if (argc > 1) {
		/* skip options until the end-of-option marker.
		   For example:
		     ./batch-unrar -f -b -k -- hello.rar
		   Moves argp over there:
					       ^^^^^^^^^
		   For example:
		     ./batch-unrar -f -b -k -- --hello.rar
		   Moves argp over there:
					       ^^^^^^^^^^^
		 */
		argp = (const char**) &argv[1];
		while (*argp && (*argp)[0] == '-') {
			if ((*argp)[0] == '-' && (*argp)[1] == '-' && (*argp)[2] == 0) {
				++argp;
				break;
			}
			++argp;
		}
		
		if (*argp) {
			absolute_path = expand_name(*argp);
			if (unrar_frontend(absolute_path) != 0) {
				g_free(absolute_path);
				return 1;
			}
			g_free(absolute_path);
		}
	} else {
		/* process dbus stuff */
		cmdline = FALSE;
		
		osso_mime_set_cb(osso_context, document_open_callback, NULL);

		/* hang around until a request comes....
		   Note that this could just make us hang around doing nothing,
		   should we have been started from the application menu */
		gtk_main();
	}
	
	return 0;
}
