#include "servicewindow.h"
#include "ui_servicewindow.h"

ServiceWindow::ServiceWindow(QWidget *parent) : QMainWindow(parent), ui(new Ui::ServiceWindow)
{
    ui->setupUi(this);

    manager = new QNetworkAccessManager(this);
    connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(readResponse(QNetworkReply*)));

    ui->scrollArea->setProperty("FingerScrollable", true);
    setupDialog();
}

/*
 * Initialises the progress dialog for the network request
 */
void ServiceWindow::setupDialog()
{
    dialog = new QDialog(0);
        layout = new QHBoxLayout();
            progress = new QProgressBar();
            progress->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
            cancelButton = new QPushButton("Cancel");

            layout->addWidget(progress);
            layout->addWidget(cancelButton);
        dialog->setLayout(layout);
    dialog->show();

    connect(cancelButton, SIGNAL(clicked()), dialog, SLOT(reject()));
    connect(dialog, SIGNAL(rejected()), this, SLOT(cancel()));
}

/*
 * Loads the service window for the requested train
 */
void ServiceWindow::show(QString sID)
{
    serviceID = sID;
    //Load the departure board data.
    setupRequest();
    sendRequest();
    //Note that QMainWindow::show() is absent here. Instead we show a progress dialog, then show the
    //departure board only when we have loaded the data & created the display.
    errorFlag = false;
    firstTime = true;
}

/*
 * Creates the network request, including the SOAP request XML.
 */
void ServiceWindow::setupRequest()
{
    QDomDocument document;

    QDomElement env = document.createElement("soapenv:Envelope");
    env.setAttribute("xmlns:soapenv", "http://schemas.xmlsoap.org/soap/envelope/");
    env.setAttribute("xmlns:typ", "http://thalesgroup.com/RTTI/2007-10-10/ldb/types");

        QDomElement header = document.createElement("soapenv:Header");
        QDomElement body = document.createElement("soapenv:Body");

            QDomElement typ = document.createElement("typ:GetServiceDetailsRequest");

                QDomElement id = document.createElement("typ:serviceID");
                id.appendChild(document.createTextNode(serviceID));

    document.appendChild(env);
        env.appendChild(header);
        env.appendChild(body);
            body.appendChild(typ);
                typ.appendChild(id);

    requestData = document.toString(-1).prepend("<?xml version=\"1.0\" encoding=\"utf-8\"?>");

    request = QNetworkRequest(QUrl("http://realtime.nationalrail.co.uk/LDBWS/ldb2.asmx"));
    request.setHeader(QNetworkRequest::ContentTypeHeader, QLatin1String("text/xml;charset=utf-8"));
    request.setRawHeader("SOAPAction", "http://thalesgroup.com/RTTI/2008-02-20/ldb/GetServiceDetails");
}

/*
 * Sends the SOAP request
 */
void ServiceWindow::sendRequest()
{
    m_reply = manager->post(request, requestData.toUtf8().constData());
    connect(m_reply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(networkError(QNetworkReply::NetworkError)));
    connect(m_reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(downloadProgress(qint64, qint64)));
    connect(m_reply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(uploadProgress(qint64, qint64)));
}

/*
 * Cancel the current request
 */
void ServiceWindow::cancel()
{
    //Abort the request; an error signal will be emitted, so we don't need to delete the network reply here
    m_reply->abort();
}

/*
 * Updates the progress bar when making the request
 */
void ServiceWindow::uploadProgress(qint64 rec, qint64 total)
{
    dialog->setWindowTitle("Sending request");
    progress->setRange(0, total);
    progress->setValue(rec);
}

/*
 * Updates the progress bar when receiving the reply to the request
 */
void ServiceWindow::downloadProgress(qint64 rec, qint64 total)
{
    dialog->setWindowTitle("Receiving reply");
    progress->setRange(0, total);
    progress->setValue(rec);
}

/*
 * Slot for when there is a network error
 */
void ServiceWindow::networkError(QNetworkReply::NetworkError error)
{
    //If the request was cancelled, we show no error message
    if(error == QNetworkReply::OperationCanceledError)
    {
        dialog->hide();
        //Set the error flag to true, so that we can exit out of the readResponse method
        errorFlag = true;
    }
    else
    {
        //If this error is a result of the first network request, we assume there is no connection to the internet
        if(firstTime)
        {
            firstTime = false;
            ensureConnection();
            errorFlag = true;
        }
        else
        {
            QMessageBox msg;
            msg.setText(QString("Network error: %1, %2").arg(m_reply->errorString()).arg(error));
            msg.exec();
            dialog->hide();
            errorFlag = true;
        }
    }
}

/*
 * Attempt to wait for connection on either wifi or gprs
 */
void ServiceWindow::ensureConnection()
{
    dialog->setWindowTitle("Connecting");

    QNetworkInterface wlan = QNetworkInterface::interfaceFromName("wlan0");
    QNetworkInterface gprs = QNetworkInterface::interfaceFromName("gprs0");

    //qDebug() << "wlan0" << wlan.flags().testFlag(QNetworkInterface::IsUp) << wlan.flags().testFlag(QNetworkInterface::IsRunning);
    //qDebug() << "\tgprs0" << gprs.flags().testFlag(QNetworkInterface::IsUp) << gprs.flags().testFlag(QNetworkInterface::IsRunning);

    if( (wlan.isValid() && wlan.flags().testFlag(QNetworkInterface::IsUp)) || (gprs.isValid() && gprs.flags().testFlag(QNetworkInterface::IsUp)) )
    {
        //We are connected, so try sending the network request again
        errorFlag = false;
        sendRequest();
    }
    else
    {
        QTimer::singleShot(100, this, SLOT(ensureConnection()));
    }
}

#define NAMESPACE   "declare namespace t=\"http://thalesgroup.com/RTTI/2007-10-10/ldb/types\";" \
                    "declare function local:if-absent( $arg as item()*, $value as item()* ) as item()* { if (exists($arg)) then $arg else $value } ;"

/*
 * Reads the response to the network request, and extracts the data we want.
 */
void ServiceWindow::readResponse(QNetworkReply* reply)
{
    QByteArray in = reply->readAll();

    if(errorFlag == true)
    {
        m_reply->deleteLater();
        return;
    }

    received.setData(in);
    received.open(QIODevice::ReadOnly);

    QXmlQuery query;
    query.bindVariable("data", &received);

    QStringList previousNames = queryList(query, "for $x in doc($data)//t:previousCallingPoints/t:callingPointList/t:callingPoint return fn:string(local:if-absent($x/t:locationName, ''))");
    QStringList previousST = queryList(query, "for $x in doc($data)//t:previousCallingPoints/t:callingPointList/t:callingPoint return fn:string(local:if-absent($x/t:st, ''))");
    QStringList previousAT = queryList(query, "for $x in doc($data)//t:previousCallingPoints/t:callingPointList/t:callingPoint return fn:string(local:if-absent($x/t:at, ''))");

    QStringList nextNames = queryList(query, "for $x in doc($data)//t:subsequentCallingPoints/t:callingPointList/t:callingPoint return fn:string(local:if-absent($x/t:locationName, ''))");
    QStringList nextST = queryList(query, "for $x in doc($data)//t:subsequentCallingPoints/t:callingPointList/t:callingPoint return fn:string(local:if-absent($x/t:st, ''))");
    QStringList nextET = queryList(query, "for $x in doc($data)//t:subsequentCallingPoints/t:callingPointList/t:callingPoint return fn:string(local:if-absent($x/t:at, ''))");

    for(int i = 0; i < previousNames.size(); i++)
    {
        DepartureWidget * w = new DepartureWidget(previousST.at(i), previousNames.at(i), previousAT.at(i), "", "");
        ui->scrollArea->widget()->layout()->addWidget(w);
        w->setShowPlatform(false);
    }

    QString name = queryOne(query, "for $x in doc($data)//t:GetServiceDetailsResult return fn:string(local:if-absent($x/t:locationName, ''))");
    QString platform = queryOne(query, "for $x in doc($data)//t:GetServiceDetailsResult return fn:string(local:if-absent($x/t:platform, '-'))");
    QString std = queryOne(query, "for $x in doc($data)//t:GetServiceDetailsResult return fn:string(local:if-absent($x/t:std, '-'))");
    QString etd = queryOne(query, "for $x in doc($data)//t:GetServiceDetailsResult return fn:string(local:if-absent($x/t:etd, '-'))");

    DepartureWidget * w = new DepartureWidget(std, name, etd, platform, "");
    ui->scrollArea->widget()->layout()->addWidget(w);

    for(int i = 0; i < nextNames.size(); i++)
    {
        DepartureWidget * w = new DepartureWidget(nextST.at(i), nextNames.at(i), nextET.at(i), "", "");
        ui->scrollArea->widget()->layout()->addWidget(w);
        w->setShowPlatform(false);
    }

    dialog->accept();
    m_reply->deleteLater();
    received.close();

    QMainWindow::show();
}

QString ServiceWindow::queryOne(QXmlQuery & q, QString query)
{
    QStringList output;
    output.clear();
    q.setQuery(query.prepend(NAMESPACE));
    q.evaluateTo(&output);
    return (output.size() == 0) ? QString() : output.at(0);
}

QStringList ServiceWindow::queryList(QXmlQuery & q, QString query)
{
    QStringList output;
    output.clear();
    q.setQuery(query.prepend(NAMESPACE));
    q.evaluateTo(&output);
    return output;
}

ServiceWindow::~ServiceWindow()
{
    delete ui;
    delete manager;
    delete dialog;
}

void ServiceWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}
