# -*- coding: utf-8 -*-

# Copyright (c) 2012 Hugo Osvaldo Barrera <hugo@osvaldobarrera.com.ar>
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

from PySide.QtCore import Signal, Qt, QSettings
from PySide.QtGui import QAction, QCheckBox, QDialog, QGridLayout, QIcon, QLabel, QLineEdit, QImage, QMenuBar, QMessageBox, QPixmap, QPushButton, QScrollArea, QVBoxLayout, QWidget

from PySide.QtMaemo5 import QMaemo5ValueButton, QMaemo5InformationBox

from foursquare import Cache
import foursquare
from venues import NewVenueWindow, VenueListWindow
from locationProviders import LocationProviderSelector, LocationProvider
from threads import VenueProviderThread, UserUpdaterThread, UpdateSelf, VenueSearchThread
from custom_widgets import SignalEmittingValueButton, CategorySelector, UberSquareWindow, Title
from users import UserListWindow
from about import AboutDialog
from datetime import datetime

from model.user import User


class ProfileWidget(QWidget):

    clicked = Signal()
    selfUpdated = Signal()

    def __init__(self, parent, session):
        super(ProfileWidget, self).__init__(parent)
        self._session = session
        self.manualUpdate = False
        self.user = self._session.get_user("self", foursquare.Cache.CacheOrGet)['user']
        self.photo_label = QLabel()

        self.textLabel = QLabel()
        self.textLabel.setWordWrap(True)
        self.__updateInfo(True)

        name = ""
        if 'firstName' in self.user:
            name += self.user['firstName'] + " "
        if 'lastName' in self.user:
            name += self.user['lastName']

        self.nameTitle = Title(name)

        profileLayout = QGridLayout()
        self.setLayout(profileLayout)
        profileLayout.addWidget(self.photo_label, 0, 0, 2, 1)
        profileLayout.addWidget(self.nameTitle, 0, 1)
        profileLayout.addWidget(self.textLabel, 1, 1)
        profileLayout.setColumnStretch(1, 5)
        profileLayout.setSpacing(5)
        profileLayout.setContentsMargins(11, 11, 11, 11)

        self.clicked.connect(self.__clicked)
        self.selfUpdated.connect(self.__updateInfo)

    def __clicked(self):
        t = UpdateSelf(self)
        t.start()
        QMaemo5InformationBox.information(self, "Updating stats...", 1500)
        self.manualUpdate = True

    def mousePressEvent(self, event):
        self.clicked.emit()

    def checkin(self):
        t = UpdateSelf(self)
        t.start()

    def __updateInfo(self, initial=False):
        if not initial:
            self.user = self._session.get_user("self", Cache.CacheOrGet)['user']
        if self.manualUpdate:
            QMaemo5InformationBox.information(self, "Stats updated!", 1500)
            self.manualUpdate = False

        badges = "<b>" + str(self.user['badges']['count']) + "</b> badges"
        mayorships = "<b>" + str(self.user['mayorships']['count']) + "</b> mayorships"
        checkins = "<b>" + str(self.user['checkins']['count']) + "</b> checkins"

        if 'items' in self.user['checkins']:
            location = self.user['checkins']['items'][0]['venue']['name']
            lastSeen = self.user['checkins']['items'][0]['createdAt']
            lastSeen = datetime.fromtimestamp(lastSeen).strftime("%Y-%m-%d %X")
            location = "Last seen @" + location  # + "</b>, at <i>" + lastSeen + "</i>"
        else:
            location = "Never checked in anywhere!"

        text = location + "<br>" + badges + " | " + mayorships + " | " + checkins
        self.textLabel.setText(text)

        self.photo = QImage(Cache.get_image(self.user['photo']))
        self.photo_label.setPixmap(QPixmap(self.photo))


class MainWindow(UberSquareWindow):

    showSearchResults = Signal()

    __venues = None

    _searchDialog = None

    def __init__(self, session):
        super(MainWindow, self).__init__(None)
        self._session = session

        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)
        self.setWindowTitle("UberSquare")

        self._session.imageDownloaded.connect(self.__image_downloaded)

        self.centralWidget = QWidget()
        self.setCentralWidget(self.centralWidget)

        #Main Layout
        layout = QVBoxLayout()
        layout.setSpacing(0)
        self.centralWidget.setLayout(layout)

        #Content Layout
        self.container = QWidget()

        self.scrollArea = QScrollArea()
        self.scrollArea.setWidget(self.container)

        layout.addWidget(self.scrollArea)

        self.scrollArea.setWidgetResizable(True)

        gridLayout = QGridLayout()
        self.container.setLayout(gridLayout)

        previous_venues_button = QPushButton("Visited")
        previous_venues_button.setIcon(QIcon.fromTheme("general_clock"))
        previous_venues_button.clicked.connect(self.__previous_venues_pushed)

        todo_venues_button = QPushButton("To-Do List")
        todo_venues_button.setIcon(QIcon.fromTheme("calendar_todo"))
        todo_venues_button.clicked.connect(self.__todo_venues_pushed)

        search_venues_button = QPushButton("Search/Explore")
        search_venues_button.clicked.connect(self.__search_venues_pushed)

        self.location_button = SignalEmittingValueButton("Location", self.__locationSelected, self)
        self.location_button.setPickSelector(LocationProviderSelector(session))
        self.location_button.setValueLayout(QMaemo5ValueButton.ValueUnderTextCentered)

        images_button = QPushButton("Update image cache")
        images_button.clicked.connect(self.__imageCache_pushed)

        logout_button = QPushButton("Forget credentials")
        logout_button.clicked.connect(self.__logout_pushed)

        new_venue_button = QPushButton("Create Venue")
        new_venue_button.clicked.connect(self.__new_venue_pushed)

        leaderboard_button = QPushButton("Leaderboard")
        leaderboard_button.clicked.connect(self.__leaderboard_button_pushed)

        settings_button = QPushButton("Sharing")
        settings_button.clicked.connect(self.__settings_button_pushed)

        self._profileWidget = ProfileWidget(self, session)
        self._session.checkinDone.connect(self._profileWidget.checkin)

        row = 0
        gridLayout.addWidget(self._profileWidget, row, 0, 1, 2)
        row += 1
        gridLayout.addWidget(previous_venues_button, row, 0)
        gridLayout.addWidget(todo_venues_button, row, 1)
        row += 1
        gridLayout.addWidget(new_venue_button, row, 0)
        gridLayout.addWidget(search_venues_button, row, 1)
        row += 1
        gridLayout.addWidget(leaderboard_button, row, 0)
        row += 1
        gridLayout.addWidget(QLabel("<b>Settings</b>"), row, 0, 1, 2, Qt.AlignHCenter)
        row += 1
        gridLayout.addWidget(self.location_button, row, 0)
        gridLayout.addWidget(settings_button, row, 1)
        row += 1
        gridLayout.addWidget(images_button, row, 0)
        gridLayout.addWidget(logout_button, row, 1)
        row += 1
        gridLayout.addWidget(QLabel("Developer edition 0.5"), row, 0, 1, 2, Qt.AlignHCenter)

        self.setupMenu()
        self.showSearchResults.connect(self.__showSearchResults)

    def __imageCache_pushed(self):
        c = QMessageBox(self)
        c.setWindowTitle("Update image cache?")
        c.setText("This will update all the category images in the cache. Make sure you have a good connection, and don't have to pay-by-megabyte")
        c.addButton("Yes", QMessageBox.YesRole)
        c.addButton("No", QMessageBox.NoRole)
        c.setIcon(QMessageBox.Question)
        c.exec_()
        if c.buttonRole(c.clickedButton()) == QMessageBox.YesRole:
            self._session.init_category_icon_cache()

    def __showSearchResults(self):
        self.progressDialog().close()

    def __showAbout(self):
        AboutDialog().exec_()

    def __leaderboard_button_pushed(self):
        users = self._session.users_leaderboard(Cache.CacheOrNull)
        w = UserListWindow("Leaderboard", users, self)
        t = UserUpdaterThread(w, self._session.users_leaderboard, self)
        t.start()
        if users:
            w.show()
        else:
            self.showWaitingDialog.emit()

    def __logout_pushed(self):
        self._session.config_del("code")
        self._session.config_del("access_token")
        msgBox = QMessageBox()
        msgBox.setText("I've gotten rid of your credentials. I'm going to close now, and if you run me again, it'll be like our first time all over again. Bye!")
        msgBox.setWindowTitle("Credentials forgotten")
        msgBox.exec_()
        self.close()

    def __previous_venues_pushed(self):
        venues = self._session.get_history(Cache.CacheOrNull)
        w = VenueListWindow("Visited Venues", venues, self)
        t = VenueProviderThread(w, self._session.get_history, self)
        t.start()
        if venues:
            w.show()
        else:
            self.showWaitingDialog.emit()

    def __todo_venues_pushed(self):
        try:
            venues = self._session.lists_todos(Cache.CacheOrNull)
            w = VenueListWindow("To-Do Venues", venues, self)
            t = VenueProviderThread(w, self._session.lists_todos, self)
            t.start()
            if venues:
                w.show()
            else:
                self.showWaitingDialog.emit()
        except IOError:
            self.networkError.emit()

    def __search_venues_pushed(self):
        if not self._searchDialog:
            self._searchDialog = SearchDialog(self, self._session)

        self._searchDialog.exec_()

        if (self._searchDialog.result() != QDialog.Accepted):
            return None

        venueName = self.searchDialog.text().encode('utf-8')
        categoryId = self.searchDialog.category()
        ll = LocationProvider().get_ll()

        try:
            venues = self._session.venues_search(venueName, ll, categoryId, foursquare.DEFAULT_FETCH_AMOUNT, Cache.CacheOrNull)
            v = VenueListWindow("Search results", venues, self)
            t = VenueSearchThread(v, self._session.venues_search, venueName, ll, categoryId, foursquare.DEFAULT_FETCH_AMOUNT, self)
            t.start()
            if venues:
                v.show()
            else:
                self.showWaitingDialog.emit()
        except IOError:
            self.networkError.emit()

    def __locationSelected(self, index):
        LocationProvider().select(index)
        QSettings().setValue("locationProvider", index)

    def __new_venue_pushed(self):
        try:
            w = NewVenueWindow(self, self._session.get_venues_categories(), LocationProvider().get_ll())
            w.show()
        except IOError:
            self.networkError.emit()

    def __settings_button_pushed(self):
        SettingsDialog(self).show()

    def __image_downloaded(self, name):
        print "showinfo called"
        QMaemo5InformationBox.information(self, name)

    def setupMenu(self):
        about = QAction(self)
        about.setText("About")
        about.triggered.connect(self.__showAbout)

        #settings = QAction(self)
        #settings.setText("Settings")

        menubar = QMenuBar(self)
        self.setMenuBar(menubar)

        #menubar.addAction(settings)
        menubar.addAction(about)

    def setVenues(self, venues):
        self.__venues = venues

    def venues(self):
        return self.__venues

    def setUsers(self, venues):
        self.__users = venues

    def users(self):
        return self.__users


class SettingsDialog(UberSquareWindow):
    def __init__(self, parent=None):
        super(SettingsDialog, self).__init__(parent)
        self.setWindowTitle("Settings")

        self.cw = QWidget(self)
        self.setCentralWidget(self.cw)

        layout = QGridLayout(self.cw)
        layout.setContentsMargins(22, 22, 22, 22)

        notice = "Yo'll need to link your facebook/twitter account from the foursquare website for these setting to have any effect."
        notice += "  The foursquare API does not offer any means for applications to do this."
        label = QLabel(notice)
        label.setWordWrap(True)

        self.tw = QCheckBox("Post to Twitter")
        self.fb = QCheckBox("Post to Facebook")

        broadcast = QSettings.value("broadcast")
        if broadcast:
            self.tw.setChecked("twitter" in broadcast)
            self.fb.setChecked("facebook" in broadcast)

        self.saveButton = QPushButton("Save")
        self.saveButton.connect(self.__save())

        layout.addWidget(label, 0, 0)
        layout.addWidget(self.tw, 1, 0)
        layout.addWidget(self.fb, 2, 0)
        layout.addWidget(self.saveButton, 3, 0)
        layout.setRowStretch(4, 3)

    def __save(self):
        broadcast = "public"
        if self.tw.isChecked():
            broadcast += ",twitter"
        if self.fb.isChecked():
            broadcast += ",facebook"

        QSettings.setValue("broadcast", broadcast)
        self.close()


class SearchDialog(QDialog):
    def __init__(self, parent, session):
        super(SearchDialog, self).__init__(parent)
        self.setWindowTitle("Search")
        self.centralWidget = QWidget()

        layout = QGridLayout()
        self.setLayout(layout)

        self.searchQuery = QLineEdit(self)
        self.searchQuery.setPlaceholderText("Search query")

        button = QPushButton("Search")
        button.clicked.connect(self.accept())

        self.categorySelector = CategorySelector(self, session)

        layout.addWidget(self.categorySelector, 0, 0, 1, 2)
        layout.addWidget(self.searchQuery, 1, 0)
        layout.addWidget(button, 1, 1)

    def category(self):
        return self.categorySelector.selectedCategory()

    def text(self):
        return self.searchQuery.text()
