#include "TxtReader.h"
#include <libosso.h>

osso_context_t *oss = NULL;

TxtReader::TxtReader(QString infile)
{
        version = QString("0.6.9");
        setWindowTitle( QString("TXTReader %1").arg(version));
        qApp->installTranslator(&appTranslator);
        oss = osso_initialize("txtreader", "0.6.9", FALSE, NULL);

        QSettings settings("lsyer", "txtreader");
	QPoint pos = settings.value("pos", QPoint(100, 60)).toPoint();
        QSize size = settings.value("size", QSize(840, 618)).toSize();
	curFile = settings.value("curFile", QString("")).toString();
        curOffset = -1;
        txtFont.fromString(settings.value("txtFont",QString("MHeiGB18030C-Medium,28,-1,5,50,0,0,0,0,0")).toString());
        txtColor=QColor(settings.value("txtColor", QColor("#000000")).toString());
	o_isfullscreen = settings.value("o_isfullscreen", 0).toBool();
	bgColor=QColor(settings.value("bgColor", QColor("#e0d686")).toString());
	bgImage = settings.value("bgImage", QString("")).toString();
	codecstr = settings.value("codecstr", QString("GBK")).toString();
        uilang = settings.value("uilang", QString("zh_CN")).toString();
        isLandscape = settings.value("isLandscape", 1).toBool();
        isNightmode = settings.value("isNightmode", 0).toBool();

        if(isLandscape)
            setOrientation(ScreenOrientationLockLandscape);
        else
            setOrientation(ScreenOrientationLockPortrait);
        screenblank = new QTimer(this);
        connect(screenblank,SIGNAL(timeout()),this,SLOT(screenLight()));
        createActions();
	createLanguageMenu();
        createTrayIcon();
        setStyleSheet( "TxtReader {background-color: "+bgColor.name()+";background-image: url("+bgImage+");background-position:center;;}");

        viewContentEdit = new ReaderView(this);
        viewContentEdit->setStyleSheet("color:"+txtColor.name()+";");
        viewContentEdit->setFont(txtFont);
        statusBar()->setStyleSheet("background:black;");

        showbutton=new QTimer(this);
        connect(showbutton,SIGNAL(timeout()),this,SLOT(touchFull()));
        fullButton=new QPushButton(QIcon(":/images/fullscreen.png"),"",this);
        fullButton->resize(48,48);
        fullButton->hide();
        connect(fullButton,SIGNAL(clicked()),this,SLOT(hideAndRestore()));

        setCentralWidget(viewContentEdit);
        appTranslator.load(":/i18n/txtreader_" + uilang);
        retranslateUi();
        //qDebug()<<"infile2:"<< infile <<"\n";
        if(infile.isEmpty()){
            loadFile(curFile);
        }else{
            loadFile(infile);
        }

        if(o_isfullscreen){
                showFullScreen();
        }else{
                resize(size);
        }
        if(isNightmode){
            isNightmode=false;
            setbgnight();
        }
}
void TxtReader::setOrientation(ScreenOrientation orientation)
{
#if defined(Q_OS_SYMBIAN)
    // If the version of Qt on the device is < 4.7.2, that attribute won't work
    if (orientation != ScreenOrientationAuto) {
        const QStringList v = QString::fromAscii(qVersion()).split(QLatin1Char('.'));
        if (v.count() == 3 && (v.at(0).toInt() << 16 | v.at(1).toInt() << 8 | v.at(2).toInt()) < 0x040702) {
            qWarning("Screen orientation locking only supported with Qt 4.7.2 and above");
            return;
        }
    }
#endif // Q_OS_SYMBIAN

    Qt::WidgetAttribute attribute;
    switch (orientation) {
#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    case ScreenOrientationLockPortrait:
        attribute = static_cast<Qt::WidgetAttribute>(128);
        break;
    case ScreenOrientationLockLandscape:
        attribute = static_cast<Qt::WidgetAttribute>(129);
        break;
    default:
    case ScreenOrientationAuto:
        attribute = static_cast<Qt::WidgetAttribute>(130);
        break;
#else // QT_VERSION < 0x040702
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_LockPortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_LockLandscapeOrientation;
        break;
    default:
    case ScreenOrientationAuto:
        attribute = Qt::WA_AutoOrientation;
        break;
#endif // QT_VERSION < 0x040702
    };
    setAttribute(attribute, true);
}

void TxtReader::resizeEvent(QResizeEvent *event)
{
    //qDebug()<<":resizeEvent\n";
    fullButton->move(this->width()-48,this->height()-48);
    reGenPageList();
    QWidget::resizeEvent(event);
}

void TxtReader::jumpToOffset(int offset)
{
    //qDebug()<<offset<<":offset\n";
    if(offset<0 || offset>FileContent.size()){
        jumpToPage(1);
        return;
    }

    int i;
    for(i=0;i<TotalPageNum;i++){
        if(PageOffsetList[i] > offset)
            break;
    }
    //qDebug()<<i<<":i\n";
    jumpToPage(i);
    return;
}

void TxtReader::jumpToPage(int pageNum)
{
        if(pageNum < 1)
                curPageNum = 1;
        else if(pageNum > TotalPageNum)
                curPageNum = TotalPageNum;
	else
                curPageNum = pageNum;

        viewContentEdit->setText(FileContent.mid(PageOffsetList[curPageNum-1],PageOffsetList[curPageNum]-PageOffsetList[curPageNum-1]));
        if(!o_isfullscreen) statusBar()->showMessage(QString(tr("Page %1 /Total %2")).arg(curPageNum).arg(TotalPageNum));

        return;
}
void TxtReader::jumpToPrePage(){
    jumpToPage(curPageNum-1);
    curOffset = PageOffsetList[curPageNum-1];
    return;
}
void TxtReader::jumpToNextPage(){
    jumpToPage(curPageNum+1);
    curOffset = PageOffsetList[curPageNum-1];
    return;
}

void TxtReader::genPageList(QString &content,QList<int> &list){
    //QApplication::setOverrideCursor(Qt::WaitCursor);
    //qDebug()<<"StartTime:"<<QTime().currentTime().toString()<<"\n";

    list.clear();
    QFontMetrics fm=QFontMetrics(txtFont);
    int PageLineNum = (viewContentEdit->height()-8) / qMax(fm.lineSpacing(),fm.height());
    int PageLineNow=0;
    int PageWidth=viewContentEdit->width()-8;
    int LineWidthNow=0;
    int AsciiWidth=fm.width("i"),ChineseWidth=qMax(fm.maxWidth(),fm.width("中"));
    int newPageOffset=0;
    list.append(0);
    int FileContentSize=content.size();
    for(int offset=0;offset<FileContentSize;offset++){
        int b = FileContent[offset].toAscii();

        if(b ==0 ) {// Chinese
            LineWidthNow += ChineseWidth;
            if(LineWidthNow > PageWidth){
                LineWidthNow = ChineseWidth;
                PageLineNow++;
                newPageOffset = offset;
            }
        }else if (b == 10){// \n
            LineWidthNow=0;
            PageLineNow++;
            newPageOffset = offset+1;
        }else if (b != 13){// not \r
            // Ascii
            LineWidthNow += AsciiWidth;
            if(LineWidthNow > PageWidth){
                LineWidthNow = AsciiWidth;
                PageLineNow++;
                newPageOffset = offset;
            }
        }

        if(PageLineNow==PageLineNum){
            PageLineNow=0;
            list.append(newPageOffset);
        }
    }
    if(list.last() != FileContentSize || FileContentSize == 0)
        list.append(FileContentSize);

    TotalPageNum=PageOffsetList.size() - 1;
    //qDebug()<<"CompleteTime:"<<QTime().currentTime().toString()<<"\n";
    //QApplication::restoreOverrideCursor();
    return;
}

void TxtReader::reGenPageList()
{
    genPageList(FileContent,PageOffsetList);
    jumpToOffset(curOffset);
}

int TxtReader::loadFile(const QString &fileName)
{
    if (fileName.isEmpty()) {
                return 0;
    }
    QFile file(fileName);
    if (!file.open(QFile::ReadOnly | QFile::Text)) {
        QMessageBox::warning(this,tr("Read File Failed"),
                             tr("Failed to read file %1.\nPerhaps it is not there, or you don't have access permission.").arg(fileName));
        QSettings setting("lsyer", tr("txtreader_%1").arg(strippedName(fileName)));
        QDir rmd;
        if(rmd.exists(setting.fileName()))
            rmd.remove(setting.fileName());
        QSettings settings("lsyer", "txtreader");
        QStringList files = settings.value("recentFileList").toStringList();
        files.removeAll(fileName);
        settings.setValue("recentFileList", files);

        updateRecentFileActions();
        return 0;
    }

    QTextStream in(&file);
    in.setCodec((const char *)codecstr.toLocal8Bit());
    FileContent=in.readAll();

    saveBookDependSetting();
    setCurrentFile(fileName);
    getBookDependSetting();

    return 1;
}

void TxtReader::backToRead()
{
    viewContentEdit->setVisible(true);
    if(!o_isfullscreen) statusBar()->showMessage(QString(tr("Page %1 /Total %2")).arg(curPageNum).arg(TotalPageNum));
}

void TxtReader::showChanges(){
    QFile file(QString(":/CHANGES.TXT"));
    if (!file.open(QFile::ReadOnly | QFile::Text)) {
            QMessageBox::warning(this, QString("TxtReader %1").arg(version),
                            tr("Cannot read file,please contact to author!"));
            return;
    }

    QTextStream in(&file);
    in.setCodec("UTF-8");

    QDialog changesDialog;
    changesDialog.setWindowTitle(tr("TxtReader Changes"));
    QGridLayout l;
    QTextEdit t;
    t.setReadOnly(true);
    t.setText(in.readAll());
    QPushButton b(tr("&Close"),this);
    connect(&b,SIGNAL(clicked()),&changesDialog,SLOT(close()));
    l.addWidget(&t);
    l.addWidget(&b,1,0,Qt::AlignRight);
    changesDialog.setLayout(&l);
    if(!isLandscape)
      changesDialog.setAttribute(static_cast<Qt::WidgetAttribute>(128),true);
    changesDialog.exec();
}

void TxtReader::myShowContextMenu(QPoint point)
{
    trayIconMenu->exec(point);
    if(!o_isfullscreen) statusBar()->showMessage(QString(tr("Page %1 /Total %2")).arg(curPageNum).arg(TotalPageNum));
}

void TxtReader::createActions()
{
    openAct = new QAction("open", this);
    connect(openAct, SIGNAL(triggered()), this, SLOT(open()));

    jumpAct = new QAction("jump", this);
    connect(jumpAct, SIGNAL(triggered()), this, SLOT(slotJumpToPage()));

    addBookmarkAct = new QAction("mark", this);
    connect(addBookmarkAct, SIGNAL(triggered()), this, SLOT(addBookmark()));

    delBookmarkAct = new QAction("del", this);
    connect(delBookmarkAct, SIGNAL(triggered()), this, SLOT(delBookmark()));

    clearBookmarkAct = new QAction("clear", this);
    connect(clearBookmarkAct, SIGNAL(triggered()), this, SLOT(clearBookmark()));

    addTxtSizeAct = new QAction("add", this);
    connect(addTxtSizeAct, SIGNAL(triggered()), this, SLOT(addTxtSize()));

    subTxtSizeAct = new QAction("txtsize", this);
    connect(subTxtSizeAct, SIGNAL(triggered()), this, SLOT(subTxtSize()));

    setfontAct = new QAction("font", this);
    connect(setfontAct, SIGNAL(triggered()), this, SLOT(setfont()));

    setfontcolorAct = new QAction("fcolor", this);
    connect(setfontcolorAct, SIGNAL(triggered()), this, SLOT(setfontcolor()));

    setbgnightAct = new QAction("bgnight", this);
    setbgnightAct->setCheckable(true);
    connect(setbgnightAct, SIGNAL(triggered()), this, SLOT(setbgnight()));

    setbgcolorAct = new QAction("bgcolor", this);
    connect(setbgcolorAct, SIGNAL(triggered()), this, SLOT(setbgcolor()));

    setbgimageAct = new QAction("bgimg", this);
    connect(setbgimageAct, SIGNAL(triggered()), this, SLOT(setbgimage()));
    
    delbgimageAct = new QAction("delimg", this);
    connect(delbgimageAct, SIGNAL(triggered()), this, SLOT(resetbg()));

    codecActionGroup = new QActionGroup(this);
    connect(codecActionGroup, SIGNAL(triggered(QAction *)),
            this, SLOT(setcodecstr(QAction *)));
    GB2312Act = new QAction(tr("&GB2312"), this);
    GB2312Act->setData("GBK");
    GB2312Act->setCheckable(true);
    codecActionGroup->addAction(GB2312Act);
    if (codecstr == "GBK") GB2312Act->setChecked(true);

    UTF8Act = new QAction(tr("&UTF-8"), this);
    UTF8Act->setData("UTF-8");
    UTF8Act->setCheckable(true);
    codecActionGroup->addAction(UTF8Act);
    if (codecstr == "UTF-8") UTF8Act->setChecked(true);

    BIG5Act = new QAction(tr("&BIG5"), this);
    BIG5Act->setData("BIG5");
    BIG5Act->setCheckable(true);
    codecActionGroup->addAction(BIG5Act);
    if (codecstr == "BIG5") BIG5Act->setChecked(true);

    searchOriAct = new QAction("sch", this);
    connect(searchOriAct, SIGNAL(triggered()), this, SLOT(slotSearchOri()));

    searchCurAct = new QAction("shc", this);
    connect(searchCurAct, SIGNAL(triggered()), this, SLOT(slotSearchCur()));

    for (int i = 0; i < MaxRecentFiles; ++i) {
        recentFileActs[i] = new QAction(this);
        recentFileActs[i]->setVisible(false);
        connect(recentFileActs[i], SIGNAL(triggered()),
                this, SLOT(openRecentFile()));
    }

    scnAction = new QAction("ScreenLight", this);
    scnAction->setCheckable(true);
    connect(scnAction, SIGNAL(triggered()), this, SLOT(setScreenLight()));
    
    quitAction = new QAction("quit", this);
    connect(quitAction, SIGNAL(triggered()), this, SLOT(quitaction()));

    aboutAct = new QAction("about", this);
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

    rotateAct = new QAction("Rotate", this);
    connect(rotateAct, SIGNAL(triggered()), this, SLOT(rotate()));

	return ;
}

void TxtReader::createLanguageMenu()
{
	languageMenu = new QMenu(this);

    languageActionGroup = new QActionGroup(this);
    connect(languageActionGroup, SIGNAL(triggered(QAction *)),
            this, SLOT(switchLanguage(QAction *)));

    QDir qmDir(":/i18n/");
    QStringList fileNames =
            qmDir.entryList(QStringList("txtreader_*.qm")); 

    for (int i = 0; i < fileNames.size(); ++i) {
    	QString locale = fileNames[i];
        locale.remove(0, locale.indexOf('_') + 1);
        locale.chop(3);
        
        QAction *action = new QAction(tr("&%1 %2")
                                      .arg(i + 1).arg(locale), this);
        action->setCheckable(true);
        action->setData(locale);

        languageMenu->addAction(action);
        languageActionGroup->addAction(action);

        if (locale == uilang)
            action->setChecked(true);
    }
}

void TxtReader::createTrayIcon()
{
    fileMenu = menuBar()->addMenu(tr("&File"));
    fileMenu->addAction(openAct);
    fileMenu->addAction(jumpAct);
    fileMenu->addAction(aboutAct);
    fileMenu->addAction(rotateAct);

    trayIconMenu = new QMenu(this);

    bookmarkMenu = trayIconMenu->addMenu(tr("bookmenu"));
    bookmarkMenu->addAction(addBookmarkAct);
    bookmarkMenu->addAction(delBookmarkAct);
    bookmarkMenu->addAction(clearBookmarkAct);
//    bookmarkMenu->addSeparator();

    setfontMenu = trayIconMenu->addMenu(tr("fontmenu"));
    setfontMenu->addAction(addTxtSizeAct);
    setfontMenu->addAction(subTxtSizeAct);
    setfontMenu->addAction(setfontAct);
    setfontMenu->addAction(setfontcolorAct);

    bgMenu = trayIconMenu->addMenu(tr("bgmenu"));
    bgMenu->addAction(scnAction);
    bgMenu->addAction(setbgnightAct);
    bgMenu->addAction(setbgcolorAct);
    bgMenu->addAction(setbgimageAct);
    bgMenu->addAction(delbgimageAct);

    codecMenu = trayIconMenu->addMenu(tr("codecmenu"));
    codecMenu->addAction(GB2312Act);
    codecMenu->addAction(UTF8Act);
    codecMenu->addAction(BIG5Act);

    searchMenu = trayIconMenu->addMenu(tr("searchmenu"));
    searchMenu->addAction(searchOriAct);
    searchMenu->addAction(searchCurAct);

    historyMenu = trayIconMenu->addMenu(tr("hismenu"));
    for (int i = 0; i < MaxRecentFiles; ++i)
            historyMenu->addAction(recentFileActs[i]);

    separatorAct = trayIconMenu->addSeparator();
    trayIconMenu->addMenu(languageMenu);
    trayIconMenu->addAction(aboutAct);
    updateRecentFileActions();

        return ;
}
void TxtReader::rotate()
{
    if (isLandscape){
        setOrientation(ScreenOrientationLockPortrait);
        isLandscape=false;
    }else{
        setOrientation(ScreenOrientationLockLandscape);
        isLandscape=true;
    }
}

void TxtReader::setbgnight()
{
    if(isNightmode){
        isNightmode = false;
        setbgnightAct->setChecked(false);
        setStyleSheet( "TxtReader {background-color: "+bgColor.name()+";background-image: url("+bgImage+");background-position:center;}");
        viewContentEdit->setStyleSheet("color:"+txtColor.name()+";");
    }else{
        isNightmode = true;
        setbgnightAct->setChecked(true);
        QColor tmpColor;
        tmpColor=QColor("#111f3b");
        setStyleSheet( "TxtReader {background-color: "+tmpColor.name()+";background-image: url("+bgImage+");background-position:center;}");
        tmpColor=QColor("#7d7d7d");
        viewContentEdit->setStyleSheet("color:"+tmpColor.name()+";");
    }
}
void TxtReader::setScreenLight()
{
    if(screenblank->isActive()){
        screenblank->stop();
        scnAction->setChecked(false);
    }else{
        screenblank->start(29000);// 29秒
        scnAction->setChecked(true);
    }
}
void TxtReader::screenLight()
{
    if(oss) osso_display_blanking_pause(oss);
}

void TxtReader::setCurrentFile(const QString &fileName)
{
    curFile = fileName;

    setWindowTitle(QString("%1 - TxtReader %2").arg(strippedName(curFile))
                                    .arg(version));

    QSettings settings("lsyer", "txtreader");
    QStringList files = settings.value("recentFileList").toStringList();
    files.removeAll(fileName);
    files.prepend(fileName);
    while (files.size() > MaxRecentFiles)
        files.removeLast();

    settings.setValue("recentFileList", files);

    updateRecentFileActions();
        return ;
}

void TxtReader::updateRecentFileActions()
{
        QSettings settings("lsyer", "txtreader");
	QStringList files = settings.value("recentFileList").toStringList();

	int numRecentFiles = qMin(files.size(), (int)MaxRecentFiles);

	for (int i = 0; i < numRecentFiles; ++i) {
		QString text = tr("&%1 %2").arg(i + 1).arg(strippedName(files[i]));
		recentFileActs[i]->setText(text);
		recentFileActs[i]->setData(files[i]);
		recentFileActs[i]->setVisible(true);
        }
	for (int j = numRecentFiles; j < MaxRecentFiles; ++j)
		recentFileActs[j]->setVisible(false);

        separatorAct->setVisible(numRecentFiles > 0);
        return ;
}

QString TxtReader::strippedName(const QString &fullFileName)
{
	return QFileInfo(fullFileName).fileName();
}

void TxtReader::open()
{
    QString fileName = QFileDialog::getOpenFileName(this,tr("Open new file"),tr(""),tr("text files (*.txt)"));
    if(loadFile(fileName)){
        reGenPageList();
            backToRead();
            }
}
void TxtReader::about()
{
    QDialog aboutDialog;
    aboutDialog.setWindowIcon(this->windowIcon());
    aboutDialog.setWindowTitle(tr("About"));
    QGridLayout l;
    QLabel t;
    t.setText(tr("<p align=center>Txt Reader %1 </p><p align=center> Design by <a href='http://code.google.com/p/txtreader4linux' target=_blank>lsyer</a></p><p align=center>CopyLeft(C)2006-YEAR</p><p align=center>Eveing 为 N900 移植</p>").arg(version));
    QPushButton c(tr("&Changes"));
    c.setMinimumSize(190,70);
    connect(&c,SIGNAL(clicked()),this,SLOT(showChanges()));
    QPushButton b(tr("&Back"),this);
    b.setMinimumSize(190,70);
    connect(&b,SIGNAL(clicked()),&aboutDialog,SLOT(close()));
    b.setFocus();
    l.addWidget(&t,0,0,2,1,Qt::AlignCenter);
    l.addWidget(&c,0,1,1,1,Qt::AlignRight);
    l.addWidget(&b,1,1,1,1,Qt::AlignRight);
    aboutDialog.setLayout(&l);
    aboutDialog.setFixedSize(aboutDialog.sizeHint());
    if(!isLandscape)
      aboutDialog.setAttribute(static_cast<Qt::WidgetAttribute>(128),true);
    aboutDialog.exec();

    return ;
}
void TxtReader::setcodecstr(QAction *action)
{
    codecstr = action->data().toString();
    if (codecstr == "UTF-8"){
            UTF8Act->setChecked(true);
            GB2312Act->setChecked(false);
            BIG5Act->setChecked(false);
    } else if (codecstr == "GBK"){
            UTF8Act->setChecked(false);
            GB2312Act->setChecked(true);
            BIG5Act->setChecked(false);
    } else {
            UTF8Act->setChecked(false);
            GB2312Act->setChecked(false);
            BIG5Act->setChecked(true);
    }
    if(loadFile(curFile))
        reGenPageList();
    return ;
}
void TxtReader::slotJumpToPage()
{
    QDialog d;
    d.setWindowTitle(tr("Jump to:"));
    QGridLayout l;
    QSlider t;//(Qt::Horizontal);
    t.setOrientation(Qt::Horizontal);
    t.setRange(1,TotalPageNum);
    t.setValue(curPageNum);
    //connect(&t,SIGNAL(valueChanged(value)),&d,SLOT(setWindowTitle(QString::number(value))));
    QPushButton b(tr("&Jump"),this);
    connect(&b,SIGNAL(clicked()),&d,SLOT(accept()));
    l.addWidget(&t);
    l.addWidget(&b,0,1,Qt::AlignRight);
    d.setLayout(&l);
    if(!isLandscape)
      d.setAttribute(static_cast<Qt::WidgetAttribute>(128),true);
    if(d.exec() == QDialog::Accepted){
      int i = t.value();
      if(i != curPageNum){
        jumpToPage(i);
        curOffset = PageOffsetList[curPageNum-1];
      }
    }
}
void TxtReader::slotSearchOri()
{
    QDialog d;
    d.setWindowTitle("Search:");
    QGridLayout l;
    QLineEdit t;
    QPushButton b(tr("&Search"),this);
    connect(&b,SIGNAL(clicked()),&d,SLOT(accept()));
    l.addWidget(&t);
    l.addWidget(&b,0,1,Qt::AlignRight);
    d.setLayout(&l);
    if(!isLandscape)
      d.setAttribute(static_cast<Qt::WidgetAttribute>(128),true);
    if(d.exec() == QDialog::Accepted){
      searchText = t.displayText();
      if ( !(searchText.isEmpty()) ){
        int pos=FileContent.indexOf(searchText,curOffset + 1);
        if(pos != -1){
            jumpToOffset(pos);
            curOffset = pos;
        }
      }
    }
}
void TxtReader::slotSearchCur()
{
    if ( !(searchText.isEmpty()) ){
        int pos=FileContent.indexOf(searchText,curOffset + 1);
        if(pos != -1){
            jumpToOffset(pos);
            curOffset = pos;
        }
    }

   return ;
}
void TxtReader::addBookmark()
{
    QString text;
    QDialog d;
    d.setWindowTitle(tr("Please input the mark:"));
    QGridLayout l;
    QLineEdit t;
    t.setText(tr("%1").arg(curPageNum));
    QPushButton b(tr("&Add Bookmark"),this);
    connect(&b,SIGNAL(clicked()),&d,SLOT(accept()));
    l.addWidget(&t);
    l.addWidget(&b,0,1,Qt::AlignRight);
    d.setLayout(&l);
    if(!isLandscape)
      d.setAttribute(static_cast<Qt::WidgetAttribute>(128),true);
    if(d.exec() == QDialog::Accepted){
        text = t.displayText();
	if ( !text.isEmpty()){
                QAction * bookmark = new QAction(tr("%1").arg(text), this);
                bookmark->setData(PageOffsetList[curPageNum-1]);
		connect(bookmark, SIGNAL(triggered()), this, SLOT(openBookmark()));
                bookmarkArray.append(bookmark);
		bookmarkMenu->addAction(bookmark);
	}
    }
}
void TxtReader::delBookmark()
{
    for(int i=0;i<bookmarkArray.size();i++){
        //qDebug()<<"i:"<<i<<" PageOffsetList[curPageNum-1]:"<<PageOffsetList[curPageNum-1]<<" PageOffsetList[curPageNum]:"<<PageOffsetList[curPageNum]<<"\n";
        if(bookmarkArray[i]->data().toInt()>=PageOffsetList[curPageNum-1] && bookmarkArray[i]->data().toInt()<=PageOffsetList[curPageNum]){
            bookmarkMenu->removeAction(bookmarkArray[i]);
            delete bookmarkArray[i];
            bookmarkArray.removeAt(i);
            break;
        }
    }
    return ;
}
void TxtReader::clearBookmark()
{
    for(int i=0;i<bookmarkArray.size();i++){
        bookmarkMenu->removeAction(bookmarkArray[i]);
    }
    if(bookmarkArray.size()>0){
        qDeleteAll(bookmarkArray);
        bookmarkArray.clear();
    }
    return ;
}
void TxtReader::openBookmark()
{
	QAction *action = qobject_cast<QAction *>(sender());
	if (action){
                jumpToOffset(action->data().toInt());
	}
	return ;
}

void TxtReader::getBookDependSetting(){
    QSettings settings("lsyer", tr("txtreader_%1").arg(strippedName(curFile)));
    curOffset = settings.value("curOffset", 0).toInt();
    int j = settings.value("bookmarknum", 0).toInt()+1;
    for(int k=1;k < j;++k){
            QAction * bookmark = new QAction(settings.value(QString("bookmark%1words").arg(k)/*, QString(tr("未知标签"))*/).toString(), this);
            bookmark->setData(settings.value(QString("bookmark%1offset").arg(k)/*, 1*/).toInt());
            connect(bookmark, SIGNAL(triggered()), this, SLOT(openBookmark()));
            bookmarkArray.append(bookmark);
            bookmarkMenu->addAction(bookmark);
    }
}

void TxtReader::saveBookDependSetting(){
    if(curOffset != -1){
        QSettings settings("lsyer", tr("txtreader_%1").arg(strippedName(curFile)));
        settings.setValue("curOffset", curOffset);
        int j=bookmarkArray.size();
        settings.setValue("bookmarknum",j);
        int k=0;
        for(int i=0;i<j;i++){
            ++k;
            settings.setValue(QString("bookmark%1offset").arg(k),bookmarkArray[i]->data().toString());
            settings.setValue(QString("bookmark%1words").arg(k),bookmarkArray[i]->text());
        }
        qDeleteAll(bookmarkArray);
        bookmarkArray.clear();
    }
    return;
}

void TxtReader::addTxtSize()
{
	if(txtFont.pointSize() < 38){
		txtFont.setPointSize(txtFont.pointSize()+1);
                this->setSizeIncrement(txtFont.pointSize(),txtFont.pointSize());
                reGenPageList();
                viewContentEdit->setFont(txtFont);
	}
	return;
}
void TxtReader::subTxtSize()
{
	if(txtFont.pointSize() > 5){
		txtFont.setPointSize(txtFont.pointSize()-1);
                this->setSizeIncrement(txtFont.pointSize(),txtFont.pointSize());
                reGenPageList();
                viewContentEdit->setFont(txtFont);
	}
	return;
}
void TxtReader::setfont()
{
	bool ok;
        QFont font = QFontDialog::getFont(&ok, txtFont, this,QString(tr("Set Font - Txt Reader %1")).arg(version));
	if (ok && font!=txtFont) {
		txtFont = font;
                this->setSizeIncrement(txtFont.pointSize(),txtFont.pointSize());
                reGenPageList();
                viewContentEdit->setFont(txtFont);
	}
	return;
}
void TxtReader::setfontcolor()
{
	QColor Color = QColorDialog::getColor(txtColor, this);
	if (Color.isValid()) {
                if(isNightmode) setbgnight();
                txtColor=Color;
                viewContentEdit->setStyleSheet("color:"+txtColor.name()+";");
	}
	return ;
}
void TxtReader::setbgcolor()
{
	QColor Color = QColorDialog::getColor(bgColor, this);
	if (Color.isValid()) {
                if(isNightmode) setbgnight();
		bgColor=Color;
                setStyleSheet( "TxtReader {background-color: "+bgColor.name()+";background-image: url("+bgImage+");background-position:center;;}");
	}
	return ;
}
void TxtReader::setbgimage()
{
    QString fileName = QFileDialog::getOpenFileName(this,tr("Select bg-image"),tr("/home/user/MyDocs"),tr("picture files (*.png *.jpg *.jpeg *.gif)"));
    if (!fileName.isEmpty()) {
    	bgImage=fileName;
	setStyleSheet( "TxtReader {background-color: "+bgColor.name()+";background-image: url("+bgImage+");background-position:center;;}");  	
    }
	return ;
}
void TxtReader::resetbg()
{
    bgImage = "";
    if(isNightmode) setbgnight();
    bgColor=QColor("#e0d686");
    setStyleSheet( "TxtReader {background-color: "+bgColor.name()+";background-image: url("+bgImage+");background-position:center;}");
    txtColor=QColor("#000000");
    viewContentEdit->setStyleSheet("color:"+txtColor.name()+";");
}
void TxtReader::hideAndRestore()
{
        if(o_isfullscreen){
            o_isfullscreen = false;
            statusBar()->show();
            showNormal();
  	}else{
            o_isfullscreen = true;
            statusBar()->hide();
            showFullScreen();
  	}
}
void TxtReader::switchLanguage(QAction *action)
{
    QString locale = action->data().toString();
	uilang = locale;
    appTranslator.load(":/i18n/txtreader_" + locale);
    retranslateUi();
}

void TxtReader::retranslateUi()
{
    openAct->setText(tr("&Open"));
    jumpAct->setText(tr("&Jump"));
    addBookmarkAct->setText(tr("&Add Bookmark"));
    delBookmarkAct->setText(tr("&Del Current"));
    rotateAct->setText(tr("&Rotate"));
    clearBookmarkAct->setText(tr("&Clear All"));
    searchOriAct->setText(tr("&Find"));
    searchCurAct->setText(tr("&Next"));
    addTxtSizeAct->setText(tr("&Zoom In"));
    subTxtSizeAct->setText(tr("&Zoom Out"));
    setfontAct->setText(tr("&Set Font"));
    setfontcolorAct->setText(tr("&Color"));
    scnAction->setText(tr("&Screen Light"));
    setbgnightAct->setText(tr("&Night Mode"));
    setbgcolorAct->setText(tr("&Bg color"));
    setbgimageAct->setText(tr("&Bg image"));
    delbgimageAct->setText(tr("&Reset"));
    quitAction->setText(tr("&Quit"));
    aboutAct->setText(tr("&About"));

    languageMenu->setTitle(tr("&Language"));
    historyMenu->setTitle(tr("&History"));
    bookmarkMenu->setTitle(tr("&Bookmark"));
    setfontMenu->setTitle(tr("&Font"));
    bgMenu->setTitle(tr("&Background"));
    codecMenu->setTitle(tr("&Codec"));
    searchMenu->setTitle(tr("&Search"));
}

void TxtReader::quitaction()
{
        QSettings settings("lsyer", "txtreader");
	settings.setValue("pos", pos());
	settings.setValue("size", size());
        settings.setValue("curFile", curFile);
	settings.setValue("o_isfullscreen", o_isfullscreen);
	settings.setValue("txtFont", txtFont);
	settings.setValue("txtColor", txtColor);
	settings.setValue("bgColor", bgColor);
	settings.setValue("bgImage", bgImage);
	settings.setValue("codecstr", codecstr);
        settings.setValue("uilang", uilang);
        settings.setValue("isLandscape", isLandscape);
        settings.setValue("isNightmode", isNightmode);

        if(oss) osso_deinitialize(oss);
        saveBookDependSetting();
	qApp->quit();
	return ;
}

void TxtReader::openRecentFile()
{
	QAction *action = qobject_cast<QAction *>(sender());
	if (action){
            if(action->data().toString()!=curFile){
                    if(loadFile(action->data().toString())){
                        reGenPageList();
			backToRead();
			}
            }
	}
	return ;
}
void TxtReader::touchFull()
{
    if(showbutton->isActive()) showbutton->stop();
    fullButton->hide();
}

void TxtReader::keyPressEvent(QKeyEvent * event)
{
        if (event->key() == Qt::Key_Right || event->key() == Qt::Key_Down){
            jumpToNextPage();
        } else if (event->key() == Qt::Key_Left || event->key() == Qt::Key_Up){
            jumpToPrePage();
        } else if (event->key() == Qt::Key_F) {
            hideAndRestore();
        } else if (event->key() == Qt::Key_Q) {
            quitaction();
        } else {
            QWidget::keyPressEvent(event);
	return ;
	}
        event->accept();
}
void TxtReader::mousePressEvent(QMouseEvent *event)
{
    if (event->button() == Qt::LeftButton){
        int h = height();
        if(event->globalX() > (width()-64) && event->globalY() > h-64){
            fullButton->show();
            showbutton->start(3000);
        }else if(h*0.7 < event->globalY()){
            jumpToNextPage();
        }else if(h*0.4 > event->globalY()){
            jumpToPrePage();
	}
    }else{
        QWidget::mousePressEvent(event);
    }
    return ;
}
void TxtReader::contextMenuEvent(QContextMenuEvent *event)
{
    myShowContextMenu(event->globalPos());
    return ;
}

void TxtReader::closeEvent ( QCloseEvent * event )
{
        quitaction();
        event->accept();
}


