/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "timelineview.h"

/*!
 * \brief TimelineView constructor.
 */
TimelineView::TimelineView(void) :
        timelineOnScreen(NULL),
        leftPos(-801,0),
        centerPos(0,0),
        rightPos(801,0)
{
    _exitAnim = 0;
}

/*!
 * \brief Creates the TimelineView objects and animations.
 */
void TimelineView::createUi(void)
{
    _createEntry();
    _createButtons();
    _createSettings();
    _createTopBar();
    _createTimeline();

    startLoader();

    /* Animations */
    _entryAnim = _createEntryAnimation();
    _entryAnim->start();
    _exitAnim = _createExitAnimation();

    connect(qobject_cast<QObject *>(_exitAnim), SIGNAL(finished()), SIGNAL(done()));

//    connect(_locationCheckBox, SIGNAL(pressed()), this, SIGNAL(showLocationDialog()));
//    connect(_ipLocation, SIGNAL(finished(double, double)), this, SLOT(_showIpLocationInfo(double, double)));
}

/*!
 * \brief This method creates the entry animation for all objects on the screen.
 * \return anim Entry animation.
 */
QAbstractAnimation *TimelineView::_createEntryAnimation(void)
{
    QParallelAnimationGroup *anim = new QParallelAnimationGroup(this);

    QEasingCurve easingCurve;
    easingCurve.setType(QEasingCurve::OutBounce);

    /* Buttons */
    _addOpacityAnim(anim, _updateButton, 0);
    _addPosAnim(anim, _updateButton, QPointF(800 - _updateButton->boundingRect().width(), 480 - (_updateButton->boundingRect().height()) - 50),
                QPointF(800 - _updateButton->boundingRect().width(), 480 - (_updateButton->boundingRect().height())),
                easingCurve, 0);
    _addOpacityAnim(anim, _exposeButton, 0);
    _addPosAnim(anim, _exposeButton, QPointF(0, 50), QPointF(0, 0), easingCurve, 0);

    /* TopBar */
    _addOpacityAnim(anim, _topBar, 0);
    _addPosAnim(anim, _topBar, QPointF(0, 50), QPointF(0, 0), easingCurve, 0);

     /* Update Entry */
    _addOpacityAnim(anim, _entryUpdate, 0);
    _addPosAnim(anim, _entryUpdate, QPointF(0, 480 - _entryUpdate->boundingRect().height() - 50),
                QPointF(0, 480 - _entryUpdate->boundingRect().height()),easingCurve, 0);

    return anim;
}

/*!
 * \brief This method creates the exit animation for all objects on the screen.
 * \return anim Exit animation.
 * This method gets the entry animation and sets it backwards as the exit animation.
 */
QAbstractAnimation* TimelineView::_createExitAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = qobject_cast<QParallelAnimationGroup *>(_createEntryAnimation());

    anim->setDirection(QAbstractAnimation::Backward);
    return anim;
}

/*!
 * \brief Creates the timeline.
 */
void TimelineView::_createTimeline()
{
    _homeTimeline = new KineticList(KineticList::Vertical, QSizeF(800,360),this);
    _homeTimeline->setPos(0,60);
    _homeTimeline->setZValue(1.0);

    _mentionsTimeline = new KineticList(KineticList::Vertical, QSizeF(800,360), this);
    _mentionsTimeline->setPos(801,60);
    _mentionsTimeline->setZValue(1.0);

    timelineOnScreen = _homeTimeline;

    _timelines << _homeTimeline << _mentionsTimeline;
    _downloadedTimelines << true << false;

    connect(_homeTimeline, SIGNAL(createTweetInfo(Tweet*)), SLOT(_createTweetInfo(Tweet*)));
    connect(_homeTimeline, SIGNAL(moreTweets()),SLOT(_emitMoreTweets()));
    connect(_mentionsTimeline, SIGNAL(moreTweets()), SLOT(_emitMoreTweets()));
    connect(_mentionsTimeline, SIGNAL(createTweetInfo(Tweet*)), SLOT(_createTweetInfo(Tweet*)));
}

/*!
 * \brief Creates the top bar containing user information.
 */
void TimelineView::_createTopBar()
{
    _topBar = new TwitterTopBar(this);
    _topBar->setZValue(1.0);

    connect(_topBar, SIGNAL(settings()), _settingsDialog, SLOT(startAnimation()));
    connect(_topBar, SIGNAL(refresh()), SLOT(_emitRefresh()));
    connect(_topBar, SIGNAL(toHome()), SLOT(_toHome()));
    connect(_topBar, SIGNAL(toMentions()), SLOT(_toMentions()));
}

void TimelineView::_createSettings()
{
    _settingsDialog = new TimelineSettings(this);
    _settingsDialog->setPos(0,560);
    _settingsDialog->setZValue(4.0);

    connect(_settingsDialog, SIGNAL(logout()), SLOT(_backToLogin()));
}

/*!
 * \brief Creates the buttons on the screen.
 */
void TimelineView::_createButtons()
{
    _updateButton = new TwitterButton(TwitterButton::GenericDownRight,"Tweet",this);
    _updateButton->setOpacity(0);
    _updateButton->setZValue(3.0);
    _updateButton->enable(false);

    _exposeButton = new TwitterButton(TwitterButton::Expose,"",this);
    _exposeButton->setOpacity(0);
    _exposeButton->setZValue(6.0);

    connect(_updateButton, SIGNAL(released()), this, SLOT(_emitPostUpdate()));
}

/*!
 * \brief Creates the entry to write a new status message.
 */
void TimelineView::_createEntry()
{
    _entryUpdate = new LineEdit(LineEdit::Long,this);
    _entryUpdate->setOpacity(0);
    _entryUpdate->setPos(QPointF(100,435));
    _entryUpdate->setZValue(3.0);

    connect(_entryUpdate, SIGNAL(enterPressed()), SLOT(_emitPostUpdate()));
    connect(_entryUpdate, SIGNAL(textChanged(QString)), SLOT(_enableTweetButton(QString)));
}

/*!
 * \brief Creates tweetinfo object.
 */
void TimelineView::_createTweetInfo(Tweet *tweet)
{
    _tweetInfo = new TwitterTweetInfo(tweet, true, this);
    _tweetInfo->setZValue(2.0);
    _tweetInfo->startAnimation();

    connect(_tweetInfo, SIGNAL(retweet(QString)), this, SLOT(_handleNewMessage(QString)));
    connect(_tweetInfo, SIGNAL(reply(QString,quint64)), this, SLOT(_handleReply(QString, quint64)));
    connect(_tweetInfo, SIGNAL(del(quint64, Tweet*)), this, SLOT(_emitDelete(quint64, Tweet*)));
//    connect(_tweetInfo, SIGNAL(favorite(QString, bool)), this, SLOT(_emitFavorite(QString, bool)));
//    connect(_tweetInfo, SIGNAL(unfollow(QString, bool)), this, SLOT(_emitUnfollow(QString, bool)));
//    connect(_tweetInfo, SIGNAL(imageUrl(TwitPic::Services)), this, SLOT(_openTwitPic(TwitPic::Services)));
//    connect(this, SIGNAL(loadMap()), _tweetInfo, SLOT(_loadMap()));
}

void TimelineView::_enableTweetButton(QString msg)
{
    if(!msg.length() || msg.length() > 140)
        _updateButton->enable(false);
    else
        _updateButton->enable(true);
}

/*!
 * \brief Adds opacity animation.
 * \param item the item that will be animated.
 * \param anim the item's animation.
 * \param delay to start each item's animation.
 */
void TimelineView::_addOpacityAnim(QParallelAnimationGroup *anim, TwitterApplicationItem *item, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("opacity");
    animation->setStartValue(0.0);
    animation->setEndValue(1.0);
    animation->setDuration(500);
    animation->setEasingCurve(QEasingCurve::Linear);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Adds the position animation.
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 * \param startValue the start position of the animation
 * \param endValue the end position of the animation
 * \param easing the easing curve for the movimentation animation
 */
void TimelineView::_addPosAnim(QParallelAnimationGroup *anim,
                            TwitterApplicationItem *item,
                            QPointF startValue, QPointF endValue,
                            QEasingCurve easing, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("pos");
    animation->setStartValue(startValue);
    animation->setEndValue(endValue);
    animation->setDuration(500);
    animation->setEasingCurve(easing);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

void TimelineView::_toHome(void)
{
    if(timelineOnScreen == _homeTimeline)
        return;

    KineticList *oldTimeline = timelineOnScreen;
    timelineOnScreen = _homeTimeline;

    _moveTimelines(oldTimeline,timelineOnScreen);
}

void TimelineView::_toMentions(void)
{
    if(timelineOnScreen == _mentionsTimeline)
        return;

    int pos = _timelines.indexOf(_mentionsTimeline);
    if(!_downloadedTimelines.at(pos))
    {
        startLoader();
        _downloadedTimelines.insert(pos,true);
        emit getTimeline(Timeline::Mentions);
    }

    KineticList *oldTimeline = timelineOnScreen;
    timelineOnScreen = _mentionsTimeline;

    _moveTimelines(oldTimeline,timelineOnScreen);
}

void TimelineView::_moveTimelines(KineticList *oldTimeline, KineticList *newTimeline)
{
    int indexOld = _timelines.indexOf(oldTimeline);
    int indexNew = _timelines.indexOf(newTimeline);

    if(indexOld > indexNew) {
        _moveToRight(oldTimeline,newTimeline);
    } else {
        _moveToLeft(oldTimeline,newTimeline);
    }
}

void TimelineView::_moveToLeft(KineticList *oldTimeline, KineticList *newTimeline)
{
    QPropertyAnimation *anim1 = new QPropertyAnimation(oldTimeline,"pos");
    anim1->setStartValue(oldTimeline->pos());
    anim1->setEndValue(oldTimeline->pos() + leftPos);
    anim1->setDuration(250);
    anim1->setEasingCurve(QEasingCurve::Linear);

    QPropertyAnimation *anim2 = new QPropertyAnimation(newTimeline,"pos");
    anim2->setStartValue(newTimeline->pos());
    anim2->setEndValue(newTimeline->pos() + leftPos);
    anim2->setDuration(250);
    anim2->setEasingCurve(QEasingCurve::Linear);

    QParallelAnimationGroup *toleft = new QParallelAnimationGroup();
    toleft->addAnimation(anim1);
    toleft->addAnimation(anim2);
    toleft->start();
}

void TimelineView::_moveToRight(KineticList *oldTimeline, KineticList *newTimeline)
{
    QPropertyAnimation *anim1 = new QPropertyAnimation(oldTimeline,"pos");
    anim1->setStartValue(oldTimeline->pos());
    anim1->setEndValue(oldTimeline->pos() + rightPos);
    anim1->setDuration(250);
    anim1->setEasingCurve(QEasingCurve::Linear);

    QPropertyAnimation *anim2 = new QPropertyAnimation(newTimeline,"pos");
    anim2->setStartValue(newTimeline->pos());
    anim2->setEndValue(newTimeline->pos() + rightPos);
    anim2->setDuration(250);
    anim2->setEasingCurve(QEasingCurve::Linear);

    QParallelAnimationGroup *toRight = new QParallelAnimationGroup();
    toRight->addAnimation(anim1);
    toRight->addAnimation(anim2);
    toRight->start();
}

/*! \brief Gets the message typed on message edit box.
 *  \return message Message typed.
 */
QString TimelineView::getUpdateMessage(void)
{
   return _entryUpdate->getText();
}

/*! \brief Sets the message in the message edit box.
 *  \param[in] message The user's message.
 */
void TimelineView::setUpdateMessage(const QString &message)
{
    _entryUpdate->setMessage(message);
}

void TimelineView::clearText(void)
{
    _updateButton->enable(true);
    _entryUpdate->clearText();
}

/*!
 * \brief This slot is called when the user pushes the "update" button.
 * This method starts the loader then sends a signal to the controller to try to login.
 */
void TimelineView::_emitPostUpdate()
{
    if(_entryUpdate->getText().length() < 141) {
        startLoader();
        _updateButton->enable(false);
        emit postUpdate();
    }
}

/*!
 * \brief Sets the new message on the textbox after the url was shortened.
 * \param newMessage message
 */
void TimelineView::_handleNewMessage(QString newMessage)
{
    this->setUpdateMessage(newMessage);
//    this->_updateButton->enable(true);
}

/*! \brief Calls the loader when user clicks on update.
 */
void TimelineView::startLoader()
{
    _loader = new Loader(this,QPointF(400,426));
    _loader->setZValue(5.0);
    _loader->startEntryAnimation();
    connect(_loader, SIGNAL(finished()), this, SLOT(_deleteLoader()));
}

/*! \brief Deletes loader object when loader finishes.
 */
void TimelineView::_deleteLoader()
{
    delete _loader;
    _loader = NULL;
}

/*! \brief This slot is called to start loader's exit animation.
 */
void TimelineView::endLoader()
{
    if(_loader)
        _loader->startExitAnimation();
}

/*!
 * \brief Updates the topbar with new information about the user.
 */
void TimelineView::updateTopBar(void)
{
    _topBar->updateTopBar();
}

/*!
 * \brief Starts exit animation to go back to login screen.
 */
void TimelineView::_backToLogin()
{
    if (_exitAnim) {
        _exitAnim->start();
    } else {
        emit done();
    }
}

void TimelineView::_emitMoreTweets(void)
{
    startLoader();

    if(timelineOnScreen == _homeTimeline)
        emit moreHomeTweets();
    else if(timelineOnScreen == _mentionsTimeline)
        emit moreMentions();
}

void TimelineView::_emitRefresh(void)
{
    startLoader();

    if(timelineOnScreen == _homeTimeline)
        emit homeRefresh();
    else if(timelineOnScreen == _mentionsTimeline)
        emit mentionsRefresh();
}

/*!
 * \brief Emits an Unfollow signal.
 */
void TimelineView::_emitUnfollow(QString username, bool _unfollow)
{
//    startLoader();
//    emit unfollow(username, _unfollow);
}

/*!
 * \brief Emits a delete signal.
 */
void TimelineView::_emitDelete(quint64 id, Tweet *tweet)
{
    startLoader();
    m_deleteTweet = tweet;
    emit deleteTweet(id);
}

/*!
 * \brief Emits a favorite signal.
 * \param isFavorite true if status is favorite and false otherwise.
 */
void TimelineView::_emitFavorite(QString tweetId, bool isFavorite)
{
//    startLoader();
//    emit favorite(tweetId, isFavorite);
}

/*!
 * \brief Gets geographical goordinates.
 */
void TimelineView::getIPCoordinates()
{
//    qDebug() << "+ " << __PRETTY_FUNCTION__;

//    #ifdef LIBLOCATION
//        QMaemo5InformationBox::information(NULL, tr("Acquiring geographical information through your IP"), QMaemo5InformationBox::DefaultTimeout);
//    #endif // LIBLOCATION

//    _ipLocation->getLocation();
}

/*!
 * \brief Shows geographical goordinates.
 */
void TimelineView::_showIpLocationInfo(double latitude, double longitude)
{
//    qDebug() << "+ " << __PRETTY_FUNCTION__;

//    #ifdef LIBLOCATION
//        QMaemo5InformationBox::information(NULL, tr("GOT IT! Latitude = %1, Longitude = %2").arg(latitude).arg(longitude),
//                                           QMaemo5InformationBox::DefaultTimeout);
//    #endif // LIBLOCATION
}

/*!
 * \brief Emits a signal to change user's avatar.
 * \param image New image to upload.
 */
void TimelineView::_emitChangeAvatar(QString image)
{
//    startLoader();
//    emit changeAvatar(image);
}

/*!
 * \brief Starts GPS with ANGSS method
 */
void TimelineView::getGPSCoordinates()
{
//    qDebug() << "+ " << __PRETTY_FUNCTION__;

//    #ifdef LIBLOCATION
//        _gpsLocation->get_agnss();
//        QMaemo5InformationBox::information(NULL, tr("Acquiring GPS information"), QMaemo5InformationBox::DefaultTimeout);
//        connect(_gpsLocation, SIGNAL(getOK_agnss()), this, SIGNAL(gotGPS()));
//    #endif // LIBLOCATION
}

/*!
 * \brief Unchecks location checkbox
 */
void TimelineView::setLocationCheckboxFalse()
{
//    _locationCheckBox->setChecked(false);
}

/*!
 * \brief Sets the new message on the textbox after the user presses reply button.
 * \param newMessage message
 */
void TimelineView::_handleReply(QString newMessage, quint64 statusId)
{
    this->setUpdateMessage(newMessage);
//    this->_updateButton->enable(true);

//    isReply = true;
    m_statusId = statusId;
}

void TimelineView::_openTwitPic(TwitPic::Services service)
{
//    m_twitPic = new TwitPic(this);
//    connect(m_twitPic, SIGNAL(animExit()), this, SLOT(_closeTwitPic()));
//    m_twitPic->setZValue(2.5);

//    if(service == TwitPic::twitPic)
//        m_twitPic->downloadTwitPic(_tweetInfo->getTwitPicUrl() + "/full");
//    else
//        if(service == TwitPic::tweetPhoto)
//            m_twitPic->downloadTweetPhoto(_tweetInfo->getTweetPhotoUrl());

//    QPropertyAnimation *animate = new QPropertyAnimation(m_twitPic, "scale");
//    m_twitPic->setTransformOriginPoint(m_twitPic->boundingRect().center());
//    animate->setStartValue(0);
//    animate->setEndValue(1);
//    animate->setDuration(700);
//    animate->setEasingCurve(QEasingCurve::OutExpo);
//    animate->start(QPropertyAnimation::DeleteWhenStopped);
}

void TimelineView::_closeTwitPic()
{
//    QPropertyAnimation *animate = new QPropertyAnimation(m_twitPic, "scale");
//    m_twitPic->setTransformOriginPoint(m_twitPic->boundingRect().center());
//    animate->setStartValue(1);
//    animate->setEndValue(0);
//    animate->setDuration(700);
//    animate->setEasingCurve(QEasingCurve::OutExpo);
//    animate->start(QPropertyAnimation::DeleteWhenStopped);

//    connect(animate, SIGNAL(finished()), this, SLOT(_deleteTwitPic()));
}

void TimelineView::_deleteTwitPic()
{
//    delete m_twitPic;
}

bool TimelineView::getHome(void)
{
//    return this->m_home;
}

void TimelineView::addMoreButton(Timeline::TimelineType type)
{
    switch(type)
    {
        case Timeline::Friends:
            _homeTimeline->addMoreButton();
            break;
        case Timeline::Mentions:
            _mentionsTimeline->addMoreButton();
            break;
    }
}

void TimelineView::tweetDeleted(quint64 id)
{
    if(m_deleteTweet->getEntry().id == id)
        _homeTimeline->removeFromList(m_deleteTweet);
    else {
        // TODO: error
    }
    _deleteLoader();
}
